
/******************************************************************************\
*
*  PROGRAM:     WALLYAPP.C
*
*  PURPOSE:     A simple demonstration of LoadLibrary()-ing the Wally95 DLL at
*               runtime, linking and calling all available functions within it.
*
\******************************************************************************/

#include <windows.h>
#include <commdlg.h>
#include "stdio.h"
#include "resource.h"
#include "wally95.h"
#include "wallyapp.h"

HANDLE hLib = NULL;

/******************************************************************************\
*
*  FUNCTION:    WinMain (standard WinMain INPUTS/RETURNS)
*
*  GLOBAL VARS: ghwndMain - handle of main app window
*
*  LOCAL VARS:  msg - msg to get/dispatch
*
\******************************************************************************/

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;

  if (!hPrevInstance)
  {
    WNDCLASS wc;

    wc.style         = 0;
    wc.lpfnWndProc   = (WNDPROC)MainWndProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = hInstance;
    wc.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor       = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject (GRAY_BRUSH);
    wc.lpszMenuName  = (LPCTSTR) "Menu";
    wc.lpszClassName = (LPCTSTR) "WallyAPP";

    if (!RegisterClass (&wc))
    {
      MessageBox (NULL, (LPCTSTR) "WinMain(): RegisterClass() failed",
                  (LPCTSTR) "Err! - WallyAPP", MB_OK | MB_ICONEXCLAMATION);
      return(FALSE);
    }
  }

  if (!(ghwndMain = CreateWindow ("WallyAPP", "Weird Wally Sample Application",
                                  WS_OVERLAPPEDWINDOW,
                                  CW_USEDEFAULT, CW_USEDEFAULT,
                                  CW_USEDEFAULT, CW_USEDEFAULT,
                                  NULL, NULL, hInstance, NULL)))
    return (0);

  ShowWindow (ghwndMain, nCmdShow);

  while (GetMessage (&msg, NULL, 0, 0))
  {
    TranslateMessage (&msg);
    DispatchMessage  (&msg);
  }

  return (msg.wParam);
}



/******************************************************************************\
*
*  FUNCTION:   LoadWallyLibrary - loads the animation dll
*
\******************************************************************************/

void LoadWallyLibrary(HWND hwnd)
   {
   if(!hLib)
     {
     if(!(hLib = LoadLibrary("WALLY95.DLL")))
        {
        MessageBox(hwnd, (LPCTSTR) "Wally95.dll - LoadLibrary function failed!",
                  (LPCTSTR) "Load Error in WallyAPP", MB_OK | MB_ICONEXCLAMATION);
        FixMenu(hwnd, FALSE);
        }
     else
        {
        gpfnDLLwallysize     = (PFNDLL) GetProcAddress(hLib, "Wally_Size");
        gpfnDLLwallypaint    = (PFNDLL) GetProcAddress(hLib, "Wally_Paint");
        gpfnDLLwallytimer    = (PFNDLL) GetProcAddress(hLib, "Wally_Timer");
        gpfnDLLwallyendframe = (PFNDLL) GetProcAddress(hLib, "Wally_EndFrame");
        gpfnDLLtoonframe     = (PFNDLL) GetProcAddress(hLib, "WallyPlayToonFrame");
        gpfnDLLtoonstep      = (PFNDLL) GetProcAddress(hLib, "WallyPlayToonStep");
        gpfnDLLtoonfile      = (PFNDLL) GetProcAddress(hLib, "WallyPlayToonFile");
        gpfnDLLtoonresource  = (PFNDLL) GetProcAddress(hLib, "WallyPlayToonResource");
        gpfnDLLisplaying     = (PFNDLL) GetProcAddress(hLib, "IsWallyPlaying");
        gpfnDLLpause         = (PFNDLL) GetProcAddress(hLib, "PauseWallyPlayback");
        gpfnDLLcontinue      = (PFNDLL) GetProcAddress(hLib, "ContinueWallyPlayback");
        FixMenu(hwnd, TRUE);
        }
     }
   }

/******************************************************************************\
*
*  FUNCTION:   FreeWallyLibrary - frees the animation dll
*
\******************************************************************************/

void FreeWallyLibrary(HWND hwnd)
   {
   if(hLib)
      {
      FreeLibrary(hLib);
      gpfnDLLwallysize     = NULL;
      gpfnDLLwallypaint    = NULL;
      gpfnDLLwallytimer    = NULL;
      gpfnDLLwallyendframe = NULL;
      gpfnDLLtoonframe     = NULL;
      gpfnDLLtoonstep      = NULL;
      gpfnDLLtoonfile      = NULL;
      gpfnDLLtoonresource  = NULL;
      gpfnDLLisplaying     = NULL;
      gpfnDLLpause         = NULL;
      gpfnDLLcontinue      = NULL;
      FixMenu(hwnd, FALSE);
      hLib = NULL;
      }
   }


/******************************************************************************\
*
*  FUNCTION:   InterruptPlayback - pauses or continues the playback. 
*                                  changes IDM_INTERRUPT menu item's text to suit.
*
*  DLL FUNCTIONS USED: IsWallyPlaying
*                      PauseWallyPlayback
*                      ContinueWallyPlayback
*
\******************************************************************************/

void InterruptPlayback(HWND hWnd)
    { 
    HMENU hMenu = GetMenu(hWnd);
    
    // check and exit if any functions not linked
    if(!gpfnDLLisplaying) return;
    if(!gpfnDLLpause)     return;
    if(!gpfnDLLcontinue)  return;

    // now interrupt or continue playback
    if((gpfnDLLisplaying)())
       {
       (gpfnDLLpause)(hWnd);
       ModifyMenu(hMenu, 
                  IDM_INTERRUPT, MF_STRING|MF_BYCOMMAND, 
                  IDM_INTERRUPT, "&Continue Playback");
       }
    else
       {
       (gpfnDLLcontinue)(hWnd);
       ModifyMenu(hMenu, 
                  IDM_INTERRUPT, MF_STRING|MF_BYCOMMAND, 
                  IDM_INTERRUPT, "&Pause Playback");
       }
    DrawMenuBar(hWnd);
    }


/******************************************************************************\
*
*  FUNCTION:   StartFramePlayback
*                 - walks onscreen to center of screen, winks, speaks his mind, 
*                   waits for five seconds and then walks offscreen again.
*
*  DLL FUNCTION USED: WallyPlayToonFrame
*
\******************************************************************************/

BOOL StartFramePlayback(HWND hwnd)
   {
   POINT ptEnd;
   RECT  rc;
   // find the center of the window
   GetClientRect(hwnd, &rc);
   ptEnd.x = ((rc.right-rc.left)/2);
   ptEnd.y = ((rc.bottom-rc.top)/2);

   return (gpfnDLLtoonframe)(hwnd, ptEnd, 7, TRUE, 
          "Weird Wally's animations provided by Slick Software!");
   }


/******************************************************************************\
*
*  FUNCTION:   WalkToMousePosition
*                 - walks from previous position to mouse click position.
*                   when he gets there, he winks and displays current position.
*                   everytime he stops, he poses in the next available pose.
*                   if the last pose is reached, he walks offscreen again.
*
*  DLL FUNCTION USED: WallyPlayToonStep
*
\******************************************************************************/

int       iCurPose=0;
TOONSTEP  tsMouseWalk;

BOOL WalkToMousePosition(HWND hwnd, LPARAM lParam)
   {
   POINT ptEnd;
   RECT  rcWin;
   // first grey all our menu items
   FixMenu(hwnd, FALSE);
   // set the end point to be the mouse position
   ptEnd.x = (signed short)LOWORD(lParam);
   ptEnd.y = (signed short)HIWORD(lParam);
   // set some of the other toonstep variables
   tsMouseWalk.itPose = iCurPose;
   tsMouseWalk.uiWait = 5;
   tsMouseWalk.btWink = TRUE;
   // and report mouse position and pose in text
   sprintf(tsMouseWalk.sztText, 
           "Walked to x=%i, y=%i, and now standing in Pose %i", 
           ptEnd.x, ptEnd.y, iCurPose+1);

   // now we'll check if we have to set the start or end point
   GetClientRect(hwnd, &rcWin);
   if(iCurPose==0)
      { // is we haven't started yet, position Wally offscreen
      tsMouseWalk.itYpos = ptEnd.y;
      if(ptEnd.x<((rcWin.right-rcWin.left)/2))
         tsMouseWalk.itXpos = -150; 
      else
         tsMouseWalk.itXpos = (rcWin.right-rcWin.left)+150; 
      }
   else
      {
      if(iCurPose==8)
         { // if we're at the last pose, make Wally walk offscreen
         ptEnd.y = tsMouseWalk.itYpos;
         if(tsMouseWalk.itXpos<((rcWin.right-rcWin.left)/2))
            ptEnd.x = -150;
         else
            ptEnd.x = (rcWin.right-rcWin.left)+150;
         
         iCurPose = -1; // to start at zero next time
         tsMouseWalk.sztText[0]=0; // set to speak no text
         tsMouseWalk.uiWait = 1;   // reduce wait period to 1 second
         }
      }
   iCurPose += 1; // increase to show next pose next time
   // Finally call the DLL function to play the toonstep
   return (gpfnDLLtoonstep)(hwnd, (PTOONSTEP)&tsMouseWalk, ptEnd);
   }


/******************************************************************************\
*
*  FUNCTION: ToonFileOpenDialog - calls file open dialog for toonfile selection
*
\******************************************************************************/

OPENFILENAME ofn; // struct. passed to GetOpenFileName 

BOOL ToonFileOpenDialog(HWND hWnd, LPTSTR lpFileName)
   {
   ofn.lStructSize       = sizeof(OPENFILENAME);
   ofn.hwndOwner         = hWnd;
   ofn.lpstrFilter       = "Wally Toon Files (*.wtf)\0*.wtf\0";
   ofn.lpstrCustomFilter = NULL;
   ofn.nFilterIndex      = 1;
   ofn.lpstrFile         = lpFileName;
   ofn.nMaxFile          = 256;
   ofn.lpstrInitialDir   = NULL;
   ofn.Flags             = OFN_FILEMUSTEXIST;
   ofn.lpfnHook          = NULL;
   ofn.lpstrTitle        = "Open Toon File";

   return GetOpenFileName((LPOPENFILENAME)&ofn);
   }


/******************************************************************************\
*
*  FUNCTION:    DoCommand (handles windows command messages)
*
\******************************************************************************/

void DoCommand(HWND hwnd, HINSTANCE ghInst, WPARAM wParam)
   {
   switch (LOWORD(wParam))
      {
      case IDM_FRAME:
          {
		  // start a ToonFrame playback 
          if(gpfnDLLtoonframe) 
             if(StartFramePlayback(hwnd))
                FixMenu(hwnd, FALSE);
          }
      break;

      case IDM_TOONRES:
          {
		  // start a ToonRes Playback
          if(gpfnDLLtoonresource)
             {
             HINSTANCE hInst = (HINSTANCE)GetWindowLong(hwnd, GWL_HINSTANCE);
             if((gpfnDLLtoonresource)(hwnd, hInst, "intro"))
                FixMenu(hwnd, FALSE);
             }
          }
      break;

      case IDM_TOONFILE:
          {
		  // displays the file open common control and then
		  // start a ToonFile Playback
          if(gpfnDLLtoonfile)    
             {
             char szToonFile[256]="\0";

             if(ToonFileOpenDialog(hwnd, (LPTSTR)szToonFile))
                {
                if((gpfnDLLtoonfile)(hwnd, szToonFile))
                   FixMenu(hwnd, FALSE);
                }
              }
          }
      break;

      case IDM_ABOUT:
           DialogBox(ghInst, "About", hwnd, AboutDlgProc);
      break;

      case IDM_INTERRUPT:
	       // interrupt or continue playback
           InterruptPlayback(hwnd);
      break;

      case IDM_EXITPROG:
           DestroyWindow(hwnd);
      break;
      }
   }


/******************************************************************************\
*
*  FUNCTION:    TileBitmap - tiles the given bitmap into the given window
*
\******************************************************************************/

void TileBitmap(HWND hWnd, HDC hdc, HBITMAP hBitmap)
	{
    BITMAP bm;
    RECT   rc;
    HDC    hdcMem;
    POINT  ptSize, ptOrg, ptStart;

    GetClientRect(hWnd, &rc);

    hdcMem = CreateCompatibleDC(hdc);
    SelectObject(hdcMem, hBitmap);
    SetMapMode(hdcMem, GetMapMode(hdc));
   
    GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
    ptSize.x = bm.bmWidth;
    ptSize.y = bm.bmHeight;
    DPtoLP(hdc, &ptSize, 1);
   
    ptOrg.x = 0;
    ptOrg.y = 0;
    DPtoLP(hdcMem, &ptOrg, 1);
    
    ptStart.x = 0;
    ptStart.y = 0;
    DPtoLP(hdcMem, &ptStart, 1);

    while(ptStart.y < (rc.bottom-rc.top))
    {  while(ptStart.x < (rc.right-rc.left))
       {  BitBlt(hdc, ptStart.x, ptStart.y, ptSize.x, ptSize.y, 
          hdcMem, ptOrg.x, ptOrg.y, SRCCOPY);
          ptStart.x = ptStart.x + ptSize.x;
       }  ptStart.x = 0;
       ptStart.y = ptStart.y + ptSize.y;
    } 
    DeleteDC(hdcMem);
	}


/******************************************************************************\
*
*  FUNCTION:    MainWndProc (standard window procedure INPUTS/RETURNS)
*
\******************************************************************************/

LRESULT CALLBACK MainWndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
  {
  static HBITMAP hBackground;
  HINSTANCE ghInst = (HINSTANCE)GetWindowLong(hwnd, GWL_HINSTANCE);

  switch (message)
    {
    case WM_CREATE:
         LoadWallyLibrary(hwnd);
         hBackground = LoadBitmap(ghInst, "backgr");
    break;

    case WM_COMMAND:
         DoCommand(hwnd, ghInst, wParam);
    break;

    case WM_TIMER:
         if(gpfnDLLwallytimer)
           (gpfnDLLwallytimer)(hwnd, wParam);
    break;

    case WM_LBUTTONDOWN:
         if(gpfnDLLtoonstep)
            WalkToMousePosition(hwnd, lParam);
    break;

    case WM_PAINT:
        if(gpfnDLLwallypaint)
          {
          PAINTSTRUCT ps;
          HDC hdc = BeginPaint(hwnd, &ps);
          TileBitmap(hwnd, hdc, hBackground);
          (gpfnDLLwallypaint)(hdc);
          EndPaint(hwnd, &ps);
          }
    break;

    case WM_SIZE:
        if(gpfnDLLwallysize)
          (gpfnDLLwallysize)(hwnd, lParam);
    break;

    case WM_ENDFRAME:
        if(gpfnDLLwallyendframe)
          (gpfnDLLwallyendframe)(hwnd);
    break;

    case WM_ENDPLAYBACK:
        FixMenu(hwnd, TRUE);
  break;

    case WM_DESTROY:
        FreeWallyLibrary(hwnd);
        DeleteObject(hBackground);
        PostQuitMessage(0);
    break;

    default:
        return (DefWindowProc(hwnd, message, wParam, lParam));
    }
  return (0);
}


/****************************************************************************\
*
*    FUNCTION: AboutDlgProc(HWND, unsigned, WORD, LONG)
*
*    PURPOSE:  Processes messages for "About" dialog box
*
*    MESSAGES:
*
*    WM_INITDIALOG - initialize dialog box
*    WM_COMMAND    - Input received
*
*    COMMENTS:
*
*    No initialization is needed for this particular dialog box, but TRUE
*    must be returned to Windows.
*
*    Wait for user to click on "Ok" button, then close the dialog box.
*
\****************************************************************************/

BOOL CALLBACK AboutDlgProc(hDlg, message, wParam, lParam)
   HWND hDlg;               /* window handle of the dialog box */
   unsigned message;        /* type of message                 */
   WPARAM wParam;             /* message-specific information    */
   LPARAM lParam;
   {
   const HINSTANCE dhInst = (HINSTANCE)GetWindowLong(hDlg, GWL_HINSTANCE);
   const HWND hParent     = GetParent(hDlg);
   
    switch (message)
      {
      case WM_INITDIALOG:
          return (TRUE);

      case WM_COMMAND:
          switch (wParam)
            {
            case IDOK:
              EndDialog(hDlg, TRUE); 
            return TRUE;
            }  
            return (FALSE);
      }
    return (FALSE);               /* Didn't process a message    */
   }                          
   


/******************************************************************************\
*
*  FUNCTION:    FixMenu
*
*  INPUTS:      hWnd     - handle of window with menu
*               choice   - TRUE enables menu items, FALSE disables them
*
*  COMMENTS:    Enables/disables menuitems depending on whether Wally is 
*               currently walking. Disables all if DLL could not be loaded.
*
\******************************************************************************/

void FixMenu(HWND hWnd, BOOL choice)
  {
  UINT i;
  HMENU hMenu    = GetMenu(hWnd);
  HMENU hSubMenu = GetSubMenu(hMenu, 0);

  if(choice == TRUE)
    {
    EnableMenuItem (hMenu, IDM_INTERRUPT, MF_DISABLED | MF_GRAYED |MF_BYCOMMAND);
    for(i=IDM_FRAME; i<=IDM_ABOUT; i++)
        EnableMenuItem (hSubMenu, i, MF_ENABLED | MF_BYCOMMAND);
    }
  else
    {
    EnableMenuItem (hMenu, IDM_INTERRUPT, MF_ENABLED | MF_BYCOMMAND);
    for(i=IDM_FRAME; i<=IDM_ABOUT; i++)
        EnableMenuItem (hSubMenu, i, MF_DISABLED | MF_GRAYED |MF_BYCOMMAND);
    }
  DrawMenuBar(hWnd);
  }



