package FSignal;

#This package is useful when importing waveform formats that specify events using
#a "future state" format (FST). Waveforms in this format specify an event as a time and
#a state that the waveform will be in until the next event. This format is commonly
#used by logic analyzers in particular. Internally The WaveFormer(TWF) represents events
#as a current state format (CST). A currently state format describe events as a time/state
#pair where the waveform is in the  given state until the given time. 
#
#To use this package, instead of directly creating waveforms in TWF using twf::NewSignal
#function, create FSignal objects (new FSignal) and add events to the waveform
#using the SetInitialStateFST and AddFutureEvent2 functions. When you have created all
#your signals and added all your events, call SetLastTimeEvent on all your FSignals to
#create the last event for each signal. If you just want all the events to end at the
#same time you can call the static member function SetLastEventTimeForAll.
#
#Whenever you need to directly access the underlying twf signal pointer (e.g. to put the
#signal on the signal list in TWF), use the TSignalPtr data member (e.g. if your FSignal
#is named $sig, use $sig->{TSignalPtr} to get the TSignalPtr object).

#----------------------------------------------------------
#create a new FSignal 
#----------------------------------------------------------
sub new
{
  print "new FSignal\n";
  my ($type) = @_;
  my $this = {};
  $this->{TSignalPtr} = twf::NewSignal();	#create the underlying TSignalPtr object in TWF
  bless $this;
  push(@SigArray,$this);
  return $this;
}

#----------------------------------------------------------
#SetInitialStateFST: use to import from future state time format
#    Call this function to set the initial state
#    of the waveform (state since time 0)
#----------------------------------------------------------
sub SetInitialStateFST #(TState,ExState)
{
  my ($this,$state,$exState) = @_;
  print "FSignal::SetInitialStateFST to $state\n";
  $this->{StaticState} = $state;
  $this->{StaticExState} = $exState;
}

#----------------------------------------------------------
#Add an event to a the signal (events must be
#added in time order)
#----------------------------------------------------------
sub AddFutureEvent2 #(TState,Time,Time,ExState,DelayResolution)
{
  my ($this,$state,$minT,$maxT,$exState,$delayRes) = @_;
  if ( defined($this->{StaticExState}) )
    {
    #this if clause used to get rid of uninit. value warnings
    if (defined $delayRes)    
      {
      $this->{TSignalPtr}->AddEvent2( $this->{StaticState}, $minT, $maxT, $this->{StaticExState}, $delayRes );
      print "AddEvent2 (StaticExState): $this->{StaticState}, $minT, $maxT, $this->{StaticExState}, $delayRes\n";
      }
    else
      {
      $this->{TSignalPtr}->AddEvent2( $this->{StaticState}, $minT, $maxT, $this->{StaticExState},  1);
      print "AddEvent2 (StaticExState): $this->{StaticState}, $minT, $maxT, $this->{StaticExState}, 1\n";
      }      
    }
  else
    {
    $this->{TSignalPtr}->AddEvent2( $this->{StaticState}, $minT, $maxT, $delayRes );
    print "AddEvent2 (!def StaticExState): $this->{StaticState}, $minT, $maxT, $delayRes\n";
    }
  $this->{StaticState} = $state;
  $this->{StaticExState} = $exState;
}

#----------------------------------------------------------
#Add an event to a the signal (events must be
#added in time order), does nothing if previous event
#is the same.
#----------------------------------------------------------
sub AddFutureEventNoDup2 #(TState,Time,Time,ExState,DelayResolution)
{
  my ($this,$state,$minT,$maxT,$exState,$delayRes) = @_;
  if ( defined($this->{StaticExState}) )
    {
    if ($this->{StaticExState} ne $exState)
      {
      $this->{TSignalPtr}->AddEvent2( $this->{StaticState}, $minT, $maxT, $this->{StaticExState}, $delayRes );
      }
    }
  else
    {
    if ($this->{StaticState} ne $state)
      {
      $this->{TSignalPtr}->AddEvent2( $this->{StaticState}, $minT, $maxT, $delayRes );
      }
    }
  $this->{StaticState} = $state;
  $this->{StaticExState} = $exState;
}

#----------------------------------------------------------
#Ends the waveform at time $time. If no $time is given,
#$time is set is using the GetLastOutputTime API function.
#----------------------------------------------------------
sub SetLastEventTime #(Time)
{
  my ($this,$time) = @_;
  $time = twf::GetLastOutputTime() if !$time;
  print "SetLastEvenTime=$time\n";
  $this->AddFutureEvent2(0,$time,$time,0);
}

#----------------------------------------------------------
#static member function that sets last event
#time for all FSignals
#----------------------------------------------------------
sub SetLastEventTimeForAll  #Time
{
  my ($type,$time) = @_;
  print "FSignal::SetLastEventTimeForAll $type $time\n";
  
  my $sig;
  foreach $sig (@SigArray)
    {
    $sig->SetLastEventTime($time);
    }
}

1;