unit Mainform;
(* Version 1.01
  a) The ability to use command line parameters with the install program
     got disabled by a "cosmetic" improvement at the last minute. Fixed!
  b) If you added an INI file to INCTRL2.INI for special tracking *and*
     that INI file was not in the Windows directory, it didn't work. Fixed.
*)
(* Version 1.02
  a) The report used to say it came from "beta version"; now it
     says "version 1.02"
  b) When you reach the report preview page, the multi-line edit
     containing the report now has the focus, so PgDn works right
     away
  c) If you set an INI file to watch in INCTRL2.INI and it doesn't
     exist, version 1.02 will warn you at startup and
     remove it from the list.
*)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, ExtCtrls;
Const
  WM_FILESYSCHANGE = $34;
type
  TTrackMode = (tmUnknown, tmFastNotify, tmDiskRead);
  TInstStatus = (isNotyet, isPrepare, isInstall, isInstallFailed,
    isRestart, isGather, isDone, isTwoPhase);
  String4  = String[4];
  String12 = String[12];
  String18 = String[18];

  TForm1 = class(TForm)
    NotebookMain: TNotebook;
    BtnDoIt: TBitBtn;
    NotebookSub: TNotebook;
    Bevel0: TBevel;
    InstNameLabel: TLabel;
    EditInstallName: TEdit;
    BtnBrowse: TButton;
    BtnSelect: TButton;
    FFNotifyLabel: TLabel;
    BtnExplain: TButton;
    FFCheckBox: TCheckBox;
    ScrollBox1: TScrollBox;
    fpTrackModeLabel: TLabel;
    fpReptNameLabel: TLabel;
    fpProgTypeLabel: TLabel;
    fpInstProgLabel: TLabel;
    fpWinVerLabel: TLabel;
    BtnNext: TBitBtn;
    BtnHelp: TBitBtn;
    BtnPrev: TBitBtn;
    BtnCancel: TBitBtn;
    BtnAbout: TBitBtn;
    Panel1: TPanel;
    Image1: TImage;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    BtnDone: TBitBtn;
    BtnAbort: TBitBtn;
    BtnHelpToo: TBitBtn;
    Memo1: TMemo;
    BtnClose: TBitBtn;
    EditDescript: TEdit;
    PanelReportName: TPanel;
    fpDescriptLabel: TLabel;
    ReportNameLabel: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure BtnBrowseClick(Sender: TObject);
    procedure EditInstallNameChange(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure BtnNextClick(Sender: TObject);
    procedure BtnPrevClick(Sender: TObject);
    procedure NotebookSubPageChanged(Sender: TObject);
    procedure BtnSelectClick(Sender: TObject);
    procedure BtnExplainClick(Sender: TObject);
    procedure DriveBtnClick(Sender: TObject);
    procedure BtnCancelClick(Sender: TObject);
    procedure BtnAboutClick(Sender: TObject);
    procedure BtnDoItClick(Sender: TObject);
    procedure BtnDoneClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure BtnHelpTooClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure BtnHelpClick(Sender: TObject);
    procedure FFNotifyLabelDblClick(Sender: TObject);
  private
    { Private declarations }
    Platform, Version   : Word;
    eType               : Integer;
    tType               : TTrackMode;
    FileCdrAvailable    : Boolean;
    DOSFileCdrAvailable : Boolean;
    Enhanced            : Boolean;
    DriveButtons        : TList;
    WatchLabels         : TList;
    DirLabels           : TList;
    IniToWatch          : TStringList;
    OutPath             : String;
    InstStatus          : TInstStatus;
    FileCdrFile         : TextFile;
    WindowsDir          : String[145];
    function RememberDisk(const fName : TFilename) : LongInt;
    procedure RememberINI(const InName : String; Ext : String4);
    procedure NormalizeINI(const InN, OutN : String);
    procedure CompareDisk(const OldN, NewN : TFilename;
      vAdd, vDel, vCha : TStringList);
    procedure CompareINI(const OldN, NewN : TFilename;
      vAddK, vDelK, vChaK, vAddS, vDelS : TStringList);
    function ExecInstall : Integer;
    function ProcMsgTerminated : Boolean;
    function CanRestart : Boolean;
    procedure WMFileSysChange(VAR Msg : TMessage);
      message WM_FILESYSCHANGE;
    procedure WMQueryEndSession(VAR Msg: TWMQueryEndSession);
      message WM_QUERYENDSESSION;
    procedure WMEndSession(VAR Msg: TWMEndSession);
      message WM_ENDSESSION;
    procedure WMNCPaint(VAR Msg: TWMNCPaint);
      message WM_NCPAINT;
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation
Uses IniFiles, FileCtrl, ShellApi,
  WinVerU, fExeType, FileCdrI, DlPop, AboutBox, TextLoc;
{$R *.DFM}
{$I HELPCTX.PAS}

CONST
  TempExt1 = '.$$1';
  TempExt2 = '.$$2';
  TempName = 'INCTRL$$';

function TForm1.ProcMsgTerminated : Boolean;
{call processMessages and return whether or not
 the app was terminated}
BEGIN
  Application.ProcessMessages;
  Result := Application.Terminated;
END;

procedure TForm1.WMNCPaint(VAR Msg: TWMNCPaint);
BEGIN
  Inherited;
END;

procedure TForm1.WMEndSession(VAR Msg: TWMEndSession);
VAR
  S : String;
  N : Word;
CONST DumbAct : TCloseAction = caFree;
BEGIN
  IF Msg.EndSession THEN
    BEGIN
      {Uninstall the FileCdr hook (harmless if not installed}
      FileCdrUninstall(Handle);
      {close the FileCdr log file, ignoring exceptions}
      try
        CloseFile(FileCdrFile);
      except
        ON Exception DO;
      end;
      {If installation not reached yet, just clean up}
      IF InstStatus < isInstall THEN
        FormClose(Self, DumbAct)
      ELSE
        {otherwise, set up for restarting}
        BEGIN
          IF CanRestart THEN
            WITH TIniFile.Create('WIN.INI') DO
              try
                S := ReadString('Windows', 'Run', '');
                S := Application.ExeName + ' ' + S;
                WriteString('Windows', 'Run', S);
              finally
                Free;
              end;
        WITH TIniFile.Create(ChangeFileExt(Application.ExeName,'.INI')) DO
          try
            WriteString('Survival', 'SurvivedRestart', 'YES');
            WriteString('Survival', 'Description', EditDescript.Text);
            WriteString('Survival', 'ReportName',
              PanelReportName.Caption);
            WriteBool('Survival', 'FastNotify', tType=tmFastNotify);
            IF tType <> tmFastNotify THEN
              FOR N := 0 TO DriveButtons.Count-1 DO
                BEGIN
                  WriteInteger('WatchDrive', 'd'+IntToStr(N),
                    TLabel(WatchLabels[N]).Tag);
                  IF TLabel(WatchLabels[N]).Tag > 0 THEN
                    WriteString('WatchDir', 'd'+IntToStr(N),
                      TLabel(DirLabels[N]).Caption);
                END;
          finally
            Free;
          end;
        END;
    END;
END;

procedure TForm1.WMQueryEndSession(VAR Msg: TWMQueryEndSession);
begin
  Msg.Result := 1;
  IF (InstStatus >= isInstall) AND (NOT CanRestart) THEN
    IF tType = tmFastNotify THEN
      BEGIN
        MessageDlg('INCTRL 2 is using fast file notification, and '+
          'cannot survive a restart. INCTRL 2 will write its report '+
          'now. You will have to manually restart Windows to finish '+
          'the installation.', mtInformation, [mbOK, mbHelp],
            Cannot_Survive_Restart);
         msg.Result := 0;
         BtnDoneClick(BtnDone);
      END
    ELSE
      MessageDlg('INCTRL 2 cannot survive restart of Windows; you will '+
        'need to launch it again after the installation is complete',
        mtInformation, [mbOk, mbHelp], Cannot_Survive_Restart);
end;

procedure TForm1.WMFileSysChange(VAR Msg : TMessage); (* FIG09.TXT *)
{Copy file system change info to temporary file}
BEGIN
  WITH Msg DO
    try
      IF (wParam = 2) OR (wParam = $56) THEN
        WriteLn(FileCdrFile, Format('%.2x%s TO %s',
          [wParam, StrUpper(PChar(lParam)),
            StrUpper(StrEnd(PChar(LParam))+1)]))
      ELSE WriteLn(FileCdrFile, Format('%.2x%s',
        [wParam, StrUpper(PChar(lParam))]));
    except
      ON E:EInOutError DO
        IF E.ErrorCode <> 6 THEN Raise;
    end;
END;

function TForm1.CanRestart : Boolean;
VAR
  OKRunLen : Word;
  S        : String;
begin
  Result := True;
  OkRunLen := 127 - Length(Application.ExeName);
  WITH TIniFile.Create('WIN.INI') DO
    try
      S := ReadString('windows','run','');
    finally
      Free;
    end;
  Result := Length(S) < OkRunLen;
end;

procedure TForm1.FormCreate(Sender: TObject);

  PROCEDURE CheckFileCdr;
  BEGIN
    IF NOT Enhanced THEN
      BEGIN
        FileCdrAvailable := False;
        DOSFileCdrAvailable := False;
      END
    ELSE IF FileCdrInstall(Handle) THEN
      BEGIN
        FileCdrAvailable := True;
        FileCdrUninstall(Handle);
        WITH TIniFile.Create('SYSTEM.INI') DO
          try
            DosFileCdrAvailable := UpperCase(
              ReadString('386Enh', 'FileSysChange', 'OFF')) = 'ON';
          finally
            Free;
          end;
      END
    ELSE
      BEGIN
        FileCdrAvailable := False;
        DOSFileCdrAvailable := False;
      END;
  END;

  PROCEDURE SetupDriveButtons; (* FIG06.TXT *)
  VAR N : Word;
  BEGIN
    {initialize component lists (data fields of TForm1)}
    DriveButtons := TList.Create;
    WatchLabels := TList.Create;
    DirLabels := TList.Create;
    {create temporary drive combo box}
    WITH TDriveComboBox.Create(Self) DO
      try
        Parent := Self;
        FOR N := 0 TO Items.Count-1 DO
          BEGIN
            DriveButtons.Add(TBitBtn.Create(Self));
            WITH TBitBtn(DriveButtons[N]) DO
              BEGIN
                Parent  := ScrollBox1;
                Left    := 8;
                Top     := 8 + N*28;
                Width   := 49;
                Height  := 25;
                Caption := Items[N][1] + ':\';
                Margin  := 4;
                Glyph   := Items.Objects[N] AS TBitmap;
                OnClick := DriveBtnClick;
                Tag     := N;
              END;
            WatchLabels.Add(TLabel.Create(Self));
            WITH TLabel(WatchLabels[N]) DO
              BEGIN
                Parent     := ScrollBox1;
                Left       := 64;
                Top        := 12 + N*28;
                Width      := 40;
                Height     := 20;
                Font.Size  := 12;
                Font.Style := [fsBold];
                Font.Color := clRed;
                Tag        := 0;
                Caption    := 'NO';
                {disable buttons for CD-ROM drives; set fixed drives
                 initally to YES, removable and network drives to NO}
                CASE GetDriveType(Ord(Items[N][1])-Ord('a')) OF
                  DRIVE_REMOVABLE : ; {do nothing; it's ready}
                  DRIVE_REMOTE    : BEGIN {network OR CD-ROM}
                    IF IsCDRom(Ord(Items[N][1])-Ord('a')) THEN
                      TBitBtn(DriveButtons[N]).Enabled := False;
                    END;
                  DRIVE_FIXED     : BEGIN
                    Caption := 'YES';
                    Font.Color := clGreen;
                    Tag := 1;
                  END;
                END;
              END;
            DirLabels.Add(TLabel.Create(Self));
            WITH TLabel(DirLabels[N]) DO
              BEGIN
                Parent  := ScrollBox1;
                Left    := 112;
                Top     := 12 + 28*N;
                Caption := Items[N][1] + ':\';
                Tag     := N;
              END;
          END;
      finally
        Free;
      end;
  END;

  PROCEDURE GetIniFileData;
  VAR
    N : Word;
    S : String;
  BEGIN
    IniToWatch := TStringList.Create;
    IniToWatch.Sorted := True;
    IniToWatch.Duplicates := dupIgnore;
    WITH TIniFile.Create(ChangeFileExt(Application.ExeName,'.INI')) DO
      try
        ReadSection('INI Files To Watch', IniToWatch);
        (* 1.02 *)
        IF IniToWatch.Count > 0 THEN
          FOR N := IniToWatch.Count-1 DOWNTO 0 DO
            BEGIN
              S := IniToWatch[N];
              IF ExtractFilePath(S) = '' THEN
                S := WindowsDir + S;
              IF NOT FileExists(S) THEN
                BEGIN
                  MessageDlg(Format(
                   'The file %s, specified in INCTRL2.INI as an Ini File '+
                   'to Watch, does not exist!', [S]), mtWarning, 
                     [mbOk, mbHelp], Tracking_Other_INI_files);
                  IniToWatch.Delete(N);
                END;
            END;
        (* 1.02 *)
        {The next two lines will have no effect if WIN.INI
         and SYSTEM.INI are in INCTRL2.INI already}
        IniToWatch.Add('WIN.INI');
        IniToWatch.Add('SYSTEM.INI');
        S := ReadString('Survival', 'SurvivedRestart', 'NO');
        IF S <> 'NO' THEN
          BEGIN
            EditDescript.Text := ReadString('Survival', 'Description', '');
            PanelReportName.Caption := ReadString('Survival', 'ReportName', '');
            ReportNameLabel.Caption := PanelReportName.Caption;
            IF ReadBool('Survival', 'FastNotify', FALSE) THEN
              tType := tmFastNotify {FileCDR set up in OnActivate}
            ELSE
              BEGIN
                tType := tmDiskRead;
                FOR N := 0 TO DriveButtons.Count-1 DO
                  BEGIN
                    TLabel(WatchLabels[N]).Tag :=
                      ReadInteger('WatchDrive', 'd'+IntToStr(N), 0);
                    IF TLabel(WatchLabels[N]).Tag > 0 THEN
                      TLabel(DirLabels[N]).Caption :=
                        ReadString('WatchDir', 'd'+IntToStr(N),
                          Char(Ord('a')+N)+':\');
                  END;
              END;
            EraseSection('Survival');
            EraseSection('WatchDrive');
            EraseSection('WatchDir');
            FormStyle := fsStayOnTop;
            NotebookMain.PageIndex := 1;
            BtnPrev.Enabled := False;
            BtnNext.Enabled := False;
            BtnDoIt.Enabled := False;
            WITH BtnHelpToo DO
              BEGIN
                Self.ClientWidth := Left + Width + 1;
                Self.ClientHeight := Height + Top + 1;
              END;
            Caption := 'INCTRL 2 - Installation restarted';
            BtnDone.Enabled := True;
            ActiveControl := BtnDone;
            BtnAbort.Caption := 'Abort: No Report';
            InstStatus := isRestart;
          END;
        IF S = 'YES' THEN {actual restart; fix WIN.INI}
          WITH TIniFile.Create('WIN.INI') DO
            try
              S := ReadString('Windows', 'Run', '');
              N := Pos(Application.ExeName, S);
              IF N > 0 THEN
                System.Delete(S, N, Length(Application.ExeName) + 1);
              WriteString('Windows', 'Run', S);
            finally
              Free;
            end;
      finally
        Free;
      end;
  END;

  procedure DefaultOutputName;
  VAR N : Word;
  BEGIN
    N := 0;
    REPEAT
      PanelReportName.Caption := OutPath +
        Format('%.4d.RPT', [N]);
      Inc(N);
    UNTIL NOT FileExists(PanelReportName.Caption);
  END;

begin
  tType := tmUnknown;
  eType := ET_BLANK;
  InstStatus := isNotyet;
  InstNameLabel.Caption := ExeTypeName(ET_BLANK);
  OutPath := ExtractFilePath(Application.ExeName);
  NotebookMain.PageIndex := 0;
  NotebookSub.PageIndex := 0;
  fpWinVerLabel.Caption := WinVerString(Platform, Version, Enhanced);
  (* 1.02 *) (* moved WindowsDir initialization to here *)
  GetWindowsDirectory(@WindowsDir[1], 145);
  WindowsDir[0] := Char(StrLen(@WindowsDir[1]));
  IF WindowsDir[Length(WindowsDir)] <> '\' THEN
    WindowsDir := WindowsDir + '\';
  (* 1.02 *)
  DefaultOutputName;
  SetupDriveButtons;
  GetINIFileData;
  IF tType = tmUnknown THEN CheckFileCdr;
  IF InstStatus = isNotyet THEN
    ActiveControl := EditInstallName;
end;

procedure TForm1.BtnBrowseClick(Sender: TObject);
begin
  WITH OpenDialog1 DO
    BEGIN
      InitialDir := ExtractFilePath(EditInstallName.Text);
      IF Execute THEN
        EditInstallName.Text := Filename;
    END;
end;

procedure TForm1.EditInstallNameChange(Sender: TObject);
(* Version 1.01: Ability to use command line parameters
     was disabled w/o notice. These changes restore it. *)
VAR EndOfName : Byte; (* 1.01*)
begin
  WITH EditInstallName DO
    BEGIN
      EndOfName := Pos(' ', Text); (* 1.01*)
      IF EndOfName = 0 THEN        (* 1.01*)
        EndOfName := Length(Text)  (* 1.01*)
      ELSE Dec(EndOfName);         (* 1.01*)
      IF Text = '' THEN
        eType := ET_BLANK
      ELSE IF (Pos('.', Text) > 0) AND
(*        (Length(Text) - Pos('.', Text) = 3) THEN*) (* 1.01*)
        (EndOfName - Pos('.', Text) = 3) THEN        (* 1.01*)
        eType := ExeType(Text)
      ELSE eType := ET_UNKNOWN;
    END;
  InstNameLabel.Caption := ExeTypeName(eType);
  BtnNext.Enabled := eType >= ET_BLANK;
end;

procedure TForm1.FormDestroy(Sender: TObject);
begin
  FileCdrUninstall(Handle);
end;

procedure TForm1.BtnNextClick(Sender: TObject);
{Upon leaving a page for the next page, update any
 other data in the program that depends on the
 departing page or is required by the next page}
(* Version 1.01: Ability to use command line parameters
     was disabled w/o notice. These changes restore it. *)
VAR Skip, SpacePos : Word; (* 1.01 *)
begin
  Skip := 1;
  CASE NotebookSub.PageIndex OF
    0 : BEGIN
      IF EditInstallName.Text <> '' THEN
        BEGIN                                         (* 1.01 *)
          SpacePos := Pos(' ', EditInstallName.Text); (* 1.01 *)
          IF SpacePos = 0 THEN                        (* 1.01 *)
            EditInstallName.Text :=
              ExpandFilename(EditInstallName.Text)
          ELSE                                             (* 1.01 *)
            EditInstallName.Text :=                        (* 1.01 *)
              ExpandFilename(Copy(EditInstallName.Text, 1, (* 1.01 *)
                SpacePos-1))+                              (* 1.01 *)
                Copy(EditInstallName.Text, SpacePos, 255); (* 1.01 *)
        END;
      fpInstProgLabel.Caption := EditInstallName.Text;
      fpProgTypeLabel.Caption := ExeTypeName(eType);
      BtnNext.Enabled := True;
    END;
    1 : fpDescriptLabel.Caption := EditDescript.Text;
    2 : BEGIN
          fpReptNameLabel.Caption := PanelReportName.Caption;
          ReportNameLabel.Caption := PanelReportName.Caption;
          IF (((eType >= ET_WINEXE) AND FileCdrAvailable) OR
             ((eType >= ET_DOSEXE) AND DOSFileCdrAvailable)) AND
              (Platform <> WINVER_NT) THEN
            BEGIN
              FFNotifyLabel.Caption := 'is available';
              FFCheckBox.Checked := True;
              FFCheckBox.Enabled := True;
              tType := tmFastNotify;
            END
          ELSE
            BEGIN
              FFNotifyLabel.Caption := 'is not available';
              FFCheckBox.Checked := False;
              FFCheckBox.Enabled := False;
              tType := tmDiskRead;
            END;
          END;
    3 : IF FFCheckBox.Checked THEN
          BEGIN
            Skip := 2;
            tType := tmFastNotify;
            fpTrackModeLabel.Caption := 'Fast file notify';
            BtnDoIt.Enabled := True;
            BtnDoIt.BringToFront;
            BtnNext.Enabled := False;
          END
        ELSE
          BEGIN
            tType := tmDiskRead;
            fpTrackModeLabel.Caption := 'Disk contents comparison';
          END;
    4 : BEGIN
      BtnDoIt.Enabled := True;
      BtnDoIt.BringToFront;
      BtnNext.Enabled := False;
    END;
  END;
  NotebookSub.PageIndex := NotebookSub.PageIndex + Skip;
  BtnPrev.Enabled := True;
end;

procedure TForm1.BtnPrevClick(Sender: TObject);
VAR Skip : Word;
begin
  Skip := 1;
  CASE NotebookSub.PageIndex OF
    5 : BEGIN
      BtnNext.Enabled := True;
      BtnNext.BringToFront;
      BtnDoIt.Enabled := False;
      IF FFCheckBox.Checked THEN Skip := 2;
    END;
    4 : ;
    3 : ;
    1 : BtnPrev.Enabled := False;
  END;
  NotebookSub.PageIndex := NotebookSub.PageIndex - Skip;
  BtnNext.Enabled := True;
end;

procedure TForm1.NotebookSubPageChanged(Sender: TObject);
{when a page becomes current, make a particular control
 active}
begin
  CASE NotebookSub.PageIndex OF
    0 : ActiveControl := EditInstallName;
    1 : ActiveControl := EditDescript;
    2 : ActiveControl := BtnSelect;
    3 : IF FFCheckBox.Enabled THEN ActiveControl := FFCheckBox;
    4 : ActiveControl := TBitBtn(DriveButtons[0]);
    5 : ;
  END;
  WITH NotebookSub DO
    TPage(NotebookMain.Pages.Objects[0]).HelpContext :=
      TPage(Pages.Objects[PageIndex]).HelpContext;
end;

procedure TForm1.BtnSelectClick(Sender: TObject);
VAR Driv : Integer;
begin
  WITH SaveDialog1 DO
    BEGIN
      Filename := '';
      InitialDir := ExtractFilepath(PanelReportName.Caption);
      IF Execute THEN
        BEGIN
          PanelReportName.Caption := ExpandFilename(Filename);
          Driv := Ord(UpCase(PanelReportName.Caption[1]))-Ord('A');
          CASE GetDriveType(Driv) OF
            0, DRIVE_REMOVABLE : BtnNext.Enabled := FALSE;
            ELSE BtnNext.Enabled := NOT IsCDRom(Driv);
          END;
        END;
      ActiveControl := BtnNext;
    END;
end;

procedure TForm1.BtnExplainClick(Sender: TObject);
VAR Explan : Word;
begin
  IF EditInstallName.Text = '' THEN
    Explan := Fast_File_Notification__Why_Not_5_
  ELSE IF NOT Enhanced THEN
    Explan := Fast_File_Notification__Why_not_2_
  ELSE IF Platform = WINVER_95 THEN
    Explan := Fast_File_Notification__Why_not_4_
  ELSE IF NOT FileCdrAvailable THEN
    Explan := Fast_File_Notification__Why_not_3_
  ELSE IF (eType IN [ET_DOSEXE, ET_COM, ET_BAT]) AND
    (NOT DOSFileCdrAvailable) THEN
      Explan := Fast_File_Notification__Why_not_1_
  ELSE IF (Platform = WINVER_NT) THEN
    Explan := Fast_File_Notification__Why_Not_6_
  ELSE
    Explan := Fast_File_Notification_Available;
  Application.HelpContext(Explan);
end;

procedure TForm1.DriveBtnClick(Sender: TObject);
VAR
  which : Integer;
  Any   : Boolean;
begin
  which := (Sender AS TBitBtn).Tag;
  WITH TLabel(WatchLabels[which]) DO
    BEGIN
      Tag := (Tag+1) MOD 3;
      CASE Tag OF
        0 : BEGIN
          Font.Color := clRed;
          Caption := 'NO';
          TLabel(DirLabels[Which]).Caption :=
            (Sender AS TBitBtn).Caption;
        END;
        1 : BEGIN
          Font.Color := clGreen;
          Caption := 'YES';
          TLabel(DirLabels[Which]).Caption :=
            (Sender AS TBitBtn).Caption;
        END;
        2 : BEGIN
          Font.Color := clBlue;
          Caption := 'DIR>';
          WITH DirListPop DO
            BEGIN
              DirectoryListBox1.Directory := TLabel(DirLabels[Which]).Caption;
              Left := TLabel(DirLabels[Which]).Left + Self.Left +
                ScrollBox1.Left;
              Top  := TLabel(DirLabels[Which]).Top + NotebookSub.Top +
                ScrollBox1.Top + Self.Top + Self.Height -
                Self.ClientHeight -4;
            END;
          IF DirListPop.ShowModal = mrOk THEN
            TLabel(DirLabels[Which]).Caption :=
              DirListPop.DirectoryListBox1.Directory
          ELSE
            BEGIN
              Tag := 0;
              Font.Color := clRed;
              Caption := 'NO';
            END;
        END;
      END;
    END;
  Any := False;
  FOR which := 0 TO WatchLabels.Count-1 DO
    IF TLabel(WatchLabels[which]).Tag > 0 THEN
      Any := True;
  BtnNext.Enabled := Any;
end;

procedure TForm1.BtnCancelClick(Sender: TObject);
begin
  Close;
end;

procedure TForm1.BtnAboutClick(Sender: TObject);
begin
  WITH TAboutForm.Create(Application) DO
    try
      ShowModal;
    finally
      Free;
    end;
end;

procedure TForm1.NormalizeINI(const InN, OutN : String);
{Convert an INI file into "normalized form", sorted by
 section and then by key, with the section name prefixed
 to each key in that section}
VAR
  InF, OutF        : TextFile;
  S, SecName       : String;
  SecList, KeyList : TStringList;
  SecNum, KeyNum   : Word;

  PROCEDURE Strip(VAR S : String);
  {Strips any mixture of leading and trailing
   spaces and tabs}
  VAR First, Last : Word;
  BEGIN
    First := 1;
    WHILE (First < Length(S)) AND
      ((S[First] = ' ') OR
       (S[First] = #9)) DO Inc(First);
    Last := Length(S);
    WHILE (Last >= First) AND
      ((S[Last] = ' ') OR
       (S[Last] = #9)) DO Dec(Last);
    IF Last >= First THEN
      S := Copy(S, First, Last-First+1)
    ELSE S := '';
  END;

  FUNCTION IsSection(VAR S : String) : Boolean;
  {Assume S has been stripped of leading and trailing
   spaces and tabs}
  BEGIN
    IsSection := (S <> '') AND (S[1]='[')
      AND (S[Length(S)]=']');
  END;

  FUNCTION IsKey(VAR S : String) : Boolean;
  {Assums S is stripped. True for key values only;
   not for blank lines, comments, sections, or lines
   lacking an equal-sign}
  BEGIN
    IsKey := (S <> '') AND (S[1]<>';') AND (NOT IsSection(S))
      AND (Pos('=',S)>0);
  END;

begin
  AssignFile(InF, InN);
  Reset(InF);
  AssignFile(OutF, OutN);
  Rewrite(OutF);
  try try
    SecList := TStringList.Create;
    try
      SecList.Sorted := True;
      WHILE NOT EoF(InF) DO
        BEGIN
          ReadLn(InF, S);
          IF ProcMsgTerminated THEN Exit;
          Strip(S);
          {The TFilePos function is found in the TextLoc unit;
           it is a FilePos function that works on text files}
          IF IsSection(S) THEN
            SecList.Add(Format('%s %.8x',
              [S, TFilePos(InF)]));
        END;
      KeyList := TStringList.Create;
      try
        KeyList.Sorted := True;
        FOR SecNum := 0 TO SecList.Count-1 DO
          BEGIN
            S := SecList[SecNum];
            KeyList.Clear;
            SecName := Copy(S, 1, Length(S)-9);
            WriteLn(OutF, SecName);
            S := '$' + Copy(S, Length(S)-7, 8);
            {The TFilePos procedure is found in the TextLoc unit;
             it is a seek procedure that works on text files}
            TSeek(InF, StrToIntDef(S, 0));
            WHILE NOT (EoF(InF) OR IsSection(S)) DO
              BEGIN
                ReadLn(InF,S);
                Strip(S);
                IF IsKey(S) THEN
                  KeyList.Add(S);
              END;
            IF KeyList.Count > 0 THEN
              FOR KeyNum := 0 TO KeyList.Count-1 DO
                WriteLn(OutF, SecName, KeyList[KeyNum]);
            END;
          finally
            KeyList.Free;
          end;
    finally
      SecList.Free;
    end;
  finally
    CloseFile(InF);
    CloseFile(OutF);
  end;
  except
    ON E: EInOutError DO
      IF E.ErrorCode <> 6 THEN Raise;
  end;
end;

procedure TForm1.RememberINI(const InName : String;
  Ext : String4);
(* Version 1.01: Trouble with hand-added INI files that
   are not in Windows directory fixed. *)
VAR
  iName, tName : String;
  iFil, tFil   : TextFile;
BEGIN
  Caption := 'INCTRL 2 - Copying ' +
    ExtractFileName(inName);
  IF ProcMsgTerminated THEN Exit;
  iName := ExtractFilePath(InName);    (* 1.01 *)
  IF iName = '' THEN (* 1.01 *)
    BEGIN                              (* 1.01 *)
      GetWindowsDirectory(@iName[1], 255);
      iName[0] := Char(StrLen(@iName[1]));
      IF iName[Length(iName)] <> '\' THEN
        iName := iName + '\';
      iName := iName + InName;
    END                                (* 1.01 *)
  ELSE iName := InName;                (* 1.01 *)
(*tName := ChangeFileExt(OutPath + inName, Ext);*) (* 1.01 *)
  tName := ChangeFileExt(OutPath + ExtractFileName(inName), Ext); (* 1.01 *)
  IF FileExists(iName) THEN
    NormalizeIni(iName, tName);
END;

Function TForm1.RememberDisk(const fName : tFilename) : LongInt; {WIP}
{"memorize" the contents of the disk, for comparison later to
see what was added/deleted/changed}
VAR
  OutF     : TextFile;
  StartDir : TFilename;
  N        : Word;

  (* FIG07.TXT begin *)
  function DirWMask(const Dir : TFileName) : TFileName;
  {Add \*.* mask to a directory that may have
   trailing blanks}
  VAR P : Byte;
  BEGIN
    P := Length(Dir);
    WHILE (P > 0) AND (Dir[P] = ' ') DO
      Dec(P);
    Result := Copy(Dir,1,P) + '\*.*';
  END;

  function StringFor(var vTS :TSearchRec; vLevel : Char) : String18;
  {Create an 18 character string to store info about one file.}
  const
    vTimeSize : RECORD
      Len  : Byte;
      Valu : LongInt;
    END = (Len: 4; Valu: 0);
  VAR
    sTimeSize : String[4] ABSOLUTE vTimeSize;
    N         : Word;
  BEGIN
    vTimeSize.valu := vTS.Time XOR vTS.Size;
    FOR N := 1 TO 4 DO
      CASE sTimeSize[N] OF
        #0, #10, #13, #26 : Inc(sTimeSize[N], $20);
      END;
    Result := Format('%.1s%-12.12s%.1x%s',[vLevel,
                vTS.Name, (vTS.Attr AND faDirectory) SHR 4,
                sTimeSize])
  END;

  function AddDir(const Dir: TFileName; const Line: String18): TFileName;
  VAR P : Byte;
  BEGIN
    P := 13;
    WHILE (P > 0) AND (line[P] = ' ') DO Dec(P);
    AddDir := Dir + '\' + Copy(line, 2, P-1);
  END;

  procedure FindFiles(const Dir : TFileName; Level : Char);
  VAR
    TS : TSearchRec;
    I  : Integer;
    TL : TStringList; {local TStringList}
  BEGIN
    {Let Windows process messages; Exit if app terminated}
    IF ProcMsgTerminated THEN Exit;
    TL := TStringList.Create;
    try try
      TL.Sorted := True;
      {add file strings to sorted list box}
      I := FindFirst(DirWMask(Dir), faDirectory OR faHidden OR
        faSysFile, TS);
      WHILE I = 0 DO
        BEGIN
          IF TS.Name[1] <> '.' THEN
            TL.Add(StringFor(TS, Level));
          I := FindNext(TS);
        END;
      FindClose(TS);
      {write out strings in sorted order; call FindFiles
       recursively for any directories found}
      FOR I := 0 TO TL.Count-1 DO
        BEGIN
          WriteLn(OutF, TL.Strings[I]);
          Result := Result + 1;
          IF TL.Strings[I][14] = '1' THEN
            FindFiles(AddDir(Dir, TL.Strings[I]), Succ(Level));
        END;
    finally
      TL.Free;
    END;
    except
      ON E: EInOutError DO
        IF E.ErrorCode <> 6 THEN Raise;
    end;
  END;
  (* FIG07.TXT end *)

  function SkinnyName(const S : String) : String;
  CONST MaxWid = 30;
  VAR
    namOnly : String12;
    Path    : TFileName;
    Len     : Word;
  BEGIN
    IF Length(S) < MaxWid THEN Result := S
    ELSE
      BEGIN
        namOnly := ExtractFilename(S);
        Len := MaxWid - 3 - Length(NamOnly);
        Path := ExtractFilePath(S);
        WHILE Length(Path) > Len DO
          Path := ExtractFilePath(Copy(Path, 1, Length(Path)-1));
        Result := Path + '..\' + namOnly;
      END;
  END;

BEGIN
  AssignFile(OutF, fName);
  Rewrite(OutF);
  try try
    FOR N := 0 TO DriveButtons.Count-1 DO
      IF TLabel(WatchLabels[N]).Tag > 0 THEN
        BEGIN
          StartDir := Uppercase(TLabel(DirLabels[N]).Caption);
          Caption := 'INCTRL 2 - ' +
            SkinnyName(TLabel(DirLabels[N]).Caption);
          IF ProcMsgTerminated THEN Exit;
          WriteLn(OutF, '0',StartDir);
          IF Length(StartDir) = 3 THEN Dec(StartDir[0]);
          FindFiles(StartDir, '1');
        END;
  finally
    CloseFile(OutF);
  end;
  except
    ON E: EInOutError DO
      IF E.ErrorCode <> 6 THEN Raise;
  end;
END;

procedure TForm1.CompareINI(const OldN, NewN : TFilename;
      vAddK, vDelK, vChaK, vAddS, vDelS : TStringList);
{Take two normalized INI files and compare them, reporting
  sections and keys added, deleted, or changed}
VAR
  OldF, NewF : TextFile;
  OldL, NewL : String;
  OldC, NewC : String;
  OldK, NewK : String;
  OldV, NewV : String;

  PROCEDURE ReadNext(VAR T : TextFile; VAR Line, Sec, Key, Valu: String);
  {Read next item from file, extracting section and key.
   If at end of file, return a string of 255
   Z characters in the Sec parameter. (In the ANSI sorting
   sequence, left-bracket comes last, Z next-to-last. Since
   left-bracket is significant in INI files, we use Z)}
  VAR
    Posn : Word;
  BEGIN
    IF EoF(T) THEN
      BEGIN
        FillChar(Line, SizeOf(Line), 'Z');
        Line[0] := #255;
        Exit;
      END;
    ReadLn(T, Line);
    Posn := Pos(']', Line);
    Sec := Copy(Line, 1, Posn);
    Key := Copy(Line, Posn+1, 255);
    Posn := Pos('=', Key);
    IF Posn > 0 THEN
      BEGIN
        Valu := Copy(Key, Posn+1, 255);
        Key := Copy(Key, 1, Posn-1);
      END
    ELSE Valu := '';
  END;

BEGIN
  vAddK.Clear;
  vDelK.Clear;
  vChaK.Clear;
  vAddS.Clear;
  vDelS.Clear;
  AssignFile(OldF, OldN);
  AssignFile(NewF, NewN);
  try
    Reset(OldF);
    try try
      Reset(NewF);
      try
      ReadNext(OldF, OldL, OldC, OldK, OldV);
      ReadNext(NewF, NewL, NewC, NewK, NewV);
      WHILE NOT (EoF(OldF) AND EoF(NewF)) DO
        CASE AnsiCompareText(OldL, NewL) OF
          -1 : BEGIN
            IF (OldK<>'') AND (AnsiCompareText(OldK,NewK)=0) THEN {just a change}
              BEGIN
                vChaK.Add(OldL + ' to ' + NewV);
                ReadNext(OldF, OldL, OldC, OldK, OldV);
                ReadNext(NewF, NewL, NewC, NewK, NewV);
              END
            ELSE
              BEGIN  {old is less; was deleted}
                IF OldK='' THEN vDelS.Add(OldL)
                ELSE vDelK.Add(OldL);
                ReadNext(OldF, OldL, OldC, OldK, OldV);
              END;
          END;
          0  : BEGIN
            ReadNext(OldF, OldL, OldC, OldK, OldV);
            ReadNext(NewF, NewL, NewC, NewK, NewV);
          END;
          1  : BEGIN
            IF (NewK<>'') AND (AnsiCompareText(OldK,NewK)=0) THEN {just a change}
              BEGIN
                vChaK.Add(OldL + ' to ' + NewV);
                ReadNext(OldF, OldL, OldC, OldK, OldV);
                ReadNext(NewF, NewL, NewC, NewK, NewV);
              END
            ELSE
              BEGIN  {new is less; was added}
                IF NewK = '' THEN vAddS.Add(NewL)
                ELSE vAddK.Add(NewL);
                ReadNext(NewF, NewL, NewC, NewK, NewV);
              END;
          END;
        END;
    IF OldL <> NewL THEN
      BEGIN
        IF Length(OldL) <> 255 THEN
          IF OldK='' THEN vDelS.Add(OldL)
          ELSE vDelK.Add(OldL);
        IF Length(NewL) <> 255 THEN
          IF NewK = '' THEN vAddS.Add(NewL)
          ELSE vAddK.Add(NewL);
      END;
    finally
      CloseFile(NewF);
    end;
    except
      On EInOutError DO
        MessageBeep(0);
    end;
  finally
    CloseFile(OldF);
  end;
  except
    ON EInOutError Do
      MessageBeep(0);
  end;
END;

procedure TForm1.CompareDisk(const OldN, NewN : TFilename;
      vAdd, vDel, vCha : TStringList);
{Take two disk-layout files in the format created by the
 RememberDisk method and compare them, reporting files and
 directories added or deleted and files changed}
VAR
  OldF, NewF : TextFile;
  OldP, NewP : String;
  OldD, NewD : String4;
  OldL, NewL : Char;

  PROCEDURE ReadNext(VAR T : TextFile; VAR Pth: String;
    VAR DateSize : String4; VAR Level : Char);
  {Read next item from specified file, converting to full
   path in Pth parameter; report date/time/size data in
   DateSize parameter and subdirectory level in Level
   parameter. If at end of file, return a string of 255
   left bracket characters in the Pth parameter. (In the
   ANSI sorting sequence, left bracket comes last).}
  VAR
    TempStr  : String;
    NewLevel : Char;
    Posn     : Word;
    NewName  : String12;
  BEGIN
    IF EoF(T) THEN
      BEGIN
        FillChar(Pth, SizeOf(Pth), '[');
        Pth[0] := #255;
        {Set level to char before '0'; this handles
         case where ENTIRE DRIVE has been demolished}
        Level := '/';
        Exit;
      END;
    ReadLn(T, TempStr);
    NewLevel := TempStr[1];
    IF NewLevel = '0' THEN
      BEGIN
        Pth := Copy(TempStr,2,255);
        DateSize := '';
        Level := '0';
        Exit;
      END;
    DateSize := Copy(TempStr, 14, 4);
    Posn := 13;
    WHILE TempStr[Posn] = ' ' DO Dec(Posn);
    NewName := Copy(TempStr, 2, Posn-1);
    IF NewLevel <= Level THEN Pth := ExtractFilePath(Pth);
    WHILE NewLevel < Level DO
      BEGIN
        Pth := ExtractFilePath(Copy(Pth,1,Length(Pth)-1));
        Dec(Level);
      END;
    IF Pth[length(Pth)] <> '\' THEN Pth := Pth + '\';
    Pth   := Pth + NewName;
    Level := NewLevel;
  END;

  FUNCTION ComparePath(L1, L2 : Char; const P1, P2 : String) : Integer;
  {First compare depth of subdirectory. If same
   level, then compare paths}
  BEGIN
    IF L1 < L2 THEN
      Result := 1
    ELSE IF L1 > L2 THEN
      Result := -1
    ELSE
      {Use AnsiCompareText to be same as sorted list}
      CASE AnsiCompareText(P1, P2) OF
        -32768..-1 : Result := -1;
        0          : Result := 0;
        1..32767   : Result := 1;
      END;
  END;

BEGIN
  vAdd.Clear;
  vDel.Clear;
  vCha.Clear;
  AssignFile(OldF, OldN);
  AssignFile(NewF, NewN);
  try
    Reset(OldF);
    try try
      Reset(NewF);
      try
      OldL := '0';
      NewL := '0';
      ReadNext(OldF, OldP, OldD, OldL);
      ReadNext(NewF, NewP, NewD, NewL);
      IF ProcMsgTerminated THEN Exit;
      WHILE NOT (EoF(OldF) AND EoF(NewF)) DO
        CASE ComparePath(OldL, NewL, OldP, NewP) OF
          -1 : BEGIN {old is less; was deleted}
            vDel.Add(OldP);
            ReadNext(OldF, OldP, OldD, OldL);
          END;
          0  : BEGIN
            IF OldD <> NewD THEN
              vCha.Add(OldP);
            ReadNext(OldF, OldP, OldD, OldL);
            ReadNext(NewF, NewP, NewD, NewL);
          END;
          1  : BEGIN {new is less; was added}
            vAdd.Add(NewP);
            ReadNext(NewF, NewP, NewD, NewL);
          END;
        END;
    IF OldP <> NewP THEN
      BEGIN
        IF Length(OldP) <> 255 THEN
          vDel.Add(OldP);
        IF Length(NewP) <> 255 THEN
          vAdd.Add(NewP);
      END;
    finally
      CloseFile(NewF);
    end;
    except
      On EInOutError DO MessageBeep(0);
    end;
  finally
    CloseFile(OldF);
  end;
  except
    ON EInOutError Do MessageBeep(0);
  end;
END;

function TForm1.ExecInstall : Integer;
VAR
  fName, fDir, CmdLine : String;
  SpacePos : Word;
BEGIN
  fName := EditInstallName.Text;
  IF eType = et_Bat THEN
    fName := 'COMMAND /C ' + fName;
  (* 1.02 *)
  (* The following lines were re-ordered in version 1.02,
     to restore the ability to use command-line parameters
     with install file names. The numbers in {} brackets
     indicate the original order *)
  {3}SpacePos := Pos(' ', fName);
  {4}IF SpacePos = 0 THEN CmdLine := #0
  {5}ELSE
  {6}  BEGIN
  {7}    CmdLine := Copy(fName, SpacePos+1, 255) + #0;
  {8}    fName := Copy(fName, 1, SpacePos-1);
  {9}  END;
  {1}fDir := ExtractFilePath(fName) + #0;
  {2}fName := fName + #0;
  Result := ShellExecute(Handle, NIL, @fName[1], @CmdLine[1],
    @fDir[1], SW_SHOW);
END;

procedure TForm1.BtnDoItClick(Sender: TObject);
VAR
  Rslt, N : Integer;
begin
  InstStatus := isPrepare;
  IF tType = tmFastNotify THEN
    BEGIN
      AssignFile(FileCdrFile, OutPath+TempName+TempExt1);
      Rewrite(FileCdrFile);
      IF Not FileCdrInstall(Handle) THEN
        BEGIN
          tType := tmDiskRead;
          CloseFile(FileCdrFile);
          MessageDlg('INCTRL 2 was unable to use fast file notification. '+
            'Switching to disk contents comparison.', mtInformation,
            [mbOk], 0);
        END;
    END;
  FormStyle := fsStayOnTop;
  NotebookMain.PageIndex := 1;
  BtnPrev.Enabled := False;
  BtnNext.Enabled := False;
  BtnDoIt.Enabled := False;
  WITH BtnHelpToo DO
    BEGIN
      Self.ClientWidth := Left + Width + 1;
      Self.ClientHeight := Height + Top + 1;
    END;
  IF tType = tmDiskRead THEN RememberDisk(OutPath+TempName+TempExt1);
  IF Application.Terminated THEN Exit;
  FOR N := 0 TO IniToWatch.Count-1 DO
    BEGIN
      RememberINI(IniToWatch[N], TempExt1);
      IF Application.Terminated THEN Exit;
    END;
  IF EditInstallName.Text = '' THEN
    BEGIN
      InstStatus := isTwoPhase;
      WITH TIniFile.Create(ChangeFileExt(Application.ExeName,'.INI')) DO
        try
          WriteString('Survival', 'SurvivedRestart', 'TwoPhase');
          WriteString('Survival', 'Description', EditDescript.Text);
          WriteString('Survival', 'ReportName',
            PanelReportName.Caption);
          WriteBool('Survival', 'FastNotify', tType=tmFastNotify);
          FOR N := 0 TO DriveButtons.Count-1 DO
            BEGIN
              WriteInteger('WatchDrive', 'd'+IntToStr(N),
                TLabel(WatchLabels[N]).Tag);
              IF TLabel(WatchLabels[N]).Tag > 0 THEN
                WriteString('WatchDir', 'd'+IntToStr(N),
                  TLabel(DirLabels[N]).Caption);
            END;
        finally
          Free;
        end;
      {code duplicated in BtnDone}
      Caption := 'INCTRL 2';
      FormStyle := fsNormal;
      NotebookMain.PageIndex := 0;
      NotebookSub.PageIndex := 7;
      Self.ClientWidth := BtnCancel.Left + BtnCancel.Width + 7;
      Self.ClientHeight := BtnCancel.Top + BtnCancel.Height + 6;
      BtnClose.BringToFront;
    END
  ELSE
    BEGIN
      Rslt := ExecInstall;
      IF Rslt > 32 THEN
        BEGIN
          Caption := 'INCTRL 2 - Installation in progress';
          BtnDone.Enabled := True;
          ActiveControl := BtnDone;
          BtnAbort.Caption := 'Abort: No Report';
          InstStatus := isInstall;
        END
      ELSE
        BEGIN
          Caption := 'INCTRL 2 - Execute failed: '+
            IntToStr(Rslt);
          InstStatus := isInstallFailed;
        END;
    END;
end;

procedure TForm1.BtnDoneClick(Sender: TObject);
VAR
  OutF        : TextFile;
  TAdd, TDel,
  TCha, TAddS,
  TDelS       : TStringList;
  N           : Word;

  procedure WriteOut(L : TStringList; const S : String);
  VAR N : Word;
  BEGIN
    IF L.Count > 0 THEN
      BEGIN
        WriteLn(OutF, Format(S+'(%d)', [L.Count]));
        FOR N := 0 TO L.Count-1 DO
          WriteLn(OutF, L[N]);
        WriteLn(OutF);
      END;
  END;

  procedure StripOurOwn(L : TStringList);
  (* Version 1.01: Trouble with hand-specified INI files not
    in Windows directory. Fixed. *)
  VAR I, N : Integer;
  BEGIN
    WITH L DO
      BEGIN
        I := IndexOf(OutPath+'INCTRL2.INI');
        IF I >= 0 THEN Delete(I);
        I := IndexOf(OutPath+TempName+TempExt1);
        IF I >= 0 THEN Delete(I);
        I := IndexOf(OutPath+TempName+TempExt2);
        IF I >= 0 THEN Delete(I);
        FOR N := 0 TO IniToWatch.Count-1 DO
          BEGIN
            (*I := IndexOf(ChangeFileExt(OutPath + IniToWatch[N],TempExt1));*)
            I := IndexOf(ChangeFileExt(OutPath + ExtractFileName(
              IniToWatch[N]), TempExt1)); (* 1.01 *)
            IF I >= 0 THEN Delete(I);
            (*I := IndexOf(ChangeFileExt(OutPath + IniToWatch[N],TempExt2));*)
            I := IndexOf(ChangeFileExt(OutPath + ExtractFileName(
              IniToWatch[N]),TempExt2)); (* 1.01 *)
            IF I >= 0 THEN Delete(I);
          END;
      END;
  END;

  procedure StripGRPs;
  {If Windows was restarted, ALL the .GRP files will
   be reported as having changed. That's pretty useless,
   so we strip those from the report. Likewise, changes
   to the permanent swap file 386SPART.PAR are not of
   interest to us}
  VAR
    N : Word;
    (*WindowsDir : String[145];*) (* 1.02 *)
    (* made WindowsDir a "global" field of main form object,
      initialized in FormCreate *)
  BEGIN
    IF TCha.Count = 0 THEN Exit;
    (*GetWindowsDirectory(@WindowsDir[1], 145);
    WindowsDir[0] := Char(StrLen(@WindowsDir[1]));
    IF WindowsDir[Length(WindowsDir)] <> '\' THEN
      WindowsDir := WindowsDir + '\';*)
    FOR N := TCha.Count-1 DOWNTO 0 DO
      BEGIN
        IF (ExtractFilePath(TCha[N])=WindowsDir) AND
          (ExtractFileExt(TCha[N])='.GRP') THEN
            TCha.Delete(N);
        IF ExtractFileName(TCha[N])='386SPART.PAR' THEN
          TCha.Delete(N);
      END;
  END;

  procedure ReportINI(const RealN : String);
  (* Version 1.01: Errors occurred if a hand-specified
     INI file was not in the Windows directory. Fixed here. *)
  BEGIN
(*    CompareIni(ChangeFileExt(OutPath+RealN, TempExt1),
      ChangeFileExt(OutPath+RealN, TempExt2),
      TAdd, TDel, TCha, TAddS, TDelS);*)
    CompareIni(                                                (* 1.01 *)
      ChangeFileExt(OutPath+ExtractFileName(RealN), TempExt1), (* 1.01 *)
      ChangeFileExt(OutPath+ExtractFileName(RealN), TempExt2),  (* 1.01 *)
      TAdd, TDel, TCha, TAddS, TDelS);
    IF TAddS.Count + TDelS.Count + TAdd.Count + TDel.Count
       + TCha.Count = 0 THEN
      BEGIN
        WriteLn(OutF, 'NO CHANGES MADE TO '+RealN+'...');
        WriteLn(OutF);
      END
    ELSE
      BEGIN
        WriteLn(OutF, 'CHANGES MADE TO '+RealN+'...');
        WriteOut(TAddS,'SECTIONS ADDED TO '+RealN+': ');
        WriteOut(TDelS,'SECTIONS DELETED FROM '+RealN+': ');
        WriteOut(TAdd,'KEYS ADDED TO '+RealN+': ');
        WriteOut(TDel,'KEYS DELETED FROM '+RealN+': ');
        WriteOut(TCha,'KEYS CHANGED IN '+RealN+': ');
      END;
  END;

  procedure WriteHeader;
  VAR N : Word;
  BEGIN
    WriteLn(OutF, 'Installation report: '+EditDescript.Text);
    (* 1.02 - fixed line below, which still said "beta version"! *)
    WriteLn(OutF, '    (generated by INCTRL 2, version 1.02)');
    WriteLn(OutF, FormatDateTime('dddd, mmmm d  hh:mm AM/PM', Now));
    WriteLn(OutF, fpWinVerLabel.Caption);
    Write(OutF, 'Notification by ');
    CASE tType OF
      tmFastNotify : WriteLn(OutF, 'fast file notification');
      tmDiskRead : BEGIN
        WriteLn(OutF, 'disk contents comparison, tracking:');
        FOR N := 0 TO DriveButtons.Count-1 DO
          IF TLabel(WatchLabels[N]).Tag > 0 THEN
             WriteLn(OutF, '    ', Uppercase(TLabel(DirLabels[N]).Caption));
      END;
    END;
    WriteLn(OutF);
  END;

  procedure ProcessLists; (* FIG09.TXT *)
  VAR
    N    : Word;
    I    : Integer;
    S    : String;
    Code : Word;
  BEGIN
    TAddS.LoadFromFile(OutPath+TempName+TempExt1);
    IF TAddS.Count = 0 THEN Exit;
    FOR N := 0 TO TAddS.Count-1 DO
      BEGIN
        S := TAddS[N];
        Code := StrToIntDef('$'+Copy(S,1,2), 0);
        S := Copy(S, 3, 255);
        CASE Code OF
       {DOS, WIN}
          0, $3C,  {Create file}
             $5A,  {Create unique file}
                   {from DOS box, this function comes as 0}
             $5B,  {Create new file}
                   {from DOS box, this function comes as 0}
          7, $39,  {Create directory}
             $6C : {Extended open}
                   {from DOS box, this function is ignored}
                   TAdd.Add(S);
          1, $41,  {Delete file}
          8, $3A : {Delete directory}
                   BEGIN
                     I := TAdd.IndexOf(S);
                     {if we're deleting a file that was added...}
                     IF I >= 0 THEN
                       {...then just don't report it as added}
                       TAdd.Delete(I)
                     ELSE TDel.Add(S);
                   END;
          2, $56 : TCha.Add(S); {Rename file/dir);}
          3, $43 : ;{Get/set file attrs}
                   {no event for GET file attr from a WinApp}
             $57 : ;{Set file date/time}
                   {Schulman mentions this, but it doesn't happen}
        END;
      END;
    TAddS.Clear;
  END;

begin
  InstStatus := isGather;
  IF tType = tmFastNotify THEN
    BEGIN
      FileCdrUninstall(Handle);
      CloseFile(FileCdrFile);
    END
  ELSE RememberDisk(OutPath+TempName+TempExt2);
  IF Application.Terminated THEN Exit;
  FOR N := 0 TO IniToWatch.Count-1 DO
    BEGIN
      RememberINI(IniToWatch[N], TempExt2);
      IF Application.Terminated THEN Exit;
    END;
  TAdd := TStringList.Create;
  TDel := TStringList.Create;
  TCha := TStringList.Create;
  TAddS := TStringList.Create;
  TDelS := TStringList.Create;
  try
    IF tType = tmDiskRead THEN
      BEGIN
        CompareDisk(OutPath+TempName+TempExt1,
          OutPath+TempName+TempExt2,
          TAdd, TDel, TCha);
        IF InstStatus = isRestart THEN
          StripGRPs;
      END
    ELSE ProcessLists;
    StripOurOwn(TAdd);
    StripOurOwn(TCha);
    AssignFile(OutF, PanelReportName.Caption);
    Rewrite(OutF);
    try
      WriteHeader;
      IF TAdd.Count+TDel.Count+TCha.Count = 0 THEN
        BEGIN
          IF tType = tmDiskRead THEN
            WriteLn(OutF, 'NO CHANGES IN TRACKED DISKS')
          ELSE
            WriteLn(OutF, 'NO CHANGE NOTIFICATIONS RECEIVED');
          WriteLn(OutF);
        END
      ELSE
        BEGIN
          WriteOut(TAdd,'FILES AND DIRECTORIES ADDED: ');
          WriteOut(TDel,'FILES AND DIRECTORIES DELETED: ');
          IF tType = tmDiskRead THEN
            WriteOut(TCha,'FILES CHANGED: ')
          ELSE
            WriteOut(TCha,'FILES AND DIRECTORIES RENAMED: ');
        END;
      FOR N := 0 TO IniToWatch.Count-1 DO
        ReportINI(IniToWatch[N]);
    finally
      CloseFile(OutF);
    end;
  finally
    TAdd.Free;
    TDel.Free;
    TCha.Free;
    TAddS.Free;
    TDelS.Free;
  end;
  Caption := 'INCTRL 2';
  FormStyle := fsNormal;
  NotebookMain.PageIndex := 0;
  NotebookSub.PageIndex := 6;
  Self.ClientWidth := BtnCancel.Left + BtnCancel.Width + 7;
  Self.ClientHeight := BtnCancel.Top + BtnCancel.Height + 6;
  Memo1.Lines.LoadFromFile(PanelReportName.Caption);
  ActiveControl := Memo1; (* 1.02 *)
  BtnClose.BringToFront;
  InstStatus := isDone;
end;

procedure TForm1.FormActivate(Sender: TObject);
{The form's handle is not available at OnCreate time,
 so we check in OnActivate and attach our FileCDR
 if necessary}
begin
  IF InstStatus=isRestart THEN
    BEGIN
      IF tType=tmFastNotify THEN
        BEGIN
          AssignFile(FileCdrFile, OutPath+TempName+TempExt1);
          try
            Append(FileCdrFile);
          except
            ON EInOutError DO
              Rewrite(FileCdrFile);
          end;
          IF NOT FileCdrInstall(Handle) THEN
            MessageDlg('INCTRL 2 was unable to re-activate' +
              'Fast File Notification after Windows restart.' +
              'File changes occuring after this point will not' +
              'be recorded.', mtError, [mbOk, mbHelp],
              Cannot_Reactivate_Fast_File_Notification);
        END;
    END
  ELSE
    BEGIN
      IF NOT CanRestart THEN
        MessageDlg('INCTRL 2 will not be able to handle install '+
          'programs that restart Windows. Press Help for explanation.',
          mtWarning, [mbOk, mbHelp], Cannot_Survive_Restart);
    END;
end;

procedure TForm1.BtnHelpTooClick(Sender: TObject);
begin
  CASE InstStatus OF
    isNotyet        : ;
    isPrepare       : Application.HelpContext(Preparing_for_Installation);
    isInstall       : Application.HelpContext(Installation_In_Progress);
    isInstallFailed : Application.HelpContext(Execute_Failed);
    isRestart       : Application.HelpContext(Installation_After_Restart);
    isGather        : Application.HelpContext(Gathering_Post_Install_Data);
    isDone          : ;
  END;
end;

procedure TForm1.FormClose(Sender: TObject; var Action: TCloseAction);
{When the application is shut down via Windows restart or
 termination, FormClose does NOT get called. However, the
 wmEndSession handler specifically calls FormClose if the
 restart is NOT happening in the middle of an install}
VAR
  N : Word;
  S : String;
begin
  Application.HelpCommand(HELP_QUIT, 0);
  IF InstStatus = isTwoPhase THEN Exit;
  DeleteFile(OutPath+TempName+TempExt1);
  DeleteFile(OutPath+TempName+TempExt2);
  FOR N := 0 TO IniToWatch.Count-1 DO
    BEGIN
      S := ChangeFileExt(OutPath + ExtractFileName(IniToWatch[N]),
        TempExt1);
      DeleteFile(S);
      S := ChangeFileExt(S, TempExt2);
      DeleteFile(S);
    END;
end;

procedure TForm1.BtnHelpClick(Sender: TObject);
begin
  WITH NotebookSub, Pages.Objects[PageIndex] AS TPage DO
    Application.HelpContext(HelpContext);
end;

procedure TForm1.FFNotifyLabelDblClick(Sender: TObject);
begin
  IF (Platform  = WINVER_NT) AND
    (NOT FFCheckBox.Enabled) THEN
    BEGIN
      FFCheckBox.Enabled := True;
      FFCheckBox.Checked := True;
      FFNotifyLabel.Caption := 'available';
    END;
end;

end.

