/////////////////////////////////////////////////////////////////
// RESTART.C - (c) 1993, Jeffrey M. Perkel
//		All rights reserved.
//
// Restart is a utility that allows the user to quit or restart
// Windows.
//
// Usage:  RESTART [switches]
//               "R" -- Restart
//               "E" -- Exit
//               "S" -- Reboot System
//				 "C" -- Turn confirmation OFF (default == ON)
/////////////////////////////////////////////////////////////////

#include <windows.h>
#include "ctl3d.h"
#include "restart.h"

// Global Variables

static HINSTANCE ghInstance;
static char szAppName[] = "Restart";
static int Action, Confirm = 1, SaveOnExit, Answer, iLastChoice;
static DWORD dwWinVer;
static BOOL bSuccess, bWin31;
static LPSTR lpCommandLine;

// Exported Functions

BOOL FAR PASCAL _export MainDlgProc (HWND hDlg, unsigned message, WORD
        wParam, LONG lParam);
BOOL FAR PASCAL _export AboutDlgProc (HWND hAboutDlg, UINT message, UINT
        wParam, LONG lParam);

// Internal Functions

BOOL ParseCmdLine (LPSTR lpCommandLine);
BOOL MakeItSo (int Action);
int AreYouSure (HWND hDlg, int Action);
int ErrorHandler (HWND hwnd, int iError);
BOOL DoWriteWinIni (HWND hDlg);
int GetYesNo (HWND hwnd, char *lpszPrompt);
void CenterDialog (HWND hDlg);

// Program Entry Point

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
        LPSTR lpCmdLine, int nCmdShow)
        {
        BOOL bCmdLineParam = FALSE;
        HWND FirsthWnd;
        
        ghInstance = hInstance;
        lpCommandLine = lpCmdLine;

// If another instance is running, bring to front and terminate
// new instance.
        
        if (FirsthWnd = FindWindow (szAppName, NULL))
                {
                BringWindowToTop (FirsthWnd);
                ShowWindow (FirsthWnd, SW_SHOWNORMAL);
                return FALSE;
                }
                 
        if (!hPrevInstance)
                {
                WNDCLASS wndclass;
                
                wndclass.style = CS_HREDRAW | CS_VREDRAW;
                wndclass.lpfnWndProc = DefDlgProc;
                wndclass.cbClsExtra = 0;
                wndclass.cbWndExtra = DLGWINDOWEXTRA;
                wndclass.hInstance = ghInstance;
                wndclass.hIcon = LoadIcon (ghInstance, szAppName);
                wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);
                wndclass.hbrBackground = GetStockObject (WHITE_BRUSH);
                wndclass.lpszMenuName = NULL;
                wndclass.lpszClassName = szAppName;                
        
                if (!RegisterClass (&wndclass)) {
                        ErrorHandler (NULL, ERR_REGISTERCLASSFAIL);
                        return FALSE;
                        }
                }

		Ctl3dRegister (ghInstance);
        Ctl3dAutoSubclass (ghInstance);

// Determine the version of Windows which is running

        dwWinVer = GetVersion();
        if ((LOBYTE(LOWORD(dwWinVer)) >= 3) && (HIBYTE(LOWORD(dwWinVer))
                >= 1)) bWin31 = TRUE;
        else bWin31 = FALSE;                
        
// Determine if RESTART was invoked using command line parameters


        bCmdLineParam = (ParseCmdLine (lpCommandLine));
        if (bCmdLineParam == TRUE) {
                if (Confirm) { 
                	Answer = AreYouSure ((HWND) NULL, Action);
                	if (Answer == IDYES) MakeItSo (Action);
                	else return 1;
                }
                else MakeItSo (Action);
                return 1;
                }

// Determine if Confirmation status was stored in WIN.INI

        Confirm = GetProfileInt (szAppName, "ConfirmationOn", 1);
        iLastChoice = GetProfileInt (szAppName, "LastOption", 1);
        Action = ((iLastChoice == 0) ? IDD_EXIT : (iLastChoice == 1)
                ? IDD_RESTART : (iLastChoice == 2) ? 
                IDD_REBOOT : IDD_RESTART);
 
        return DialogBoxParam (ghInstance, szAppName, NULL, (DLGPROC)
                MakeProcInstance ((FARPROC) MainDlgProc, ghInstance),
                (LONG) nCmdShow);
}

///////////////////////////////////////////////////////////////
// MainDlgProc - Dialog Procedure for the Main Program Window
///////////////////////////////////////////////////////////////

BOOL FAR PASCAL _export MainDlgProc (HWND hDlg, unsigned message, WORD
        wParam, LONG lParam)
{
        switch (message)
                {
                case WM_INITDIALOG:                  
                		Ctl3dSubclassDlg (hDlg, CTL3D_ALL);
                		CenterDialog (hDlg);
                        ShowWindow (hDlg, LOWORD(lParam));
                        if (Confirm) SendDlgItemMessage (hDlg, IDD_CONFIRM_ON,
                                BM_SETCHECK, TRUE, 0L);
                        SendDlgItemMessage (hDlg, IDD_SAVEONEXIT,
                                BM_SETCHECK, TRUE, 0L);
                        SendDlgItemMessage (hDlg, Action, BM_SETCHECK,
                                TRUE, 0L);
                                
                        // EW_REBOOTSYSTEM flag not valid in Windows 3.0!
                        // Disable option to ensure compatibility.
                        
                        if (!bWin31) EnableWindow (GetDlgItem (hDlg,
                                IDD_REBOOT), FALSE);                                
                        break;
                
                case WM_CTLCOLOR:
                		return Ctl3dCtlColorEx (message, wParam, lParam);
                		break;
                
                case WM_SYSCOLORCHANGE:
                	Ctl3dColorChange ();
                	break;    
                	
                case WM_SETTEXT:
                case WM_NCPAINT:
                case WM_NCACTIVATE:
                	SetWindowLong (hDlg, DWL_MSGRESULT,
                		Ctl3dDlgFramePaint (hDlg, message, wParam, lParam));
                	return TRUE;
                	break;
                
                case WM_CLOSE:
                        EndDialog (hDlg, FALSE);
                        Ctl3dUnregister (ghInstance);
                        break;
                
                case WM_COMMAND:
                        switch (wParam)
                                {
                              case ID_OK:
                                {
                                Confirm = (IsDlgButtonChecked (hDlg,
                                    IDD_CONFIRM_ON)) ? 1 : 0;
                                SaveOnExit = (IsDlgButtonChecked (hDlg,
                                    IDD_SAVEONEXIT)) ? 1 : 0;
                                Action = (IsDlgButtonChecked (hDlg,
                                    IDD_RESTART)) ? EW_RESTARTWINDOWS :
                                    (IsDlgButtonChecked (hDlg, IDD_EXIT)) ?
                                    NULL : (IsDlgButtonChecked (hDlg,
                                    IDD_REBOOT)) ? EW_REBOOTSYSTEM : NOCHECK;
                                if (Action == NOCHECK) 
                                        {
                                        MessageBeep(0);
                                        break;
                                        }
                                if (Confirm)
                                        {
                                        Answer = AreYouSure (hDlg, Action);
                                        if (Answer != IDYES) break;
                                        }
                                if (SaveOnExit) {
                                        bSuccess = DoWriteWinIni(hDlg);
                                        if (!bSuccess) {
                                Answer = GetYesNo (hDlg, 
                                   "WriteProfileString() failed! Continue?");
                                if (Answer != IDYES) break;
                                                       }
                                                    }
                                bSuccess = MakeItSo (Action);
                                if (!bSuccess) {
                                        ErrorHandler (hDlg,
                                        ERR_APPREFUSETERMINATE);
                                        }
                                break;
                               }
                              case ID_CANCEL:
                                        EndDialog (hDlg, FALSE);
                                        break;
                                        
                              case ID_ABOUT:
                                        {
                                        FARPROC lpfnAboutDlgProc;
                                        
                                        lpfnAboutDlgProc = 
                                                MakeProcInstance((FARPROC)
                                                AboutDlgProc, ghInstance); 
                                        DialogBox(ghInstance, 
                                                MAKEINTRESOURCE (AboutBox),
                                                hDlg, (DLGPROC) 
                                                lpfnAboutDlgProc);
                                        FreeProcInstance (
                                                lpfnAboutDlgProc);
                                        break;         
                                        }
                                break;
                                } 
                                break;       
                              default:
                                        return FALSE;
                                }
                                
                        return TRUE;
                        }
                        
                        
///////////////////////////////////////////
// AboutDlgProc - About... box proceedure
///////////////////////////////////////////

BOOL FAR PASCAL _export AboutDlgProc (HWND hAboutDlg, UINT message,
        UINT wParam, LONG lParam)
        
{
        switch (message)
                {
                case WM_INITDIALOG:
                        Ctl3dSubclassDlg (hAboutDlg, CTL3D_ALL);
                        break;
                        
                case WM_CTLCOLOR:
                		return Ctl3dCtlColorEx (message,wParam, lParam);
                		break;
                		
                case WM_SETTEXT:
                case WM_NCPAINT:
                case WM_NCACTIVATE:
                		SetWindowLong (hAboutDlg, DWL_MSGRESULT,
                			Ctl3dDlgFramePaint (hAboutDlg, message,wParam, lParam));
                		return TRUE;
                		break;
                		
                
                case WM_COMMAND:
                        switch (wParam)
                                {
                                case ID_OK:
                                        EndDialog (hAboutDlg, 0);
                                        return TRUE;
                                }
                        break;
                }
        return FALSE;
}
/////////////////////////////////////////////////////////////
// MakeItSo - This function is the "meat" of the program.
//      Passes user request to ExitWindows() API
/////////////////////////////////////////////////////////////                                        

BOOL MakeItSo (int Action)
{
        BOOL bResult;
            
        bResult = ExitWindows (Action, 0);
        return bResult;
}                                                  

/////////////////////////////////////////////////////////////
// AreYouSure - Confirm the user's response.
/////////////////////////////////////////////////////////////

int AreYouSure (HWND hDlg, int Action)
{
        char szBuffer[50];
        short nAnswer;
                
        if (Action == EW_REBOOTSYSTEM) 
                wsprintf (szBuffer, "Exit the Windows System and Reboot?");
        else wsprintf (szBuffer, "%s the Windows System?", (LPSTR) (Action == 
            EW_RESTARTWINDOWS ? "Restart" : "Exit"));
        nAnswer = MessageBox (hDlg, szBuffer,
            "Confirmation", MB_YESNOCANCEL | MB_ICONQUESTION);
        return nAnswer;
}

/////////////////////////////////////////////////////////////
// ParseCmdLine - Parses the command line
/////////////////////////////////////////////////////////////

BOOL ParseCmdLine (LPSTR lpCommandLine) {

	BOOL 	fRetValue = FALSE,
		 	fActionPicked = FALSE;
	
    if (lpCommandLine == 0) return FALSE;
    
    while ((*lpCommandLine == '/') || (*lpCommandLine == '-')) {
           lpCommandLine++;
                switch (tolower (*lpCommandLine)) {
                   case 'r':
                        if (fActionPicked) break;
                        Action = EW_RESTARTWINDOWS;
                        fActionPicked = TRUE;
                        fRetValue = TRUE;
                        break;
                   case 'e':
                        if (fActionPicked) break;
                        Action = NULL;       
                        fActionPicked = TRUE;
                        fRetValue = TRUE;
                        break;         
                   case 's':
                        if (fActionPicked) break;
                        Action = EW_REBOOTSYSTEM;
                        fActionPicked = TRUE;
                        fRetValue = TRUE;
                        break;
                   case 'c':
                   		Confirm = 0;
                   		fRetValue = TRUE;
                        break;
                   default:
                        ErrorHandler (NULL, ERR_BADCMDLINESWITCH);
                 } // switch  
                do {
                	lpCommandLine++;
                } while ((*lpCommandLine != '\0') && (*lpCommandLine <= ' '));
                   
        }  // while...
    if ((Confirm == 0) && (fActionPicked == FALSE)) { 
    	Confirm == 1;
    	fRetValue = FALSE;
    }
	return fRetValue;
}

/////////////////////////////////////////////////////////////
// Error Handler Routine
// Parameters:  hwnd - handle to the parent window
//              iError - error code
// To use this function, define a STRINGTABLE resource
// in the .RC file containing iError and the message.
/////////////////////////////////////////////////////////////

int ErrorHandler (HWND hwnd, int iError)
{
        char szBuffer[50];
        
        LoadString (ghInstance, iError, szBuffer, sizeof (szBuffer));
        return MessageBox (hwnd, szBuffer, "Error!", MB_ICONEXCLAMATION |
                MB_OK); 
}              

/////////////////////////////////////////////////////////////
// DoWriteWinIni -- Save Confirmation Setting in WIN.INI
/////////////////////////////////////////////////////////////

BOOL DoWriteWinIni (HWND hDlg)
{
        BOOL fSuccess1, fSuccess2;
        char szBuffer[3];
        
        wsprintf (szBuffer, ((Action == NULL) ? "0" : 
                (Action == EW_RESTARTWINDOWS) ? "1" : 
                (Action == EW_REBOOTSYSTEM) ? "2" : "1"));
        fSuccess1 = WriteProfileString (szAppName, "ConfirmationOn",  ((Confirm) ? "1" : "0"));
        fSuccess2 = WriteProfileString (szAppName, "LastOption", 
                szBuffer);
        if ((fSuccess1 == 0) || (fSuccess2 == 0)) {
                ErrorHandler (hDlg, ERR_WRITEPROFILESTRINGFAIL);
                return FALSE;
                }
}

int GetYesNo (HWND hwnd, char *lpszPrompt)
{
        int iResult;
        
        iResult = MessageBox (hwnd, lpszPrompt, szAppName, 
                MB_YESNO | MB_ICONQUESTION);
                
        return (iResult);
}                               

void CenterDialog (HWND hDlg)
{
	
	HWND hWnd;
	RECT hdlgRect;          
	RECT hwndRect;
	int dlgWidth, dlgHeight;
	int wndWidth, wndHeight;
	
	hWnd = GetDesktopWindow();
	
	GetWindowRect (hDlg, &hdlgRect);
	GetWindowRect (hWnd, &hwndRect);
	
		// determine dimensions of dlg box...
	dlgWidth = (hdlgRect.right - hdlgRect.left);
	dlgHeight = (hdlgRect.bottom - hdlgRect.top);
	
		// determine dimensions of main window...
	wndWidth = (hwndRect.right - hwndRect.left);
	wndHeight = (hwndRect.bottom - hwndRect.top);
	
	if (!MoveWindow (hDlg, (wndWidth - dlgWidth)/2, (wndHeight - dlgHeight)/2,
		dlgWidth, dlgHeight, TRUE)) {
			ErrorHandler (hWnd, ERR_CENTERDLGFAIL);
			return;
		}
} // CenterDialog			
