/* //////////////////////////////////////////////////////////////////////

  Module:   Xwinsave.cpp - A Microsoft Windows screen saver using the
                           screen savers from the X11 program Xlock.c

  Author:   Perry K. Sloope

  Acknowledgements:  A few pieces of the code in this file were taken
                     directly from xlock.c by Patrick J. Naughton.
                     In keeping with his copyright directives please note the
                     following:

       Xlock.c  Copyright (c) 1988-91 by Patrick J. Naughton.

       Permission to use, copy, modify, and distribute this software and its
       documentation for any purpose and without fee is hereby granted,
       provided that the above copyright notice appear in all copies and that
       both that copyright notice and this permission notice appear in
       supporting documentation.

       This file is provided AS IS with no warranties of any kind.  The author
       shall have no liability with respect to the infringement of copyrights,
       trade secrets or any patents by this file or any part thereof.  In no
       event will the author be liable for any lost revenue or profits or
       other special, indirect and consequential damages.

  The above notice also applies to this file and program. Now that that is out
  of the way, the pieces of code that were adapted from xlock.c are primarly
  the struct LockStruct LockProcs and a few pieces of code that use this
  structure, (the way in which the callback and init function pointers are used.)
  The manner in which LockStruct is designed and used is really a nice piece
  of work.

  The X11 screen saver modules (pyro.c, flame.c, etc.) are the original X11
  code.  One or two of the modules were modifed slightly to account for
  the differences in int and long variables. (int and long seem to both
  be 4 bytes in X11, at least for the Linux Gcc version.) The remainder
  were untouched.

  This was built and compiled using Borland 4.5 C++.  However I tried to design
  the code so that it was independent of the compiler.  It should be possible
  to compile this using any C++ compiler that can produce MS Windows 3.1
  executables.  No special library files or dlls are required other than
  those needed to build any MS Windows program.

//////////////////////////////////////////////////////////////////////  */


#include "xwinsave.h"
#include"ssclass.hpp"


LockStruct LockProcs[] =
{
    {"qix", initqix, drawqix, 30, 64, 1.0, "Spinning lines a la Qix(tm)"},
    {"pyro", initpyro, drawpyro, 10, 25, 1.0, "Fireworks"},
    {"flame", initflame, drawflame, 10, 20, 1.0, "Cosmic Flame Fractals"},
    {"swarm", initswarm, drawswarm, 10, 100, 1.0, "Swarm of bees"},
    {"hop", inithop, drawhop, 0, 1000, 1.0, "Hopalong iterated fractals"},
/*    {"life", initlife, drawlife, 1000000, 100, 1.0, "Conway's game of Life"}, */
    {"rotor", initrotor, drawrotor, 10, 4, 0.4, "Tom's Roto-Rooter"},
    {"worm", initworm, drawworm, 10, 20, 1.0, "Wiggly Worms"},
    {"spline", initspline, drawspline, 20, 6, 0.4, "Moving Splines"},
    {"galaxy", initgalaxy, drawgalaxy, 10, 3, 1.0, "Spinning galaxies"},
    {"kaleid", initkaleid, drawkaleid, 100, 4, 1.0, "Kaleidoscope"},
//    {"grav", initgrav, drawgrav, 10, 10, 1.0, "Orbiting planets"},
/*    {randomstring, NULL, NULL, 0, 0, 0.0, "Random mode"},
    {"blank", initblank, drawblank, 5000000, 1, 1.0, "Blank screen"},
    {"sphere", initsphere, drawsphere, 10000, 1, 1.0, "Shaded spheres"},
    {"hyper", inithyper, drawhyper, 10000, 1, 1.0, "Spinning Tesseract"}, */
    {"helix", inithelix, drawhelix, 1000, 1, 1.0, "Helix"},
/*
    {"rock", initrock, drawrock, 30000, 100, 1.0, "Asteroid field"},
    {"blot", initblot, drawblot, 10000, 6, 0.4, "Rorschach's ink blot test"},
    {"grav", initgrav, drawgrav, 10000, 10, 1.0, "Orbiting planets"},
    {"bounce", initbounce, drawbounce, 10000, 10, 1.0, "Bouncing ball"},
    {"world", initworld, drawworld, 100000, 8, 0.3, "Random Spinning Earths"},
    {"rect", initrect, drawrect, 10000, 100, 1.0, "Greynetic rectangles"},
    {"bat", initbat, drawbat, 10000, 6, 1.0, "Flying Bats"},
    {"image", initimage, drawimage, 2000000, 8, 0.3, "Random bouncing image"}, */
};

// These globals are expected by some of the screen saver modules.
perscreen Scr[MAXSCREENS];
long batchcount = 16;
int screen;     /* Pyro.c expects these. They 're placebos. */
int mono = 0, deflt;
Display *dsp = NULL;
HDC Ghdc;

char AppName[] = "ScreenSaver.XwinSave";

#define NUMPROCS (sizeof LockProcs / sizeof LockProcs[0])

BOOL FAR PASCAL _export ConfigDlgFunc (HWND hDlg, UINT message,
                                                 UINT wParam, LONG lParam);

class XwinSave : public ScreenSave
{
     protected:
        Window win;  /* window used to cover screen */
        int timedelay;
        void (*callback) (Window);
        void (*init) (Window);

     public:
        virtual void ShowScreenSaver();
        virtual int getSelection(char *NameSelect);
        XwinSave(char *AppNm, HANDLE hInst, HANDLE hPrevInst, LPSTR CmdLine,
                int CmdShow);
        ~XwinSave();
        virtual void InitSS();  // This is called in RunScreenSaver().
};



XwinSave::XwinSave(char *AppNm, HANDLE hInst, HANDLE hPrevInst, LPSTR CmdLine,
                  int CmdShow):ScreenSave(AppNm,hInst,hPrevInst,CmdLine,CmdShow)
{
    screen = 0;
    callback = NULL;
    init = NULL;
    DlgName = "CONFIGDIALOG";
    pConfigDlg = ::ConfigDlgFunc;
    Scr[screen].gc = NULL;

}


XwinSave::~XwinSave()
{
    if (Scr[screen].gc != NULL)
       free (Scr[screen].gc);
}


int XwinSave::getSelection(char *NameSelect)
{
     int i = 0;
     while (i < NUMPROCS)
     {
        if (strncmp(LockProcs[i].cmdline_arg,NameSelect,strlen(LockProcs[i].cmdline_arg)) == 0)
           break;
        else
          i++;
     }
     if (i >= NUMPROCS)
       i = random() % (NUMPROCS);
     return i;
}


void XwinSave::InitSS()
{
     HPEN pen;
     int i;
     XGCValues   xgcv;
     int selection = 3;
     u_char red[64],green[64],blue[64];
     char NameSelect[MAXSELECTION_NAMESZ];

     srandom((unsigned)time(NULL));
     win.hdc = hdc;
     win.hWnd = hWnd;
     win.hbrBackground = wndclass.hbrBackground;
     screen = 0;
     GetPrivateProfileString(AppName,DEFSCRNSAVE, "random",NameSelect,sizeof(NameSelect),INI_FILE);
     if (NameSelect[0] == NULL)
       strcpy(NameSelect,"random");
     selection = getSelection(NameSelect);

     Ghdc = win.hdc;
     SetBkColor(win.hdc,0);
     pen = GetStockObject (WHITE_PEN);
     SelectObject(win.hdc,pen);
     callback = LockProcs[selection].lp_callback;   // Screen we will use this time.
     init = LockProcs[selection].lp_init;

     timedelay = LockProcs[selection].def_delay;
     batchcount = LockProcs[selection].def_batchcount;

     Scr[screen].gc = XCreateGC(dsp, win,
                      GCForeground | GCBackground, &xgcv);
     Scr[screen].npixels = 0;
     hsbramp(0.0, 1, 1.0, 1.0, 1, 1.0, 64, red, green, blue);
     for (i = 0; i < 64; i++)
     {
       Scr[screen].pixels[i] = RGB (red[i],green[i],blue[i]);
       Scr[screen].npixels++;
     }
     init(win);
}


void XwinSave::ShowScreenSaver()
{
     callback(win);
     usleep(timedelay);
}


// This is the callback function for the Setup dialog invoked when the
// program is executed with the -c option.  If passwording is added, this
// is one of the places to put it.

BOOL FAR PASCAL _export ConfigDlgFunc (HWND hDlg, UINT message,
                                                 UINT wParam, LONG lParam)
{
     static HWND  hSelectSS;
     static char  selection[MAXSELECTION_SZ];
     WORD i,deflt = 0;
     char buf[MAXSELECTION_SZ],defName[MAXSELECTION_NAMESZ];

     switch (message)
     {
          case WM_INITDIALOG:
               hSelectSS = GetDlgItem (hDlg, ID_SELECTSCREEN);
               GetPrivateProfileString(AppName,DEFSCRNSAVE, "random",defName,sizeof(defName),INI_FILE);
               SendMessage (hSelectSS, LB_INSERTSTRING, 0,
                                 (LONG) (LPSTR)"random   Random selection of saver");
               for (i = 1; i <= NUMPROCS; i++)
               {
                    sprintf(buf,"%s\t%s",LockProcs[i-1].cmdline_arg,LockProcs[i-1].desc);
                    SendMessage (hSelectSS, LB_INSERTSTRING, i,
                                 (LONG) (LPSTR)buf);
                    if (strncmp(LockProcs[i-1].cmdline_arg,defName,strlen(LockProcs[i-1].cmdline_arg)) == 0)
                      deflt = i;
               }
               SendMessage (hSelectSS, LB_SETCURSEL, deflt,0);
               SetFocus (hSelectSS);
               return FALSE;

          case WM_COMMAND:
               switch (wParam)
               {
                    case IDOK:
                         i = (WORD) SendMessage (hSelectSS, LB_GETCURSEL, 0, 0L);
                         i = (WORD) SendMessage (hSelectSS, LB_GETTEXT, i,
                                            (LONG) (LPSTR) selection);
                         WritePrivateProfileString(AppName, DEFSCRNSAVE, selection, INI_FILE);
                         EndDialog (hDlg, TRUE) ;
                         return TRUE;

                    case IDCANCEL:
                         EndDialog (hDlg, FALSE);
                         return TRUE;

                    case ID_SELECTSCREEN:
                         return TRUE;
                    }
               break;
          }
     return FALSE;
}



int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow)
{
    XwinSave *TheScreenSaver =  new XwinSave(AppName,hInstance,hPrevInstance,lpszCmdLine, nCmdShow);
    TheScreenSaver->RunScreenSaver();
    delete TheScreenSaver;
    return 1;
}
