/* *******************************************************************

The Oracle SQL Class Library Version 1.0,
Author: Sergei Kuchin
Copyright (C) Mosakin Corp., 1995
This library is free software.  Permission to use, copy,
modify and redistribute the Oracle SQL class library for any
purpose is hereby granted without fee, provided that the
above copyright notice appear in all copies.


******************************************************************* */
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <oscl.h>

ora_text OCI::msg_buf[1000]={0};
uword OCI::err_code=0;

int Debug_Codes[]={
 32000,
 32001,
 32002,
 32003,
 32004
};

static char* DebugMsg[]={
 "ORA-32000: Incompatible data types in stream operation",
 "ORA-32001: Row must be full for flushing output stream",
 "ORA-32002: Invalid host variable declaration ==> ",
 "ORA-32003: Not all input variables are initialized",
 "ORA-32004: No input variables were defined in SELECT statement"
};

#ifdef __EXCEPTION_ENABLED__
short oraException::nesting_lvl=0;
short oraException::raise_mode=0;

#define INC_LVL     oraException::inc_lvl()
#define DEC_LVL     oraException::dec_lvl()
#define CHECK_ERR   RAISE;DEC_LVL;
#define RAISE2					\
 if(oraException::get_raise_mode()){		\
  EXCEPTION_DISABLE;				\
  throw oraException();				\
 }

#else

#define INC_LVL
#define DEC_LVL
#define CHECK_ERR RAISE;DEC_LVL 
#define RAISE2    				\
 if(except_flag){				\
  EXCEPTION_DISABLE;				\
  raise2();					\
 }

#include <setjmp.h>

jmp_buf jb;
static int except_flag=0;
static int jmp_num=0;

void exception_enable(void)
{
 except_flag=1;
}

void exception_disable(void)
{
 except_flag=0; 
}

void raise(void)
{
 if(except_flag&&err::on()){
  except_flag=0;
  ++jmp_num;
  if(jmp_num<=1){
   longjmp(jb,1);
  }
 }
}

void raise2(void)
{
 if(jmp_num<=1){
  ++jmp_num;
  longjmp(jb,1);
 }
}

#endif


static char* CopyName(char* Name)
{char* name;
 name=new char[strlen(Name)+1];
 strcpy(name,Name);
 char* c=name;
 while(*c!='\0'){
  *c=toupper(*c);
  c++;
 }
 return name;
}

/* --- */
void OCI::SetErr(char* Msg,int Code)
{
 err_code=Code;
 strcpy((char*)msg_buf,Msg);
}

/* --- */
void OCI::get_error(CDA_DEF* cur)
{
 if(err_code!=cur->cda.rc)
  oerhms(&cur->lda->lda,cur->cda.rc,msg_buf,sizeof(msg_buf));
 err_code=cur->cda.rc;
} /* get_error */

void OCI::get_error(LDA_DEF* lda)
{
 if(err_code!=lda->lda.rc)
  oerhms(&lda->lda,lda->lda.rc,msg_buf,sizeof(msg_buf));
 err_code=lda->lda.rc;
} /* get_error */

/* --- */
void OCI::ClearErr(void)
{
 err_code=0;
 *msg_buf=0;
} /* ClearErr */

/* --- */
LDA_DEF* OCI::RLogOn(char* UserID)
{
 LDA_DEF* lda=new LDA_DEF;

 INC_LVL;
 if(orlon(&lda->lda,lda->hda,(ora_text*)UserID,-1,0,-1,0)){
  get_error(lda);
  delete lda;
  CHECK_ERR;
  return 0;
 } else{
  DEC_LVL;
  return lda;
 }
} /* RLogOn */

/* --- */
int OCI::RLogOn(LDA_DEF* lda,char* UserID)
{INC_LVL;
 if(orlon(&lda->lda,lda->hda,(ora_text*)UserID,-1,0,-1,0)){
  get_error(lda);
  CHECK_ERR;
  return 1;
 } else{
  DEC_LVL;
  return 0;
 }
}

/* --- */
LDA_DEF* OCI::LogOn(char* UserID)
{
 LDA_DEF* lda=new LDA_DEF;
 INC_LVL;
 if(olon(&lda->lda,(ora_text*)UserID,-1,0,-1,0)){
  get_error(lda);
  delete lda;
  CHECK_ERR;
  return 0;
 } else{
  DEC_LVL;
  return lda;
 }
} /* RLogOn */

/* --- */
int OCI::LogOut(LDA_DEF* lda)
{int rc;
 INC_LVL;
 if(ologof(&lda->lda)){
  get_error(lda);
  CHECK_ERR;
  rc=1;
 }else{
  DEC_LVL;
  rc=0;
 }
 return rc;
} /* LogOut */

/* --- */
int OCI::AutoCommitOff(LDA_DEF* lda)
{INC_LVL;
 if(ocof(&lda->lda)){
  get_error(lda);
  CHECK_ERR;
  return 1;
 }else{
  DEC_LVL;
  return 0;
 }
} /* AutoCommitOff */

/* --- */
int OCI::AutoCommitOn(LDA_DEF* lda)
{INC_LVL;
 if(ocon(&lda->lda)){
  get_error(lda);
  CHECK_ERR;
  return 1;
 }else{
  DEC_LVL;
  return 0;
 }
} /* AutoCommitOn */

/* --- */
int OCI::RollBack(LDA_DEF* lda)
{INC_LVL;
 if(orol(&lda->lda)){
  get_error(lda);
  CHECK_ERR;
  return 1;
 }else{
  DEC_LVL;
  return 0;
 }
} /* RollBack */

/* --- */
int OCI::Commit(LDA_DEF* lda)
{INC_LVL;
 if(ocom(&lda->lda)){
  get_error(lda);
  CHECK_ERR;
  return 1;
 }else{
  DEC_LVL;
  return 0;
 }
} /* Commit */

int OCI::Break(LDA_DEF* lda)
{INC_LVL;
 if(obreak(&lda->lda)){
  get_error(lda);
  CHECK_ERR;
  return 1;
 }else{
  DEC_LVL;
  return 0;
 }
} /* Commit */

/* --- */
CDA_DEF* OCI::OpenCursor(LDA_DEF* lda)
{CDA_DEF* cda=new CDA_DEF(lda);
 INC_LVL;
 if(oopen(&cda->cda,&lda->lda,0,-1,-1,0,-1)){
  get_error(cda);
  delete cda;
  CHECK_ERR;
  return 0;
 } else{
  DEC_LVL;
  return cda;
 }
} /* OpenCursor */

/* --- */
int OCI::CloseCursor(CDA_DEF* cda)
{int rc;
 INC_LVL;
 if(oclose(&cda->cda)){
  get_error(cda);
  rc=1;
 }
 rc=0;
 delete cda;
 CHECK_ERR;
 cda=0;
 return rc;
} /* CloseCursor */

/* --- */
int OCI::Cancel(CDA_DEF* cda)
{INC_LVL;
 if(ocan(&cda->cda)){
  get_error(cda);
  CHECK_ERR;
  return 1;
 }
 DEC_LVL;
 return 0;
} /* Cancel */

int OCI::Option(CDA_DEF* cda,sword rbopt,sword waitopt)
{INC_LVL;
 if(oopt(&cda->cda,rbopt,waitopt)){
  get_error(cda);
  CHECK_ERR;
  return 1;
 }
 DEC_LVL;
 return 0;
} /* Option */

/* --- */
int OCI::Exec(CDA_DEF* cda, short VectSize)
{ INC_LVL;
  cda->row_fetched=0;
  switch(cda->stm_code){
  case 4: // SELECT
   if(oexfet(&cda->cda,VectSize,0,0)){
    cda->row_count=cda->cda.rpc;
    if(cda->cda.rc==1403){
     cda->row_fetched=(short)cda->row_count;
     DEC_LVL;
     return 0;
    }else{
     get_error(cda);
     CHECK_ERR;
     return 1;
    }
   } else {
    cda->row_count=cda->cda.rpc;
    cda->row_fetched=(short)cda->row_count;
    DEC_LVL;
    return 0;
   }
  default:
   cda->row_count=0;
   if(oexn(&cda->cda,VectSize,0)){
    cda->row_count=cda->cda.rpc;
    get_error(cda);
    CHECK_ERR;
    return 1;
   } else {
    cda->row_count=cda->cda.rpc;
    DEC_LVL;
    return 0;
   }
  } /* switch */
} /* Exec */

/* --- */
int OCI::Fetch(CDA_DEF* cda,short VectSize)
{INC_LVL;
 cda->row_fetched=0;
 if(VectSize==1){
  if(ofetch(&cda->cda)){
   if(cda->cda.rc==1403){
    cda->row_count=cda->cda.rpc;
    DEC_LVL;
    return FetchEod;
   }else {
    get_error(cda);
    CHECK_ERR;
    return FetchErr;
   }
  }
  cda->row_count=cda->cda.rpc;
  cda->row_fetched=1;
  DEC_LVL;
  return FetchOk;
 } else {
  if(ofen(&cda->cda,VectSize)){
   if(cda->cda.rc==1403){
    cda->row_fetched=(short)(cda->cda.rpc-cda->row_count);
    cda->row_count=cda->cda.rpc;
    DEC_LVL;
    return FetchEod;
   } else {
    get_error(cda);
    CHECK_ERR;
    return FetchErr;
   }
  }
  cda->row_fetched=(short)(cda->cda.rpc-cda->row_count);
  cda->row_count=cda->cda.rpc;
  DEC_LVL;
  return FetchOk;
 }
} /* Fetch */

/* --- */
int OCI::Parse(CDA_DEF* cda,char* Stm,int& StmCode,int& ErrPos)
{INC_LVL;
 ErrPos=-1; StmCode=0;
 if(oparse(&cda->cda,(ora_text*)Stm,-1,0,0)){
  get_error(cda);
  ErrPos=cda->cda.peo;
  CHECK_ERR;
  return 1;
 } else {
  StmCode=cda->cda.ft;
  cda->stm_code=StmCode;
  DEC_LVL;
  return 0;
 }
} /* Parse */

int OCI::Parse(CDA_DEF* cda,char* Stm)
{int ErrPos,StmCode;
 return Parse(cda,Stm,ErrPos,StmCode);
} /* Parse */

/* --- */
THostVarArray* OCI::DefineSelectList(CDA_DEF* cda,short ShouldDelete,PHostVar v,...)
{PHostVar* vp=&v;
 int va_size=0,i=0;
 THostVarArray* va;

 INC_LVL;
 while(*vp++)++va_size;
 va=new THostVarArray(va_size,ShouldDelete);
 vp=&v;
 while(i<va_size){
  THostVar* vv=*vp++;
  if(odefin(&cda->cda,
            i+1,
            vv->buf,
            vv->elem_size,
            vv->ftype,
            -1,
            vv->indp,
            0,
            -1,
            -1,
            vv->rlen,
            vv->rcode
           )){
   get_error(cda);
   delete va;
   CHECK_ERR;
   return 0;
  }
  va->va[i++]=vv;
 }
 DEC_LVL;
 return va;
} /* DefineSelectList */

/* --- */
int OCI::DefineSelectList(CDA_DEF* cda,PHostVarArray v)
{int v_size,i=0;
 if(!v) return 0;
 INC_LVL;
 v_size=v->valen;
 while(i<v_size){
  PHostVar vv=v->va[i];
  if(odefin(&cda->cda,
            i+1,
            vv->buf,
            vv->elem_size,
            vv->ftype,
            -1,
            vv->indp,
            0,
            -1,
            -1,
            vv->rlen,
            vv->rcode
           )){
   get_error(cda);
   CHECK_ERR;
   return 1;
  }
  ++i;
 }
 DEC_LVL;
 return 0;
} /* DefineSelectList */

THostVarArray* OCI::BindVarList(CDA_DEF* cda,short ShouldDelete,PHostVar v,...)
{PHostVar* vp=&v;
 int va_size=0,i=0;
 THostVarArray* va;
 INC_LVL;
 while(*vp++)++va_size;
 va=new THostVarArray(va_size,ShouldDelete);
 vp=&v;
 while(i<va_size){
  THostVar* vv=*vp++;
  if(OCI::RebindVar(cda,vv)){
   get_error(cda);
   delete va;
   CHECK_ERR;
   return 0;
  }
  va->va[i++]=vv;
 }
 DEC_LVL;
 return va;
} /* BindVarList */

int OCI::RebindVar(CDA_DEF* cda,PHostVar v)
{ if(!v)return 0;
  INC_LVL;
  if(obndrv(&cda->cda,
            (ora_text*)v->name,
             -1,
             v->buf,
             v->elem_size,
             v->ftype,
             -1,
             v->indp,
             0,
             -1,
             -1)){
   get_error(cda);
   CHECK_ERR;
   return 1;
  }else{
   DEC_LVL;
   return 0;
  }
} /* RebindVar */

/* --- */
static int Int2Ext(int IntType)
{
 switch(IntType){
 case inVarChar2: return extCChar;
 case inNumber:   return extFloat;
 case inLong:     return extCChar; //extLong
 case inRowId:    return extCChar; // converting  the internal type ROWID into extCChar
 case inDate:     return extDate;
 case inRaw:      return extCChar;//extRaw;
 case inLongRaw:  return extCChar;//extLongRaw;
 case inChar:     return extCChar;
 default:
  return -1;
 }
} /* Int2Ext */

/* --- */
static int TSize(int Type,int maxsz,int intType)
{
 switch(Type){
 case extCChar:
  switch(intType){
  case inRowId:
   return 19;
  case inDate:
   return 10; // e.g. 19-JAN-64 + <null_terminator>
  case inLong:
  case inLongRaw:
   return MAX_LONG_SIZE;
  case inRaw:
   return maxsz*2+1;
  default:
   return maxsz+1;
  }
 case extFloat:   return sizeof(double);
 case extDate:    return sizeof(TDateIntern);
// case extLong:    return MAX_LONG_SIZE;
// case extRaw:     return maxsz;
// case extLongRaw: return MAX_LONG_SIZE;
 default:
  return 0;
 }
} /* TSize */

/* --- */
static THostVar* MapType(inFldDesc& fd,short VectSize)
{ sword ftype=Int2Ext(fd.Dbtype());
  sword esize=TSize(ftype,(int)fd.Dbsize(),(int)fd.Dbtype());
  INC_LVL;
  THostVar* v=new THostVar(fd.Name(),ftype,esize,VectSize);
  DEC_LVL;
  return v;
} /* MapType */

/* --- */
THostVarArray* OCI::DefineSelectList(CDA_DEF* cda,inRowDesc* row_desc,short VectSize)
{ int va_size; THostVarArray* va;

  if(!row_desc) return 0;
  va_size=row_desc->RowLen();
  INC_LVL;
  va=new THostVarArray(va_size,1);

  for(int i=0;i<va_size;++i){
   THostVar* vv=va->va[i]=MapType((*row_desc)[i],VectSize);
   if(odefin(&cda->cda,
             i+1,
             vv->buf,
             vv->elem_size,
             vv->ftype,
             -1,
             vv->indp,
             0,
             -1,
             -1,
             vv->rlen,
             vv->rcode
            )){
    get_error(cda);
    delete va;
    CHECK_ERR;
    return 0;
   }
  }
  DEC_LVL;
  return va;
}



/* --- */
inFldDesc::inFldDesc(CDA_DEF* cda,int pos)
{sb1 cbuf[241]; sb4 cbufl;

 name=0;
 if(cda->stm_code!=4) return; // Statement is not "SELECT"
 cbufl=sizeof(cbuf);
 INC_LVL;
 odescr(&cda->cda,
        pos,
        &dbsize,
        &dbtype,
        &cbuf[0],
        &cbufl,
        &dsize,
        &prec,
        &scale,
        &nullok
       );
 if(cda->cda.rc!=1007&&cda->cda.rc!=0){
  OCI::get_error(cda);
  CHECK_ERR;
  return;
 } else if(cda->cda.rc==1007){
  DEC_LVL;
  return;
 }
 name=new char[(int)cbufl+1];
 cbuf[(int)cbufl]='\0';
 strcpy(name,(char*)cbuf);
 DEC_LVL;
}

/* --- */
inFldDesc::~inFldDesc()
{
 if(name)delete name;
}

short inFldDesc::NumberOfType(void)
{if(dbtype!=inNumber)
  return type_notNumber;
 else if(scale==0&&prec==0)
  return type_floatNumber;
 else if(prec>0&&prec<=4&&scale==0)
  return type_shortNumber;
 else 
  return type_Number;
}



/* --- */
inRowDesc::inRowDesc(CDA_DEF* cda)
{ PinFldDesc* tmp; int tmp_len;

 row_desc=0; row_len=0;
 if(cda->stm_code!=4)return;
 INC_LVL;
 tmp=new PinFldDesc[MAX_VAR_ARRAY_SIZE]; // creating temporary buffer
 tmp_len=0;
 for(;;){
  PinFldDesc f;
  f=new inFldDesc(cda,tmp_len+1);
  if(f->Name())
   tmp[tmp_len++]=f;
  else {
   delete f;
   break;
  }
 }
 row_len=tmp_len;
 row_desc=new PinFldDesc[row_len];
 memcpy(row_desc,tmp,row_len*sizeof(PinFldDesc));
 delete tmp;
 CHECK_ERR;
}

/* --- */
inRowDesc::~inRowDesc()
{
 for(int i=0;i<row_len;++i)
  delete row_desc[i];
 if(row_len)delete row_desc;
}


/* --- */
THostVar::THostVar()
{
 vect_size=0;
}

void THostVar::InitVar(char* Name,sword FType,sword ElemSize,sword MaxVectSize,void* Buf)
{int i,k;
 vect_size=MaxVectSize;
 elem_size=ElemSize;
 name=0;
 if(Buf){
  buf=(ub1*)Buf;
  buf_is_external=1;
 } else {
  buf_is_external=0;
  if(elem_size==-1)
   buf=0;
  else
   buf=new ub1[elem_size*(unsigned)MaxVectSize];
 }
 ftype=FType;
 indp=new sb2[MaxVectSize];
 k=elem_size==-1?0:elem_size;
 for(i=0;i<MaxVectSize;++i)
  indp[i]=k;
 rlen=new ub2[MaxVectSize];
 for(i=0;i<MaxVectSize;++i)
  rlen[i]=0;
 rcode=new ub2[MaxVectSize];
 if(Name)
  name=CopyName(Name);
}

/* --- */
THostVar::THostVar(char* Name,sword FType,sword ElemSize,sword MaxVectSize,void* Buf)
{InitVar(Name,FType,ElemSize,MaxVectSize,Buf);
}

/* --- */
THostVar::THostVar(char* Name,sword FType,sword Size,void* Buf)
{InitVar(Name,FType,Size,(sword)1,Buf);
}

/* --- */
THostVar::~THostVar()
{
 if(vect_size==0) return;
 if(!buf_is_external&&elem_size!=-1)delete buf;
 delete indp;
 delete rlen;
 delete rcode;
 delete name;
}

/* --- */
void THostVar::SetBuf(void* Buf)
{
 if(!buf_is_external)
  delete buf;
 buf_is_external=1;
 buf=(ub1*)Buf;
} /* SetBuf */

/* --- */
void THostVar::SetName(char* Name)
{
  if(name) delete name;
  name=new char[strlen(Name)+1];
  strcpy(name,Name);
} /* SetName */

/* --- */
void THostVar::GetVal(int Ndx, void* Dst)
{ sword len=Len(Ndx);

  if(ftype==extCharZ||ftype==extCChar)
   ++len;
  memcpy(Dst,Val(Ndx),len);
}

void THostVar::SetVal(int Ndx,sword len,void* Src)
{
  memcpy(Val(Ndx),Src,len);
  indp[Ndx]=len;
}

/* --- */
void THostVar::GetVal(void* Dst)
{ sword len=Len(0);

  if(ftype==extCharZ||ftype==extCChar)
   ++len;
  memcpy(Dst,Val(0),len);
}

void THostVar::SetVal(sword len,void* Src)
{
  memcpy(Val(0),Src,len);
  indp[0]=len;
}

/* --- */
void TSelectStm::CopyFrom(int Ndx)
{PHostVarArray src,dst; int i,n;
 src=internal_select_list;
 dst=select_list;
 n=dst->Size();
 for(i=0;i<n;++i){
  THostVar& s=(*src)[i];
  THostVar& d=(*dst)[i];
  sword Len=s.Len(Ndx);
  if(d.ftype==extCharZ||d.ftype==extCChar)
   ++Len;
  memcpy(d.Val(0),s.Val(Ndx),Len);
  d.indp[0]=s.indp[Ndx];
  d.rcode[0]=s.rcode[Ndx];
  d.rlen[0]=s.rlen[Ndx];
 }
}

short THostVar::double2short(int Ndx)
{
 double* p=(double*)Val(Ndx);
 short v=(short)*p;
 return v;
}

float THostVar::double2float(int Ndx)
{
 double* p=(double*)Val(Ndx);
 float v=(float)*p;
 return v;
}

int THostVar::double2int(int Ndx)
{
 double* p=(double*)Val(Ndx);
 int v=(int)*p;
 return v;
}

unsigned THostVar::double2unsigned(int Ndx)
{
 double* p=(double*)Val(Ndx);
 unsigned v=(unsigned)*p;
 return v;
}

long THostVar::double2long(int Ndx)
{
 double* p=(double*)Val(Ndx);
 long v=(long)*p;
 return v;
}

char THostVar::cchar2char(int Ndx)
{
  char* p=(char*)Val(Ndx);
  return *p;
}

/* --- */
THostVarArray::THostVarArray(int Size,short ShouldDelete)
{
 valen=Size;
 va=new PHostVar[valen];
 for(int i=0;i<valen;++i)va[i]=0;
 should_delete=ShouldDelete;
}

/* --- */
THostVarArray::THostVarArray(short ShouldDelete,PHostVar v,...)
{PHostVar* vp=&v;
 int i=0;

 should_delete=ShouldDelete;
 valen=0;
 while(*vp++)++valen;
 va=new PHostVar[valen];
 vp=&v;
 while(i<valen)
  va[i++]=*vp++;
}

/* --- */
THostVarArray::THostVarArray(PHostVar v,...)
{PHostVar* vp=&v;
 int i=0;

 should_delete=0;
 valen=0;
 while(*vp++)++valen;
 va=new PHostVar[valen];
 vp=&v;
 while(i<valen)
  va[i++]=*vp++;
}

/* --- */
THostVarArray::THostVarArray(THostVarArray& v,short NewVectSize)
{int i;
 valen=v.Size();
 va=new PHostVar[valen];
 for(i=0;i<valen;++i)va[i]=0;
 should_delete=v.should_delete;
 for(i=0;i<valen;++i)
  va[i]=new THostVar(v[i].Name(),v[i].Ftype(),v[i].ElemSize(),NewVectSize);
}

/* --- */
THostVarArray::~THostVarArray()
{
 if(should_delete)
  for(int i=0;i<valen;++i)
   if(va[i])
    delete va[i];
 delete va;
}

/* --- */
THostVar& THostVarArray::operator[](char* Name)
{ static THostVar Empty; // to return the reference to an empty host variable
                         // for the case when no name found
  char tmp_name[256];

  strcpy(tmp_name,Name);

  char* c=tmp_name;
  while(*c!='\0'){
   *c=toupper(*c);
   c++;
  }
  for(int i=0;i<valen;++i)
   if(strcmp(tmp_name,va[i]->Name())==0)
    return (THostVar&)*va[i];
  return Empty;
}


/* --- */
TConnect::TConnect(char* UserId)
{for(int i=0;i<MAX_CURSOR_NUM;++i)
  cda[i]=0;
 cda_len=0;
 INC_LVL;
 lda=OCI::RLogOn(UserId);
 is_connected=lda?1:0;
 CHECK_ERR;
}

/* --- */
TConnect::TConnect()
{for(int i=0;i<MAX_CURSOR_NUM;++i)
  cda[i]=0;
 cda_len=0;
 lda=new LDA_DEF;
 is_connected=0;
}

/* --- */
TConnect::~TConnect()
{if(is_connected){
  INC_LVL;
  for(int i=0;i<cda_len;++i)
   if(cda[i]){
    if(cda[i]->should_delete)
     delete cda[i];
   }
  OCI::LogOut(lda);
  CHECK_ERR;
 }
 delete lda;
}

/* --- */
int TConnect::LogOn(char* UserId)
{int rc;
 INC_LVL;
 if(is_connected){
  rc=OCI::LogOut(lda);
  if(rc){
   CHECK_ERR;
   return rc;
  }
  is_connected=0;
 }
 rc=OCI::RLogOn(lda,UserId);
 if(rc){
  CHECK_ERR;
  return rc;
 }
 is_connected=1;
 DEC_LVL;
 return 0;
}

int TConnect::LogOut(void)
{
 if(!is_connected)return 1;
 INC_LVL;
 if(OCI::LogOut(lda)){
  CHECK_ERR;
  return 1;
 }else{
  is_connected=0;
  DEC_LVL;
  return 0;
 }
}

/* --- */
int TConnect::ExecDirect(char* Stm)
#ifdef __EXCEPTION_ENABLED__
{INC_LVL;
 TStm stm(this,Stm);
 stm.Parse();
 if(err::on()){
  CHECK_ERR;
  return 1;
 }
 stm.Exec();
 if(err::on()){
  CHECK_ERR;
  return 1;
 }
 DEC_LVL;
 return 0;
}
#else
{INC_LVL;
 PStm stm=0;
 stm=new TStm(this,Stm);
 stm->Parse();
 if(err::on()){
  delete stm;
  CHECK_ERR;
  return 1;
 }
 stm->Exec();
 if(err::on()){
  delete stm;
  CHECK_ERR;
  return 1;
 }
 DEC_LVL;
 delete stm;
 return 0;
}
#endif


/* --- */
TOCursor::TOCursor(PConnect db, short ShouldDelete)
{INC_LVL;
 owner=db;
 cda=OCI::OpenCursor(db->lda);
 row_count=OCI::RowCount(cda);
 row_fetched=OCI::RowFetched(cda);
 err_pos=-1;
 stm=0;
 db->cda[db->cda_len++]=this;
 should_delete=ShouldDelete;
 row_desc=0;
 CHECK_ERR;
}

/* --- */
TOCursor::~TOCursor()
{int i,n=-1; PConnect db=(PConnect)owner;

 if(!cda)return;
 INC_LVL;
 if(row_desc)
  delete row_desc;
 if(stm){
  if(stm->should_delete){
   should_delete=0;
   delete stm;
  }
 }
 stm=0;
 if(!OCI::CloseCursor(cda))
  cda=0;

 for(i=0;i<db->cda_len;++i)
  if(db->cda[i]==this){
   n=i;
   break;
  }
 if(n!=-1){
  if(db->cda_len>1)
   for(i=n;i<db->cda_len-1;++i)
    db->cda[i]=db->cda[i+1];
  db->cda_len--;
 }
 CHECK_ERR;
}

/* --- */
int TOCursor::Exec(short VectSize)
{int rc;
 if(!stm) return 1;
 INC_LVL;
 rc=OCI::Exec(cda,VectSize);
 row_count=OCI::RowCount(cda);
 row_fetched=OCI::RowFetched(cda);
 CHECK_ERR;
 return rc;
}

/* --- */
int TOCursor::Parse(void)
{int stm_code,rc;

 if(!stm) return 1;
 INC_LVL;
 rc=OCI::Parse(cda,stm->stm,stm_code,err_pos);
 if(rc){
   CHECK_ERR;
   return rc;
 }
 if(row_desc) delete row_desc;
 row_desc=OCI::DescribeSelectList(cda);
 if(err::code()){
  CHECK_ERR;
  return 1;
 }
 if(stm->bound_var_list){
  PHostVarArray v=stm->bound_var_list;
  for(int i=0;i<v->valen;++i){
   OCI::RebindVar(cda,v->va[i]);
   if(err::code()){
    CHECK_ERR;
    return 1;
   }
  }
 }
 if(stm->default_select_list)
  stm->select_list=OCI::DefineSelectList(cda,row_desc,stm->select_list_vect_size);
 else
  OCI::DefineSelectList(cda,stm->select_list);
 if(err::code()){
  CHECK_ERR;
  return 1;
 }
 DEC_LVL;
 return 0;
}

/* --- */
int TOCursor::Fetch(short VectSize)
{int rc;
 if(!stm) return 1;
 INC_LVL;
 rc=OCI::Fetch(cda,VectSize);
 row_count=OCI::RowCount(cda);
 row_fetched=OCI::RowFetched(cda);
 CHECK_ERR;
 return rc;
}


/* --- */
TStm::TStm(POCursor cur,char* Stm,short ShouldDelete,PHostVarArray SelectList,PHostVarArray BoundVarList)
{
 should_delete=ShouldDelete;
 owner=cur;
 default_select_list=0;
 bound_var_list=BoundVarList;
 select_list=SelectList;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 if(cur->stm){
  if(cur->stm->should_delete)
   delete cur->stm;
 }
 cur->stm=this;
 select_list_vect_size=1;
 cur_is_child=0;
}

TStm::TStm(POCursor cur,char* Stm)
{
 should_delete=SHOULD_DELETE;
 owner=cur;
 default_select_list=1;
 bound_var_list=0;
 select_list=0;
 select_list_vect_size=1;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 cur->stm=this;
 cur_is_child=0;
}

/* --- */
TStm::TStm(POCursor cur,char* Stm,short SelectListVectSize,short ShouldDelete,PHostVarArray BoundVarList)
{
 should_delete=ShouldDelete;
 owner=cur;
 default_select_list=1;
 bound_var_list=BoundVarList;
 select_list=0;
 select_list_vect_size=SelectListVectSize;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 cur->stm=this;
 cur_is_child=0;
}

/* --- */
TStm::TStm(PConnect db,char* Stm,short ShouldDelete,PHostVarArray SelectList,PHostVarArray BoundVarList)
{POCursor cur;
 INC_LVL;
 cur=new TOCursor(db);
 if(err::on()){
  delete cur;
  cur=0;
  cur_is_child=-1;
  CHECK_ERR;
  return;
 }
 cur_is_child=1;
 should_delete=ShouldDelete;
 owner=cur;
 default_select_list=0;
 bound_var_list=BoundVarList;
 select_list=SelectList;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 if(cur->stm){
  if(cur->stm->should_delete)
   delete cur->stm;
 }
 cur->stm=this;
 select_list_vect_size=1;
 DEC_LVL;
}

/* --- */
TStm::TStm(PConnect db,char* Stm,short SelectListVectSize,short ShouldDelete,PHostVarArray BoundVarList)
{POCursor cur;
 INC_LVL;
 cur=new TOCursor(db);
 if(err::on()){
  delete cur;
  cur=0;
  cur_is_child=-1;
  CHECK_ERR;
  return;
 }
 should_delete=ShouldDelete;
 owner=cur;
 default_select_list=1;
 bound_var_list=BoundVarList;
 select_list=0;
 select_list_vect_size=SelectListVectSize;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 cur->stm=this;
 cur_is_child=1;
 DEC_LVL;
}

/* --- */
TStm::TStm(PConnect db,char* Stm)
{POCursor cur;
 INC_LVL;
 cur=new TOCursor(db);
 if(err::on()){
  delete cur;
  cur=0;
  cur_is_child=-1;
  CHECK_ERR;
  return;
 }
 should_delete=SHOULD_DELETE;
 owner=cur;
 default_select_list=1;
 bound_var_list=0;
 select_list=0;
 select_list_vect_size=1;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 cur->stm=this;
 cur_is_child=1;
 DEC_LVL;
}

/* --- */
TStm::~TStm()
{
 if(cur_is_child==-1)
  return;
 INC_LVL;
 if(cur_is_child==1&&((POCursor)owner)->should_delete){
  ((POCursor)owner)->stm=0;
  delete owner;
 }
 delete stm;
 if(select_list){
  if(select_list->should_delete)
   delete select_list;
 }
 if(bound_var_list){
  if(bound_var_list->should_delete)
   delete bound_var_list;
 }
 CHECK_ERR;
}

/* --- */
void TStm::AttachCursor(POCursor cur)
{INC_LVL;
 owner=cur;
 if(cur->stm){
  if(cur->stm->should_delete)
   delete cur->stm;
 }
 cur->stm=this;
 CHECK_ERR;
}

/* --- */
TSelectStm::TSelectStm(PSelectCursor cur,
                       char* Stm,
                       short ShouldDelete,
                       short InternalVectSize,
                       PHostVarArray SelectList,
                       PHostVarArray BoundVarList
                      ):TStm(cur,Stm,ShouldDelete,SelectList,BoundVarList)

{ INC_LVL;
  internal_select_list=0;
  internal_vect_size=InternalVectSize;
  if(SelectList)
   internal_select_list=new THostVarArray(*SelectList,InternalVectSize);
  CHECK_ERR;
}

/* --- */
TSelectStm::TSelectStm(PSelectCursor cur,
                       char* Stm,
                       short ShouldDelete,
                       short InternalVectSize,
                       PHostVarArray BoundVarList
                      ):TStm(cur,Stm,1,ShouldDelete,BoundVarList)
{ INC_LVL;
  internal_select_list=0;
  internal_vect_size=InternalVectSize;
  CHECK_ERR;
}


/* --- */
TSelectStm::TSelectStm(PConnect db,
                       char* Stm,
                       short ShouldDelete,
                       short InternalVectSize,
                       PHostVarArray SelectList,
                       PHostVarArray BoundVarList
                      )//:TStm(new TSelectCursor(db),Stm,ShouldDelete,SelectList,BoundVarList)
{PSelectCursor cur;
 INC_LVL;
 cur=new TSelectCursor(db);
 if(err::on()){
  delete cur;
  cur=0;
  cur_is_child=-1;
  CHECK_ERR;
  return;
 }
 cur_is_child=1;
 should_delete=ShouldDelete;
 owner=cur;
 default_select_list=0;
 bound_var_list=BoundVarList;
 select_list=SelectList;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 cur->stm=this;
 select_list_vect_size=1;

 internal_select_list=0;
 internal_vect_size=InternalVectSize;
 if(SelectList)
  internal_select_list=new THostVarArray(*SelectList,InternalVectSize);
 CHECK_ERR;
}

/* --- */
TSelectStm::TSelectStm(PConnect db,
                       char* Stm,
                       short ShouldDelete,
                       short InternalVectSize,
                       PHostVarArray BoundVarList
                      )//:TStm(new TSelectCursor(db),Stm,1,ShouldDelete,BoundVarList)
{PSelectCursor cur;
 INC_LVL;
 cur=new TSelectCursor(db);
 internal_select_list=0;
 internal_vect_size=InternalVectSize;

 if(err::on()){
  delete cur;
  cur=0;
  cur_is_child=-1;
  CHECK_ERR;
  return;
 }
 should_delete=ShouldDelete;
 owner=cur;
 default_select_list=1;
 bound_var_list=BoundVarList;
 select_list=0;
 select_list_vect_size=1;
 stm=new char[strlen(Stm)+1];
 strcpy(stm,Stm);
 cur->stm=this;
 cur_is_child=1;
 DEC_LVL;
}

/* --- */
TSelectStm::~TSelectStm()
{ if(cur_is_child==-1)
   return;
  if(err::code()!=Debug_Codes[0]&&
     err::code()!=Debug_Codes[1]&&
     err::code()!=Debug_Codes[2]){
   INC_LVL;
   delete internal_select_list;
   CHECK_ERR;
  }
}

/* --- */
TSelectCursor::TSelectCursor(PConnect db,short ShouldDelete)
  :TOCursor(db,ShouldDelete)
{
 row_num=-1;
 last_fetch_rc=FetchOk;
 real_row_count=0;
 real_row_fetched=0;
}

/* --- */
int TSelectCursor::GetFirst(void)
{PSelectStm st; int rc;

 row_num=-1; real_row_count=0; real_row_fetched=0;
 if(!stm) return (last_fetch_rc=FetchErr);
 INC_LVL;
 st=(PSelectStm)stm;
 rc=TOCursor::Exec(st->internal_vect_size);
 if(rc){
  CHECK_ERR;
  return (last_fetch_rc=FetchErr);
 }
 last_fetch_rc=row_fetched==st->internal_vect_size?FetchOk:FetchEod;
 if(row_fetched==0){
  DEC_LVL;
  return FetchEod; // No rows have been fetched
 }
 ++real_row_count;
 real_row_fetched=1;
 st->CopyFrom(++row_num);
 DEC_LVL;
 return FetchOk;
}

/* --- */
int TSelectCursor::GetNext(void)
{PSelectStm st=(PSelectStm)stm; int rc;

 if(!stm) return (last_fetch_rc=FetchErr);
 if(row_num==-1)return GetFirst();
 if(last_fetch_rc==FetchEod&&row_num==row_fetched-1)return FetchEod;
 if(row_num<row_fetched-1){
  ++real_row_count;
  real_row_fetched=1;
  st->CopyFrom(++row_num);
  return FetchOk;
 }else{
   row_num=-1;
   INC_LVL;
   rc=TOCursor::Fetch(st->internal_vect_size);
   if(rc==FetchErr){
    CHECK_ERR;
    return (last_fetch_rc=FetchErr);
   }
   last_fetch_rc=row_fetched==st->internal_vect_size?FetchOk:FetchEod;
   if(row_fetched==0){
    DEC_LVL;
    return FetchEod; // No rows have been fetched
   }
   real_row_fetched=1;
   ++real_row_count;
   st->CopyFrom(++row_num);
   DEC_LVL;
   return FetchOk;
 }
}

/* --- */
int TSelectCursor::Parse(void)
{int rc;
 PSelectStm st=(PSelectStm)stm;;
 PHostVarArray tmp;

 if(!stm) return 1;
 INC_LVL;
 last_fetch_rc=FetchOk;
 row_num=-1;
 if(st->default_select_list){
  st->select_list_vect_size=st->internal_vect_size;
  rc=TOCursor::Parse();
  if(rc){
   CHECK_ERR;
   return rc;
  }
  if(st->select_list){
   st->internal_select_list=st->select_list;
   st->select_list=new THostVarArray(*st->select_list,1);
  }
  st->select_list_vect_size=1;
 }else{
  tmp=st->select_list;
  st->select_list=st->internal_select_list; // Actually the internal select list is used
                                            // but resulting rows are put to the "select_list"  array
  rc=TOCursor::Parse();
  st->select_list=tmp;
  if(rc){
   CHECK_ERR;
   return rc;
  }
 }
 if(cda->stm_code!=4){
  DEC_LVL;
  return 1; // Statement is not "SELECT"
 }
 DEC_LVL;
 return 0;
}

OraStream::OraStream(char* StmText)
{
 cur_x=-1;
 cur_y=0 ;
 va_len=0;
 vect_size=0;
 va=0;
 stm_text=new char[strlen(StmText)+1];
 strcpy(stm_text,StmText);
}

OraStream::OraStream()
{
 cur_x=-1;
 cur_y=0 ;
 va_len=0;
 vect_size=0;
 va=0;
 stm_text=0;
}

static void InternSetErr(int code=0)
{
 err::set(DebugMsg[code],Debug_Codes[code]);
 RAISE2;
} /* InternSetErr */

SelStream::SelStream(PConnect db,char* StmText,short VectSize,PHostVarArray InVar)
          :OraStream(StmText)
{
 rc=FetchOk;
 vect_size=VectSize;
 va=InVar;
 stmOpen(db);
 va=stm->SelectList();
 va_len=va->Size();
 rc=stm->GetNext();
 null_fetched=0;
}

SelStream::~SelStream()
{stmClose();
}

void SelStream::GetNext(void)
{if(cur_x<va_len-1){
  ++cur_x;
  null_fetched=(*va)[cur_x].isNull();
 } else{
  rc=stm->GetNext();
  cur_x=0;
 }
} /* GetNext */

int SelStream::CheckType(int TypeCode)
{if((*va)[cur_x].Ftype()!=TypeCode){
  InternSetErr(0);
  return 0;
 }else
  return 1;
}

void SelStream::LookAhead(void)
{
 if(cur_x==va_len-1){
  rc=stm->GetNext();
  cur_x=-1;
 }
} /* LookAhead */

void SelStream::rewind(void)
{
 rc=stm->GetFirst();
 cur_x=-1;
} /* rewind */

SelStream& SelStream::operator>>(char& c)
{if(eof())return *this;
 GetNext();
 if(CheckType(extCChar)&&!eof()){
  c=*(char*)(*va)[cur_x].Val();
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(unsigned char& c)
{if(eof())return *this;
 GetNext();
 if(CheckType(extCChar)&&!eof()){
  c=*(unsigned char*)(*va)[cur_x].Val();
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(char* s)
{if(eof())return *this;
 GetNext();
 if(CheckType(extCChar)&&!eof()){
  strcpy(s,(char*)(*va)[cur_x].Val());
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(unsigned char* s)
{if(eof())return *this;
 GetNext();
 if(CheckType(extCChar)&&!eof()){
  strcpy((char*)s,(char*)(*va)[cur_x].Val());
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(int& n)
{if(eof())return *this;
 GetNext();
 if(CheckType(extFloat)&&!eof()){
  n=(*va)[cur_x].double2int();
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(unsigned& u)
{if(eof())return *this;
 GetNext();
 if(CheckType(extFloat)&&!eof()){
  u=(*va)[cur_x].double2unsigned();
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(short& sh)
{if(eof())return *this;
 GetNext();
 if(CheckType(extFloat)&&!eof()){
  sh=(*va)[cur_x].double2short();
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(long int& l)
{if(eof())return *this;
 GetNext();
 if(CheckType(extFloat)&&!eof()){
  l=(*va)[cur_x].double2long();
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(float& f)
{if(eof())return *this;
 GetNext();
 if(CheckType(extFloat)&&!eof()){
  f=(*va)[cur_x].double2float();
  LookAhead();
 }
 return *this;
}

SelStream& SelStream::operator>>(double& d)
{if(eof())return *this;
 GetNext();
 if(CheckType(extFloat)&&!eof()){
  double* p=(double*)(*va)[cur_x].Val();
  d=*p;
  LookAhead();
 }
 return *this;
}

class HVD{
public:
 
 enum hv_type{ 
  tcchar=type_TCChar, 
  tdouble=type_TDouble, 
  tfloat=type_TFloat, 
  tint=type_TInt,
  tuint=type_TUnsignedInt,
  tshort=type_TShortInt,
  tlongint=type_TLongInt
 };
 
 static  PHostVar CreateHostVar(char* s,short VectSize);
};

extern "C" int atoi(const char*);

PHostVar HVD::CreateHostVar(char* s,short VectSize)
{char name[64];
 char type;
 sword size=0;

 char *c=name,*c1=s;
 while(*c1!=' ')
  *c++=*c1++;
 *c='\0';
 while(*c1==' ')
  ++c1;
 type=toupper(*c1);
 if(type=='C'){
  char tmp[32];
  char *t=tmp;
  while(*c1!='[')
    ++c1;
  ++c1;
  while(*c1!=']')
   *t++=*c1++;
  *t='\0';
  size=atoi(tmp);
 }
 
switch(type){
 case 'C':
#ifdef __TEMPLATE_ENABLED__
  return new yTCChar<type_TCChar>(name,size,VectSize);
#else
  return new TCChar(name,size,VectSize);
#endif
 case 'D':
  return new TDouble(name,VectSize);
 case 'F':
  return new TFloat(name,VectSize);
 case 'I':
  return new TOInt(name,VectSize);
 case 'U':
  return new TUnsignedInt(name,VectSize);
 case 'S':
  return new TShortInt(name,VectSize);
 case 'L':
  return new TLongInt(name,VectSize);
 default:
  {char errmsg[128];
   strcpy(errmsg,DebugMsg[2]);
   strcat(errmsg,s);
   err::set(errmsg,
	    Debug_Codes[2]
	   );
   RAISE2;
   return 0;
  }
 }
}

OStream::OStream(PConnect db,char* StmText,short VectSize, char* hvd,...)
          :OraStream(StmText)
{char** h=&hvd;
 vect_size=VectSize;
 commit_on_flush=1;
 dirty=0;
 adb=db;
 while(*h){
  ++va_len;
  ++h;
 }
 h=&hvd;
 va=new THostVarArray(va_len);
 va_should_delete=0;
 for(int i=0;i<va_len;++i)
  va->SetHostVar(i,HVD::CreateHostVar(h[i],vect_size));
 stmOpen(db);
}

OStream::OStream(PConnect db,char* StmText,short VectSize, char* hvd[])
          :OraStream(StmText)
{vect_size=VectSize;
 commit_on_flush=1;
 dirty=0;
 adb=db;
 while(hvd[va_len])
  ++va_len;
 va=new THostVarArray(va_len);
 va_should_delete=0;
 for(int i=0;i<va_len;++i)
  va->SetHostVar(i,HVD::CreateHostVar(hvd[i],vect_size));
 stmOpen(db);
}

OStream::~OStream()
{if(dirty&&!err::on())flush();
 if(va_should_delete)
  delete va;
 stmClose();
}

void OStream::GetNext(void)
{if(cur_x<va_len-1)
  ++cur_x;
  else{
  if(cur_y<vect_size-1){
   ++cur_y;
   cur_x=0;
  }else{
   flush();
   cur_x=0;
  }
 }
 dirty=1;
} /* GetNext */

void OStream::CheckBuf(void)
{if(cur_x==va_len-1&&cur_y==vect_size-1)
  flush();
} /* CheckBuf */

int OStream::CheckType(int TypeCode)
{if((*va)[cur_x].isA()!=TypeCode){
  InternSetErr(0);
  return 0;
 }else
  return 1;
}

OStream& OStream::operator<<(char c)
{GetNext();
 if(CheckType(HVD::tcchar)){
  char* tmp=(char*)(*va)[cur_x].Val(cur_y);
  tmp[0]=c;
  tmp[1]='\0';
 }
 return *this; 
}

OStream& OStream::operator<<(unsigned char c)
{GetNext();
 if(CheckType(HVD::tcchar)){
  unsigned char* tmp=(unsigned char*)(*va)[cur_x].Val(cur_y);
  tmp[0]=c;
  tmp[1]='\0';
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(const char* s)
{GetNext();
 if(CheckType(HVD::tcchar)){
  char* tmp=(char*)(*va)[cur_x].Val(cur_y);
  strcpy(tmp,s);
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(const unsigned char* s)
{GetNext();
 if(CheckType(HVD::tcchar)){
  char* tmp=(char*)(*va)[cur_x].Val(cur_y);
  strcpy(tmp,(char*)s);
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(int n)
{GetNext();
 if(CheckType(HVD::tint)){
  int* tmp=(int*)(*va)[cur_x].Val(cur_y);
  *tmp=n;
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(unsigned u)
{GetNext();
 if(CheckType(HVD::tuint)){
  unsigned* tmp=(unsigned*)(*va)[cur_x].Val(cur_y);
  *tmp=u;
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(short sh)
{GetNext();
 if(CheckType(HVD::tshort)){
  short* tmp=(short*)(*va)[cur_x].Val(cur_y);
  *tmp=sh;
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(long int l)
{GetNext();
 if(CheckType(HVD::tlongint)){
  long* tmp=(long*)(*va)[cur_x].Val(cur_y);
  *tmp=l;
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(float f)
{GetNext();
 if(CheckType(HVD::tfloat)){
  float* tmp=(float*)(*va)[cur_x].Val(cur_y);
  *tmp=f;
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(double d)
{GetNext();
 if(CheckType(HVD::tdouble)){
  double* tmp=(double*)(*va)[cur_x].Val(cur_y);
  *tmp=d;
 }
 CheckBuf();
 return *this;
}

OStream& OStream::operator<<(onull n)
{GetNext();
 (*va)[cur_x].SetNull(cur_y); 
 CheckBuf();
 return *this;
}

void OStream::flush(void)
{if(!dirty)return;
 if(cur_x!=va_len-1)
  InternSetErr(1);
 stm->Exec(cur_y+1);
 if(commit_on_flush)
  adb->Commit();
 for(int i=0;i<vect_size;++i)
  for(int j=0;j<va_len;++j)
   (*va)[j].SetNotNull(i);
 cur_x=-1;
 cur_y=0;
 dirty=0;
}

class HVDArray{
public:
 enum VarStatus{
  in=0,
  out=1,
  def=2
 };
 HVDArray(char* stm);
 ~HVDArray();
 char* operator[](int ndx){return hv[ndx];};
 short v_status(int ndx){return inout[ndx];};
private:
 char* hv[MAX_VAR_ARRAY_SIZE];
 short inout[MAX_VAR_ARRAY_SIZE];
 void AddVar(int &n,char* v,short in_out);
};

static int isId(char c)
{
 return isalnum(c)||c=='_';
}

static int name_comp(char* n1,char* n2)
{while(*n1!=' '&&*n1!='\0'&&*n2!=' '&&*n2!='\0'){
  if(toupper(*n1)!=toupper(*n2))return 0;
  ++n1;
  ++n2;
 }
 if(*n1==' '&&*n2!=' '||*n2==' '&&*n1!=' ')
  return 0;
 return 1;
} /* name_comp */

void HVDArray::AddVar(int &n,char* v,short in_out)
{for(int i=0;i<n;++i)
  if(name_comp(hv[i],v))
   return;
 hv[n]=new char[strlen(v)+1];
 strcpy(hv[n],v);
 inout[n]=in_out;
 hv[++n]=0;
 inout[n]=def;
}

HVDArray::HVDArray(char* stm)
{int i=0; 
 short InStr=0;
 char *c=stm;

 hv[i]=0;
 while(*c){
  if(*c=='\''){
   if(!InStr)
    InStr=1;
   else{
    if(c[1]=='\'')
     ++c;
    else
     InStr=0; 
   }
  }
  if(*c==':'&&!InStr){
   short in_out=in;
   char var[64];
   char* v=var;
   *v++=*c++;
   while(isId(*c))
    *v++=*c++;
   while(isspace(*c))
    ++c;
   if(*c=='<'){
    *c=' ';
    while(*c!='>'&&*c!=','){
     *v++=*c;
     *c++=' ';
    }
    if(*c==','){
     *c++=' ';
     if(toupper(*c)=='I')
      in_out=in;
     else if(toupper(*c)=='O')
      in_out=out;
     while(*c!='>')
      *c++=' ';
    }
    *c=' ';
    *v='\0';
    AddVar(i,var,in_out);
   }
  }//else if(*c==':'&&!InStr&&c[1]=='%'){
//   
//  }
  ++c;
 }
}

HVDArray::~HVDArray()
{
 for(int i=0;hv[i]!=0;++i)
  delete hv[i];
}

OutStream::OutStream(PConnect db,char* StmText,short VectSize)
{
 stm_text=new char[strlen(StmText)+1];
 strcpy(stm_text,StmText);
 {
  HVDArray hvd(stm_text);
  vect_size=VectSize;
  commit_on_flush=1;
  dirty=0;
  adb=db;
  va_len=0;
  while(hvd[va_len])
    ++va_len;
  va=new THostVarArray(va_len);
  for(int i=0;i<va_len;++i)
    va->SetHostVar(i,HVD::CreateHostVar(hvd[i],vect_size));
 }
 stmOpen(db);
}

SelectStream::SelectStream(PConnect db,char* StmText,short VectSize)
{
 stm_text=new char[strlen(StmText)+1];
 strcpy(stm_text,StmText);
 executed=0;
 {
  HVDArray hvd(stm_text);
  iv_len=0;
  while(hvd[iv_len])
    ++iv_len;
  if(iv_len==0)
    in_var=0;
  else{
   in_var=new THostVarArray(iv_len);
   for(int i=0;i<iv_len;++i)
    in_var->SetHostVar(i,HVD::CreateHostVar(hvd[i],1));
  }
 }
 rc=FetchErr;
 null_fetched=0;
 vect_size=VectSize;
 va=in_var;
 if(stmOpen(db))
  return;
 va=stm->SelectList();
 va_len=va->Size();
 cur_in_x=0;
 if(iv_len==0){
  executed=1;
  rc=stm->GetNext();
  null_fetched=0;
 }
}

void SelectStream::GetInNext(void)
{if(cur_in_x==iv_len-1){
  rewind();
  cur_in_x=0;
  executed=1;
 }else{
  ++cur_in_x;
  executed=0;
 }
} /* GetInNext */

#define CHECK_INPUT_VARS  \
        if(iv_len==0){    \
	 InternSetErr(4); \
	 return *this;    \
	} 

int SelectStream::CheckInType(int TypeCode)
{if((*in_var)[cur_in_x].isA()!=TypeCode){
  InternSetErr(0);
  return 0;
 }else
  return 1;
}

SelectStream& SelectStream::operator<<(char c)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tcchar)){
  char* tmp=(char*)(*in_var)[cur_in_x].Val();
  tmp[0]=c;
  tmp[1]='\0';
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(unsigned char c)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tcchar)){
  unsigned char* tmp=(unsigned char*)(*in_var)[cur_in_x].Val();
  tmp[0]=c;
  tmp[1]='\0';
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(const char* s)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tcchar)){
  char* tmp=(char*)(*in_var)[cur_in_x].Val();
  strcpy(tmp,s);
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(const unsigned char* s)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tcchar)){
  char* tmp=(char*)(*in_var)[cur_in_x].Val();
  strcpy(tmp,(char*)s);
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(int n)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tint)){
  int* tmp=(int*)(*in_var)[cur_in_x].Val();
  *tmp=n;
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(unsigned u)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tuint)){
  unsigned* tmp=(unsigned*)(*in_var)[cur_in_x].Val();
  *tmp=u;
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(short sh)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tshort)){
  short* tmp=(short*)(*in_var)[cur_in_x].Val();
  *tmp=sh;
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(long int l)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tlongint)){
  long* tmp=(long*)(*in_var)[cur_in_x].Val();
  *tmp=l;
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(float f)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tfloat)){
  float* tmp=(float*)(*in_var)[cur_in_x].Val();
  *tmp=f;
 }
 GetInNext();
 return *this;
}

SelectStream& SelectStream::operator<<(double d)
{CHECK_INPUT_VARS;
 if(CheckInType(HVD::tdouble)){
  double* tmp=(double*)(*in_var)[cur_in_x].Val();
  *tmp=d;
 }
 GetInNext();
 return *this;
}

#define CHECK_EXEC                      \
       if(!executed){                   \
	InternSetErr(3);                \
	return *this;                   \
       }

SelectStream& SelectStream::operator>>(char& c)
{CHECK_EXEC;
 SelStream::operator>>(c);   
 return *this;        
}

SelectStream& SelectStream::operator>>(unsigned char& c)
{CHECK_EXEC;
 SelStream::operator>>(c);   
 return *this;        
}

SelectStream& SelectStream::operator>>(char* s)
{CHECK_EXEC;
 SelStream::operator>>(s);   
 return *this;        
}

SelectStream& SelectStream::operator>>(unsigned char* s)
{CHECK_EXEC;
 SelStream::operator>>(s);   
 return *this;        
}

SelectStream& SelectStream::operator>>(int& n)
{CHECK_EXEC;
 SelStream::operator>>(n);   
 return *this;        
}

SelectStream& SelectStream::operator>>(unsigned& u)
{CHECK_EXEC;
 SelStream::operator>>(u);   
 return *this;        
}

SelectStream& SelectStream::operator>>(short& sh)
{CHECK_EXEC;
 SelStream::operator>>(sh);   
 return *this;        
}

SelectStream& SelectStream::operator>>(long int& l)
{CHECK_EXEC;
 SelStream::operator>>(l);   
 return *this;        
}

SelectStream& SelectStream::operator>>(float& f)
{CHECK_EXEC;
 SelStream::operator>>(f);   
 return *this;        
}

SelectStream& SelectStream::operator>>(double& d)
{CHECK_EXEC;
 SelStream::operator>>(d);   
 return *this;        
} 


InOutStream::InOutStream(PConnect db,char* StmText, short VectSize)
{PHostVarArray out_va; int ov_len;
 stm_text=new char[strlen(StmText)+1];
 strcpy(stm_text,StmText);
 {
  HVDArray hvd(stm_text);
  vect_size=VectSize;
  commit_on_flush=1;
  dirty=0;
  adb=db;
  va_len=0;
  while(hvd[va_len])
    ++va_len;
  va=new THostVarArray(va_len);
  for(int i=0;i<va_len;++i)
    va->SetHostVar(i,HVD::CreateHostVar(hvd[i],vect_size));
  ov_len=0; iv_len=0;
  for(int j=0;j<va_len;++j){
   if(hvd.v_status(j)==HVDArray::out)
    ++iv_len;
   else if(hvd.v_status(j)==HVDArray::in)
    ++ov_len;
  }
  if(iv_len>0)
   in_va=new THostVarArray(iv_len,SHOULD_NOT_DELETE);
  else
   in_va=0;
  if(ov_len>0)
   out_va=new THostVarArray(ov_len,SHOULD_NOT_DELETE);
  else
    out_va=0;
  ov_len=0; iv_len=0;
  {
   for(int j=0;j<va_len;++j){
    if(hvd.v_status(j)==HVDArray::out){
     ++iv_len;
     in_va->SetHostVar(iv_len-1,va->GetHostVarPtr(j));
    }else if(hvd.v_status(j)==HVDArray::in){
     ++ov_len;
     out_va->SetHostVar(ov_len-1,va->GetHostVarPtr(j)); 
    }
   }
  }
 }
 va_should_delete=1;
 stmOpen(db);
 va=out_va;
 va_len=ov_len;
 cur_in_x=0;
 cur_in_y=0;
 in_y_len=0;
 null_fetched=0;
 if(va_len==0){
  stm->Exec(vect_size);
  in_y_len=vect_size;
  cur_in_y=0;
  cur_in_x=0;
 }
}

InOutStream::~InOutStream()
{
 delete in_va;
}

int InOutStream::eof(void)
{
 if(iv_len==0)return 1;
 if(in_y_len==0)return 1;
 if(cur_in_y<=in_y_len-1)return 0;
 return 1;
}

void InOutStream::flush(void)
{if(va_len==0)return;
 in_y_len=cur_y+1;
 cur_in_y=0;
 cur_in_x=0;
 OStream::flush();
}

int InOutStream::is_null1(void)
{if(iv_len==0)return 0;
 if(in_y_len==0)return 0;
 if(in_y_len>0)
   return (*in_va)[cur_in_x].isNull(cur_in_y);
 return 0;
}

void InOutStream::GetInNext(void)
{
 if(iv_len==0)return;
 if(in_y_len==0)return;
 if(cur_in_x<iv_len-1)
   ++cur_in_x;
 else{
  if(cur_in_y<in_y_len-1){
   ++cur_in_y;
   cur_in_x=0;
  }else{
   cur_in_y=0;
   cur_in_x=0;
   in_y_len=0;
  }
 }
} /* GetInNext */

int InOutStream::CheckInType(int TypeCode)
{if((*in_va)[cur_in_x].isA()!=TypeCode){
  InternSetErr(0);
  return 0;
 }else
  return 1;
} /* CheckInType */

InOutStream& InOutStream::operator>>(char& c)
{if(eof())return *this;
 if(CheckInType(HVD::tcchar)){
  char* tmp=(char*)(*in_va)[cur_in_x].Val(cur_in_y);
  c=tmp[0];
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(unsigned char& c)
{if(eof())return *this;
 if(CheckInType(HVD::tcchar)){
  unsigned char* tmp=(unsigned char*)(*in_va)[cur_in_x].Val(cur_in_y);
  c=tmp[0];
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(char* s)
{if(eof())return *this;
 if(CheckInType(HVD::tcchar)){
  char* tmp=(char*)(*in_va)[cur_in_x].Val(cur_in_y);
  strcpy(s,tmp);
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(unsigned char* s)
{if(eof())return *this;
 if(CheckInType(HVD::tcchar)){
  char* tmp=(char*)(*in_va)[cur_in_x].Val(cur_in_y);
  strcpy((char*)s,tmp);
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(int& n)
{if(eof())return *this;
 if(CheckInType(HVD::tint)){
  int* tmp=(int*)(*in_va)[cur_in_x].Val(cur_in_y);
  n=*tmp;
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(unsigned& u)
{if(eof())return *this;
 if(CheckInType(HVD::tuint)){
  unsigned* tmp=(unsigned*)(*in_va)[cur_in_x].Val(cur_in_y);
  u=*tmp;
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(short& sh)
{if(eof())return *this;
 if(CheckInType(HVD::tshort)){
  short* tmp=(short*)(*in_va)[cur_in_x].Val(cur_in_y);
  sh=*tmp;
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(long int& l)
{if(eof())return *this;
 if(CheckInType(HVD::tlongint)){
  long* tmp=(long*)(*in_va)[cur_in_x].Val(cur_in_y);
  l=*tmp;
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(float& f)
{if(eof())return *this;
 if(CheckInType(HVD::tfloat)){
  float* tmp=(float*)(*in_va)[cur_in_x].Val(cur_in_y);
  f=*tmp;
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator>>(double& d)
{if(eof())return *this;
 if(CheckInType(HVD::tdouble)){
  double* tmp=(double*)(*in_va)[cur_in_x].Val(cur_in_y);
  d=*tmp;
  null_fetched=is_null1();
 }
 GetInNext();
 return *this;
}

InOutStream& InOutStream::operator<<(char c)
{
 if(va_len>0)OStream::operator<<(c);
 return *this;
}

InOutStream& InOutStream::operator<<(unsigned char c)
{
 if(va_len>0)OStream::operator<<(c);
 return *this;
}


InOutStream& InOutStream::operator<<(const char* s)
{
 if(va_len>0)OStream::operator<<(s);
 return *this;
}

InOutStream& InOutStream::operator<<(const unsigned char* s)
{
 if(va_len>0)OStream::operator<<(s);
 return *this;
}

InOutStream& InOutStream::operator<<(int n)
{
 if(va_len>0)OStream::operator<<(n);
 return *this;
}

InOutStream& InOutStream::operator<<(unsigned u)
{
 if(va_len>0)OStream::operator<<(u);
 return *this;
}

InOutStream& InOutStream::operator<<(short sh)
{
 if(va_len>0)OStream::operator<<(sh);
 return *this;
}

InOutStream& InOutStream::operator<<(long int l)
{
 if(va_len>0)OStream::operator<<(l);
 return *this;
}

InOutStream& InOutStream::operator<<(float f)
{
 if(va_len>0)OStream::operator<<(f);
 return *this;
}

InOutStream& InOutStream::operator<<(double d)
{
 if(va_len>0)OStream::operator<<(d);
 return *this;
}

InOutStream& InOutStream::operator<<(onull n)
{
 if(va_len>0)OStream::operator<<(n);
 return *this;
}

// -------------------------------------------
SQL::SQL(char* Connect)
{
 db=0;
 db=new TConnect(Connect);
}

SQL::~SQL()
{
 delete db;
 db=0;
}

void SQL::commit(void)
{
 db->Commit();
}

void SQL::roll_back(void)
{
 db->RollBack();
}

#ifdef __EXCEPTION_ENABLED__

int SQL::exec(char* Stm,int ignore_error)
{int rc;
 if(ignore_error){
  short r_mode=oraException::get_raise_mode();
  EX_DISABLE;
  rc=db->ExecDirect(Stm);
  err_clear();
  if(r_mode)EX_ENABLE;
 }else
  rc=db->ExecDirect(Stm);
 return rc;
}
#else

int SQL::exec(char* Stm,int ignore_error)
{int rc;
 if(ignore_error){
  int r_mode=except_flag;
  EX_DISABLE;
  rc=db->ExecDirect(Stm);
  err_clear();
  if(r_mode)EX_ENABLE;
 }else
  rc=db->ExecDirect(Stm);
 return rc;
}

#endif

void  SQL::err_clear(void)
{
 err::clear();
}

char* SQL::err_msg(void)
{
 return err::msg();
}

int SQL::err_code(void)
{
 return err::code();
}

SQLStream::SQLStream(SQL* sql,char* Stm,short BufSize)
{
 io=0;
 ss=0;
 db=sql;
 if(strncmp(Stm,"SELECT",6)==0||strncmp(Stm,"select",6)==0)
  ss=new SelectStream(db->db,Stm,BufSize);
 else
  io=new InOutStream(db->db,Stm,BufSize);
}

SQLStream::~SQLStream()
{
 delete io;
 delete ss;
}

int SQLStream::eof(void)
{
 if(io)
  return io->eof();
 else
  return ss->eof();
}

int  SQLStream::fail(void)
{
 if(io)
  return io->fail();
 else
  return ss->fail();
}

int  SQLStream::is_null(void)
{
 if(io)
  return io->is_null();
 else
  return ss->is_null();
}

void SQLStream::flush(void)
{
 if(io)io->flush();
}

void SQLStream::set_commit(int commit_when_flush)
{
 if(io)
  io->SetCommit(commit_when_flush);
}

void SQLStream::commit(void)
{
 db->commit();
}

void SQLStream::roll_back(void)
{
 db->roll_back();
}


SQLStream& SQLStream::operator<<(char c)
{
 if(io)
   io->operator<<(c);
 else
   ss->operator<<(c);
 return *this;
}

SQLStream& SQLStream::operator<<(unsigned char c)
{
 if(io)
   io->operator<<(c);
 else
   ss->operator<<(c);
 return *this; 
}

SQLStream& SQLStream::operator<<(const char* s)
{
 if(io)
   io->operator<<(s);
 else
   ss->operator<<(s);
 return *this; 
}

SQLStream& SQLStream::operator<<(const unsigned char* s)
{
 if(io)
   io->operator<<(s);
 else
   ss->operator<<(s);
 return *this; 
}

SQLStream& SQLStream::operator<<(int n)
{
 if(io)
   io->operator<<(n);
 else
   ss->operator<<(n);
 return *this; 
}

SQLStream& SQLStream::operator<<(unsigned u)
{
 if(io)
   io->operator<<(u);
 else
   ss->operator<<(u);
 return *this; 
}

SQLStream& SQLStream::operator<<(short sh)
{
 if(io)
   io->operator<<(sh);
 else
   ss->operator<<(sh);
 return *this; 
}

SQLStream& SQLStream::operator<<(long int l)
{
 if(io)
   io->operator<<(l);
 else
   ss->operator<<(l);
 return *this; 
}

SQLStream& SQLStream::operator<<(float f)
{
 if(io)
   io->operator<<(f);
 else
   ss->operator<<(f);
 return *this; 
}

SQLStream& SQLStream::operator<<(double d)
{
 if(io)
   io->operator<<(d);
 else
   ss->operator<<(d);
 return *this; 
}

SQLStream& SQLStream::operator<<(onull n)
{
 if(io)
   io->operator<<(onull());
 return *this; 
}

SQLStream& SQLStream::operator>>(char& c)
{
 if(io)
   io->operator>>(c);
 else
   ss->operator>>(c);
 return *this; 
}

SQLStream& SQLStream::operator>>(unsigned char& c)
{
 if(io)
   io->operator>>(c);
 else
   ss->operator>>(c);
 return *this; 
}

SQLStream& SQLStream::operator>>(char* s)
{
 if(io)
   io->operator>>(s);
 else
   ss->operator>>(s);
 return *this; 
}

SQLStream& SQLStream::operator>>(unsigned char* s)
{
 if(io)
   io->operator>>(s);
 else
   ss->operator>>(s);
 return *this; 
}

SQLStream& SQLStream::operator>>(int& n)
{
 if(io)
   io->operator>>(n);
 else
   ss->operator>>(n);
 return *this; 
}

SQLStream& SQLStream::operator>>(unsigned& u)
{
 if(io)
   io->operator>>(u);
 else
   ss->operator>>(u);
 return *this; 
}

SQLStream& SQLStream::operator>>(short& sh)
{
 if(io)
   io->operator>>(sh);
 else
   ss->operator>>(sh);
 return *this; 
}

SQLStream& SQLStream::operator>>(long int& l)
{
 if(io)
   io->operator>>(l);
 else
   ss->operator>>(l);
 return *this; 
}

SQLStream& SQLStream::operator>>(float& f)
{
 if(io)
   io->operator>>(f);
 else
   ss->operator>>(f);
 return *this; 
}

SQLStream& SQLStream::operator>>(double& d)
{
 if(io)
   io->operator>>(d);
 else
   ss->operator>>(d);
 return *this; 
}
