/* *******************************************************************

The Oracle SQL Class Library Version 1.0,
Author: Sergei Kuchin
Copyright (C) Mosakin Corp., 1995
This library is free software.  Permission to use, copy,
modify and redistribute the Oracle SQL class library for any
purpose is hereby granted without fee, provided that the
above copyright notice appear in all copies.


******************************************************************* */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <csql_req.h>
#include <sqlpipe.h>

static TBufHead buf_head;
static TBuf buf;
static int inp,outp;
static int fdc[2]; 
static int fds[2];

void default_handler(char* s);

static short connected=0;
static ERR_HANDLER handler=default_handler;
static ERR_HANDLER external_handler=0;
static int timeout_on=0;

void sql_set_handler(ERR_HANDLER ahandler)
{
 external_handler=ahandler;
}

ERR_HANDLER sql_get_handler(void)
{
 return external_handler;
}

static void TimeOut(int p)
{
 if(timeout_on){
  if(sql_get_handler())(sql_get_handler())(SQL_TIME_OUT_MSG);
  exit(1);
 }
} /* TimeOut */

static void SetTimeOut(void)
{timeout_on=1;
 signal(SIGALRM,TimeOut);
 alarm(SQL_TIME_OUT);
} /* SetTimeOut */

static int Read(int inp,void* buf,unsigned nbyte)
{int n;
 SetTimeOut();
 n=read(inp,buf,nbyte);
 timeout_on=0;
 return n;
} /* Read */

static int Write(int inp,void* buf,unsigned nbyte)
{int n;
 SetTimeOut();
 n=write(inp,buf,nbyte);
 timeout_on=0;
 return n;
} /* Read */

static void WrBufHead(short code,short len,short user)
{
 buf_head.code=code;
 buf_head.len=len;
 buf_head.user=user;
 Write(outp,&buf_head,sizeof(buf_head));
}

static int RdBufHead(void)
{
 return Read(inp,&buf_head,sizeof(buf_head));
} /* RdBufHead */

static int RdBuf(void)
{
 return Read(inp,buf,buf_head.len);
} /* RdBuf */

static void WrBuf(void)
{
 Write(outp,buf,buf_head.len);
} /* WrBuf */

static int ExecProg(char* path,char* arg,...)
{char **p,*argv[64];
 int i=1,status,rc;
 pid_t ps;
 
 argv[0]=path;
 for(p=&arg;*p;++p)
  argv[i++]=*p;
 argv[i]=0;
 ps=fork();
 if(ps==-1)return -1;
 if(ps>0){
  RdBufHead();
  switch(buf_head.code){
  case cNop:
   return 0;
  case cNotFound:
/*   "SQLPIPE not found" */
   return -2;
  case cError:
   RdBuf();
   fprintf(stderr,"%s\n",buf);
   return -1;
  default:
   return -1;
  }
 }else{
  rc=execvp(path,argv);
  if(rc==-1){
   buf_head.code=cNotFound;
   buf_head.len=0;
   Write(fdc[1],&buf_head,sizeof(buf_head));
   exit(0);
  }
  return -1;
 }
} /* ExecProg */

int sql_open(char* Connect)
{int rc;
 char fdc0[10];
 char fdc1[10];
 char fds0[10];
 char fds1[10];

 rc=pipe(fdc);
 if(rc==-1)return 1;
 sprintf(fdc0,"%d",fdc[0]);
 sprintf(fdc1,"%d",fdc[1]);
 rc=pipe(fds);
 if(rc==-1)return 1;
 sprintf(fds0,"%d",fds[0]);
 sprintf(fds1,"%d",fds[1]);
 inp=fdc[0];
 outp=fds[1];
 rc=ExecProg("sqlpipe",fds0,fdc1,Connect,(char*)0);
 if(rc!=0){
  if(rc=-2)
    return 2;  
  else
    return 1;
 }
   else{
  connected=1;
  return 0;
 }
}

int sql_close(void)
{
 connected=0;
 WrBufHead(cEnd,0,0);
 RdBufHead();
 close(inp);
 close(outp);
 if(buf_head.code==cNop)
  return 0; 
 else
  return 1;
}

void sql_commit(void)
{
 WrBufHead(cCommit,0,0);
 RdBufHead();
 (*handler)(0);
}

void sql_roll_back(void)
{
 WrBufHead(cRollBack,0,0);
 RdBufHead();
 (*handler)(0);
}

int sql_exec(char* Stm,int ignore_error)
{short len=strlen(Stm)+1;
 WrBufHead(cExec,len,ignore_error);
 strcpy(buf,Stm);
 WrBuf();
 RdBufHead();
 (*handler)(0);
 return buf_head.len; 
}

void default_handler(char* s)
{if(!connected)return;
 if(buf_head.code!=cError)
  return;
 RdBuf();
 if(sql_get_handler())(*sql_get_handler())(buf);
  exit(1);
} /* default_handler */

int sql_stream_open(char* Stm,short BufSize)
{int len;
 if(!connected)return -1;
 len=strlen(Stm)+1;
 WrBufHead(cOpenStream,len,BufSize);
 strcpy(buf,Stm);
 WrBuf();
 RdBufHead();
 (*handler)(0);
 return buf_head.user;
}

void sql_stream_close(int sd)
{if(!connected)return; 
 WrBufHead(cCloseStream,0,sd);
 RdBufHead();
 (*handler)(0);
}

int sql_eof(int sd)
{if(!connected)return 1;
 WrBufHead(cEof,0,sd);
 RdBufHead();
 (*handler)(0);
 return buf_head.user; 
}

int sql_is_null(int sd)
{if(!connected)return 0;
 WrBufHead(cIsNull,0,sd);
 RdBufHead();
 (*handler)(0);
 return buf_head.user; 
}

void sql_flush(int sd)
{if(!connected)return;
 WrBufHead(cFlush,0,sd);
 RdBufHead();
 (*handler)(0);
}

void sql_set_commit(int sd,int commit_when_flush)
{if(!connected)return;
 WrBufHead(cSetCommit,commit_when_flush,sd);
 RdBufHead();
 (*handler)(0);
}

void sql_putc(int sd,char c)
{if(!connected)return;
 WrBufHead(cWrChar,1,sd);
 buf[0]=c;
 WrBuf();
 RdBufHead();
 (*handler)(0);
}

void sql_putuc(int sd,unsigned char c)
{if(!connected)return;
 WrBufHead(cWrUChar,1,sd);
 {
  unsigned char* tmp=(unsigned char*)buf;
  *tmp=c;
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_puts(int sd,const char* s)
{if(!connected)return;
 {
  int len=strlen(s)+1;
  WrBufHead(cWrStr,len,sd);
  strcpy(buf,s);
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_putus(int sd,const unsigned char* s)
{if(!connected)return;
 {
  int len=strlen((char*)s)+1;
  WrBufHead(cWrUStr,len,sd);
  strcpy(buf,(char*)s);
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_puti(int sd,int n)
{if(!connected)return;
 WrBufHead(cWrInt,sizeof(int),sd);
 {
  int* tmp=(int*)buf;
  *tmp=n;
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_putu(int sd,unsigned u)
{if(!connected)return;
 WrBufHead(cWrUInt,sizeof(unsigned),sd);
 {
  unsigned* tmp=(unsigned*)buf;
  *tmp=u;
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_putsh(int sd,short sh)
{if(!connected)return;
 WrBufHead(cWrShort,sizeof(short),sd);
 {
  short* tmp=(short*)buf;
  *tmp=sh;
  WrBuf();
  RdBufHead();
 (*handler)(0);
 }
}

void sql_putl(int sd,long int l)
{if(!connected)return;
 WrBufHead(cWrLongInt,sizeof(long),sd);
 {
  long* tmp=(long*)buf;
  *tmp=l;
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_putf(int sd,float f)
{if(!connected)return;
 WrBufHead(cWrFloat,sizeof(float),sd);
 {
  float* tmp=(float*)buf;
  *tmp=f;
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_putd(int sd,double d)
{if(!connected)return;
 WrBufHead(cWrDouble,sizeof(double),sd);
 {
  double* tmp=(double*)buf;
  *tmp=d;
  WrBuf();
  RdBufHead();
  (*handler)(0);
 }
}

void sql_putnull(int sd)
{if(!connected)return;
 WrBufHead(cWrONull,0,sd);
 RdBufHead();
 (*handler)(0);
}

void sql_getc(int sd,char* c)
{if(!connected)return;
 WrBufHead(cRdChar,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 *c=buf[0];
}

void sql_getuc(int sd,unsigned char* c)
{if(!connected)return;
 WrBufHead(cRdUChar,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 *c=(unsigned char)buf[0];
}

void sql_gets(int sd,char* s)
{if(!connected)return;
 WrBufHead(cRdStr,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 strcpy(s,buf);
}

void sql_getus(int sd,unsigned char* s)
{if(!connected)return;
 WrBufHead(cRdUStr,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 strcpy((char*)s,buf);
}

void sql_geti(int sd,int* n)
{if(!connected)return;
 WrBufHead(cRdInt,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 {
  int* tmp=(int*)buf;
  *n=*tmp;
 }
}

void sql_getu(int sd,unsigned* u)
{if(!connected)return;
 WrBufHead(cRdUInt,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 {
  unsigned* tmp=(unsigned*)buf;
  *u=*tmp;
 }
}

void sql_getsh(int sd,short* sh)
{if(!connected)return;
 WrBufHead(cRdShort,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 {
  short* tmp=(short*)buf;
  *sh=*tmp;
 }
}

void sql_getl(int sd,long int* l)
{if(!connected)return;
 WrBufHead(cRdLongInt,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 {
  long* tmp=(long*)buf;
  *l=*tmp;
 }
}

void sql_getf(int sd,float* f)
{if(!connected)return;
 WrBufHead(cRdFloat,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 {
  float* tmp=(float*)buf;
  *f=*tmp;
 }
}

void sql_getd(int sd,double* d)
{if(!connected)return;
 WrBufHead(cRdDouble,0,sd);
 RdBufHead();
 (*handler)(0);
 RdBuf();
 {
  double* tmp=(double*)buf;
  *d=*tmp;
 }
}

void sql_printf(int sd,char* fmt,...)
{char *c=fmt;
 char *arg=(char*)&fmt+sizeof(char*);

 while(*c){
  if(*c=='%'){
   ++c;
   switch(*c){
   case 'N':
    sql_putnull(sd);
    break;
   case 'f':
    {double* f=(double*)arg;
     sql_putf(sd,*f);
     arg+=sizeof(double);
     break;
    }
   case 'd':
    {int* n=(int*)arg;
     sql_puti(sd,*n);
     arg+=sizeof(int);
     break;
    }
   case 'u':
    {unsigned* u=(unsigned*)arg;
     sql_putu(sd,*u);
     arg+=sizeof(unsigned);
     break;
    }
   case 'l':
    ++c;
    switch(*c){
    case 'd':
     {long* l=(long*)arg;
      sql_putl(sd,*l);
      arg+=sizeof(long);
      break;
     }
   case 'f':
     {double* f=(double*)arg;
      sql_putd(sd,*f);
      arg+=sizeof(double);
      break;
     }
    }
    break;    
   case 'c':
    {int* n=(int*)arg;
     sql_putc(sd,*n);
     arg+=sizeof(int);
     break;
    }
   case 's':
    {char** s=(char**)arg;
     sql_puts(sd,*s);
     arg+=sizeof(char*);
     break;
    }
   }
  }
  ++c;
 }
}

void sql_scanf(int sd,char* fmt,...)
{char *c=fmt;
 char *arg=(char*)&fmt+sizeof(char*);

 while(*c){
  if(*c=='%'){
   ++c;
   switch(*c){
   case 'f':
    {float** f=(float**)arg;
     sql_getf(sd,*f);
     arg+=sizeof(float*);
     break;
    }
   case 'd':
    {int** n=(int**)arg;
     sql_geti(sd,*n);
     arg+=sizeof(int*);
     break;
    }
   case 'u':
    {unsigned** u=(unsigned**)arg;
     sql_getu(sd,*u);
     arg+=sizeof(unsigned*);
     break;
    }
   case 'l':
    ++c;
    switch(*c){
    case 'd':
     {long** l=(long**)arg;
      sql_getl(sd,*l);
      arg+=sizeof(long*);
      break;
     }
   case 'f':
     {double** f=(double**)arg;
      sql_getd(sd,*f);
      arg+=sizeof(double*);
      break;
     }
    }
    break;    
   case 'c':
    {char** n=(char**)arg;
     sql_getc(sd,*n);
     arg+=sizeof(char*);
     break;
    }
   case 's':
    {char** s=(char**)arg;
     sql_gets(sd,*s);
     arg+=sizeof(char*);
     break;
    }
   }
  }
  ++c;
 }

}


/* The following format specifiers are provided:
    %d  - int
    %u  - unsigned 
    %ld - long int
    %f  - float
    %lf - double
    %c  - char
    %s  - string
*/

