/*
 * This file forms part of CorVu dbCGI
 *
 * CorVu dbCGI is Copyright (C) 1995  CorVu Pty Ltd.
 * Written by Troy Rollo
 *
 * CorVu dbCGI is free software. It may be modified and redistributed under
 * the terms of the CorVu General Public License, either version 1, or, at
 * your option, any later version. Ensure that you read this license before
 * modifying or redistributing the software.
 *
 * THIS PROGRAM IS PROVIDED "AS IS" WITH NO WORRANTY OF ANY KIND. YOU USE
 * THIS PROGRAM ENTIRELY AT YOUR OWN RISK. NEITHER CORVU, NOR ANY OTHER
 * PARTY MAY BE HELD RESPONSIBLE FOR ANY DAMAGES ARISING FROM YOUR USE OR
 * MISUSE OF THIS PROGRAM.
 */

#include "dbcgi.h"
#include <sybfront.h>
#include <sybdb.h>
#include <syberror.h>

typedef struct __dbw_conninfo
{
	DBPROCESS	*dbproc;
} dbw_conninfo;

static dbw_typemap atm[] =
{
 { SYBCHAR,	0,		NTBSTRINGBIND,	DBWT_Char },
 { SYBTEXT,	0,		NTBSTRINGBIND,	DBWT_Char },
 { SYBBINARY,	0,		BINARYBIND,	DBWT_Raw_Lidx },
 { SYBIMAGE,	0,		BINARYBIND,	DBWT_Raw_Lidx },
 { SYBINT1,	0,		INTBIND,	DBWT_Int },
 { SYBINT2,	0,		INTBIND,	DBWT_Int },
 { SYBINT4,	0,		INTBIND,	DBWT_Int },
 { SYBFLT8,	0,		FLT8BIND,	DBWT_Float },
 { SYBMONEY,	0,		FLT8BIND,	DBWT_Float },
 { SYBBIT,	2,		NTBSTRINGBIND,	DBWT_Char },
 { SYBDATETIME,	PRLDATETIME+3,	NTBSTRINGBIND,	DBWT_Char},
#ifdef SYBREAL
 { SYBREAL,	0,		FLT8BIND,	DBWT_Float },
#endif
#ifdef SYBMONEY4
 { SYBMONEY4,	PRMONEY+1,	NTBSTRINGBIND,	DBWT_Float },
#endif
#ifdef SYBDATETIME4
 { SYBDATETIME4,PRLDATETIME+3,	NTBSTRINGBIND,	DBWT_Char},
#endif
};

static	char	*pchCurrentSQL;

int	SybHandleError(DBPROCESS *dbproc,
			int	iSeverity,
			int	iError,
			int	iOSErr,
			char	*pchError,
			char	*pchOSErr)
{
	if (iOSErr != DBNOERR)
		FormatErrors(iOSErr, pchOSErr, pchCurrentSQL);
	else
		FormatErrors(iError, pchError, pchCurrentSQL);
	return INT_CANCEL;
}

int	SybHandleMessage(DBPROCESS *dbproc,
			DBINT	iMsgNo,
			int	iMsgState,
			int	iSeverity,
			char	*pchText,
			char	*pchServer,
			char	*pchProcName,
			int	iLine)
{
	if (iSeverity > 1)
		FormatErrors(iMsgNo, pchText, pchCurrentSQL);
	return 0;
}

static	void	RunQuery(dbw_value *pval,
			int iGetResults)
{
	int	iColumn;
	int	nColumns;
	char	*pchCommand = GetValue(pval, "Query");
	int	iColumnWidth;
	int	iColType;
	char	achColumnName[256];
	dbw_result *prslt;
	DBINT	*piIndicators;
	dbw_bindinfo *pbi;
	dbw_conninfo *pinfo = FindConnection(pval);
	char	*pchColName;

	pchCurrentSQL = pchCommand;
	if (!pinfo ||
	    !pchCommand ||
            dbcmd(pinfo->dbproc, pchCommand) == FAIL)
	{
		FreeString(pchCommand);
		return;
	}
	if (dbsqlexec(pinfo->dbproc) == FAIL ||
	    dbresults(pinfo->dbproc) == FAIL)
	{
		FreeString(pchCommand);
		dbfreebuf(pinfo->dbproc);
		return;
	}

	if (!iGetResults)
	{
		dbcanquery(pinfo->dbproc);
		return;
	}

	nColumns = dbnumcols(pinfo->dbproc);

	prslt = new2(dbw_result, nColumns);
	piIndicators = new2(DBINT, nColumns);

	for (iColumn = 0; iColumn < nColumns; iColumn++)
	{
		pchColName = dbcolname(pinfo->dbproc, iColumn + 1);
		iColType = dbcoltype(pinfo->dbproc, iColumn + 1);
		iColumnWidth = dbcollen(pinfo->dbproc, iColumn + 1);
		if (pchColName)
		{
			strncpy(achColumnName,
				pchColName,
				sizeof(achColumnName) - 1);
			achColumnName[sizeof(achColumnName) - 1] = 0;
		}
		else
		{
			*achColumnName = 0;
		}
		pbi = FindColInfo(atm,
				rangeof(atm),
				&prslt[iColumn],
				achColumnName,
				iColType,
				iColumnWidth,
				-1);
		dbbind(pinfo->dbproc,
			iColumn + 1,
			pbi->iBindType,
			pbi->iWidth,
			pbi->pchData +
			  (pbi->iBindType == BINARYBIND ? 4 : 0));
		dbnullbind(pinfo->dbproc,
			iColumn + 1,
			piIndicators + iColumn);
	}

	FormatHeadings(prslt, nColumns);
	while (dbnextrow(pinfo->dbproc) != NO_MORE_ROWS)
	{
		for (iColumn = 0; iColumn < nColumns; iColumn++)
		{
			prslt[iColumn].iIsNull = (piIndicators[iColumn] == -1);
			if (prslt[iColumn].type == DBWT_Raw_Lidx)
				*(long *) prslt[iColumn].pchCharValue =
					dbdatlen(pinfo->dbproc, iColumn + 1);
		}
		FormatOutput(prslt, nColumns);
	}
	dbcanquery(pinfo->dbproc);
}


void
DBQuery(dbw_value *pval)
{
	RunQuery(pval, 1);
}

void
DBExecute(dbw_value *pval)
{
	RunQuery(pval, 0);
}

void
DBDisconnect(dbw_value *pval)
{
	dbw_conninfo *pinfo = FindConnection(pval);

	pchCurrentSQL = "Disconnect";
	if (!pinfo)
		return;
	dbcancel(pinfo->dbproc);
	dbclose(pinfo->dbproc);
	DropConnection(pval);
}

void
DBInit(dbw_value *pval)
{
	char		*pchSybase = GetValue(pval, "SYBASE");

	pchCurrentSQL = "Init";
	PutEnvVar("SYBASE", pchSybase);
	FreeString(pchSybase);
	dberrhandle((int (*)()) SybHandleError);
		dbmsghandle((int (*)()) SybHandleMessage);
}

void
DBUnInit(dbw_value *pval)
{
	pchCurrentSQL = "UnInit";
	dbexit();
}

void
DBConnect(dbw_value *pval)
{
	char		*pchUser = GetValue(pval, "USER");
	char		*pchPass = GetValue(pval, "PASS");
	char		*pchServer = GetValue(pval, "SERVER");
	char		*pchAppName = GetValue(pval, "APPNAME");
	dbw_conninfo	*pinfo = new(dbw_conninfo);
	LOGINREC	*plr;

	pchCurrentSQL = "Connect";
	if (!AddConnection(pval, pinfo))
	{
		free(pinfo);
		return;
	}
	plr = dblogin();
	if (!plr)
	{
		DropConnection(pval);
		return;
	}
	DBSETLUSER(plr, pchUser);
	DBSETLPWD(plr, pchPass);
	DBSETLAPP(plr, pchAppName ? pchAppName : "sybdbcgi");
	pinfo->dbproc = dbopen(plr, pchServer);
	dbloginfree(plr);
	if (!pinfo->dbproc)
	{
		DropConnection(pval);
		return;
	}
	FreeString(pchAppName);
	FreeString(pchServer);
	FreeString(pchUser);
	FreeString(pchPass);
}

