unit Main;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ShellAPI, ExtCtrls, StdCtrls, Buttons, IniFiles, Menus;

const
  ProgramStr      = 'AppLaunch';
  VersionStr      = '1.3';
  CopyrightStr    = ' Copyright 1994-1995 by Anders Ohlsson';
  EmailAdrStr     = 'ao@e.kth.se, ao@it.kth.se or ao@sto.foa.se';

  CopyrightNotice : String = ProgramStr+' '+VersionStr+' - '+
                             CopyrightStr+' - '+EmailAdrStr;

  IniFile         : String = 'LAUNCH.INI';
  WriFile         : String = 'LAUNCH.WRI';

  AbsMaxNrOfApps = 32;
  AppWidth       = 36;

type
  TSlotRec = record
    Active    : Boolean;
    Command,
    Directory,
    Hint,
    IconFile  : String;
    IconNr    : Word;
    Panel     : TPanel;
    Image     : TImage;
    Icon      : TIcon;
  end;

  TMainForm = class(TForm)
    PopupMenu1: TPopupMenu;
    Help1: TMenuItem;
    ViewDocs: TMenuItem;
    About1: TMenuItem;
    Save1: TMenuItem;
    Load1: TMenuItem;
    N1: TMenuItem;
    Exit1: TMenuItem;
    Properties1: TMenuItem;
    N2: TMenuItem;
    StatusBar: TPanel;
    Toggle1: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure Load1Click(Sender: TObject);
    procedure Save1Click(Sender: TObject);
    procedure ViewDocsClick(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure StatusBarDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure StatusBarDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure PopupMenu1Popup(Sender: TObject);
    procedure Properties1Click(Sender: TObject);
    procedure Toggle1Click(Sender: TObject);
    procedure FormActivate(Sender: TObject);
  private
    { Private declarations }
    Slot         : array [0..AbsMaxNrOfApps-1] of TSlotRec;
    ExecFailed,
    Modified,
    DragOn       : Boolean;
    MousePos     : TPoint;
    procedure DropFiles(var Msg : TWMDropFiles); message WM_DROPFILES;
    procedure ApplicationHint(Sender : TObject);
    procedure ImageDragOver(Sender, Source : TObject; X, Y : Integer;
                            State : TDragState; var Accept : Boolean);
    procedure ImageDragDrop(Sender, Source : TObject; X, Y : Integer);
    procedure ImageDblClick(Sender : TObject);
    procedure ImageMouseDown(Sender : TObject; Button : TMouseButton;
                             Shift : TShiftState; X, Y : Integer);
    procedure ImageMouseUp(Sender : TObject; Button : TMouseButton;
                           Shift : TShiftState; X, Y : Integer);
    procedure ImageMouseMove(Sender : TObject; Shift : TShiftState;
                             X, Y : Integer);
    procedure LoadLayout;
    procedure SaveLayout;
  public
    { Public declarations }
  end;

var
  MainForm     : TMainForm;
  ScreenWidth,
  MaxNrOfApps,
  NrOfApps     : Integer;

implementation

uses Info, About, Resource;

{$R *.DFM}

{
  DropFiles is called when a file is dragged from the file manager and
  dropped onto the application
}
procedure TMainForm.DropFiles(var Msg : TWMDropFiles);
var
  FileName,
  IconFile   : array [0..255] of Char;
  NrOfFiles,
  BufSize,
  IconNr     : Word;
  TheIcon    : HIcon;
  Point      : TPoint;
  j          : Integer;
begin
  { If the bar is full, inform the user and exit }
  if NrOfApps = MaxNrOfApps then begin
    MessageDlg('Sorry! The bar is full...',mtError,[mbOK],0);
    DragFinish(Msg.Drop);
    Exit;
  end;

  { Get the coordinates where the file was dropped }
  DragQueryPoint(Msg.Drop,Point);

  { Get the app number from the drop point }
  j := Point.x div AppWidth;

  { If the slot is taken, inform the user and exit }
  if Slot[j].Active then begin
    MessageDlg('Sorry! That slot is already taken...',mtError,[mbOK],0);
    DragFinish(Msg.Drop);
    Exit;
  end;

  { Get the number of files that were dropped on us }
  NrOfFiles := DragQueryFile(Msg.Drop,Word(-1),FileName,BufSize);

  { If more than one file was dropped, inform the user and exit }
  if NrOfFiles > 1 then begin
    MessageDlg('Hey! Only one file at a time...',mtError,[mbOK],0);
    DragFinish(Msg.Drop);
    Exit;
  end;

  { Get the name of the file that was dropped on us }
  DragQueryFile(Msg.Drop,0,FileName,BufSize);

  { Initialize the info form }
  InfoForm.CmdStr.Text := StrPas(FileName);
  InfoForm.DirStr.Text := ExtractFilePath(StrPas(FileName));
  InfoForm.InfoStr.Text := '';
  InfoForm.IconFileStr.Text := '';
  InfoForm.IconNrStr.Text := '';

  { Show the info form }
  InfoForm.ShowModal;

  { If the OK button was pressed we go ahead }
  if InfoForm.ModalResult = idOK then begin
    { Set some attributes for this slot }
    Slot[j].Active := True;
    Slot[j].Command := InfoForm.CmdStr.Text;
    Slot[j].Directory := InfoForm.DirStr.Text;
    Slot[j].Hint := InfoForm.InfoStr.Text;
    Slot[j].Panel.Hint := '| '+Slot[j].Hint;
    if InfoForm.IconFileStr.Text <> '' then begin
      Slot[j].IconFile := InfoForm.IconFileStr.Text;
      StrPCopy(IconFile,Slot[j].IconFile);
    end
    else begin
      Slot[j].IconFile := '';
      StrPCopy(IconFile,InfoForm.CmdStr.Text);
    end;
    try
      IconNr := StrToInt(InfoForm.IconNrStr.Text);
    except
      on EConvertError do
        IconNr := 0;
    end;
    Slot[j].IconNr := IconNr;

    { Extract the icon }
    TheIcon := ExtractIcon(HInstance,IconFile,IconNr);
    if TheIcon < 2 then
      TheIcon := LoadIcon(0,idi_Question);

    { Set some attributes for this slot }
    Slot[j].Icon.Handle := TheIcon;

    { Initialize the image and draw the icon }
    with Slot[j].Image.Canvas do begin
      Pen.Color := clSilver;
      Pen.Style := psSolid;
      Brush.Color := clSilver;
      Brush.Style := bsSolid;
      Rectangle(0,0,AppWidth-3,AppWidth-3);
      Draw(0,0,Slot[j].Icon);
    end;

    { Increment the number of apps }
    Inc(NrOfApps);

    { Raise modification flag }
    Modified := True;
  end;

  { Tell Windows that we're done for now }
  DragFinish(Msg.Drop);
end;

{
  LoadLayout loads the layout from the INI file
}
procedure TMainForm.LoadLayout;
var
  MyIniFile      : TIniFile;
  MyStringList1,
  MyStringList2  : TStringList;
  IconFile       : array [0..255] of Char;
  IconNr         : Word;
  TheIcon        : HIcon;
  i, j           : Integer;
  WasChecked     : Boolean;
begin
  { Reset all slots }
  for i:=0 to MaxNrOfApps-1 do
    with Slot[i] do begin
      { Invalidate the slot }
      Active := False;
      Command := '';
      Directory := '';
      Hint := '';
      IconFile := '';
      IconNr := 0;
      Panel.Hint := '';

      { Initialize the image }
      with Image.Canvas do begin
        Pen.Color := clSilver;
        Pen.Style := psSolid;
        Brush.Color := clSilver;
        Brush.Style := bsSolid;
        Rectangle(0,0,AppWidth-3,AppWidth-3);
      end;
    end;

  { Create the string lists }
  MyStringList1 := TStringList.Create;
  MyStringList1.Sorted := True; { This will automagically ignore duplicates }
  MyStringList2 := TStringList.Create;

  { Create the ini file object }
  MyIniFile := TIniFile.Create(IniFile);

  { Remember if the resources was visible before }
  WasChecked := Toggle1.Checked;

  { Determine if the resources should be visible or not }
  Toggle1.Checked := MyIniFile.ReadBool(ProgramStr,'Resources',True);

  { Read the section that tells us which apps are to be loaded }
  MyIniFile.ReadSection(ProgramStr,MyStringList1);

  { Calculate the number of apps we've got }
  NrOfApps := MyStringList1.Count;
  if NrOfApps < 0 then
    NrOfApps := 0;

  { Process the apps one by one }
  for i:=0 to NrOfApps-1 do begin
    { Skip if invalid section prefix }
    if Copy(MyStringList1[i],1,3) <> 'App' then
      continue;

    { Determine section number, skip if invalid }
    try
      j := StrToInt(Copy(MyStringList1[i],4,2));
    except
      { Skip this app if conversion fails }
      on EConvertError do
        continue;
    end;

    { Skip this app if the number is too high }
    if j > MaxNrOfApps-1 then
      continue;

    { Read this app's section }
    MyStringList2.Clear;
    MyIniFile.ReadSectionValues(MyStringList1[i],MyStringList2);

    { Set some attributes for this slot }
    Slot[j].Active := True;
    Slot[j].Command := MyStringList2.Values['ProgFile'];
    Slot[j].Directory := MyStringList2.Values['WorkDir'];
    Slot[j].Hint := MyStringList2.Values['ProgInfo'];
    Slot[j].Panel.Hint := '| '+Slot[j].Hint;

    { Determine where the icon is found }
    Slot[j].IconFile := MyStringList2.Values['IconFile'];
    StrPCopy(IconFile,Slot[j].IconFile);
    if IconFile[0] = #0 then
      StrPCopy(IconFile,Slot[j].Command);

    { Determine which icon to get }
    try
      IconNr := StrToInt(MyStringList2.Values['IconNr']);
    except
      on EConvertError do
        IconNr := 0;
    end;
    Slot[j].IconNr := IconNr;

    { Extract the icon }
    TheIcon := ExtractIcon(HInstance,IconFile,IconNr);
    if TheIcon < 2 then
      TheIcon := LoadIcon(0,idi_Question);

    { Set some attributes for this slot }
    Slot[j].Icon.Handle := TheIcon;

    { Draw the icon on the image }
    Slot[j].Image.Canvas.Draw(0,0,Slot[j].Icon);
  end;

  { Free the ini file object }
  MyIniFile.Free;

  { Free the string lists }
  MyStringList1.Free;
  MyStringList2.Free;

  { If we're reloading we need to adjust the resource and main forms }
  if ResourceForm <> nil then begin
    { Enable/disable the resource form accordingly }
    ResourceForm.Visible := Toggle1.Checked;

    { Modify the size of the main form accordingly }
    if not WasChecked and Toggle1.Checked then
      Width := Width-3*AppWidth;
    if WasChecked and not Toggle1.Checked then
      Width := Width+3*AppWidth;
  end;

  { Clear modification flag }
  Modified := False;
end;

{
  SaveLayout saves the layout to the INI file
}
procedure TMainForm.SaveLayout;
var
  MyIniFile : TIniFile;
  Section   : String;
  i         : Integer;
  fp        : TextFile;
begin
  { Destroy the INI file so that we don't collect a lot of garbage }
  AssignFile(fp,IniFile);
  Erase(fp);

  { Create the ini file object }
  MyIniFile := TIniFile.Create(IniFile);

  { Remember if the resource form was visible or not }
  MyIniFile.WriteBool(ProgramStr,'Resources',Toggle1.Checked);

  for i:=0 to MaxNrOfApps-1 do
    with MyIniFile do
      with Slot[i] do
        if Active then begin
          { Determine the section name }
          Section := 'App'+IntToStr(i div 10)+IntToStr(i mod 10);

          { Write the AppXX=1 entry in the first section }
          WriteInteger(ProgramStr,Section,1);

          { Write the AppXX section }
          WriteString(Section,'ProgFile',Command);
          if Directory <> '' then
            WriteString(Section,'WorkDir',Directory);
          WriteString(Section,'ProgInfo',Hint);
          if IconFile <> '' then
            WriteString(Section,'IconFile',IconFile);
          if IconNr <> 0 then
            WriteInteger(Section,'IconNr',IconNr);
        end;

  { Free the ini file object }
  MyIniFile.Free;

  { Clear modification flag }
  Modified := False;
end;

{
  FormCreate is called when the form is created
}
procedure TMainForm.FormCreate(Sender: TObject);
var
  i : Integer;
begin
  { Tell Windows that we accept dragged files }
  DragAcceptFiles(Handle,True);

  { We're not dragging anything yet }
  DragOn := False;

  { Set the application's title }
  Application.Title := ProgramStr;

  { Set the application's OnHint event handler }
  Application.OnHint := ApplicationHint;

  { Set the initial caption for our statusbar }
  StatusBar.Font.Size := 7;
  StatusBar.Caption := CopyrightNotice;

  { Calculate the maximum number of apps we can have }
  ScreenWidth := GetSystemMetrics(sm_cxScreen);
  MaxNrOfApps := ScreenWidth div AppWidth;
  if MaxNrOfApps > AbsMaxNrOfApps then
    MaxNrOfApps := AbsMaxNrOfApps;

  { Modify our coordinates and size }
  Top := 0;
  Left := 0;
  Width := AppWidth*MaxNrOfApps+1;
  Height := AppWidth+StatusBar.Height;

  { Create a panel and an image for each slot }
  for i:=0 to MaxNrOfApps-1 do
    with Slot[i] do begin
      { Create the panel }
      Panel := TPanel.Create(MainForm);
      Panel.Left := AppWidth*i;
      Panel.Top := 0;
      Panel.Width := AppWidth+1;
      Panel.Height := AppWidth+1;
      Panel.BevelOuter := bvRaised;
      Panel.BorderStyle := bsSingle;
      Panel.ShowHint := True;
      Panel.Parent := MainForm;

      { Create the image }
      Image := TImage.Create(Panel);
      Image.Align := alClient;
      Image.Parent := Panel;

      { Set event handlers for the image }
      Image.OnDragDrop := ImageDragDrop;
      Image.OnDragOver := ImageDragOver;
      Image.OnDblClick := ImageDblClick;
      Image.OnMouseDown := ImageMouseDown;
      Image.OnMouseUp := ImageMouseUp;
      Image.OnMouseMove := ImageMouseMove;

      { Create the icon }
      Icon := TIcon.Create;
    end;

  { Add path information to our filenames }
  IniFile := ExtractFilePath(Application.ExeName)+IniFile;
  WriFile := ExtractFilePath(Application.ExeName)+WriFile;

  { Read the INI file }
  LoadLayout;
end;

{
  ApplicationHint is called when the application gets an OnHint message
}
procedure TMainForm.ApplicationHint(Sender : TObject);
begin
  with StatusBar do
    if Application.Hint <> '' then begin
      { If we got a hint, show it }
      Font.Size := 9;
      Alignment := taLeftJustify;
      Caption := Application.Hint;
    end
    else begin
      { If we didn't got a hint, show default }
      Font.Size := 7;
      Alignment := taCenter;
      Caption := CopyrightNotice;
    end;
end;

{
  ImageDragOver is called when the user drags a slot over a slot
}
procedure TMainForm.ImageDragOver(Sender, Source : TObject; X, Y : Integer;
                                  State : TDragState; var Accept : Boolean);
var
  i : Integer;
begin
  Accept := False;

  if Sender is TImage then begin
    { Find the slot corresponding to the Sender }
    i := 0;
    while i < MaxNrOfApps do
      if Slot[i].Image = Sender then
        break
      else
        Inc(i);

    { if OK and not active, accept the item }
    if i < MaxNrOfApps then
      Accept := (Sender <> Source) and not Slot[i].Active;
  end;
end;

{
  ImageDragDrop is called when the user drops a slot in a slot
}
procedure TMainForm.ImageDragDrop(Sender, Source : TObject; X, Y : Integer);
var
  dst, src : Integer;
begin
  { Find the slot corresponding to the Sender }
  dst := 0;
  while dst < MaxNrOfApps do
    if Slot[dst].Image = Sender then
      break
    else
      Inc(dst);

  { Find the slot corresponding to the Source }
  src := 0;
  while src < MaxNrOfApps do
    if Slot[src].Image = Source then
      break
    else
      Inc(src);

  { If source and destination are both OK, go ahead and move }
  if (dst < MaxNrOfApps) and (src < MaxNrOfApps) then begin
    { Copy some attributes }
    Slot[dst].Active := True;
    Slot[dst].Command := Slot[src].Command;
    Slot[dst].Directory := Slot[src].Directory;
    Slot[dst].Hint := Slot[src].Hint;
    Slot[dst].Panel.Hint := Slot[src].Panel.Hint;
    Slot[dst].IconFile := Slot[src].IconFile;
    Slot[dst].IconNr := Slot[src].IconNr;
    Slot[dst].Icon.Assign(Slot[src].Icon);

    { Invalidate the source }
    with Slot[src] do begin
      Active := False;
      Command := '';
      Directory := '';
      Hint := '';
      IconFile := '';
      IconNr := 0;
      Panel.Hint := '';
    end;

    { Delete the source icon from the image }
    with Slot[src].Image.Canvas do begin
      Pen.Color := clSilver;
      Pen.Style := psSolid;
      Brush.Color := clSilver;
      Brush.Style := bsSolid;
      Rectangle(0,0,AppWidth-3,AppWidth-3);
    end;

    { Initialize the destination image and draw the icon }
    with Slot[dst].Image.Canvas do begin
      Pen.Color := clSilver;
      Pen.Style := psSolid;
      Brush.Color := clSilver;
      Brush.Style := bsSolid;
      Rectangle(0,0,AppWidth-3,AppWidth-3);
      Draw(0,0,Slot[dst].Icon);
    end;

    { Raise modification flag }
    Modified := True;
  end;
end;

{
  ImageDblClick is called when the user double clicks on the application
}
procedure TMainForm.ImageDblClick(Sender : TObject);
var
  CmdStr : array [0..255] of Char;
  i      : Integer;
begin
  ExecFailed := False;

  { Find the app corresponding to the Sender }
  i := 0;
  while i < MaxNrOfApps do
    if Slot[i].Image = Sender then
      break
    else
      Inc(i);

  { If valid and active, launch the app }
  if i < MaxNrOfApps then
    if Slot[i].Active then begin
      try
        with Slot[i] do begin
          StrPCopy(CmdStr,Command);
          if Directory[Length(Directory)] = '\' then
           { Kill the trailing backslash... The on-line help for ExtractFilePath
             says it should work anyway, but it doesn't... BUG! }
            ChDir(Copy(Directory,1,Length(Directory)-1))
          else
            ChDir(Directory);
        end;
        if WinExec(CmdStr,SW_SHOWNORMAL) < 32 then begin
          MessageDlg('Could not execute '+StrPas(CmdStr)+'.',mtError,[mbOk],0);
          ExecFailed := True; { Otherwise we'll start dragging }
        end;
      except
        on EInOutError do begin
          MessageDlg('Invalid working directory ('+Slot[i].Directory+').',mtError,[mbOk],0);
          ExecFailed := True; { Otherwise we'll start dragging }
        end;
      end;
    end;
end;

{
  ImageMouseDown is called when the user presses a mouse button on an image
}
procedure TMainForm.ImageMouseDown(Sender : TObject; Button : TMouseButton;
                                   Shift : TShiftState; X, Y : Integer);
begin
  { If the previous WinExec failed we must invalidate the first MouseDown
    message, otherwise we'll start dragging }
  if ExecFailed then begin
    ExecFailed := False;
    Exit;
  end;

  if Button = mbLeft then
    DragOn := True;
end;

{
  ImageMouseUp is called when the user releases a mouse button on an image
}
procedure TMainForm.ImageMouseUp(Sender : TObject; Button : TMouseButton;
                                 Shift : TShiftState; X, Y : Integer);
begin
  if Button = mbLeft then
    DragOn := False;
end;

{
  ImageMouseMove is called when the user moves the mouse on an image
}
procedure TMainForm.ImageMouseMove(Sender : TObject; Shift : TShiftState;
                                   X, Y : Integer);
var
  i : Integer;
begin
  if DragOn then begin
    { Find the app corresponding to the Sender }
    i := 0;
    while i < MaxNrOfApps do
      if Slot[i].Image = Sender then
        break
      else
        Inc(i);

    { If valid and active, start dragging }
    if (i < MaxNrOfApps) and Slot[i].Active then
      Slot[i].Image.BeginDrag(False);
  end;
end;

{
  Load1Click is called when the user selects the 'Load layout' option from the menu
}
procedure TMainForm.Load1Click(Sender: TObject);
begin
  if Modified and (MessageDlg('The layout has changed. Reload?',mtWarning,[mbYes,mbNo],0) <> idYes) then
    Exit;

  LoadLayout;
end;

{
  Save1Click is called when the user selects the 'Save layout' option from the menu
}
procedure TMainForm.Save1Click(Sender: TObject);
begin
  if Modified and (MessageDlg('Are you sure you want to save?',mtWarning,[mbYes,mbNo],0) <> idYes) then
    Exit;

  SaveLayout;
end;

{
  ViewDocsClick is called when the user selects the 'View documentation' option from the menu
}
procedure TMainForm.ViewDocsClick(Sender: TObject);
var
  CmdStr : array [0..255] of Char;
begin
  { Start WRITE.EXE with our docfile }
  StrPCopy(CmdStr,'write.exe '+WriFile);
  if WinExec(CmdStr,SW_SHOWNORMAL) < 32 then
    MessageDlg('Could not execute "'+StrPas(CmdStr)+'".',mtError,[mbOk],0);
end;

{
  About1Click is called when the user selects the 'About AppLaunch' option from the menu
}
procedure TMainForm.About1Click(Sender: TObject);
begin
  AboutForm.ShowModal;
end;

{
  Exit1Click is called when the user selects the 'Exit' option from the menu
}
procedure TMainForm.Exit1Click(Sender: TObject);
begin
  Close;
end;

{
  StatusBarDragOver is called when the user drags a slot over the status bar
}
procedure TMainForm.StatusBarDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  Accept := False;

  { Accept all slots }
  if Source is TImage then
    Accept := True;
end;

{
  StatusBarDragDrop is called when the user drops a slot on the status bar
}
procedure TMainForm.StatusBarDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
  i : Integer;
begin
  { Find the slot corresponding to the Source }
  i := 0;
  while i < MaxNrOfApps do
    if Slot[i].Image = Source then
      break
    else
      Inc(i);

  { If valid and active, delete the app }
  if (i < MaxNrOfApps) and Slot[i].Active then begin
    { Invalidate the slot}
    with Slot[i] do begin
      Active := False;
      Command := '';
      Directory := '';
      Hint := '';
      IconFile := '';
      IconNr := 0;
      Panel.Hint := '';
    end;

    { Delete the icon from the image }
    with Slot[i].Image.Canvas do begin
      Pen.Color := clSilver;
      Pen.Style := psSolid;
      Brush.Color := clSilver;
      Brush.Style := bsSolid;
      Rectangle(0,0,AppWidth-3,AppWidth-3);
    end;

    { Decrease the number of apps }
    Dec(NrOfApps);

    { Raise modification flag }
    Modified := True;
  end;
end;

{
  FormCloseQuery is called before the application terminates
}
procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  CanClose := True;

  if Modified and (MessageDlg('The layout has changed. Save?',mtWarning,[mbYes,mbNo],0) = idYes) then
    SaveLayout;
end;

{
  PopupMenu1Popup is called when the popup menu is activated
}
procedure TMainForm.PopupMenu1Popup(Sender: TObject);
begin
  { Enable the load/save layout options if the layout has been modified,
    otherwise disable them }
  Load1.Enabled := Modified;
  Save1.Enabled := Modified;

  { Save current mouse coordinates }
  GetCursorPos(MousePos);

  { Adjust mouse coordinates to the position of the form }
  Dec(MousePos.X,MainForm.Left);
  Dec(MousePos.Y,MainForm.Top);
end;

{
  Properties1Click is called when the user selects the 'Properties' option from the menu
}
procedure TMainForm.Properties1Click(Sender: TObject);
var
  WasActive : Boolean;
  IconFile  : array [0..255] of Char;
  IconNr    : Word;
  TheIcon   : HIcon;
  j         : Integer;
begin
  j := MousePos.X div AppWidth;

  if j < MaxNrOfApps then begin
    { Initialize the info form }
    InfoForm.CmdStr.Text := Slot[j].Command;
    InfoForm.DirStr.Text := Slot[j].Directory;
    InfoForm.InfoStr.Text := Slot[j].Hint;
    InfoForm.IconFileStr.Text := Slot[j].IconFile;
    if Slot[j].IconNr <> 0 then
      InfoForm.IconNrStr.Text := IntToStr(Slot[j].IconNr)
    else
      InfoForm.IconNrStr.Text := '';

    { Show the info form }
    InfoForm.ShowModal;

    { Remember if this slot was active before }
    WasActive := Slot[j].Active;

    { If the OK button was pressed we go ahead }
    if InfoForm.ModalResult = idOK then begin
      { Set some attributes for this slot }
      Slot[j].Active := True;
      Slot[j].Command := InfoForm.CmdStr.Text;
      Slot[j].Directory := InfoForm.DirStr.Text;
      Slot[j].Hint := InfoForm.InfoStr.Text;
      Slot[j].Panel.Hint := '| '+Slot[j].Hint;
      if InfoForm.IconFileStr.Text <> '' then begin
        Slot[j].IconFile := InfoForm.IconFileStr.Text;
        StrPCopy(IconFile,Slot[j].IconFile);
      end
      else begin
        Slot[j].IconFile := '';
        StrPCopy(IconFile,InfoForm.CmdStr.Text);
      end;
      try
        IconNr := StrToInt(InfoForm.IconNrStr.Text);
      except
        on EConvertError do
          IconNr := 0;
      end;
      Slot[j].IconNr := IconNr;

      { Extract the icon }
      TheIcon := ExtractIcon(HInstance,IconFile,IconNr);
      if TheIcon < 2 then
        TheIcon := LoadIcon(0,idi_Question);

      { Set some attributes for this slot }
      Slot[j].Icon.Handle := TheIcon;

      { Initialize the image and draw the icon }
      with Slot[j].Image.Canvas do begin
        Pen.Color := clSilver;
        Pen.Style := psSolid;
        Brush.Color := clSilver;
        Brush.Style := bsSolid;
        Rectangle(0,0,AppWidth-3,AppWidth-3);
        Draw(0,0,Slot[j].Icon);
      end;

      { Increment the number of apps }
      if not WasActive then
        Inc(NrOfApps);

      { Raise modification flag }
      Modified := True;
    end;
  end;
end;

procedure TMainForm.Toggle1Click(Sender: TObject);
begin
  { Toggle the check mark on the popup menu }
  Toggle1.Checked := not Toggle1.Checked;

  { Modify the size of the main form accordingly }
  if Toggle1.Checked then
    Width := Width-3*AppWidth
  else
    Width := Width+3*AppWidth;

  { Enable/disable the resource form accordingly }
  ResourceForm.Visible := Toggle1.Checked;

  { Raise modification flag }
  Modified := True;
end;

procedure TMainForm.FormActivate(Sender: TObject);
begin
  PixelsPerInch := 96;
end;

end.

