unit Zip20;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Zip;

type

  {Zip}
  TZippedEvent = procedure(Sender: TObject; FileName: TFileName;
                  Compression: Integer) of object;
  TZippingEvent = procedure(Sender: TObject; FileName: TFileName;
                  Size: Integer) of object;
  TWritingEvent = procedure(Sender: TObject; FileName: TFileName) of object;
  TComputeEvent = procedure(Sender: TObject; NumFiles: Integer) of object;
  TReplaceEvent = procedure(Sender: TObject; FileName: TFileName;
                 Newer: boolean) of object;
  TSplitEvent = procedure(Sender: TObject; PartNumer: Integer;
                 Size: LongInt) of object;
  {Zip}
  TOverwriteOptions = (owAlways, owNever, owPrompt, owUpDate);

  TLanguageOptions = (lgFrench, lgEnglish, lgGerman, lgNone);

  TZipModeOptions = (zmNormal, zmFast, zmSlow, zmStore);

  TZip20 = class(TComponent)
  private
    { Private declarations }
    {Zip}
    FZipEnabled: boolean;
    FWindowHandle: HWND;
    FYourName: PChar;
    FYourPassword: PChar;
    FFilesToZip: TStrings;
    FOverwriteMode: TOverwriteOptions;
    FLanguage: TLanguageOptions;
    FZipMode: TZipModeOptions;
    FStorePaths: Boolean;
    FFileName: PChar;
    FRecurse: Boolean;
    FPassword: PChar;
    FComment: PChar;
    FDestDir: PChar;
    FSplitFirstSize: LongInt;
    FSplitNextSize: LongInt;
    FDiskettePause: Boolean;
    FSplitTitle: PChar;
    FSplitMessage: PChar;
    FZipCancel: boolean;
    FOnZipping: TZippingEvent;
    FOnFileZipped: TZippedEvent;
    FOnWriting: TWritingEvent;
    FOnDeleting: TNotifyEvent;
    FOnReparing: TNotifyEvent;
    FOnCompute: TComputeEvent;
    FOnReplace: TReplaceEvent;
    FOnSplitting: TZippingEvent;
    FOnSplitted: TSplitEvent;
    procedure SetYourName(Value: String);
    function GetYourName: String;
    procedure SetYourPassword(value: String);
    function GetYourPassword: String;
    procedure SetFilesToZip(value: TStrings);
    procedure SetOverwriteMode(value: TOverwriteOptions);
    procedure SetFileName(value: TFileName);
    function GetFileName: TFileName;
    procedure SetPassword(value: string);
    function GetPassword: string;
    procedure SetComment(value: string);
    function GetComment: string;
    procedure SetDestDir(value: string);
    function GetDestDir: string;
    procedure SetSplitTitle(value: string);
    function GetSplitTitle: string;
    procedure SetSplitMessage(value: string);
    function GetSplitMessage: string;
    procedure SetLanguage(Value: TLanguageOptions);
    procedure SetZipMode(value: TZipModeOptions);
    procedure CMZipping(var Message: TMessage); message ZN_ZIPPING;
    procedure CMFileZipped(var Message: TMessage); message ZN_FILEZIPPED;
    procedure CMWriting(var Message: TMessage); message ZN_WRITING;
    procedure CMDeleting(var Message: TMessage); message ZN_DELETING;
    procedure CMReparing(var Message: TMessage); message ZN_REPARING;
    procedure CMCompute(var Message: TMessage); message ZN_COMPUTE;
    procedure CMReplace(var Message: TMessage); message ZN_REPLACE;
    procedure CMSplitting(var Message: TMessage); message ZN_SPLITTING;
    procedure CMSplitted(var Message: TMessage); message ZN_SPLITTED;
  protected
    { Protected declarations }
    procedure WndProc(var Message: TMessage);
    procedure DefaultHandler(var Message); override;
  public
    { Public declarations }
    {Zip}
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AddFiles: integer;
    function DeleteFiles: integer;
    function RepareZip: integer;
    function SetZipComment: integer;
    procedure ZipCancel(value: boolean);
    function ZipSplit:integer;
    function ZipGetReplaceFlag: integer;
    function ZipSetReplaceText(Title, Text, Yes, No, Always,
      Never: PChar): boolean;
  published
    { Published declarations }
    {Zip}
    property YourName: string read GetYourName write SetYourName;
    property YourPassword: string read GetYourPassword write SetYourPassword;
    property OverWriteMode: TOverwriteOptions read FOverwriteMode
      write SetOverwriteMode default owAlways;
    property StorePaths: boolean read FStorePaths
      write FStorePaths default false;
    property FileName: TFileName read GetFileName write SetFileName;
    property Recurse: boolean read FRecurse write FRecurse
      default false;
    property Password: string read GetPassword
      write SetPassword;
    property Comment: string read GetComment
      write SetComment;
    property Language: TLanguageOptions read FLanguage
      write SetLanguage default lgEnglish;
    property FilesToZip: TStrings read FFilesToZip
      write SetFilesToZip;
    property ZipMode: TZipModeOptions read FZipMode
      write SetZipMode default zmNormal;
    property DestDir: string read GetDestDir write SetDestDir;
    property SplitFirstSize: LongInt read FSplitFirstSize
      write FSplitNextSize default 1440000;
    property SplitNextSize: LongInt read FSplitNextSize
      write FSplitNextSize default 1440000;
    property SplitDiskettePause: boolean read FDiskettePause
      write FDiskettePause default false;
    property SplitTitle: string read GetSplitTitle
      write SetSplitTitle;
    property SplitMessage: string read GetSplitMessage
      write SetSplitMessage;
    property OnZipping: TZippingEvent read FOnZipping write FOnZipping;
    property OnFileZipped: TZippedEvent read FOnFileZipped write FOnFileZipped;
    property OnWriting: TWritingEvent read FOnWriting write FOnWriting;
    property OnDeleting: TNotifyEvent read FOnDeleting write FOnDeleting;
    property OnReparing: TNotifyEvent read FOnReparing write FOnReparing;
    property OnCompute: TComputeEvent read FOnCompute write FOnCompute;
    property OnReplace: TReplaceEvent read FOnReplace write FOnReplace;
    property OnSplitting: TZippingEvent read FOnSplitting write FOnSplitting;
    property OnSplitted: TSplitEvent read FOnSplitted write FOnSplitted;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Additional', [TZip20]);
end;

constructor TZip20.Create(AOwner: TComponent);
begin
 Inherited Create(AOwner);
 FZipEnabled := false;
 FZipCancel := false;
 FYourPassword := nil;
 YourName := 'TEST';
 FFileName := nil;
 FPassword := nil;
 FComment := nil;
 FDestDir := nil;
 SplitTitle := 'Disk %1/%2';
 SplitMessage := 'Insert Diskette %1';
 FSplitFirstSize := 1440000;
 FSplitNextSize := 1440000;
 FFilesToZip := TStringList.Create;
 FOverwriteMode := owAlways;
 FZipMode := zmNormal;
 FLanguage := lgEnglish;
 FStorePaths := false;
 FRecurse := false;
 FWindowHandle := AllocateHWnd(WndProc);
end;

destructor TZip20.Destroy;
begin
 DeallocateHWnd(FWindowHandle);
 FFilesToZip.Free;
 if FSplitTitle <> nil then
 StrDispose(FSplitTitle);
 if FSplitMessage <> nil then
 StrDispose(FSplitMessage);
 if FComment <> nil then
 StrDispose(FComment);
 if FPassword <> nil then
 StrDispose(FPassword);
 if FFileName <> nil then
 StrDispose(FFileName);
 if FYourPassword <> nil then
 StrDispose(FYourPassword);
 if FYourName <> nil then
 StrDispose(FYourName);
 inherited destroy;
end;

procedure TZip20.WndProc(var Message: TMessage);
begin
    Dispatch(Message);
end;

procedure TZip20.DefaultHandler(var Message);
begin
with TMessage(Message) do
 Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
end;

function TZip20.AddFiles: integer;
var
 loop: integer;
 line: array[0..255] of Char;
begin
result := -1;
 if not FZipEnabled then
 begin
 ZipInit(FYourName, FYourPassword);
 FZipEnabled := true;
 end;
 for loop := 0 to FFilesToZip.count-1 do
 begin
  StrPCopy(line, FFilesToZip.Strings[loop]);
  result := AddFileToZip(FFileName, line,
  Integer(FOverWriteMode), FStorePaths, FRecurse,
  FWindowHandle, FPassword);
  if (result > 1) or FZipCancel then exit;
 end;
end;

function TZip20.DeleteFiles: integer;
var
 loop: integer;
 line: array[0..255] of Char;
begin
 result := -1;
 if not FZipEnabled then
 begin
 ZipInit(FYourName, FYourPassword);
 FZipEnabled := true;
 end;
 for loop := 0 to FFilesToZip.count-1 do
 begin
  StrPCopy(line, FFilesToZip.Strings[loop]);
  result := ZipDeleteFiles(FFileName, line,
  FWindowHandle);
 end;
end;

function TZip20.RepareZip: integer;
begin
 result := -1;
 if not FZipEnabled then
 begin
 ZipInit(FYourName, FYourPassword);
 FZipEnabled := true;
 end;
 Result := ZipRepare(FFileName, FWindowHandle);
end;

function TZip20.ZipSplit: integer;
begin
  result := -1;
 if not FZipEnabled then
 begin
 ZipInit(FYourName, FYourPassword);
 FZipEnabled := true;
 end;
   result := Zip.ZipSplit(FWindowHandle, FFileName, FDestDir,
   FSplitFirstSize, FSplitNextSize, FDiskettePause,
   FSplitTitle, FSplitMessage);
end;

function TZip20.SetZipComment: integer;
begin
 result := -1;
 if not FZipEnabled then
 begin
 ZipInit(FYourName, FYourPassword);
 FZipEnabled := true;
 end;
 Result := ZipSetComment(FFileName, FComment,FWindowHandle);
end;

procedure TZip20.ZipCancel(value: boolean);
begin
 FZipCancel:= value;
 Zip.ZipCancel(Value);
end;

procedure TZip20.SetYourName(value: string);
begin
 if value <> GetYourName then
 begin
 if FYourName <> nil then StrDispose(FYourName);
 if value <> '' then
 FYourName := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FYourName := nil;
 if GetYourPassword <> '' then
 begin
 ZipInit(FYourName, FYourPassword);
 FZipEnabled := true;
 end;
 end;
end;

function TZip20.GetYourName: string;
begin
 if FYourName <> nil then
 result := StrPas(FYourName) else
 result := '';
end;

procedure TZip20.SetYourPassword(value: string);
begin
 if value <> GetYourPassword then
 begin
 if FYourPassword <> nil then StrDispose(FYourPassword);
 if value <> '' then
 FYourPassword := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FYourPassword := nil;
 ZipInit(FYourName, FYourPassword);
 FZipEnabled := true;
 end;
end;

function TZip20.GetYourPassword: string;
begin
 if FYourPassword <> nil then
 result := StrPas(FYourPassword) else
 result := '';
end;

procedure TZip20.SetComment(value: string);
begin
 if value <> GetComment then
 begin
 if FComment <> nil then StrDispose(FComment);
 if value <> '' then
 FComment := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FComment := nil;
 end;
end;

function TZip20.GetComment: string;
begin
 if FComment <> nil then
 result := StrPas(FComment) else
 result := '';
end;

procedure TZip20.SetDestDir(value: string);
begin
 if value <> GetDestDir then
 begin
 if FDestDir <> nil then StrDispose(FDestDir);
 if value <> '' then
 FDestDir := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FDestDir := nil;
 end;
end;

function TZip20.GetDestDir: string;
begin
 if FDestDir <> nil then
 result := StrPas(FDestDir) else
 result := '';
end;

procedure TZip20.SetSplitTitle(value: string);
begin
 if value <> GetSplitTitle then
 begin
 if FSplitTitle <> nil then StrDispose(FSplitTitle);
 if value <> '' then
 FSplitTitle := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FSplitTitle := nil;
 end;
end;

function TZip20.GetSplitTitle: string;
begin
 if FSplitTitle <> nil then
 result := StrPas(FSplitTitle) else
 result := '';
end;

procedure TZip20.SetSplitMessage(value: string);
begin
 if value <> GetSplitMessage then
 begin
 if FSplitMessage <> nil then StrDispose(FSplitMessage);
 if value <> '' then
 FSplitMessage := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FSplitMessage := nil;
 end;
end;

function TZip20.GetSplitMessage: string;
begin
 if FSplitMessage <> nil then
 result := StrPas(FSplitMessage) else
 result := '';
end;

procedure TZip20.SetFilesToZip(value: TStrings);
begin
 FFilesToZip.Assign(value);
end;

procedure TZip20.SetFileName(value: TFileName);
begin
 if value <> GetFileName then
 begin
 if FFileName <> nil then StrDispose(FFileName);
 if value <> '' then
 FFileName := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FFileName := nil;
 end;
end;

function TZip20.GetFileName: TFileName;
begin
 if FFileName <> nil then
 result := StrPas(FFileName) else
 result := '';
end;

procedure TZip20.SetPassword(value: string);
begin
 if value <> GetPassword then
 begin
 if FPassword <> nil then StrDispose(FPassword);
 if value <> '' then
 FPassword := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FPassword := nil;
 end;
end;

function TZip20.GetPassword: string;
begin
 if FPassword <> nil then
 result := StrPas(FPassword) else
 result := '';
end;

procedure TZip20.SetOverwriteMode(value: TOverwriteOptions);
begin
 if value <> FOverwriteMode then
  FOverwriteMode := value;
end;

procedure TZip20.SetLanguage(value: TLanguageOptions);
begin
 if value <> FLanguage then
 begin
 ZipSetlanguage(Word(Ord(value)));
 FLanguage := value;
 end;
end;

procedure TZip20.SetZipMode(value: TZipModeOptions);
begin
 if value <> FZipMode then
  if ZipSetMode(integer(Ord(value))) then
  FZipMode := value;
end;

function TZip20.ZipGetReplaceFlag: integer;
begin
 result := Zip.ZipGetReplaceFlag;
end;

function TZip20.ZipSetReplaceText(Title, Text, Yes, No, Always,
  Never: PChar): boolean;
begin
 Result := Zip.ZipSetReplaceText(Title, Text, Yes, No, Always,
   Never);
end;

procedure TZip20.CMZipping(var Message: TMessage);
var
  Filename: TFileName;
  Compressed: Integer;
begin
 if Assigned(FOnZipping) then
 begin
  Compressed := Message.WParam;
  FileName := StrPas(PChar(Message.LParam));
  FOnZipping(Self, Filename,
    Compressed);
 end;
end;

procedure TZip20.CMFileZipped(var Message: TMessage);
var
  Filename: TFileName;
  Compressed: Integer;
begin
 if Assigned(FOnFileZipped) then
 begin
  Compressed := Message.WParam;
  FileName := StrPas(PChar(Message.LParam));
  FOnFileZipped(Self, Filename,
    Compressed);
 end;
end;

procedure TZip20.CMWriting(var Message: TMessage);
var
  Filename: TFileName;
begin
 if Assigned(FOnWriting) then
 begin
  FileName := StrPas(PChar(Message.LParam));
  FOnWriting(Self, Filename);
 end;
end;

procedure TZip20.CMDeleting(var Message: TMessage);
begin
 if Assigned(FOnDeleting) then
 begin
  FOnDeleting(Self);
 end;
end;

procedure TZip20.CMReparing(var Message: TMessage);
begin
 if Assigned(FOnReparing) then
 begin
  FOnReparing(Self);
 end;
end;

procedure TZip20.CMCompute(var Message: TMessage);
var
  NumFiles: Integer;
begin
 if Assigned(FOnCompute) then
 begin
  NumFiles := Message.WParam;
  FOnCompute(Self, NumFiles);
 end;
end;

procedure TZip20.CMReplace(var Message: TMessage);
var
  Filename: TFileName;
  Replace: boolean;
begin
 if Assigned(FOnReplace) then
 begin
  FileName := StrPas(PChar(Message.LParam));
  Replace := Boolean(Message.WParam);
  FOnReplace(Self, FileName, Replace);
 end;
end;

procedure TZip20.CMSplitting(var Message: TMessage);
var
  Filename: TFileName;
  PartSize: Integer;
begin
 if Assigned(FOnSplitting) then
 begin
  FileName := StrPas(PChar(Message.LParam));
  PartSize:= Message.WParam;
  FOnSplitting(Self, FileName, PartSize);
 end;
end;

procedure TZip20.CMSplitted(var Message: TMessage);
var
  PartNumber: Integer;
  PartSize: LongInt;
begin
 if Assigned(FOnSplitted) then
 begin
  PartNumber := integer(Message.LParam);
  PartSize:= Message.WParam;
  FOnSplitted(Self, PartNumber, PartSize);
 end;
end;


end.
