unit UnZip20;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, UnZip;

type
  pVCLZipItem = ^tVCLZipItem;
  tVCLZipItem = record
    Name:         string[49];    { Filename without path.  Empty if directory }
    Dir:          string;    { Directory                                  }
    IsDir,
    IsEncrypted:  Boolean;
    Method,                                  { A ZMETHOD_* constant                       }
    Day,                                     { 1 - 31                                     }
    Month,                                   { 1 - 12                                     }
    Year,                                    { 0 - 99                                     }
    Hour,                                    { 0 - 23                                     }
    Min:          Integer;                   { 0 - 59                                     }
    OriginalSize: LongInt;
    Rate:         Integer;                   { Compression rate                           }
  end;
  {Zip}
 { TJoinedEvent = procedure(Sender: TObject; FileName: TFileName;
                  Compression: Integer) of object;}
  TJoiningEvent = procedure(Sender: TObject; FileName: TFileName;
                  Size: Integer) of object;
  TJoinDoneFileEvent = procedure(Sender: TObject; FileName: TFileName) of object;
 { TComputeEvent = procedure(Sender: TObject; NumFiles: Integer) of object;
  TReplaceEvent = procedure(Sender: TObject; FileName: TFileName;
                 Newer: boolean) of object;}
  TJoinedEvent = procedure(Sender: TObject; PartNumer: Integer;
                 Size: LongInt) of object;
  {UnZip}
  TUnZipOpenCloseTestEvent = procedure(Sender: TObject; FileName: TFileName;
                 OK: boolean) of object;
  TExpandingEvent = procedure(Sender: TObject; FileName: TFileName;
                  Percentage: integer) of object;
  TNewPasswordEvent = procedure(Sender: TObject; FileName: TFileName) of object;
  TJoinDoneSizeEvent = procedure(Sender: TObject; TotalBytes: LongInt) of object;

  {Zip}
{  TOverwriteOptions = (owAlways, owNever, owPrompt, owUpDate);}

  TLanguageOptions = (lgFrench, lgEnglish, lgGerman, lgNone);

{  TZipModeOptions = (zmNormal, zmFast, zmSlow, zmStore); }
  {UnZip}
  TUnZipQueryOptions = (umFalse, umTrue, umQuery);
  TUnZipSortOptions = (zsFullName, zsName, zsDirectory, zsDate,
    zsSize, zsRate, zsNone);

  TUnZip20 = class(TComponent)
  private
    { Private declarations }
    FUnZipEnabled: boolean;
    FWindowHandle: HWND;
    FYourName: PChar;
    FYourPassword: PChar;
    FFilesToExtract: TStrings;
    FLanguage: TLanguageOptions;
    FWithPaths: Boolean;
    FFileName: PChar;
    FPassword: PChar;
    FComment: PChar;
    FDestDir: PChar;
    FDiskettePause: Boolean;
    FJoinTitle: PChar;
    FJoinMessage: PChar;
    FActive: boolean;
    FFilesInZip: integer;
    FBackgroundMode: boolean;
    FAskPassword: boolean;
    FListBox: TListBox;
    FOverwriteMode: TUnZipQueryOptions;
    FSortMode: TUnZipSortOptions;
    FOnOpenFile: TUnZipOpenCloseTestEvent;
    FOnExpanding: TExpandingEvent;
    FOnCloseFile: TUnZipOpenCloseTestEvent;
    FOnNewPassword: TNewPasswordEvent;
    FOnTest: TUnZipOpenCloseTestEvent;
    FOnJoining: TJoiningEvent;
    FOnJoined: TJoinedEvent;
    FOnJoinDoneSize: TJoinDoneSizeEvent;
    FOnJoinDoneName: TJoinDoneFileEvent;
    procedure SetYourName(Value: String);
    function GetYourName: String;
    procedure SetYourPassword(value: String);
    function GetYourPassword: String;
    procedure SetFilesToExtract(value: TStrings);
    procedure SetFileName(value: TFileName);
    function GetFileName: TFileName;
    procedure SetPassword(value: string);
    function GetPassword: string;
    procedure SetComment(value: string);
    function GetComment: string;
    procedure SetDestDir(value: string);
    function GetDestDir: string;
    procedure SetJoinTitle(value: string);
    function GetJoinTitle: string;
    procedure SetJoinMessage(value: string);
    function GetJoinMessage: string;
    procedure SetLanguage(Value: TLanguageOptions);
    procedure SetActive(value: boolean);
    procedure SetBackgroundMode(value: boolean);
    procedure SetSortMode(value: TUnZipSortOptions);
    procedure CMOpenFile(var Message: TMessage); message ZN_OPENFILE;
    procedure CMExpanding(var Message: TMessage); message ZN_EXPANDING;
    procedure CMCloseFile(var Message: TMessage); message ZN_CLOSEFILE;
    procedure CMNewPassword(var Message: TMessage); message ZN_NEWPASSWORD;
    procedure CMTest(var Message: TMessage); message ZN_TEST;
    procedure CMJoining(var Message: TMessage); message ZN_JOINING;
    procedure CMJoined(var Message: TMessage); message ZN_JOINED;
    procedure CMJoinDoneSize(var Message: TMessage); message ZN_JOINDONESIZE;
    procedure CMJoinDoneName(var Message: TMessage); message ZN_JOINDONENAME;
  protected
    { Protected declarations }
    procedure WndProc(var Message: TMessage);
    procedure DefaultHandler(var Message); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function IsFileInZip(FileName: TFileName): boolean;
    function GetFileNameFromZip(Index: integer): string;
    function IsThisAZipFile(Quick: boolean): boolean;
    function IsFileUnZipable: boolean;
    function IsFileNameInFilter(FileName, Mask: String): boolean;
    function ExtractFiles: word;
    procedure UnZipCancel(value: boolean);
    function GetZipItem(Index: integer; ZipItem: PVCLZipItem): boolean;
    function GetZipCommentLength: word;
    function GetZipComment(Buffer: PChar): boolean;
    function DoesFileExist: boolean;
    function GetShortFileName: string;
    function GetPathFromFileName: string;
    function ViewFileFromZip(Mask: string; TextOnly:boolean): boolean;
    function UnZipSetMsgBoxTitle(Title: string): boolean;
    function UnZipSetReplaceText(Title, Text, YesBtn, NoBtn,
      AlwaysBtn, NeverBtn: string): boolean;
    {function UnZipSetPasswordText(Title, Text, YesBtn, NoBtn,
      AlwaysBtn, NeverBtn: string): boolean;}
    function ZipJoin: integer;
    function GetZFileOriginalSize(FileName: TFileName): LongInt;
    function GetZFileIsEncrypted(FileName: TFileName): boolean;
    function GetZFileCompressedSize(FileName: TFileName): LongInt;
    function GetZFileDate(FileName: TFileName): string;
    function GetZFileTime(FileName: TFileName): string;
    function GetZCompressMethod(FileName: TFileName): word;
    function GetZFileIsDir(FileName: TFileName): boolean;
    function GetZFullInfos(FileName: TFileName; Buffer: PChar): boolean;
  published
    { Published declarations }
    property FilesInZip: integer read FFilesInZip;
    property YourName: string read GetYourName write SetYourName;
    property YourPassword: string read GetYourPassword write SetYourPassword;
    property WithPaths: boolean read FWithPaths
      write FWithPaths default false;
    property FileName: TFileName read GetFileName write SetFileName;
    property Password: string read GetPassword
      write SetPassword;
    property Comment: string read GetComment
      write SetComment;
    property Language: TLanguageOptions read FLanguage
      write SetLanguage default lgEnglish;
    property FilesToExtract: TStrings read FFilesToExtract
      write SetFilesToExtract;
    property DestDir: string read GetDestDir write SetDestDir;
    property JoinDiskettePause: boolean read FDiskettePause
      write FDiskettePause default false;
    property JoinTitle: string read GetJoinTitle
      write SetJoinTitle;
    property JoinMessage: string read GetJoinMessage
      write SetJoinMessage;
    property Active: boolean read FActive write SetActive default false;
    property BackgroundMode: boolean read FBackgroundMode
      write SetBackgroundMode default true;
    property AskForPassword: boolean read FAskPassword
      write FAskPassword default true;
    property OverwriteMode: TUnZipQueryOptions read FOverWriteMode
      write FOverWriteMode default umQuery;
    property SortMode: TUnZipSortOptions read FSortMode
      write SetSortMode default zsNone;
    {property ListBox: TListBox read FListBox write FListBox;}
    property OnOpenFile: TUnZipOpenCloseTestEvent read FOnOpenFile
      write FOnOpenFile;
    property OnExpanding: TExpandingEvent read FOnExpanding
      write FOnExpanding;
    property OnCloseFile: TUnZipOpenCloseTestEvent read FOnCloseFile
      write FOnCloseFile;
    property OnNewPassword: TNewPasswordEvent read FOnNewPassword
      write FOnNewPassword;
    property OnTest: TUnZipOpenCloseTestEvent read FOnTest
      write FOnTest;
    property OnJoining: TJoiningEvent read FOnJoining
      write FOnJoining;
    property OnJoined: TJoinedEvent read FOnJoined
      write FOnJoined;
    property OnJoinDoneSize: TJoinDoneSizeEvent read FOnJoinDoneSize
      write FOnJoinDoneSize;
    property OnJoinDoneName: TJoinDoneFileEvent read FOnJoinDoneName
      write FOnJoinDoneName;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Additional', [TUnZip20]);
end;

constructor TUnZip20.Create(AOwner: TComponent);
begin
 Inherited Create(AOwner);
 FUnZipEnabled := false;
 FYourPassword := nil;
 YourName := 'TEST';
 FFileName := nil;
 FPassword := nil;
 FComment := nil;
 FDestDir := nil;
 JoinTitle := 'Disk %1/%2';
 JoinMessage := 'Insert Diskette %1';
 FFilesToExtract := TStringList.Create;
 FOverWriteMode := umQuery;
 FSortMode := zsNone;
 FLanguage := lgEnglish;
 FWithPaths := false;
 FActive := false;
 FBackgroundMode := true;
 FAskPassword := true;
 FFilesInZip := 0;
 FWindowHandle := AllocateHWnd(WndProc);
 UnZipSetReceivingWindow(FWindowHandle);
 FListBox := TListBox.Create(Self);
 TWinControl(Owner).InsertControl(FListBox);
 FListBox.Height := 0;
 FListBox.Width := 0;
 FListBox.Parent := AOwner as TWinControl;
end;

destructor TUnZip20.Destroy;
begin
 DeallocateHWnd(FWindowHandle);
 FFilesToExtract.Free;
 if FJoinTitle <> nil then
 StrDispose(FJoinTitle);
 if FJoinMessage <> nil then
 StrDispose(FJoinMessage);
 if FComment <> nil then
 StrDispose(FComment);
 if FPassword <> nil then
 StrDispose(FPassword);
 if FFileName <> nil then
 StrDispose(FFileName);
 if FYourPassword <> nil then
 StrDispose(FYourPassword);
 if FYourName <> nil then
 StrDispose(FYourName);
 inherited destroy;
end;

procedure TUnZip20.WndProc(var Message: TMessage);
begin
    Dispatch(Message);
end;

procedure TUnZip20.DefaultHandler(var Message);
begin
with TMessage(Message) do
 Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
end;


procedure TUnZip20.SetYourName(value: string);
begin
 if value <> GetYourName then
 begin
 if FYourName <> nil then StrDispose(FYourName);
 if value <> '' then
 FYourName := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FYourName := nil;
 if GetYourPassword <> '' then
 UnZipInit(FYourName, FYourPassword);
 end;
end;

function TUnZip20.GetYourName: string;
begin
 if FYourName <> nil then
 result := StrPas(FYourName) else
 result := '';
end;

procedure TUnZip20.SetYourPassword(value: string);
begin
 if value <> GetYourPassword then
 begin
 if FYourPassword <> nil then StrDispose(FYourPassword);
 if value <> '' then
 FYourPassword := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FYourPassword := nil;
 UnZipInit(FYourName, FYourPassword);
 end;
end;

function TUnZip20.GetYourPassword: string;
begin
 if FYourPassword <> nil then
 result := StrPas(FYourPassword) else
 result := '';
end;

procedure TUnZip20.SetComment(value: string);
begin
 if value <> GetComment then
 begin
 if FComment <> nil then StrDispose(FComment);
 if value <> '' then
 FComment := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FComment := nil;
 end;
end;

function TUnZip20.GetComment: string;
begin
 if FComment <> nil then
 result := StrPas(FComment) else
 result := '';
end;

procedure TUnZip20.SetDestDir(value: string);
begin
 if value <> GetDestDir then
 begin
 if FDestDir <> nil then StrDispose(FDestDir);
 if value <> '' then
 FDestDir := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FDestDir := nil;
 end;
end;

function TUnZip20.GetDestDir: string;
begin
 if FDestDir <> nil then
 result := StrPas(FDestDir) else
 result := '';
end;

procedure TUnZip20.SetJoinTitle(value: string);
begin
 if value <> GetJoinTitle then
 begin
 if FJoinTitle <> nil then StrDispose(FJoinTitle);
 if value <> '' then
 FJoinTitle := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FJoinTitle := nil;
 end;
end;

function TUnZip20.GetJoinTitle: string;
begin
 if FJoinTitle <> nil then
 result := StrPas(FJoinTitle) else
 result := '';
end;

procedure TUnZip20.SetJoinMessage(value: string);
begin
 if value <> GetJoinMessage then
 begin
 if FJoinMessage <> nil then StrDispose(FJoinMessage);
 if value <> '' then
 FJoinMessage := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FJoinMessage := nil;
 end;
end;

function TUnZip20.GetJoinMessage: string;
begin
 if FJoinMessage <> nil then
 result := StrPas(FJoinMessage) else
 result := '';
end;

procedure TUnZip20.SetFilesToExtract(value: TStrings);
begin
 FFilesToExtract.Assign(value);
end;

procedure TUnZip20.SetFileName(value: TFileName);
begin
 if value <> GetFileName then
 begin
 if FFileName <> nil then StrDispose(FFileName);
 if value <> '' then
 FFileName := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FFileName := nil;
 end;
end;

function TUnZip20.GetFileName: TFileName;
begin
 if FFileName <> nil then
 result := StrPas(FFileName) else
 result := '';
end;

procedure TUnZip20.SetPassword(value: string);
begin
 if value <> GetPassword then
 begin
 if FPassword <> nil then StrDispose(FPassword);
 if value <> '' then
 FPassword := StrPCopy(StrAlloc(Length(value) + 1), value) else
 FPassword := nil;
 end;
end;

function TUnZip20.GetPassword: string;
begin
 if FPassword <> nil then
 result := StrPas(FPassword) else
 result := '';
end;


procedure TUnZip20.SetLanguage(value: TLanguageOptions);
begin
 if value <> FLanguage then
 begin
 UnZipSetlanguage(Word(Ord(value)));
 FLanguage := value;
 end;
end;

function TUnZip20.IsFileInZip(FileName: TFileName): boolean;
var
 TempFileName: PChar;
begin
 if not FUnZipEnabled then
 begin
 UnZipInit(FYourName, FYourPassword);
 FUnZipEnabled := true;
 end;
 TempFileName := StrPCopy(StrAlloc(Length(FileName) + 1), FileName);
 try
 result := UnZip.IsFileInZip(FFileName, TempFileName);
 finally
 StrDispose(TempFileName);
 end;
end;

procedure TUnZip20.SetActive(value: boolean);
var
 ret: integer;
begin
 if not FUnZipEnabled then
 begin
 UnZipInit(FYourName, FYourPassword);
 FUnZipEnabled := true;
 end;
 if (FActive <> value) and Assigned(FListBox) then
 begin
   if value = false then
   begin
    FListBox.Clear;
    FActive := false;
   end;
   if (value = true) and Assigned(FListBox) then
   begin
    FFilesInZip := CountFileInZip(FFileName);
    ret := GetZList(FFileName, FListBox.Handle, 0,
      Integer(Ord(FSortMode)), FBackgroundMode);
   FActive := ret = 0;
   end;
 end;
end;

procedure TUnZip20.SetBackgroundMode(value: boolean);
begin
 if value <> FBackgroundMode then
 begin
 UnZipSetBackgroundMode(value);
 FBackgroundMode := value
 end;
end;

function TUnZip20.GetFileNameFromZip(Index: integer): string;
begin
 result := StrPas(UnZip.GetFileNameFromZip(FFileName, index));
end;

function TUnZip20.IsThisAZipFile(Quick: boolean): boolean;
begin
 result := UnZip.IsThisFileAZipFile(FFileName, Quick);
end;

function TUnZip20.IsFileUnZipable: boolean;
begin
 result := UnZip.IsFileUnZipable(FFileName);
end;

function TUnZip20.GetZipCommentLength: word;
begin
 result := 0;
  result := UnZip.GetZipCommentLength(FFileName);
end;

function TUnZip20.GetZipComment(Buffer: PChar): boolean;
begin
 result := false;
  result := UnZip.GetZipComment(FFileName, Buffer);
end;

function TUnZip20.DoesFileExist: boolean;
begin
 result := false;
 result := UnZip.DoesFileExist(FFileName);
end;

function TUnZip20.GetShortFileName: string;
begin
  result := '';
    result := StrPas(UnZip.GetShortFileName(FFileName));
end;

function TUnZip20.GetPathFromFileName: string;
begin
 result := '';
 if @UnZip.GetPathFromFileName <> nil then
    result := StrPas(UnZip.GetPathFromFileName(FFileName));
end;

function TUnZip20.IsFileNameInFilter(FileName, Mask: string): boolean;
var
tempFileName: PChar;
tempMask: PChar;
begin
 result := false;
 TempFileName := StrPCopy(StrAlloc(Length(FileName) + 1), FileName);
 TempMask := StrPCopy(StrAlloc(Length(Mask) + 1), Mask);
 try
 result := UnZip.IsFileNameInFilter(TempFileName, TempMask);
 finally
 StrDispose(TempMask);
 StrDispose(TempFileName);
 end;
end;

function TUnZip20.ExtractFiles: word;
var
 loop: integer;
 line: PChar;
 ORMode: TUnZipQueryOptions;
begin
 if not FUnZipEnabled then
 begin
 UnZipInit(FYourName, FYourPassword);
 FUnZipEnabled := true;
 end;
 result := 1;
 ORMode := FOverwriteMode;
 for loop := 0 to FFilesToExtract.count-1 do
 begin
  line := StrPCopy(StrAlloc(Length(FFilesToExtract.Strings[loop])+1)
  , FFilesToExtract.Strings[loop]);
  try
  result := ExtractZipFiles(FFileName, line, FDestDir,
  Integer(Ord(ORMode)), FWithPaths);
  if (ORMode = umQuery) and (@UnZip.GetQueryFlag <> nil) then
  ORMode := TUnZipQueryOptions(UnZip.GetQueryFlag);
  finally
  StrDispose(line);
  end;
  {if result > 1 then exit;}
 end;
end;

function TUnZip20.ViewFileFromZip(Mask: string; TextOnly: boolean): boolean;
var
 TempMask: PChar;
begin
 result := false;
 if not FUnZipEnabled then
 begin
 UnZipInit(FYourName, FYourPassword);
 FUnZipEnabled := true;
 end;
 TempMask := StrPCopy(StrAlloc(Length(Mask) + 1), Mask);
 try
 result := UnZip.ViewFileFromZip(FFileName, TempMask, TextOnly);
 finally
  StrDispose(TempMask);
 end;
end;

function TUnZip20.UnZipSetMsgBoxTitle(Title: string): boolean;
var
 TempTitle: PChar;
begin
 result := false;
 TempTitle := StrPCopy(StrAlloc(Length(Title) + 1), Title);
 try
 result := UnZip.UnZipSetMsgBoxTitle(TempTitle);
 finally
  StrDispose(TempTitle);
 end;
end;

function TUnZip20.UnZipSetReplaceText(Title, Text, YesBtn, NoBtn,
      AlwaysBtn, NeverBtn: string): boolean;
var
 TempTitle, TempText,
 TempYesBtn, TempNoBtn,
 TempAlwaysBtn, TempNeverBtn: PChar;
begin
 result := false;
 TempTitle := StrPCopy(StrAlloc(Length(Title) + 1), Title);
 TempText := StrPCopy(StrAlloc(Length(Text) + 1), Text);
 TempYesBtn := StrPCopy(StrAlloc(Length(YesBtn) + 1), YesBtn);
 TempNoBtn := StrPCopy(StrAlloc(Length(NoBtn) + 1), NoBtn);
 TempAlwaysBtn := StrPCopy(StrAlloc(Length(AlwaysBtn) + 1), AlwaysBtn);
 TempNeverBtn := StrPCopy(StrAlloc(Length(NeverBtn) + 1), NeverBtn);
 try
 result := UnZip.UnZipSetReplaceText(TempTitle, TempText, TempYesBtn,
   TempNoBtn, TempAlwaysBtn, TempNeverBtn);
 finally
 StrDispose(TempTitle);
 StrDispose(TempText);
 StrDispose(TempYesBtn);
 StrDispose(TempNoBtn);
 StrDispose(TempAlwaysBtn);
 StrDispose(TempNeverBtn);
 end;
end;

{function TUnZip20.UnZipSetPasswordText(Title, Text, YesBtn, NoBtn,
      AlwaysBtn, NeverBtn: string): boolean;
var
 TempTitle, TempText,
 TempYesBtn, TempNoBtn,
 TempAlwaysBtn, TempNeverBtn: PChar;
begin
 result := false;
 if FUnZipEnabled and (@UnZip.UnZipSetPasswordText <> nil) then
 begin
 TempTitle := StrPCopy(StrAlloc(Length(Title) + 1), Title);
 TempText := StrPCopy(StrAlloc(Length(Text) + 1), Text);
 TempYesBtn := StrPCopy(StrAlloc(Length(YesBtn) + 1), YesBtn);
 TempNoBtn := StrPCopy(StrAlloc(Length(NoBtn) + 1), NoBtn);
 TempAlwaysBtn := StrPCopy(StrAlloc(Length(AlwaysBtn) + 1), AlwaysBtn);
 TempNeverBtn := StrPCopy(StrAlloc(Length(NeverBtn) + 1), NeverBtn);
 try
 result := UnZip.UnZipSetPasswordText(TempTitle, TempText, TempYesBtn,
   TempNoBtn, TempAlwaysBtn, TempNeverBtn);
 finally
 StrDispose(TempTitle);
 StrDispose(TempText);
 StrDispose(TempYesBtn);
 StrDispose(TempNoBtn);
 StrDispose(TempAlways);
 StrDispose(TempNever);
 end;
end;}

procedure TUnZip20.UnzipCancel(value: boolean);
begin
  UnZip.UnzipCancel(value);
end;

function TUnZip20.ZipJoin: integer;
begin
  result := 4;
if not FUnZipEnabled then
 begin
 UnZipInit(FYourName, FYourPassword);
 FUnZipEnabled := true;
 end;
  UnZip.ZipJoin(FWindowHandle, FFileName, FDestDir, FBackgroundMode,
   FJoinTitle, FJoinMessage);
end;

function TUnZip20.GetZipItem(Index: integer; ZipItem: pVCLZipItem): boolean;
var
 returned: tZipItem;
begin
 result := false;
 if not FUnZipEnabled then
 begin
 UnZipInit(FYourName, FYourPassword);
 FUnZipEnabled := true;
 end;
 if Assigned(FListBox) then
 result := UnZip.GetZipItem(FListBox.Handle, Index, @Returned)
  else exit;
 if result then
 with ZipItem^ do
 begin
   Name := StrPas(Returned.Name);
   Dir := StrPas(Returned.Dir);
   IsDir := Returned.IsDir;
   IsEncrypted := Returned.IsEncrypted;
   Method := Returned.Method;
   Day := Returned.Day;
   Month := Returned.Month;
   Year := Returned.Year;
   Hour := Returned.Hour;
   Min := Returned.Min;
   OriginalSize := Returned.OriginalSize;
   Rate := Returned.Rate;
 end;
end;

procedure TUnZip20.SetSortMode(value: TUnZipSortOptions);
begin
 if (value <> FSortMode) and Assigned(FListBox) then
 begin
 FSortMode := value;
  UnZip.UnZipSortZList(FListBox.Handle, Integer(Ord(FSortMode)),
  FBackgroundMode);
 end;
end;

function TUnZip20.GetZFileOriginalSize(FileName: TFileName): LongInt;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := UnZip.GetZFileOriginalSize(FFileName, TempFileName);
  finally
   StrDispose(TempFileName);
  end;
end;

function TUnZip20.GetZFileIsEncrypted(FileName: TFileName): boolean;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := UnZip.GetZFileIsEncrypted(FFileName, TempFileName);
  finally
   StrDispose(TempFileName);
  end;
end;

function TUnZip20.GetZFileCompressedSize(FileName: TFileName): LongInt;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := UnZip.GetZFileCompressedSize(FFileName, TempFileName);
  finally
   StrDispose(TempFileName);
  end;
end;

function TUnZip20.GetZFileDate(FileName: TFileName): string;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := StrPas(UnZip.GetZFileDate(FFileName, TempFileName));
  finally
   StrDispose(TempFileName);
  end;
end;

function TUnZip20.GetZFileTime(FileName: TFileName): string;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := StrPas(UnZip.GetZFileTime(FFileName, TempFileName));
  finally
   StrDispose(TempFileName);
  end;
end;

function TUnZip20.GetZCompressMethod(FileName: TFileName): word;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := UnZip.GetZCompressedMethod(FFileName, TempFileName);
  finally
   StrDispose(TempFileName);
  end;
end;

function TUnZip20.GetZFileIsDir(FileName: TFileName): boolean;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := UnZip.GetZFileIsDir(FFileName, TempFileName);
  finally
   StrDispose(TempFileName);
  end;
end;

function TUnZip20.GetZFullInfos(FileName: TFileName; Buffer: PChar): boolean;
var
 TempFileName: PChar;
begin
  TempFileName := StrPCopy(StrAlloc(Length(FileName)), FileName);
  try
  result := UnZip.GetZFileFullInfos(FFileName, TempFileName, Buffer);
  finally
   StrDispose(TempFileName);
  end;
end;

procedure TUnZip20.CMOpenFile(var message: TMessage);
var
 FileName: TFileName;
 OK: boolean;
begin
  if Assigned(FOnOpenFile) then
  begin
   FileName := StrPas(PChar(message.LParam));
   OK := boolean(message.WParam);
   FOnOpenFile(Self, FileName, OK);
  end;
end;

procedure TUnZip20.CMExpanding(var message: TMessage);
var
 FileName: TFileName;
 Percent: integer;
begin
  if Assigned(FOnExpanding) then
  begin
   FileName := StrPas(PChar(message.LParam));
   Percent := message.WParam;
   FOnExpanding(self, FileName, Percent);
  end;
end;

procedure TUnZip20.CMCloseFile(var message: TMessage);
var
 FileName: TFileName;
 OK: boolean;
begin
  if Assigned(FOnCloseFile) then
  begin
   FileName := StrPas(PChar(message.LParam));
   OK := boolean(message.WParam);
   FOnCloseFile(Self, FileName, OK);
  end;
end;

procedure TUnZip20.CMNewPassword(var message: TMessage);
var
 FileName: TFileName;
 CurrentPassword: string;
 TempPos: integer;
begin
  if Assigned(FOnOpenFile) then
  begin
   FileName :=  StrPas(PChar(message.LParam));
   TempPos := Pos(#9,FileName);
   if TempPos <> 0 then
   begin
   CurrentPassword := Copy(FileName, TempPos+1, Length(FileName)-TempPos-1);
   Delete(FileName, TempPos, Length(FileName)-TempPos);
   end;
  end;
end;

procedure TUnZip20.CMTest(var message: TMessage);
var
 FileName: TFileName;
 OK: boolean;
begin
  if Assigned(FOnTest) then
  begin
   FileName := StrPas(PChar(message.LParam));
   OK := boolean(message.WParam);
   FOnTest(Self, FileName, OK);
  end;
end;

procedure TUnZip20.CMJoining(var message: TMessage);
var
 FileName: TFileName;
 PartSize: integer;
begin
  if Assigned(FOnJoining) then
  begin
   FileName := StrPas(PChar(message.LParam));
   PartSize := message.WParam;
   FOnJoining(Self, FileName, PartSize);
  end;
end;

procedure TUnZip20.CMJoined(var message: TMessage);
var
 PartSize: LongInt;
 PartNumber: integer;
begin
  if Assigned(FOnJoined) then
  begin
   PartSize := message.LParam;
   PartNumber := message.WParam;
   FOnJoined(Self, PartNumber, PartSize);
  end;
end;

procedure TUnZip20.CMJoinDoneSize(var message: TMessage);
var
 Total: LongInt;
begin
  if Assigned(FOnJoinDoneSize) then
  begin
   Total := message.LParam;
   FOnJoinDoneSize(Self, Total);
  end;
end;

procedure TUnZip20.CMJoinDoneName(var message: TMessage);
var
 FileName: TFileName;
begin
  if Assigned(FOnJoinDoneName) then
  begin
   FileName := StrPas(PChar(message.LParam));
   FOnJoinDoneName(Self, FileName);
  end;
end;

end.
