(*-------------------------------------------------------------------------------------*)
(*                                                                                     *)
(*               Unit: bigmemo                                                         *)
(*                                                                                     *)
(*               Copyright McCallum Whyman Associates Ltd 1996                         *)
(*                                                                                     *)
(*               Version 1.0                                                           *)
(*                                                                                     *)
(*               Author: Tony Whyman                                                   *)
(*                                                                                     *)
(*               Description: This unit provides the TBigMemo class. This is a subclass*)
(*                            of TBigEditor and is designed to provided a compatible   *)
(*                            set of properties and methods to the standard TMEMO.     *)
(*                            It may thus be used as a TMEMO replacement for editing   *)
(*                            large amounts of text.                                   *)
(*										       *)
(*  Note that a key difference between this version and TFileEditor is that when       *)
(*  TBigMemo is used to edit a file, the file must first be loaded into memory before  *)
(*  it is available for editing, whilst TFileEditor pages the file in on demand and    *)
(*  only keeps changes in memory. With large files, TBigMemo will therefore take longer*)
(*  to load than TFileEditor. The reason for this difference is due to the need to keep*)
(*  TBigMemo as similar as possible to TMemo.					       *)
(*-------------------------------------------------------------------------------------*)

Unit bigmemo;

interface

uses fileview, TextBuff, classes, stdctrls, forms, messages, controls, graphics, menus,
     JWEdit;

{ TBigMemo adds to TBigEditor so that it has a compatible set of methods to
  TMemo. Note that the following methods and properties are already supported
  by TBigEditor and have the same semantics as the TMemo methods. There is therefore
  no need to redefine them:

    procedure Clear;
    procedure CopyToClipboard;
    procedure CutToClipboard;
    procedure PasteFromClipboard;
    procedure SelectAll;

  The Modified property is now included in TBigEditor.

  Note that the  following property is not currently supported:

  Alignment i.e. all text is left justified.

  Note that the TBigEditor Scroll Bars are displayed automatically, when they need
  to be. If you wish to use the ScrollBars property you need to set the mode of each
  scrollbar (the VertScrollBar and HorzScrollBar properites) to sbManual using the 
  Delphi Object Inspector.

}

type

TBigMemo = class(TBigEditor)
  private
    FBorderStyle: TBorderStyle;
    FHideSelection: boolean;
    FScrollBars: TScrollStyle;
    FLines: TStrings;
    FWantReturns: boolean;
    FSelMode: TSelModes;
    function GetSelLength: longint;
    procedure SetSelLength(value: longint);
    function GetSelStart: longint;
    procedure SetSelStart(value: longint);
    function GetSelText: string;
    procedure SetSelText(value: string);
    function GetMaxLength: longint;
    function GetWordWrap: boolean;
    procedure SetLines(value: TStrings);
    procedure SetHideSelection(value: boolean);
    procedure SetBorderStyle(value: TBorderStyle);
    procedure SetMaxLength(value: longint);
    procedure SetScrollBars(value: TScrollStyle);
    procedure SetWordWrap(value: boolean);
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message wm_GetDlgCode;
    procedure CMWantSpecialKey(var Msg: TCMWantSpecialKey); message cm_wantSpecialKey;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DoKillFocus; override;
    procedure DoSetFocus; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ClearSelection;
    procedure DeleteLine(ALineNumber: longint);
    procedure InsertLine(ALineNumber: longint; const S: string);
    function GetSelTextBuf(ABuffer: PChar; BufSize: Integer): Integer;
    function GetLineLength(ALineNumber: longint): longint;
    function GetLineCount: longint;
    procedure SetSelTextBuf(ABuffer: PChar);
    Property Text;
  published
    property SelLength: longint read GetSelLength write SetSelLength;
    property SelStart: longint read GetSelStart write SetSelStart;
    property SelText: string read GetSelText write SetSelText;
    property Lines: TStrings read FLines write SetLines;
    Property AutoScroll default true;
    Property AutoIdle default true;
    property Align default alNone;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection: Boolean read FHideSelection write SetHideSelection default True;
    Property LineNumberMode;
    Property MaxUndoLevels;
    property MaxLength: longint read GetMaxLength write SetMaxLength;
    Property MaxLineLength;
    Property OEMConvert;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ScrollBars: TScrollStyle read FScrollBars write SetScrollBars default ssNone;
    Property SelectAfterPaste;
    property ShowHint;
    Property ShowLineBreaks;
    property TabOrder;
    property TabStop;
    property Visible;
    property WantTabs;
    property WantReturns: boolean read FWantReturns write FWantReturns default true;
    property WordWrap:boolean read GetWordWrap write SetWordWrap;
    Property WrapMode;
    Property WPMode default true;
    Property AltCopyKey;
    Property CancelKey;
    Property CopyKey;
    Property LeftArrowKey;
    Property RightArrowKey;
    Property WordLeftKey;
    Property WordRightkey;
    Property HardPageBreakStyle;
    Property SoftPageBreakStyle;
    Property ShowPageBreaks;
    Property HorzScrollBar;
    Property VertScrollBar;
    Property LineStartKey;
    Property LineEndKey;
    Property LineUpKey;
    Property LineDownKey;
    Property PageUpKey;
    Property PageDownKey;
    Property TextStartKey;
    Property TextEndKey;
    Property ScrollUpKey;
    Property ScrollDownKey;
    Property InsertOn;
    Property BackSpaceKey;
    Property InsertKey;
    Property NewLineKey;
    Property DeleteLineKey;
    Property DeleteKey;
    Property DeleteWordKey;
    Property CutKey;
    Property AltCutKey;
    Property PasteKey;
    Property AltPasteKey;
    Property DelLineStartKey;
    Property DelLineEndKey;
    Property QuickKey;
    Property RedoKey;
    Property UndoKey;
    Property DragMargin;
    Property LeftMargin;
    Property RightMargin;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    {$IFNDEF VER80}
    property OnStartDrag;
    {$ENDIF}
    Property OnStateChange;
    Property OnProgressEvent;
    Property OnCaretMoved;
    Property OnWrapModeChange;
    Property OnPagePrint;
    Property OnAbortRequest;
    Property OnReplaceAll;
    Property OnWriteProtect;
End;

implementation

uses  wintypes, winprocs, sysutils;

{The TBigMemoStrings class is used to permit access to the contents of the editor
 as a TStrings object, using the TBigMemo Lines property}

type
 TBigMemoStrings = class(TStrings)
 private
    FBigMemo: TBigMemo;
 protected
    function Get(Index: Integer): string; override;
    function GetCount: Integer; override;
 public
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: string); override;
    procedure LoadFromStream(Stream: TStream); override;
    procedure SaveToStream(Stream: TStream); override;
    Property BigMemo: TBigMemo read FBigMemo write FBigMemo;
 End;

 procedure TBigMemoStrings.Clear;

 Begin
      BigMemo.Clear
 End;

 function TBigMemoStrings.Get(Index: Integer): string;

 Begin
      With BigMemo Do
        Result := GetString(Index,0,GetLineLength(Index))
 End;

 {GetCount returns the number of lines in the text}

 function TBigMemoStrings.GetCount: Integer;

 var done: boolean;

 Begin
       With BigMemo Do
       Begin
            {Ensure last line is well known by calling idle procedure until no
             work to do}
            Repeat HandleOnIdle(nil,done) until done;
            Result := GetLineCount
       End
 End;

 procedure TBigMemoStrings.Delete(Index: Integer);

 var done: boolean;

 Begin
       With BigMemo Do
       Begin
            {Ensure last line is well known by calling idle procedure until no
             work to do}
            Repeat HandleOnIdle(nil,done) until done;
            DeleteLine(Index)
       End
 End;

 procedure TBigMemoStrings.Insert(Index: Integer; const S: string);

 var done: boolean;

 Begin
       With BigMemo Do
       Begin
            {Ensure last line is well known by calling idle procedure until no
             work to do}
            Repeat HandleOnIdle(nil,done) until done;
            InsertLine(Index,S)
       End
 End;

 procedure TBigMemoStrings.LoadFromStream(Stream: TStream);

 Begin
      BigMemo.LoadFromStream(Stream)
 End;

 procedure TBigMemoStrings.SaveToStream(Stream: TStream);

 Begin
      BigMemo.CopyToStream(Stream)
 End;


{------------------------------TBigMemo--------------------------------------}
 constructor TBigMemo.Create(AOwner: TComponent);

Begin
     inherited Create(AOwner);
     FLines := TBigMemoStrings.Create;
     TBigMemoStrings(FLines).BigMemo := self;
     Buffer := TEditStream.Create(Self);
     Buffer.Canvas := Canvas;
     ControlStyle := ControlStyle + [csFramed];
     FHideSelection := true;
     FBorderStyle := bsSingle;
     FWantReturns := true;
     Align := alNone;
     AutoScroll := true;
     AutoIdle := true;
     WPMode := true
End;

destructor TBigMemo.Destroy;

Begin
     if assigned(FLines) Then FLines.Free;
     inherited Destroy
End;

procedure TBigMemo.CreateParams(var Params: TCreateParams);

Begin
  inherited CreateParams(Params);
  if BorderStyle = bsSingle Then
     Params.Style := Params.Style or WS_BORDER
End;

{The SelLength property returns the length (in characters) of the control's
 selected text. By using SelLength along with the SelStart property, you specify
 which part of the text in the control is selected. You can change the number of
 selected characters by changing the value of SelLength. When the SelStart value
 changes, the SelLength value changes accordingly.

 The edit box or memo must be the active control when you change the value of
 SelLength, or nothing appears to happen.}

function TBigMemo.GetSelLength: longint;

Begin
     Result := SelectionSize
End;

procedure TBigMemo.SetSelLength(value: longint);

Begin
     {Set the selection from the current starting point to the (x,y)
      co-ordinates of SelStart + value}
     if value <> SelectionSize Then
     Begin
          SetSelection(SelStartPos,OffsetToPoint(SelStart+value),smExtend);
          redraw
     End
End;

{The SelStart property returns the starting position of the selected part of the
 control's text, with the first character in the text having a value of 0. You
 can use SelStart along with the SelLength property to select a portion of the
 text. Specify the character you want the selected text to start with by its
 position in the text as the value of SelStart.

 When the SelStart value changes, the SelLength value changes accordingly.

 The edit box or memo must be the active control when you change the value of
 SelStart, or nothing appears to happen.}

function TBigMemo.GetSelStart: longint;

Begin
     Result := TextOffset(SelStartPos)
End;

procedure TBigMemo.SetSelStart(value: longint);

Begin
     if SelMode <> smNone Then
     Begin
          SetSelection(OffsetToPoint(value),SelEndPos,SelMode);
          redraw
     End
     Else
         SelStartPos := OffsetToPoint(value);
End;

{The SelText property contains the selected part of the control's text. You can
 use it to determine what the selected text is, or you can change the contents
 of the selected text by specifying a new string. If no text is currently selected,
 the SelText string is inserted in the text at the cursor.}

function TBigMemo.GetSelText: string;

Begin
     Result := GetSelection
End;

procedure TBigMemo.SetSelText(value: string);

Begin
     DeleteSelection;
     AssumedShiftState := [ssShift];
     ExtendSelection;              {marks start of selection, as no text selected}
     InsertString(value);
     ExtendSelection;              {Extends selection to end of inserted text}
     AssumedShiftState := []
End;

procedure TBigMemo.SetLines(value: TStrings);

Begin
     FLines.Assign(value)
End;

{The HideSelection property determines whether text that is selected in an edit
 or memo remains selected when the focus shifts to another control. If True,
 the text is no longer selected until the focus returns to the control. If False, t
 he text remains selected. The default value is True.}

procedure TBigMemo.SetHideSelection(value: boolean);

Begin
  if FHideSelection <> Value then
  begin
    FHideSelection := Value;
    If not Focused Then
      If Value Then
      Begin
         FSelMode := SelMode;
         If FSelMode <> smNone Then
         Begin
              SelMode := smNone;
              redraw
         End
      End
      Else
      Begin
         SelMode := FSelMode;
         if SelMode <> smNone Then
            redraw
      End
  End
End;

procedure TBigMemo.SetScrollBars(value: TScrollStyle);

Begin
  FScrollBars := value;
    case value of
    ssNone:
      Begin
           HorzScrollBar.Visible := false;
           VertScrollBar.Visible := false;
      End;
    ssHorizontal:
      Begin
           HorzScrollBar.Visible := true;
           VertScrollBar.Visible := false;
      End;
    ssVertical:
      Begin
           HorzScrollBar.Visible := false;
           VertScrollBar.Visible := true;
      End;
    ssBoth:
      Begin
           HorzScrollBar.Visible := true;
           VertScrollBar.Visible := true;
      End;
    End
End;

{J-Write has more word wrap modes than TMemo, hence only a partial mapping can
 be done from TBigEditor's WrapMode property to TMemo's WordWrap Property. Essentially,
 wrap on line break only corresponds to WordWrap false, while the wrap on window
 boundaries mode corresponds to WordWrap true.}

function TBigMemo.GetWordWrap: boolean;

Begin
     Result := WrapMode <> opNone
End;

procedure TBigMemo.SetWordWrap(value: boolean);

Begin
     If value Then
        WrapMode := opWrapScreenWidth
     Else
        WrapMode := opNone
End;


{The ClearSelection method deletes text selected in an edit box or memo control.
 If no text is selected in the control when ClearSelection is called, nothing
 happens.}

procedure TBigMemo.ClearSelection;

Begin
     DeleteSelection
End;

{The GetSelTextBuf method copies the selected text from the edit box or memo
 control into the buffer pointed to by Buffer, up to a maximum of (BufSize - 1)
 characters, and returns the number of characters copied.}

function TBigMemo.GetSelTextBuf(ABuffer: PChar; BufSize: Integer): Integer;

var CopyBufHandle: THandle;
    P: pointer;

Begin
     Case SelMode Of          {SelMode is a protected variable that identifies
                               the current selection mode}
     smNone:
       Result := 0;
     smExtend,
     smDouble:
       Begin
           If SelectionSize >= BufSize  Then
              Result := BufSize -1
           Else
              Result := SelectionSize;
         Buffer.CopyText(ABuffer,SelStart,Result);  {Copy requested text from the buffer}
         (ABuffer + BufSize)^ := #0;
       End;
     smColumn:
       Begin
        {Determine copysize from number of lines in column mode selection * line length}

        Result := (SelEndPos.X - SelStartPos.X + 2)
                                          *(SelEndPos.Y - SelStartPos.Y+1) +1;
        CopyBufHandle := GlobalAlloc(GHND,Result);
        If CopyBufHandle = 0 Then
           OutOfMemoryError
        Else
        try
           Result := CopyColumnText(CopyBufHandle); {Get Selected text as lines
                                                       of text}
           P := GlobalLock(CopyBufHandle);
           try
             Move(P^,ABuffer,Result);
           finally
             GlobalUnLock(CopyBufHandle)
           End
          finally
           GlobalFree(CopyBufHandle)
          End
        End
     End
End;

{The SetSelTextBuf method sets the selected text in the edit box or memo control
 to the text in the null-terminated string pointed to by Buffer.}

procedure TBigMemo.SetSelTextBuf(ABuffer: PChar);

var Pos: longint;
    size: longint;

Begin
     DeleteSelection;                            {CLear the current selection}
     Pos := GetInsertPos;                        {Get the current insertion point}
     size := StrLen(ABuffer);
     AssumedShiftState := [ssShift];
     ExtendSelection;                            {Mark start of selection}
     Buffer.Insert(ABuffer,Pos,Size);            {Insert text into buffer}
     PositionTextAt(LinePos);                    {re-synchronise viewer}
     ChangedFrom(Pos);                           {start update of line count, page count, etc.}
     FocusAt(Pos+size,false);                    {Position insertion point at end of insert}
     ExtendSelection;                            {extend selection}
     AssumedShiftState := [];
     Redraw                                      {redraw the screen}
End;

procedure TBigMemo.SetBorderStyle(value: TBorderStyle);

Begin
     FBorderStyle := value;
     if BorderStyle = bsSingle then
          ControlStyle := ControlStyle + [csFixedHeight]
     else
          ControlStyle := ControlStyle - [csFixedHeight];
     RecreateWnd
End;

function TBigMemo.GetMaxLength: longint;

Begin
     Result := MaxTextSize
End;

procedure TBigMemo.SetMaxLength(value: longint);

Begin
     MaxTextSize := value
End;


procedure TBigMemo.DoKillFocus;

Begin
     if FHideSelection And (SelMode <> smNone) Then
     Begin
        FSelMode := SelMode;
        SelMode := smNone;
        redraw
     End
     Else
       FSelMode := smNone;
     inherited DoKillFocus
End;

procedure TBigMemo.DoSetFocus;

Begin
     if FHideSelection And (FSelMode <> smNone) Then
     Begin
          SelMode := FSelMode;
          redraw
     End;
     inherited DoSetFocus
End;

{The following message handler is required by Delphi in order to receive
 special keys such as the up arrow key}

procedure TBigMemo.CMWantSpecialKey(var Msg: TCMWantSpecialKey);

Begin
     If not FWantReturns And (Msg.CharCode = VK_RETURN) Then Msg.Result := 0
     Else
     inherited
End;

function TBigMemo.GetLineLength(ALineNumber: longint): longint;

Begin
     {Position text viewer on ALineNumber}
     LocateLine(ALineNumber,false);
     Result := tvLineLength
End;

procedure TBigMemo.DeleteLine(ALineNumber: longint);

Begin
     {Position text viewer on ALineNumber}
     LocateLine(ALineNumber,false);
     {Delete all line text on current line including any CR/LF separator}
     DeleteText(ALineNumber,0,NextLinePos-LinePos)
End;

procedure TBigMemo.InsertLine(ALineNumber: longint; const S: string);

Begin
     {Position text viewer on ALineNumber}
     LocateLine(ALineNumber,false);
     {Insert text plus a CR/LF separator}
     If OEMConvert Then OemToAnsiBuff(@S[1],@S[1],length(S));
     If (ALineNumber = 0) or (Buffer.Text[LinePos-1]^ = #10) then {insert after line break}
        InsertString(S+#13#10)
     else
        InsertString(#13#10 + S + #13#10)
End;

{This returns the number of lines in the current text. If the last line is empty
 i.e. CR/LF is the last line in the text, then this is ignored.}

function TBigMemo.GetLineCount: longint;

Begin
     If TextSize = 0 Then
        Result := 0
     Else
     If Buffer.Text[TextSize-1]^ = #10 Then
        Result := LastLine
     Else
        Result := LastLine + 1
End;

procedure TBigMemo.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  inherited;
  if not FWantReturns then
    Message.Result := Message.Result and not DLGC_WANTALLKEYS;
end;

end.

