/**************************************
 * config.c : base configuration
 *
 * Copyright (C) 1992 Jouni Leppjrvi
 **************************************/

#include <stdio.h>
#include <string.h>

#define SYS_INCL_STDLIB
#define SYS_INCL_FILEIO
#include "system.h"
#include "usrmsg.h"
#include "connect.h"
#include "bitcode.h"
#include "parse.h"
#include "stresc.h"
#include "config.h"

#define StrCpy(d,s) strncpy(d,s,sizeof(d) - 1);d[sizeof(d) - 1] = '\0'
#define StrCat(d,s) strncat(d,s,sizeof(d) - strlen(d));d[sizeof(d) - 1] = '\0'

#define KW_PACKETSIZE 1
#define KW_BITCODE    2
#define KW_BIT5       3
#define KW_BIT7       4
#define KW_NONE       5
#define KW_PADSTR     6
#define KW_SALVSTR    7
#define KW_INDIR      8
#define KW_OUTDIR     9

#if SYS_PROTOS
static char *ConfigParse(int what,void *p,char *line);
static int _ConfigRead(char *file,CONFIG *c,char *func(int,void *,char *),void *p);
#else
static char *ConfigParse();
static int _ConfigRead();
#endif

static char *ConfigParse(what,p,line)
int what;
void *p;
char *line;
{
    static KEYWORD kwTbl[] = {
    KW_PACKETSIZE,"PACKETSIZE",
    KW_BITCODE,"BITCODE",
    KW_PADSTR,"PADSTR",
    KW_SALVSTR,"SALVSTR",
    KW_NONE,"NONE",
    KW_BIT5,"BIT5",
    KW_BIT7,"BIT7",
    KW_INDIR,"INDIR",
    KW_OUTDIR,"OUTDIR",
    0,NULL};
    static char delim[2] = {SYS_PATHDELIM,'\0'};
    PARSE_ITEM itemTbl[32];
    CONFIG *c = p;
    char *path;
    int n, i;

    switch(what)
    {
        case CONFIG_INIT:
            memset(c,0,sizeof(c));

            path = getenv("PCUCPDIR");

            if (path == NULL)
                return("PCUCPDIR : not defined");

            if (access(path,0))
                return("PCUCPDIR : invalid path");

            i = strlen(path);
            if (i && path[i - 1] == SYS_PATHDELIM)
                path[i - 1] = '\0';

            StrCpy(c -> baseDir,path);
            StrCpy(c -> inDir,path);
            StrCat(c -> inDir,delim);
            StrCat(c -> inDir,"in");
            StrCpy(c -> outDir,path);
            StrCat(c -> outDir,delim);
            StrCat(c -> outDir,"out");

            return(NULL);

        case CONFIG_CHECK:
            if (access(c -> inDir,0))
                return("INDIR : invalid path");
            if (access(c -> outDir,0))
                return("OUTDIR : invalid path");
#ifdef SYS_UNIX
            if (access(c -> inDir,7))
                return("INDIR : needs rwx access");
            if (access(c -> outDir,7))
                return("OUTDIR : needs rwx access");
#endif
            if (c -> packetSize <= 0)
                return("PACKETSIZE : invalid");

            if (c -> bitcode != (c -> bitcode & (M_BIT5 | M_BIT7)))
                return("BITCODE : invalid");

            return(NULL);


        case CONFIG_PARSE:
            n = ParseLine(line,kwTbl,itemTbl,sizeof(itemTbl) / sizeof(itemTbl[0]));
            if (!n)
                return(NULL);
            else
            {
                switch(itemTbl[0].i)
                {
                    case KW_PACKETSIZE:
                        c -> packetSize = itemTbl[1].i;
                        return(NULL);

                    case KW_BITCODE:
                        if (n < 2)
                            return("argument required");

                        c -> bitcode = M_NONE;

                        for(i = 1; i < n ;i++)
                        {
                            if (itemTbl[i].i == KW_BIT5)
                                c -> bitcode |= M_BIT5;
                            if (itemTbl[i].i == KW_BIT7)
                                c -> bitcode |= M_BIT7;
                        }

                        return(NULL);

                    case KW_PADSTR:
                        if (n < 2)
                            return("argument required");

                        if (StrEsc(itemTbl[1].str))
                            return("invalid escape sequence");

                        StrCpy(c -> padString,itemTbl[1].str);

                        return(NULL);

                    case KW_SALVSTR:
                        if (n < 2)
                            return("argument reqiured");

                        if (StrEsc(itemTbl[1].str))
                            return("invalid escape sequence");

                        ConSetSalvStr(itemTbl[1].str);

                        return(NULL);

                    case KW_INDIR:
                        if (n < 2)
                            return("argument required");

                        StrCpy(c -> inDir,itemTbl[1].str);
                        return(NULL);

                    case KW_OUTDIR:
                        if (n < 2)
                            return("argument required");

                        StrCpy(c -> outDir,itemTbl[1].str);
                        return(NULL);

                    default:
                        return("invalid keyword");
            }
        }
    }
    return("unexpected error");
}

static int _ConfigRead(file,c,func,p)
char *file;
CONFIG *c;
#if SYS_PROTOS
char *(*func)(int,void *,char *);
#else
char *(*func)();
#endif
void *p;
{
    FILE *f;
    char line[2][256], *err;
    int l, i, eflg;

    f = fopen(file,"r");
    if (f == NULL)
    {
        MsgDisplay(MSG_ERROR,"cannot open : %s",file);
        return(1);
    }

    for (l = 1, eflg = 0; fgets(line[0],sizeof(line),f) != NULL ;l++)
    {
        for (i = 0; line[0][i] && line[0][i] != '#' && line[0][i] != '\n' ;i++);
        line[0][i] = '\0';

        StrCpy(line[1],line[0]);

        err = ConfigParse(CONFIG_PARSE,c,line[0]);
        if (err != NULL && func != NULL)
            err = (*func)(CONFIG_PARSE,p,line[1]);
        if (err != NULL)
        {
            eflg = 1;
            MsgDisplay(MSG_ERROR,"%s : line %d : %s",file,l,err);
        }
    }

    fclose(f);

    return(eflg);
}

int ConfigRead(file,c,func,p)
char *file;
CONFIG *c;
#if SYS_PROTOS
char *(*func)(int,void *,char *);
#else
char *(*func)();
#endif
void *p;
{
    static char delim[2] = {SYS_PATHDELIM,'\0'};
    char buf[SYS_MAXPATH], *err;

    err = ConfigParse(CONFIG_INIT,c,NULL);
    if (err == NULL && func != NULL)
        err = (*func)(CONFIG_INIT,p,NULL);
    if (err != NULL)
    {
        MsgDisplay(MSG_ERROR,"config init : %s",err);
        return(-1);
    }

    StrCpy(buf,c -> baseDir);
    StrCat(buf,delim);
    StrCat(buf,"pcucp.cfg");
    if ((file == NULL || !file[0]) || !access(buf,0))
        if (_ConfigRead(buf,c,func,p))
            return(-1);

    if (file != NULL && file[0])
    {
        StrCpy(buf,c -> baseDir);
        StrCat(buf,delim);
        StrCat(buf,file);
        StrCat(buf,".cfg");
        if (_ConfigRead(buf,c,func,p))
            return(-1);
    }

    err = ConfigParse(CONFIG_CHECK,c,NULL);
    if (err == NULL && func != NULL)
        err = (*func)(CONFIG_CHECK,p,NULL);
    if (err != NULL)
    {
        MsgDisplay(MSG_ERROR,"config check : %s",err);
        return(-1);
    }

    return(0);
}
