;;;    Copyright (C) 1996 by Autodesk, Inc.
;;;
;;;    Permission to use, copy, modify, and distribute this software
;;;    for any purpose and without fee is hereby granted, provided
;;;    that the above copyright notice appears in all copies and
;;;    that both that copyright notice and the limited warranty and
;;;    restricted rights notice below appear in all supporting
;;;    documentation.
;;;
;;;    AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
;;;    AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
;;;    MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
;;;    DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
;;;    UNINTERRUPTED OR ERROR FREE.
;;;
;;;    Use, duplication, or disclosure by the U.S. Government is subject to
;;;    restrictions set forth in FAR 52.227-19 (Commercial Computer
;;;    Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
;;;    (Rights in Technical Data and Computer Software), as applicable.
;;;
;;;.
;;;;Written by K.C. JONES and RANDY KINTZLEY 02-09-96
;_____________________________________________________________________________
;The problem:
;Leader and Tolerance entities that do not have an associated dimstyle.
;
;_____________________________________________________________________________
;The approach to fixing:
;Run a lisp routine on c3 to fix the corrupted DWG's.
;C:FIX_DIMS will search the drawing and block table for bad entities 
;and fix them.
;
;1. Search the drawing for leader and tolerances using ssget.
;Loop through those entities looking for ones that have no dimstyle
;assigned to them. When one is found: The entity name is passed to a 
;function that will fix it. This function is called fix_badone. 
;This function makes a copy of the current dimstyle called "_DIM_ERR" 
;and sets it current. (this is done once) 
;Then the "_DIM_ERR" dimstyle gets assigned to the bad entity using the 
;dimstyle command.
;
;2. Search the block table for bad entities in block definitions and redefine
;the blocks that contain bad entities. Bad blocks (that is.. block definitions that 
;contain bad leaders or tolerance entities.) are inserted with "*" in front of
;the name to explode them. The entities created from the exploded insert are 
;then searched for bad leaders and tolerances and fixed using the fix_badone function.
;Finally the block is redefined incorporating the fixed leader and tolerance entities.
;NOTES: anonymous blocks are handled slightly differently. An insert of the block
;is created via entmake. The block is exploded, entities fixed and it is re-defined
;via entmake.
; In case you are wondering: why do an insert/explode? Why not just redefine via entmake
;at the block table level? The reason is because of a side effect of the dimstyle problem.
;(entget na) of TOLERANCE entities returns nil when the tolerance has no dimstyle.
;You cannot do an entmake if you don't know what the entity data should be.
;Fortunately ssget "x" will return  a selection set of all tolerance entities.
;So when looping through the selection set if (entget na) returns nil, then you
;know two things. 1 the entity is a tolerance and 2 it is bad. Then all you
;have to do is pass the entity name to dimstyle apply and the entity is fixed!
;
;So the idea is: insert and explode the block, 
;                build a selection of entities from the explode 
;                pass that selection set to the select command
;                use (ssget "p" '((0 . "TOLERANCE"))) to get another selection set
;                fix the bad entities in that set
;                redefine the block using entmake or the block command.
;
;_____________________________________________________________________________
;Autolisp commands defined in this file:
;C:FIX_DIMS - this is the primary fixer. It can be run in two different 
;modes. 
;1. From the command line. 
;2. From a script. When run from a script c:fix_dims will automatically save the 
;drawing if it has completed it's task successfully. Also the function will report 
;it's find/fix activities to a file called "report.txt" located in the same directory 
;as the text file that lists the drawing names. (created and specified by the user)
;
;C:MULTIPLE_FIX - This command will generate and run a script that will execute
; c:fix_dims on a user specified list of drawings. The user specifies the drawings
;to be fixed with a text file that lists the drawing names with full path and 
;extension. This command will read the text file specified by the user and
;will write a script that: 
;(a) opens each drawing 
;(b) loads this lisp file
;(c) runs c:fix_dims
;
;Two other files are also created. 2bfixed.txt and fixed.txt. They are both placed
;in the same directory where the text file listing the names of drawings to be fixed is.
;Initially 2bfixed.txt is a list of all drawings found in the user specified file
;that are valid and existing. As the routine runs drawing names are removed from
;the 2bfixed file and placed in the fixed.txt file. If the script/routine stops for 
;any reason the user then can easily determine where the problem occurred by examining 
;these two files. He/she can then copy the 2bfixed.txt file to another name and 
;specify that file as the DWG list when running multiple_fix again. This will allow
;the user to easily re-start the process where it stopped.
;
;C:SCR_OPEN - This command is executed from the script that c:multiple_fix creates.
;It checks the status of the dbmod variable so that the Do you really want to discard
;changes? prompt can be anticipated and answered within the routine. That way the 
;command has a non-variable number of prompts. After scr_open is entered the next prompt
;will always be for the drawing name. If c:fix_dims runs successfully from a script
;it will issue a qsave, so if everything went well the dbmod setting should be 0.
;When dbmod is NOT equal to 0, then something went wrong so changes are always discarded
;by c:scr_open. The *error* routine described later will report the error to report.txt
;and issue a resume to get the script going again.
;
;C:RESTORE_FONTALT - yeah it does just that. When multiple_fix is run
;the status of fontalt is checked to see that it is set and that the font file
;can be found. If it is not set or the file cannot be found then fontalt is set
;to "txt.shx". The original value is stored in the appdata section of the config file.
;The last line in the script is restore_fontalt. The function gets the original setting
;from the config file and sets fontalt back to it's original value prior to
;running multiple_fix. All of this is done to ensure that the script
;will run uninterrupted.
;
;NOTE: Variables with a # prefix are global.

;

;========================================================================
(defun c:fix_dims ( / ssbig ctr n j es el msg path fh lst)

(if #locked
    (progn
     (setq #locked nil
           *error* nil
     );setq
     (exit)
    );progn then exit out and do nothing
);if

;Do some error handling stuff and record the state of the drawing's
;various settings so they can be restored later.
(if (not #old_err)
    (setq #old_err *error*);setq then
);if
(setq *error* my_err);setq
(if (not #cmdecho)   (setq #cmdecho (getvar "cmdecho")))
(if (not #highlight) (setq #highlight (getvar "highlight")))
(if (not #regenmode) (setq #regenmode (getvar "regenmode")))
(if (not #tilemode)  (setq #tilemode (getvar "tilemode")))
(if (not #dimstyle)  (setq #dimstyle (getvar "dimstyle")))
(if (not #clayer)    (setq #clayer (getvar "clayer")))

(if (not #dimvars)   (setq #dimvars (dims nil)));get the current settings for dim vars
(if (not #la_lst)    (setq #la_lst (layer_dat)));get the current status of all layers

;Now set things up to do the task at hand.
(setvar "cmdecho" 0)
(setvar "highlight" 0) 
(setvar "regenmode" 0)
(command "_layer" "_unlock" "*" "_thaw" "*" "_on" "*" "")

(if (equal 4 (logand 4 (getvar "cmdactive")))
    (progn 
     (print "Running in script mode")
     (setq #script 99);setq 
    );progn then
    (setq #script nil);setq else
);if 

(setq n 0) ;setq counter for bad entities found 
(setq j 0) ;setq paper space/model space ssget
(repeat 2  ;once for model space and once for paper space

(setq ssbig nil);setq
(if (equal j 0)
    (progn
     (princ "\nSearching for problem objects in Paper space...") 
     (setq ssbig (ssget "_x" (list '(-4 . "<OR") 
                                   '(0 . "LEADER") '(0 . "TOLERANCE")
                                   '(-4 . "OR>")     
                                   '(67 . 1) 
                             );list
                 );ssget get the paper space  
     );setq  
     (if ssbig
         (progn
          (setvar "tilemode" 0)
          (command "_pspace")
         );progn then
     );if
    );progn then do paper space
    (progn
     (princ "\nSearching for problem objects in Model space...")
     (setq ssbig (ssget "_x" (list '(-4 . "<OR") 
                              '(0 . "LEADER") '(0 . "TOLERANCE")
                              '(-4 . "OR>")     
                              '(-4 . "/=") 
                              '(67 . 1) 
                        );list
            );ssget get the model space        
     );setq
     (if ssbig
         (setvar "tilemode" 1)
     );if
    );progn else do model space 
);if   

(if ssbig
    (progn

     ;; Filter out all objects that already have a dimstyle reference
     (setq ctr 0);setq counter for selection set 
     (repeat (sslength ssbig)
       (setq   es (ssname ssbig ctr)
               el (entget es);This returns nil on bad tolerance entities
       );setq
       (if (not (assoc 3 el))
           (progn 
            (princ "\nBad") 
            (setq n (+ n 1));setq
            (fix_badone es) 
           );progn then bad entity
           (princ "\nGood");else
       );if
       (setq ctr (+ ctr 1));setq
     );repeat

    );progn then ssget was successful
    (setq n 0);setq else
);if

(setq j (+ j 1));setq
);repeat 2 once for paper space and once for model space

(princ "\nSearching for problem objects in block table...")
(setq n (+ n (block_fix)));setq

;;let the user know what's going on.
(if (and n 
         (not (equal n 0))
    );and
    (setq msg (strcat (itoa n) " problem objects found and fixed"));setq then
    (setq msg "No problem objects found");else
);if
(princ (strcat "\n" msg "\n"))

;;Return the drawings' settings to their original state
(layer_restore #clayer #la_lst)
(command "_dimstyle" "_restore" #dimstyle)
(dims #dimvars)
(setvar "cmdecho"   #cmdecho)
(setvar "highlight" #highlight)
(setvar "regenmode" #regenmode)
(setvar "tilemode"  #tilemode)

(if #script
    (progn
     (if (and n 
              (not (equal n 0))
         );and
         (progn 
          (princ "\nSaving...\n")
          (command "_qsave") 
         );progn then
     );if  
     ;;Report the find/fix activities to "report.txt"
     (setq path (getlocation); path is the location where all ASCII files are read/written 
             fh (open (strcat path "report.txt") "a");log the current drawing as fixed.
     );setq
     (write-line (strcat (getvar "dwgname") ".dwg" (chr 9) msg) fh)
     (close fh)

     ;;Write this drawings name to the fixed.txt file.
     (setq fh (open (strcat path "fixed.txt") "a"));setq
     (write-line (strcat (getvar "dwgname") ".dwg") fh)
     (close fh)  

     ;;Pull the first entry off of "2bfixed.txt"  and re-write the file out.
     (setq lst (cdr (read_dwg_list (strcat path "2bfixed.txt")))
            fh (open (strcat path "2bfixed.txt") "w")
             n 0
     );setq
     (repeat (length lst)
      (write-line (nth n lst) fh) 
      (setq n (+ n 1));setq
     );repeat
     (close fh)

     (princ "\nDone")
    );progn then a script is active
);if

(setq *error* #old_err);setq

(princ)
(command "_qsave")
);defun c:fix_dims

;-----------------------------------------------------------------------------
;Writes and runs a script that executes fix_dims on a user specified list
;of drawings.
;
(defun c:multiple_fix ( / fna fna2 lst path fh fh2 n a b)

;error handler
(if (not #old_err)
    (setq #old_err *error*);setq then
);if
(setq *error* my_err);setq

;Get the name of the text file that contains a listing of drawing names.
(setq fna 
      (getfiled "Select the text file that contains a list of drawing names"
                "DWGLIST.TXT" "TXT" 0
      );getfiled
);setq
(if (and fna 
         (setq fna (findfile fna));setq
    );and
    (progn
     
     (setq path (getpath fna));setq
     (setq fh (open (strcat path "report.txt") "a"));setq
     (if fh
         (progn
          (write-line (strcat "Date: " 
                              (substr (itoa (fix (getvar "cdate"))) 5 2) "-"
                              (substr (itoa (fix (getvar "cdate"))) 7 2) "-"
                              (substr (itoa (fix (getvar "cdate"))) 3 2)
                      );strcat
                      fh
          );write-line 
          (close fh)
         );progn then
         (progn
          (alert (strcat "Cannot open: " path "report.txt for write."))
          (exit)
         );progn else exit with grace
     );if

     (setq fna2 (findfile "fix_dims.lsp"));setq
     (if (not fna2)
         (progn
          (setq fna2 (getfiled "Can't find FIX_DIMS.LSP. Please select the file." 
                               "fix_dims.lsp" "lsp" 0
                     );getfiled
          );setq
          (if (not (and fna2 
                        (findfile fna2)
                        (equal "FIX_DIMS.LSP"
                               (strcase (substr fna2 (+ 1 (strlen (getpath fna2)))))
                        );equal
                   );and
              );not 
              (exit);then abort
          );if      
         );progn then 
     );if  
      
     (setq  lst (read_dwg_list fna);read the file and return a list of drawings to be fixed.
             fh (open (strcat path "2bfixed.txt") "w")
            fh2 (open (strcat path "fix_dims.scr") "w")
     );setq
     
     (if (not fh)
         (progn 
          (alert (strcat  "Cannot open: "  path "2bfixed.txt for write."))
          (exit)
         );progn then exit with grace
     );if
     (if (not fh2)
         (progn 
          (alert (strcat  "Cannot open: " path "fix_dims.scr for write."))
          (exit)
         );progn then exit with grace
      );if
      (setcfg "AppData/fix_dims/location" path);put the location of the report file in 
                                               ;the config so it can be retrieved later.

      ;;Write out the list of dwgs to be fixed to "2bfixed.txt" fh
      ;;Also write the script "fix_dims.scr" fh2
     (setq n 0);setq            
     (repeat (length lst)
      (setq a (nth n lst));setq a is a drawing name

      (write-line a fh);put the drawing name in "2bfixed.txt"
         
      (write-line "scr_open" fh2)
      (write-line a fh2)

      ;;load this lisp file.
      (write-line (strcat (chr 40) "load " 
                           (chr 40) "getstring " (chr 34) "Load file:" (chr 34) (chr 41)
                          (chr 41)
                  );strcat
                  fh2
      );write-line 
      (write-line fna2 fh2);the location of this lisp file
      (write-line "fix_dims" fh2)
      
     (setq n (+ n 1));setq
     );repeat
     (write-line "restore_fontalt" fh2);this command restores fontalt to it's original setting
     (close fh)
     (close fh2)

     ;; Make sure there is a valid font file name stored in fontalt
     ;; to ensure that the script will run without interruption by a dialog. 
     (setq a (getvar "fontalt" ))
     (setcfg "AppData/fix_dims/old_fontalt" a);store the original setting of fontalt
                                              ;so it can be restored later 
     (if (and (not (findfile a))
              (not (findfile (strcat a ".shx")))
              (not (findfile (strcat a ".ttf")))
              (not (findfile (strcat a ".pfb")))
              (not (findfile (strcat a ".pfm")))
              (findfile "txt.shx")
         );and
         (setvar "fontalt" (findfile "txt.shx"));then set fontalt
     );if
         
     ;Now run the script
     (command "_script" (strcat path "fix_dims.scr"))

    );progn then fna was found
    (alert "File not found");else
);if

(princ)
);defun c:multiple_fix

;-------------------------------------------------------------------------
;This function prompts for file name only and handles the 
;"Really want to discard changes?" prompt. 
;Changes are discarded - because it is assumed the fix_dims function will
;save the drawing upon successful completion. So... if dmod /= 0 then
;fix_dims was not successful so better not save.
;
(defun c:scr_open ( / a path fh lst n)

;error handler
(if (not #old_err)
    (setq #old_err *error*);setq then
);if
(setq *error* my_err);setq
(if (not #cmdecho)   (setq #cmdecho (getvar "cmdecho")))
(setvar "cmdecho" 0)

(setq a (getstring "\nDrawing name:"));setq
(if (not (findfile (strcat (substr a 1 (- (strlen a) 4))
                          ".dwk"
                   );strcat
         );findfile
    );not
    (progn
     (command "_open")
     (if (not (equal 0 (getvar "dbmod")))
         (command "_Y");then discard the changes.
     );if
     (command a)
     (if (wcmatch (getvar "cmdnames") "*OPEN*")
         (progn
          (setq #locked 1);setq         
          (while (wcmatch (getvar "cmdnames") "*OPEN*")
            (command nil)
          );while get out of the open command
         );progn then there is a problem with opening the file. probably read-only
         (setq #locked nil);setq
     );if
    );progn then the file is not locked
    (progn
     (setq #locked 99);setq    
    );progn else the file is locked 
);if

(if #locked
    (progn
       ;;write the error to "report.txt"
     (setq path (getlocation)
             fh (open (strcat path "report.txt") "a")
     );setq
     (if (equal #locked 99)
         (write-line (strcat a (chr 9) "WARNING! > File locked. Not proccessed") fh)
         (write-line (strcat a (chr 9) "WARNING! > File not proccessed, may be read-only") fh)
     );if
     (close fh)

     ;;Now remove the current drawing from the top of the 2bfixed.txt file
     ;; and move it to the bottom of the file. So that when the script has
     ;; completed it's run, the 2bfixed.txt file will only list the drawings
     ;;that were not successfully fixed by the routine.
     (setq lst (read_dwg_list (strcat path "2bfixed.txt")));setq
     (if lst
         (setq lst (append (cdr lst) (list (car lst))));setq then
     );if      
     (setq fh (open (strcat path "2bfixed.txt") "w"));setq
     (setq n 0);setq
     (repeat (length lst)
      (setq a (nth n lst));setq
      (write-line a fh)
      (setq n (+ n 1));setq
     );repeat
     (close fh)

     (if (equal #locked 1)
         (command "_resume");then it was probably a read only file  
     );if
    );progn the the file is locked or read-only
);if

(setvar "cmdecho" #cmdecho)

(princ)
);defun c:scr_open

;--------------------------------------------------------------------
;This command restores fontalt to it's original setting
;prior to running multiple_fix
(defun c:restore_fontalt ()
 (setvar "fontalt" (getcfg "AppData/fix_dims/old_fontalt"))
(princ)
);defun c:restore_fontalt

;-----------------------------------------------------------------------
;Searches the block table for bad leaders and tolerances within
;block definitions and redefines blocks as needed.
;
(defun block_fix ( / lst n x a lst2 j flag e1 na ss ss2 xr)

(setq lst (tblnamelst "BLOCK");get a list of block names 
        n 0 ;the lst counter
        x 0 ;the bad entity counter
);setq
(repeat (length lst)
 (setq    a (nth n lst)          ;the block name
       flag nil
 );setq 
 (if (and (not (equal "*X" (substr a 1 2))) ;;no need to search through these entities
          (not (equal "*D" (substr a 1 2)))
     );and
     (progn
      (setq lst2 (table_dat "block" a);the data for the block name 'a'
               j 0 ;the counter for the block data list lst2
      );setq
      (while (and (< j (length lst2))
                  (not flag)
             );and
       (setq e1 (nth j lst2));setq the nested entity list
       
       (if (and (not (assoc 3 e1))
                (or (not e1) ;this needs to be here because (entget ) of
                             ;a bad tolerance sometimes returns nil
                    (equal "LEADER" (cdr (assoc 0 e1)))
                    (equal "TOLERANCE" (cdr (assoc 0 e1)))
                );or 
           );and
           (setq flag 99);setq jump out of while loop because we have a potential bad block
       );if
       (setq j (+ j 1));setq
      );while 
     );progn then not a hatch pattern or a dim so search it for bad entities
 );if
 (if flag
     (progn
      (setq #na (entlast);this is for the error handler. #na is global so *error* can use
             na #na      ;entnext to clean up temp entities
             ss nil
             ss (ssadd)
             xr (cdr (assoc 70 (car lst2)))
      );setq
      (if (and (not (equal 4  (logand 4 xr)))   ;;See if it is an xref
               (not (equal 16 (logand 16 xr)))
               (not (equal 32 (logand 32 xr)))
          );and 
          (progn
           ;;if not an anonymous block then insert the block with the explode option
           (if (not (equal "*" (substr a 1 1))) 
               (command "_insert" (strcat "*" a) "0,0,0" "1" "0");command then not an anonymous block
               (progn
                (entmake (list '(0 . "INSERT") (cons 2 a) '(10 0.0 0.0 0.0) '(50 . 0.0)
                               '(41 . 1.0) '(42 . 1.0) '(43 . 1.0) 
                         );list
                );entmake 
                (command "_explode" (entlast));setq
               );progn else it is an anonymous block so do an entmake of an insert and explode
           );if
           (if (not #na)
               (setq #na (entnext)
                      na #na
                      ss (ssadd na ss)
               );setq then the drawing was empty prior to the insert above so use (entnext)
                     ;with no arg to get first entity
           );if
           (while (setq na (entnext na));setq
            (setq ss (ssadd na ss));setq
           );while build a selection set of the entities created during explode/insert 
            
           (if (and ss
                    (> (sslength ss) 0)
               );and   
               (progn 
                
                (command "_select" ss "") ;this is so I can use the ssget "p" below   
                (setq ss2 (ssget "p" (list '(-4 . "<OR") 
                                           '(0 . "LEADER") '(0 . "TOLERANCE")
                                           '(-4 . "OR>")     
                                     );list
                          );ssget
                );setq 
                (if (and ss2 
                         (> (sslength ss2) 0)
                    );and
                    (progn  
                     (setq j 0);setq           ;;loop through the leader and tolerance entities
                     (repeat (sslength ss2)    ;;created from the insert/explode 
                      (setq na (ssname ss2 j));setq
                      (if (not (assoc 3 e1))
                          (progn
                           (setq x (+ x 1));setq
                           (princ "\nBad")
                           (fix_badone na)  
                          );progn
                      );if 
                      (setq j (+ j 1));setq
                     );repeat
               
                     ;;now redefine the block     
                     (if (not (equal "*" (substr a 1 1))) 
                         (command "_block" a "_Y" "0,0,0" ss "");command then not an anonymous block
                         (redef_anonymous a ss);else
                     );if 
                         
                    );progn then there were some bad leader or tolerance entities in the block
                    (command "_erase" ss "");else no need to re-define so just erase
                );if
                
               );progn then ss is non-nil and has length>0 / the block has graphical entities
           );if
          );progn then it is NOT a xref
      );if
     );progn bad object found in block definition
 );if

 (setq n (+ n 1));setq
);repeat the number of block definitions in drawing

x
);defun block_fix

;---------------------------------------------------------------
;takes an anonymous block name and a selection set of entities and redefines the block 
(defun redef_anonymous ( a ss / b n na e1)

(entmake (list '(0 . "BLOCK")
               (cons 2 a)
               (list 10 0.0 0.0 0.0)
               '(70 . 1)
               (cons 3 a)
         );list
);entmake

(setq n 0);setq
(repeat (sslength ss)
 (setq na (ssname ss n)
       e1 (entget na '("*")) 
        b (assoc -1 e1)
       e1 (append (reverse (cdr (member b (reverse e1))))
                  (cdr (member b e1))
          );append remove the entity name from the list
        b (assoc 5 e1)
       e1 (append (reverse (cdr (member b (reverse e1))))
                  (cdr (member b e1))
          );append remove the handle from the list
 );setq

 (entmake e1)
 (entdel na)

(setq n (+ n 1));setq
);repeat
(entmake (list (cons 0 "ENDBLK")))

);defun redef_anonymous


;---------------------------------------------------------------
;takes an entity name of a bad leader or tolerance
;and fixes it
(defun fix_badone (na / ) 

;;see if _dim_err is the current dimstyle
(if (not (equal (getvar "dimstyle") "_DIM_ERR"))
    (progn
     ;;_dim_err is not the current style so see if it is defined   
     (if (not (member "_DIM_ERR" (tblnamelst "DIMSTYLE")))
         (command "_dimstyle" "_save" "_dim_err");define it
     );if 
     (command "_dimstyle" "_restore" "_dim_err");restore _dim_err
    );progn then
);if


(command "_dimstyle" "_apply" na "")

);defun fix_badone

;--------------------------------------------------
;Takes the name of a table 
;returns a list of table entry names
;
;i.e. (tblnamelst "layer") -> returns a list of layer names
;
(defun tblnamelst ( name / a lst)

(while (setq a (tblnext name (not a)));
(setq lst (append lst (list (cdr (assoc 2 a))));append
);setq
);while

lst
);defun tblnamelst

;------------------------------------------------------------------
;returns a list of data for a -table name and b -entry name     
(defun table_dat ( a b / na lst)

(setq  na (tblsearch a b)
      lst (list na)
       na (cdr (assoc -2 na))
);setq
(while (setq lst (append lst (list (entget na)));append
	      na (entnext na)
       );setq
);while

lst
);defun table_dat

;----------------------------------------------------------------------
;returns a list of layer table data
(defun layer_dat ( / a n lst lst2)

(setq lst (tblnamelst "layer");get a list of layer names
        n 0
);setq
(repeat (length lst)
 (setq    a (nth n lst)
          a (tblsearch "layer" a)
       lst2 (append lst2 (list a))
 );setq
 (setq n (+ n 1));setq
);repeat

lst2
);defun layer_dat

;----------------------------------------------------------------------
;takes current layer and a list of layer data as it originally appeared
;and restores all layer settings to their original state.
(defun layer_restore (cl lst / x y la na n)

(command "_layer" "_unlock" cl "_thaw" cl "_on" cl "_set" cl);command

(setq n 0);setq
(repeat (length lst)
(setq la (nth n lst)
      na (cdr (assoc 2 la))
       x (cdr (assoc 70 la))
       y (cdr (assoc 62 la))
);setq
(if (equal 1 (logand 1 x))
    (command "_Freeze" na);then this layer was originally frozen
);if
(if (equal 4 (logand 4 x))
    (command "_lock" na);then this layer was originally locked
);if
(if (< y 0) 
    (progn 
     (command "_off" na)
     (if (equal na cl)
         (command "_y");really want to turn off current layer?
     );if 
    );progn
    (command "_on" na)
);if
(setq n (+ n 1));setq
);repeat
(command "");get out of layer command

);defun layer_restore

;------------------------------------------------------------
;takes the name of a file and reads it line by line
;and returns a list of valid and existing drawing names contained in the file.
;
(defun read_dwg_list (fna / fh fh2 a b lst) 

(if (and fna 
         (setq fna (findfile fna))
    );and
    (progn
     (setq fh (open fna "r"));setq
     (while (setq a (read-line fh));setq
      (setq a (s_strip a));setq strip any leading or trailing spaces.
      (if (and (setq b (findfile a));setq
               (equal "DWG"
                      (strcase (substr b (- (strlen b) 2) 3));get the extension of the filename
               );equal
          );and
          (setq lst (append lst (list b)));setq then
          (progn
           (if b    ;determine the problem and report it
               (progn 
                (setq b (strcat "Not a drawing file or dwg extension not included: " a));then
                (princ (strcat "\n" b)) 
               );progn then 
               (progn 
                (if (not (equal a ""))
                    (progn 
                     (setq b (strcat "Cannot find file: " a));else
                     (princ (strcat "\n" b)) 
                    );progn then
                    (setq b nil);setq else it was an empty line
                );if
               );progn else
           );if
           (if b ;;then there is a problem worth reporting
               (progn 
                (if (setq fh2 (open (strcat (getpath fna) "report.txt") "a"));setq
                    (progn 
                     (write-line b fh2)
                     (close fh2)
                    );progn then 
                    (progn
                     (alert "Cannot open: " (getpath fna) "report.txt for write.")
                     (exit) 
                    );progn else
                );if
               );progn then
           );if 
          );progn else this line in fna is not a drawing file or was not found.
      );if
     );while build a list of valid and found dwg names
     (close fh)
    );progn then found the file 
    (progn 
      (print "Cannot find: ") 
      (print a) 
      (exit)
    );progn else
);if

lst
);defun read_dwg_list


;--------------------------------------------------------------
;Takes a full file name and returns the path
;i.e. (getpath "c:\\acad\\support\\acad.pgp") returns "c:\\acad\\support\\"
(defun getpath (fna / j n)

(setq n 1);setq
(repeat (strlen fna)
 (if (or (equal "\\" (substr fna n 1))
         (equal "/" (substr fna n 1))
     );or 
     (setq j n);setq
 );if
 (setq n (+ n 1));setq
);repeat

(substr fna 1 j)
);defun getpath

;---------------------------------------------------------------
;gets the info from config file
;the info was placed there by c:multiple_fix
(defun getlocation ( / a)

(setq a (getcfg "AppData/fix_dims/location"));setq
(if (not a)
    (progn
     (alert "Cannot find cfg data, use multiple_fix")
     (exit)
    );progn then
);if
a
);defun getlocation

;-----------------------------------------------------------------
;**S_STRIP** space strip function that strips the leading and trailing spaces
;off of any string
;tabs are treated same as spaces are.
(defun s_strip ( a / )

(while (and (not (equal "" a))
	    (or (equal (substr a (strlen a));substr
		       " "
	        );equal
                (equal (substr a (strlen a));substr
		       (chr 9);its a tab
	        );equal
            );or
       );and
(setq a (substr a 1 (- (strlen a) 1)));setq
);while
(while (and (not (equal "" a))
	    (or (equal (substr a 1 1);substr
		       " "
	        );equal
                (equal (substr a 1 1);substr
		       (chr 9);tab character
	        );equal
            );or
       );and
(setq a (substr a 2));setq
);while

a
);defun s_strip
;---------------------------------------------------------
;Returns a list of dim variable settings when lst2 is nil.
;Later the list returned can be passed to the function
;and all settings will be restored.
;
(defun dims ( lst2 / lst n)
(setq lst (list "dimALT" "dimALTD" "dimALTF" "dimALTTD" "dimALTTZ" "dimALTU" "dimALTZ" 
                "dimAPOST" "dimASO" "dimASZ" "dimAUNIT" "dimBLK" "dimBLK1" "dimBLK2"
                "dimCEN" "dimCLRD" "dimCLRE" "dimCLRT" "dimDEC" "dimDLE" "dimDLI" 
                "dimEXE" "dimEXO" "dimFIT" "dimGAP" "dimJUST" "dimLFAC" "dimLIM" 
                "dimPOST" "dimRND" "dimSAH" "dimSCALE" "dimSD1""dimSD2" "dimSE1" 
                "dimSE2" "dimSHO" "dimSOXD"  "dimTAD" "dimTDEC" "dimTFAC" 
                "dimTIH""dimTIX" "dimTM" "dimTOFL" "dimTOH" "dimTOL" "dimTOLJ" "dimTP" 
                "dimTSZ" "dimTVP" "dimTXSTY" "dimTXT" "dimTZIN" "dimUNIT" "dimUPT" "dimZIN"
          );list 
        n 0
);setq
(if lst2
    (progn
     (repeat (length lst)
      (setvar (nth n lst) (nth n lst2)) 
     (setq n (+ n 1));setq
     );repeat
     (setq lst2 nil);setq
    );progn then restore
    (progn
     (repeat (length lst)
      (setq lst2 (append lst2 (list (getvar (nth n lst)))));setq
     (setq n (+ n 1));setq
     );repeat
    );progn else get and return settings
);if
lst2
);defun dims
;--------------------------------------------------------
(defun my_err ( a / path na fh lst n)

(print a)

(if #na
    (progn
     (setq na #na);setq
     (while (setq na (entnext na));setq
      (entdel na)
     );while
    );progn then clean up any entities from an insert
);if

(while (or (equal 1 (logand 1 (getvar "cmdactive")))
           (equal 2 (logand 2 (getvar "cmdactive")))
       );or
(command nil)       
);while get out of any active command other than a script running

;restore drawings original settings
(if #highlight (setvar "highlight" #highlight))
(if #cmdecho   (setvar "cmdecho" #cmdecho))
(if #regenmode (setvar "regenmode" #regenmode))
(if #tilemode (setvar "tilemode" #tilemode))
(if #dimstyle (command "_dimstyle" "_restore" #dimstyle))
(if #dimvars (dims #dimvars));restore the dim over-rides
(if (and #clayer #la_lst) (layer_restore #clayer #lay_lst))

(if #script 
    (progn
     ;;write the error to "report.txt"
     (setq path (getlocation)
             fh (open (strcat path "report.txt") "a")
     );setq
     (write-line (strcat (getvar "dwgname") (chr 9) "ERROR! > "  a) fh)
     (close fh)

     ;;Now remove the current drawing from the top of the 2bfixed.txt file
     ;; and move it to the bottom of the file. So that when the script has
     ;; completed it's run, the 2bfixed.txt file will only list the drawings
     ;;that were not successfully fixed by the routine.
     (setq lst (read_dwg_list (strcat path "2bfixed.txt")));setq
     (if lst
         (setq lst (append (cdr lst) (list (car lst))));setq then
     );if      
     (setq fh (open (strcat path "2bfixed.txt") "w"));setq
     (setq n 0);setq
     (repeat (length lst)
      (setq a (nth n lst));setq
      (write-line a fh)
      (setq n (+ n 1));setq
     );repeat
     (close fh)

     (command "_resume");kick start the script
    );progn then
);if

(setq *error* #old_err);setq
(princ)
);defun my_err


;-----------------------------------------------------------------------------
(textscr)
(princ "\nFIX_DIMS version 1.1\n")
(princ "The FIX_DIMS command will search the drawing for damaged dim objects \n")
(princ "that have no DIMSTYLE and assign a DIMSTYLE to them. \n") 
(princ "Type FIX_DIMS to run it on the current drawing. \n")
(princ "Type MULTIPLE_FIX to run it on a several drawings. \n")
(princ "\nNote: You must list the full names of the drawings to be fixed ")
(princ "\n      in a text file in order to use multiple_fix.")
(princ)
(c:fix_dims)
