// Associated include file : ColEdit.H

#include "common/common.h"
#include "drivers/drivers.h"
#include "mecanism/mecanism.h"
#include "graphics/graphics.h"
#include "gadgets/gadgets.h"
#include "tools/app.h"
#include "tools/coledit.h"

// --- TColorView : Show one exact color whatever the graphic mode is

short RegTColorView;
char *IdentTColorView = "TColorView";

DEFINE(TColorView);

TColorView::TColorView()
{ Defaults();
}

TColorView::TColorView(int X, int Y, int L, int H)
{ Defaults();
  Init(X,Y,L,H);
}

TColorView::TColorView(TRect R)
{ Defaults();
  Init(R.X1(),R.Y1(),R.Width(),R.Height());
}

void TColorView::Defaults(void)
{ // Object identification
  Register=RegTColorView;
  Ident=IdentTColorView;
  // Other default values
  r=g=b=0;
  RealColor=0;
}

void TColorView::Init(int X, int Y, int L, int H)
{ // Base construction
  TZone::Init(X,Y,L,H);
  // Other constructions
  // ...
}

void TColorView::Done(void)
{ TZone::Done();
  SetDitherColors();
}

void TColorView::Draw(TRect )
{ Frame3D(1);
  TZone::SetColor(RealColor);
  Bar(2,2,Where.Width()-3,Where.Height()-3);
}

void TColorView::SetColor(byte _r, byte _g, byte _b)
{ TColor NRealColor;
  r=_r; g=_g; b=_b;
  switch(ColorMode)
  { case colMODE_16 :
    case colMODE_256 :
      // Change one color used in the dithering
      // No need to redraw if this color was already
      // used
      if (ColorMode==colMODE_16)  NRealColor=5;
      if (ColorMode==colMODE_256) NRealColor=190;
      if (NRealColor!=RealColor)
      { RealColor=NRealColor;
        Invalidate();
      }
      GrSetColor(RealColor,r,g,b);
      break;
    case colMODE_TrueColor :
      // This is true color : Just have to draw with the
      // specified color
      RealColor=Dither->GetRGBColor(0,0,r,g,b);
      Invalidate();
      break;
  }
}

// --- TColorHSView : Show HSV color space and a cursor on it

short RegTColorHSView;
char *IdentTColorHSView = "TColorHSView";

DEFINE(TColorHSView);

DEFINE_EVENTS_TABLE(TColorHSView,TZone)
  MOUSELDOWN()
  MOUSEMOVE()
END_EVENTS_TABLE

TColorHSView::TColorHSView()
{ Defaults();
}

TColorHSView::TColorHSView(int X, int Y)
{ Defaults();
  Init(X,Y);
}

void TColorHSView::Defaults(void)
{ // Object identification
  Register=RegTColorHSView;
  Ident=IdentTColorHSView;
  // Other default values
  h=s=v=0;
}

void TColorHSView::Init(int X, int Y)
{ // Base construction
  TZone::Init(X,Y,152,132);
  // Other constructions
  int  i,j;
  byte r,g,b;
  // :: HS zone
  HSzone=new TScreenBitmap(128,128);
  HSzone->BeginDraw();
  for(j=0;j<128;j++)
    for(i=0;i<128;i++)
    { HSV2RGB(i*2,j*2,128,r,g,b);
      GrPlot(i,127-j,Dither->GetRGBColor(i,j,r,g,b));
    }
  HSzone->EndDraw();
  // :: V zone
  Vzone =new TScreenBitmap(20,128);
  Vzone->BeginDraw();
  for(j=0;j<128;j++)
    for(i=0;i<20;i++)
    { HSV2RGB(0,0,j*2,r,g,b);
      GrPlot(i,127-j,Dither->GetRGBColor(i,j,r,g,b));
    }
  Vzone->EndDraw();
}

void TColorHSView::Done(void)
{ TZone::Done();
  delete Vzone;
  delete HSzone;
}

void TColorHSView::Draw(TRect )
{ // Show HSV bitmaps
  Frame3D(1);
  HSzone->Draw(Corner.X()+2  ,Corner.Y()+2);
  Vzone ->Draw(Corner.X()+130,Corner.Y()+2);
  // Show HSV position
  int  X,Y;
  // :: HS
  SetSysColor(White);
  X=h/2+2; Y=(127-s/2)+2;
  PutPixel(X-1,Y);  PutPixel(X,Y-1);
  PutPixel(X+1,Y);  PutPixel(X,Y+1);
  // :: V
  if (v<128) SetSysColor(White); else SetSysColor(Black);
  Y=(127-v/2)+2;
  Line(130,Y,150,Y);
}

void TColorHSView::SetHSV(byte _h, byte _s, byte _v)
{ h=_h; s=_s; v=_v;
  Invalidate();
}

void TColorHSView::GetHSV(byte& _h, byte& _s, byte& _v)
{ _h=h; _s=s; _v=v;
}

boolean TColorHSView::MouseLDown(TPoint W, int )
{ return MouseAction(W);
}

boolean TColorHSView::MouseMove(TPoint W, int Buttons)
{ if (Buttons & 1) return MouseAction(W);
              else return FALSE;
}

boolean TColorHSView::MouseAction(TPoint W)
{ TPoint P;
  if (GetStatus(sfMouseIn))
  { P=W-Corner;
    if ((P.X()<2) || (P.X()>150) || (P.Y()<2) || (P.Y()>130)) return FALSE;
    if (P.X()<130)
    { // there was a click in the (H,S) domain
      h=(P.X()-2)*2;
      s=(127-(P.Y()-2))*2;
    }
    else
    { // there was a click in the (V) domain
      v=(127-(P.Y()-2))*2;
    }
    Invalidate();
    SetCommand(cmColorViewEdited,this);
    // Notify the parent of the modification
    return TRUE;
  }
  return FALSE;
}

// --- TColorEdit : The color edition dialog

short RegTColorEdit;
char *IdentTColorEdit = "TColorEdit";

DEFINE(TColorEdit);

DEFINE_EVENTS_TABLE(TColorEdit,TDialog)
  COMMAND(cmEditChanged    ,doEditChanged)
  COMMAND(cmColorViewEdited,doColorViewEdited)
END_EVENTS_TABLE

TColorEdit::TColorEdit()
{ Defaults();
}

TColorEdit::TColorEdit(int X, int Y)
{ Defaults();
  Init(X,Y);
}

void TColorEdit::Defaults(void)
{ // Object identification
  Register=RegTColorEdit;
  Ident=IdentTColorEdit;
  // Other default values
  //...
}

void TColorEdit::Init(int X, int Y)
{ // Base construction
  TDialog::Init(X,Y,257,252,"Color Editor");
  // Other constructions
  // :: One Cancel Button
  Insert(new TCancelButton(167,219));
  // :: One OK button
  Insert(new TOKButton(167,191));
  // :: Red, Green, Blue edition
  eR=InsertEditBlock(10,30,"Red");
  eG=InsertEditBlock(10,56,"Green");
  eB=InsertEditBlock(10,82,"Blue");
  // :: Hue, Saturation, Value
  eH=InsertEditBlock(141,30,"Hue");
  eS=InsertEditBlock(141,56,"Sat.");
  eV=InsertEditBlock(141,82,"Value");
  // :: HSV viewer
  HSView=new TColorHSView(10,110);
  Insert(HSView);
  // :: Color Viewer
  ColView=new TColorView(167,110,80,55);
  Insert(ColView);
}

void TColorEdit::SetData(void *Ptr)
{ byte h,s,v;
  byte r,g,b;
  TDialog::SetData(Ptr);
  eH->GetData(&h);
  eS->GetData(&s);
  eV->GetData(&v);
  eR->GetData(&r);
  eG->GetData(&g);
  eB->GetData(&b);
  HSView->SetHSV(h,s,v);
  ColView->SetColor(r,g,b);
}

TEditInteger *TColorEdit::InsertEditBlock(int X, int Y, char *Text)
{ TEditInteger *Edit;
  Insert(new TStaticText(X,Y,40,23,Text));
  Insert(Edit=new TEditInteger(X+45,Y,50,dt_byte));
  Insert(new TIntegerLift(Edit,0,255));
  return Edit;
}

void TColorEdit::RGB2HSV(void)
{ byte  r,g,b,h,s,v;
  // 1. Get RGB values
  eR->GetData(&r);
  eG->GetData(&g);
  eB->GetData(&b);
  // 2. Conversion
  ::RGB2HSV(r,g,b,h,s,v);
  // 3. Set HSV values
  eH->SetData(&h);
  eS->SetData(&s);
  eV->SetData(&v);
  HSView->SetHSV(h,s,v);
  ColView->SetColor(r,g,b);
}

void TColorEdit::HSV2RGB(void)
{ byte  r,g,b,h,s,v;
  // 1. Get HSV values
  eH->GetData(&h);
  eS->GetData(&s);
  eV->GetData(&v);
  // 2. Conversion
  ::HSV2RGB(h,s,v,r,g,b);
  // 3. Set RGB values
  eR->SetData(&r);
  eG->SetData(&g);
  eB->SetData(&b);
  HSView->SetHSV(h,s,v);
  ColView->SetColor(r,g,b);
}

boolean TColorEdit::doEditChanged(void)
{ TEditInteger *E=(TEditInteger*)(CurrentEvent.InfoPtr);
  if ((E==eR)||(E==eG)||(E==eB))
  { RGB2HSV();
    return TRUE;
  }
  if ((E==eH)||(E==eS)||(E==eV))
  { HSV2RGB();
    return TRUE;
  }
  return FALSE;
}

boolean TColorEdit::doColorViewEdited(void)
{ TColorHSView *V=(TColorHSView*)(CurrentEvent.InfoPtr);
  if (HSView==V)
  { byte r,g,b,h,s,v;
    HSView->GetHSV(h,s,v);
    ::HSV2RGB(h,s,v,r,g,b);
    eR->SetData(&r);
    eG->SetData(&g);
    eB->SetData(&b);
    eH->SetData(&h);
    eS->SetData(&s);
    eV->SetData(&v);
    ColView->SetColor(r,g,b);
    return TRUE;
  }
  return FALSE;
}

// --- Public functions

void EditRGBColor(int X, int Y, byte& r, byte& g, byte& b)
{ TColorEdit      *ColorEdit=new TColorEdit(X,Y);
  TColorEditDatas  ColorEditDatas;
  ColorEditDatas.Red  =r;
  ColorEditDatas.Green=g;
  ColorEditDatas.Blue =b;
  RGB2HSV(r,g,b,ColorEditDatas.Hue,
                ColorEditDatas.Sat,
                ColorEditDatas.Val);
  // Execute the dialog
  if (ExecDialog(ColorEdit,&ColorEditDatas)==cmOK)
  { r=ColorEditDatas.Red;
    g=ColorEditDatas.Green;
    b=ColorEditDatas.Blue;
  }
}

void EditHSVColor(int X, int Y, byte& h, byte& s, byte& v)
{ TColorEdit      *ColorEdit=new TColorEdit(X,Y);
  TColorEditDatas  ColorEditDatas;
  ColorEditDatas.Hue=h;
  ColorEditDatas.Sat=s;
  ColorEditDatas.Val=v;
  HSV2RGB(h,s,v,ColorEditDatas.Red,
                ColorEditDatas.Green,
                ColorEditDatas.Blue);
  // Execute the dialog
  if (ExecDialog(ColorEdit,&ColorEditDatas)==cmOK)
  { h=ColorEditDatas.Hue;
    s=ColorEditDatas.Sat;
    v=ColorEditDatas.Val;
  }
}

