// -----------------------------------------------------------------------
// File lwp.c - pre-emptive LWP C source code.
// Copyright (C) 1997 Paolo De Marino
//
// Original Source Code by Sengan Short (sengan.short@durham.ac.uk)
// and Josh Turpen (snarfy@goodnet.com).
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version,
//  with the only exception that all the people in the THANKS file
//  must receive credit.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Library General Public License for more details.
//
//  You should have received a copy of the GNU Library General Public
//  License along with this library; see the file COPYING.LIB.
//  If not, write to the Free Software Foundation, Inc., 675 Mass Ave,
//  Cambridge, MA 02139, USA.
//
//  For contacting the author send electronic mail to
//     pdemarin@mbx.idn.it
//
//  Or paper mail to
//
//     Paolo De Marino
//     Via Donizetti 1/E
//     80127 Naples
//     Italy
//
// History:
// Who knows?           Original code by Sengan Short and Josh Turpen
//
// May 15th, 1997       Modified lwpasm.S to call back a new function,
// (more or less)       lwp_findnext(), to implement semaphores and
//                      waiting threads.
//                      Added ex novo lwp_wait_true,lwp_wait_false,
//                      lwp_wait_semaphore,lwp_release_semaphore,
//                      and related data types (lwp_semaphore,
//                      lwp_time). Minor modifications to _lwp_dead_thread.
//
// May 22nd, 1997       Made a few speedups, and added C++ support.
// -----------------------------------------------------------------------

#include "lwp.h"
#include <sys/timeb.h>
#include <sys/segments.h>
#include <bios.h>
#include <dos.h>

#define LWP_EXCPTN 0x99     /* arbitrary exception number */

#define IRQ8 0x70         /* used to start the 1khz timer */
#define IRQ0 0x8          /* used to start the variable timer */
#define PIT0 0x40
#define PIT1 0x41
#define PIT2 0x42
#define PITMODE 0x43
#define PITCONST 1193180L
#define PIT0DEF 18.2067597

#ifdef __cplusplus
extern "C" {
#endif
void _lwp_top_of_vars() {};
#ifdef __cplusplus
}
#endif

/*  Global Variables */
static volatile float tick_per_ms = 0.0182068;
static volatile float ms_per_tick = 54.9246551;
static volatile float freq8h = 18.2067597;
static volatile int counter_8h;
static volatile int counter_reset;
static int _lwp_irq_used = 0;
static int _lwp_speed_used = 0;
static int _lwp_on = 0;
char _fpu_init_state[108];
__dpmi_paddr _lwp_pm_old_handler, _lwp_pm_new_handler;
__dpmi_regs _lwp_regs;
void (*_lwp_old_handler)(int);
void (*_lwp_old_fpu_handler)(int);

lwp *_lwp_cur;
static volatile int _lwp_count;
volatile int _lwp_enable;
volatile int _lwp_interrupt_pending;
#ifdef __cplusplus
extern "C" {
#endif
void _lwp_bottom_of_vars() {};
#ifdef __cplusplus
}
#endif

/* Some function prototypes */
#ifdef __cplusplus
extern "C" {
#endif
extern void _lwp_scheduler(int signum);
extern void _init_fpu(void);
void _lwp_findnext(void);   // Assumes no task switching in course!
void _lwp_fpu_handler(int signum);
void lwp_deinit(void);
static int _lwp_lock_memory(void);
static int windoze();
static void setRTC(int value);
static void startIRQ8(void);
static void stopIRQ8(void);
char *get_cmostime(void);
#ifdef __cplusplus
}
#endif


static void _lwp_top_of_functions() {};
volatile void _lwp_dead_thread(void)
{
 volatile int crap_for_signal_mechanism;
 lwp_kill(lwp_getpid());
 lwp_yield();           // Jump to next task without losing time here!

// (should never come here, it'd be better to issue a warning...)
 for(;;)
    {
    crap_for_signal_mechanism++;
    }
 }

int lwp_init(int irq, int speed)
{
        volatile unsigned int pit0_set, pit0_value;
        _lwp_irq_used = irq;
        _lwp_speed_used = speed;
   if(atexit(lwp_deinit))
        return(0);
   if (!_lwp_lock_memory())
      {
      return(0);
      }
   if((_lwp_cur = (lwp *) malloc(sizeof(lwp))) == NULL)
      {
      #ifdef __DEBUG__
      printf("ERROR:  Couldn't malloc _lwp_cur\n");
      #endif
      return(0);
      }
   if(_lwp_lock_data(&_lwp_cur, sizeof(lwp)))
      {
      #ifdef __DEBUG__
      printf("couldn't lock _lwp_cur\n");
      #endif
      return(0);
      }
   _init_fpu();
   _lwp_cur->stack = NULL;
   _lwp_cur->lwpid = LWP_MAIN;
   _lwp_cur->next = _lwp_cur;
   _lwp_cur->status = RUNNING;
   _lwp_cur->userptr = (void*)0;
   _lwp_enable = 1;
   _lwp_interrupt_pending = 0;
   /* SIGILL is raised when an unhandled exception is raised */
   _lwp_old_handler = signal(SIGILL,_lwp_scheduler);
   _lwp_count = 0;
   if(irq == 8)
     {
          _lwp_pm_new_handler.offset32 = (int)_lwp_pm_irq8_timer_hook;
          _lwp_pm_new_handler.selector = _my_cs();
          __dpmi_get_protected_mode_interrupt_vector(IRQ8, &_lwp_pm_old_handler);
          __dpmi_set_protected_mode_interrupt_vector(IRQ8, &_lwp_pm_new_handler);
          _lwp_on = 1;
          _lwp_enable = 0;
          setRTC(speed);
          startIRQ8();
          return(1);
     }
   else if (irq == 0)
    {
    if(speed < 1)
       {
       return(0);
       }
    _lwp_pm_new_handler.offset32 = (int)_lwp_pm_irq0_timer_hook;
    _lwp_pm_new_handler.selector = _my_cs();

    __dpmi_get_protected_mode_interrupt_vector(IRQ0, &_lwp_pm_old_handler);
    __dpmi_set_protected_mode_interrupt_vector(IRQ0, &_lwp_pm_new_handler);

    outportb(PITMODE, 0x36);
    pit0_value = PITCONST / speed;
    pit0_set = (pit0_value & 0x00ff);
    outportb(PIT0, pit0_set);
    pit0_set = (pit0_value >> 8);
    outportb(PIT0, pit0_set);
    freq8h = speed;
    counter_8h = 0;
    counter_reset = freq8h / PIT0DEF;
    tick_per_ms = freq8h / 1000;
    ms_per_tick = 1000 / freq8h;
    _lwp_on = 1;
    _lwp_enable = 0;
    return(1);
    }
   else
    {
    return(0);
    }
}
#undef cprintf
void lwp_findnext(void)
{
 int sleepers = 0;      // Are there any sleepers?
 struct timeb buf;

 lwp *old = _lwp_cur;
 _lwp_cur = _lwp_cur->next;

 while(1)
 {
    switch(_lwp_cur->status)
    {
     case RUNNING:
//          cprintf("(%i) to run\n\r",_lwp_cur->lwpid);
          return;    // Found!
     case SLEEPING:
          sleepers = 1;
          ftime(&buf);
          if(_lwp_cur->waiting.wakeup_time.secs > buf.time) break;
          if(_lwp_cur->waiting.wakeup_time.secs  == buf.time &&
             _lwp_cur->waiting.wakeup_time.msecs >  buf.millitm) break;
          _lwp_cur->status = RUNNING; // Else...
          return;
     case WAIT_SEMAPHORE:
          if(_lwp_cur->waiting.what_sema->owned) break;
          _lwp_cur->waiting.what_sema->owned = 1;
          _lwp_cur->waiting.what_sema->owner_id = _lwp_cur->lwpid;
          _lwp_cur->status = RUNNING;
          return;
     case WAIT_TRUE:
//          cprintf("Queue %i waiting for int at %p to become != 0",
//                  _lwp_cur->lwpid,_lwp_cur->waiting.what_int);
          if( *(_lwp_cur->waiting.what_int) == 0) break;
          _lwp_cur->status = RUNNING;
          return;
     case WAIT_FALSE:
//          cprintf("Queue %i waiting for int at %p to become == 0",
//                  _lwp_cur->lwpid,_lwp_cur->waiting.what_int);
          if( *(_lwp_cur->waiting.what_int) != 0) break;
          _lwp_cur->status = RUNNING;
          return;
     default: break;
    };
// Look if the task we've just tried to clean is the current one,
// and if there are no sleeping tasks.
// If it were so, it was a grievous fault: deadlock!
    if( !sleepers && _lwp_cur == old)
    {
     printf("Deadlock!\n");
     abort();
    }
    _lwp_cur = _lwp_cur->next;
 };
}

int lwp_getpid(void)
{
   return(_lwp_cur->lwpid);
}
void lwp_setuserptr(void *usrdata)
{
 _lwp_cur->userptr = usrdata; // No need to lock multitasker
}
void *lwp_getuserptr(void)
{
 return(_lwp_cur->userptr);
}

/* spawns a light weight process.  Returns the lwpid of the proc or 0 on fail*/

int lwp_spawn(void (*proc)(void), int stack_length)
{
  lwp *tmp;
  volatile int tmp2;
  tmp2 = _lwp_enable;
  _lwp_enable = 1;
  if((proc == NULL) || (stack_length < 256))
        {
        _lwp_enable = 0;
        return(-1);
        }
  if((tmp = (lwp *) malloc(sizeof(lwp))) == NULL)
        {
        _lwp_enable = 0;
        return(-1);
        }
  if((tmp->stack = (unsigned int *) malloc(stack_length)) == NULL)
        {
        _lwp_enable = 0;
        return(-1);
        }
  if(_lwp_lock_data(&tmp, sizeof(lwp)))
        {
        _lwp_enable = 0;
        return(-1);
        }
  if(_lwp_lock_data(&tmp->stack, stack_length))
        {
        _lwp_enable = 0;
        return(-1);
        }
 /*8 regs +flags + 27 dwords for the fpu  */
  memcpy(tmp->stack, _fpu_init_state, 108);
  tmp->stack[36] = (int) proc;
  tmp->stack[37] = (int) _lwp_dead_thread;
/*
  tmp->stack[9] = (int) proc;
  tmp->stack[10] = (int) _lwp_dead_thread;
*/
  tmp->lwpid = ++_lwp_count;
  tmp->next = _lwp_cur->next;
  tmp->status = RUNNING;
  tmp->userptr = (void*)0;
  _lwp_cur->next = tmp;
  _lwp_enable = tmp2;
  return(tmp->lwpid);
}

/* Kills a lwp (takes it out of the linked list of lwp's) , or 0 if it fails */
int lwp_kill(int lwpid)
{
  volatile int i = 0;
  lwp *tmp1,*tmp2;
  volatile int tmp;
  tmp = _lwp_enable;
  _lwp_enable = 1;
  tmp1 = _lwp_cur->next;
  tmp2 = _lwp_cur;
  if((tmp1 == NULL) || (tmp2 == NULL))
        return(0);
  while((i<= _lwp_count+1) && (tmp1->lwpid != lwpid))
         {
         i++;
         tmp1 = tmp1->next;
         tmp2 = tmp2->next;
         }
  /* went through all of them and wasn't found */
  if(tmp1->lwpid != lwpid)
     return(0);
  /* now tmp IS the right one to kill */
  tmp2->next = tmp1->next;
  _lwp_unlock_data(tmp1->stack, tmp1->stklen);
  _lwp_unlock_data(&tmp1, sizeof(lwp));
  free(tmp1->stack);
  free(tmp1);
  _lwp_enable = tmp;
  /*lwp_yield();*/
  return(1);
}



void lwp_deinit(void)
{
    volatile unsigned long   tick;
    volatile char *cmostime;
    _lwp_enable = 1;
    if(!_lwp_on)
        {
        return;
        }
        if(_lwp_irq_used == 8)
        {
            stopIRQ8();
            __dpmi_set_protected_mode_interrupt_vector(IRQ8, &_lwp_pm_old_handler);
            signal(SIGILL, _lwp_old_handler);
            _lwp_unlock_data(&_lwp_cur, sizeof(lwp));
            free(_lwp_cur);
            #ifdef __DEBUG__
            printf("IRQ8 successfully uninstalled.\n");
            #endif
            return;
        }
        else if(_lwp_irq_used == 0)
          {
           outportb(PITMODE, 0x36);
           outportb(PIT0, 0x00);
           outportb(PIT0, 0x00);
           __dpmi_set_protected_mode_interrupt_vector(IRQ0, &_lwp_pm_old_handler);
           signal(SIGILL, _lwp_old_handler);
           _lwp_unlock_data(&_lwp_cur, sizeof(lwp));
           free(_lwp_cur);
           cmostime = get_cmostime();
        tick = PIT0DEF *
            (
             (((float) *cmostime) * 3600) +
             (((float) *(cmostime + 1)) * 60) +
             (((float) *(cmostime + 2)))
            );
        biostime(1, tick);

        freq8h = PIT0DEF;
        counter_reset = freq8h / PIT0DEF;
        tick_per_ms = freq8h / 1000;
        ms_per_tick = 1000 / freq8h;
        #ifdef __DEBUG__
        printf("IRQ0 successfully uninstalled.\n");
        #endif
        return;
        }
 return;
}

char           *get_cmostime(void)
{
    /* used to fix the fast timer */
    char *buff;
    static char buffer[6];
    char ch;

    buff = buffer;
    memset(&_lwp_regs, 0, sizeof(_lwp_regs));
    _lwp_regs.h.ah = 0x02;
    __dpmi_int(0x1a, &_lwp_regs);

    ch = _lwp_regs.h.ch;
    buffer[0] = (char) ((int) (ch & 0x0f) + (int) ((ch >> 4) & 0x0f) * 10);
    ch = _lwp_regs.h.cl;
    buffer[1] = (char) ((int) (ch & 0x0f) + (int) ((ch >> 4) & 0x0f) * 10);
    ch = _lwp_regs.h.dh;
    buffer[2] = (char) ((int) (ch & 0x0f) + (int) ((ch >> 4) & 0x0f) * 10);
    buffer[3] = _lwp_regs.h.dl;
    buffer[4] = (char) (_lwp_regs.x.flags & 0x0001);
    buffer[5] = 0x00;

    return (buff);
}
static void _lwp_bottom_of_functions() {};

static int _lwp_lock_memory()
{
  if(_lwp_lock_code(_lwp_top_of_functions,  (long) &_lwp_bottom_of_functions - (long) &_lwp_top_of_functions))
     {
     /* fail */
     #ifdef __DEBUG__
     printf("ERROR:  DPMI error locking code segment functions\n");
     #endif
     return(0);
     }
  if(_lwp_lock_code(&_lwpasm_start, (long) &_lwpasm_end - (long) _lwpasm_start))
     {
     /* fail */
     #ifdef __DEBUG__
     printf("ERROR:  DPMI error locking code segment IRQ handlers\n");
     #endif
     return(0);
     }
  if(_lwp_lock_data(_lwp_top_of_vars, (long) &_lwp_bottom_of_vars - (long) &_lwp_top_of_vars))
    {
    #ifdef __DEBUG__
    printf("ERROR:  DPMI error locking data segment variables\n");
    #endif
    return(0);
    }
  return(1);
}
void _lwp_fpu_handler(int signum)
{
        volatile int tmp = _lwp_enable;
        _lwp_enable = 1;
        _lwp_old_fpu_handler(signum);
        _lwp_enable = tmp;
}
/* returns true if windoze is running */
static int windoze()
{
   __dpmi_regs regs;
   regs.x.ax = 0x1600;
   __dpmi_int(0x2F, &regs);
   return((regs.x.ax - 0x1600));
}
/* copied from go32 lib funcs */
int _lwp_lock_data( void *lockaddr, unsigned long locksize )
    {
    unsigned long baseaddr;
    __dpmi_meminfo memregion;
    if(windoze()) return(0);
    if( __dpmi_get_segment_base_address( _my_ds(), &baseaddr) == -1 ) return( -1 );

    memset( &memregion, 0, sizeof(memregion) );

    memregion.address = baseaddr + (unsigned long) lockaddr;
    memregion.size    = locksize;

    if( __dpmi_lock_linear_region( &memregion ) == -1 ) return( -1 );

    return( 0 );
    }

int _lwp_unlock_data( void *lockaddr, unsigned long locksize )
    {
    unsigned long baseaddr;
    __dpmi_meminfo memregion;
    if(windoze()) return(0);
    if( __dpmi_get_segment_base_address( _my_ds(), &baseaddr) == -1 ) return( -1 );

    memset( &memregion, 0, sizeof(memregion) );

    memregion.address = baseaddr + (unsigned long) lockaddr;
    memregion.size    = locksize;

    if( __dpmi_unlock_linear_region( &memregion ) == -1 ) return( -1 );

    return( 0 );
    }

int _lwp_lock_code( void *lockaddr, unsigned long locksize )
    {
    unsigned long baseaddr;
    __dpmi_meminfo memregion;
    if(windoze()) return(0);
    if( __dpmi_get_segment_base_address( _my_cs(), &baseaddr) == -1 ) return( -1 );

    memset( &memregion, 0, sizeof(memregion) );

    memregion.address = baseaddr + (unsigned long) lockaddr;
    memregion.size    = locksize;

    if( __dpmi_lock_linear_region( &memregion ) == -1 ) return( -1 );

    return( 0 );
    }

int _lwp_unlock_code( void *lockaddr, unsigned long locksize )
    {
    unsigned long baseaddr;
    __dpmi_meminfo memregion;
    if(windoze()) return(0);
    if( __dpmi_get_segment_base_address( _my_cs(), &baseaddr) == -1 ) return( -1 );

    memset( &memregion, 0, sizeof(memregion) );

    memregion.address = baseaddr + (unsigned long) lockaddr;
    memregion.size    = locksize;

    if( __dpmi_unlock_linear_region( &memregion ) == -1 ) return( -1 );

    return( 0 );
    }

static void setRTC(int value)
{
        __asm__ __volatile__ ("cli\n\t"
                "movb $0x0A, %%al\n\t"
                "outb %%al, $0x70\n\t"  /* output status a   */
                "inb $0x71, %%al\n\t"  /* get current stats */
                "addb $0xF0, %%bl\n\t"   /* mask top 4 bits   */
                "andb %%bl, %%al\n\t"    /* output new div    */
                "subb $0xF0, %%bl\n\t"   /* only set bottom 4 bits */
                "orb  %%bl, %%al\n\t"
                "movb %%al, %%bl\n\t"
                "movb $0xA, %%al\n\t"
                "outb %%al, $0x70\n\t"  /* outputing to status a */
                "movb %%bl, %%al\n\t"
                "outb %%al, $0x71\n\t"  /* done */
                "sti\n\t"
                : /* none */
                : "b" (value)
                : "%eax" , "%edx");
}
static void startIRQ8(void)
{
        unsigned char shtuff;
        disable();
        outportb(0x70, 0x0B);  /* status B */
        shtuff = inportb(0x71); /* get status */
        shtuff = shtuff | 0x40; /* mask off interrupt enable bit */
        outportb(0x70, 0x0B);   /* status B again */
        outportb(0x71, shtuff);  /* output new value */
        outportb(0x70, 0x0C);    /* status C */
        inportb(0x71);           /* ack interrupt */
        enable();
}

static void stopIRQ8(void)
{
        unsigned char shtuff;
        disable();
        outportb(0x70, 0x0B);  /* status B */
        shtuff = inportb(0x71); /* get status */
        shtuff = shtuff & 0xBF; /* mask off interrupt enable bit */
        outportb(0x70, 0x0B);   /* status B again */
        outportb(0x71, shtuff);  /* output new value */
        outportb(0x70, 0x0C);    /* status C */
        inportb(0x71);           /* ack interrupt */
        enable();
}
void lwp_sleep(unsigned int secs,unsigned short msecs)
{
   struct timeb now;
        volatile int tmp = _lwp_enable;
        _lwp_enable = 1; // Lock multitasking engine

   ftime(&now);

   _lwp_cur->status = SLEEPING;
   _lwp_cur->waiting.wakeup_time.msecs = (now.millitm + msecs) % 1000;
   _lwp_cur->waiting.wakeup_time.secs  = now.time + secs +
                                         (now.millitm + msecs)/1000;

        _lwp_enable = tmp;
   lwp_yield();
}

void lwp_wait_true(volatile int *what)  // Wait until what != 0
{
        volatile int tmp = _lwp_enable;
        _lwp_enable = 1; // Lock multitasking engine

   if(*what == 0)
   {
    _lwp_cur->status = WAIT_TRUE;
    _lwp_cur->waiting.what_int = what;

    _lwp_enable = tmp;
    lwp_yield();
   } else _lwp_enable = tmp;  // Already != 0
}
void lwp_wait_false(volatile int *what)  // Wait until what == 0
{
        volatile int tmp = _lwp_enable;
        _lwp_enable = 1; // Lock multitasking engine

   if(*what != 0)
   {
    _lwp_cur->status = WAIT_FALSE;
    _lwp_cur->waiting.what_int = what;

    _lwp_enable = tmp;
    lwp_yield();
   } else _lwp_enable = tmp;  // Alreay == 0
}
// When a task waits for an integer to become true/false, it can be useful to
// force all the tasks waiting for that integer to pass as if it were become
// true/false.
void lwp_pulse_true(volatile int *what)
{
 lwp *cursor = _lwp_cur->next;
 volatile int tmp = _lwp_enable;
 _lwp_enable = 1; // Lock multitasking engine

 while(cursor != _lwp_cur)
 {
  if( (cursor->status == WAIT_TRUE) && (cursor->waiting.what_int == what) )
     cursor->status = RUNNING; // If the tasks waits for this integer,
                               // let it run.
  cursor = cursor->next;
 }

 _lwp_enable = tmp;
}
void lwp_pulse_false(volatile int *what)
{
 lwp *cursor = _lwp_cur->next;
 volatile int tmp = _lwp_enable;
 _lwp_enable = 1; // Lock multitasking engine

 while(cursor != _lwp_cur)
 {
  if( (cursor->status == WAIT_FALSE) && (cursor->waiting.what_int == what) )
     cursor->status = RUNNING;
  cursor = cursor->next;
 }

 _lwp_enable = tmp;
}

void lwp_wait_semaphore(lwp_semaphore *sema)
{
 volatile int tmp = _lwp_enable;
 _lwp_enable = 1;                       // Lock multitasking engine

 _lwp_cur->status = WAIT_SEMAPHORE;
 _lwp_cur->waiting.what_sema = sema;

 _lwp_enable = tmp;
 lwp_yield();
}

void lwp_init_semaphore(lwp_semaphore *sema)
{
 sema->owned = 0;
}

int lwp_release_semaphore(lwp_semaphore *sema)
{
 if(sema->owned)
 {
  if(sema->owner_id != _lwp_cur->lwpid) return -1;
  sema->owned = 0;
 };
 return 0;
}
