/*
    vdhcp.c

    Written using information from Alfons Hoogervorst - see
    misc/docs/ipdata.txt for more information.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/segments.h>

#include "vxd.h"

int vdhcp_inited = 0;
int vdhcp_entry[] = {0, 0};

/* --------------
   - vdhcp_init -
   -------------- */

/* This loads and obtains the entry point of the DHCP VxD. It returns 1 on
   successful initialisation, and 0 on failure. */

int vdhcp_init (void)
{
    if (!vdhcp_inited) {
        /* Load the DHCP VxD & get its entry */
        VxdLdrLoadDevice("VDHCP.VXD");
        VxdGetEntry(vdhcp_entry, 0x49A);

        /* Valid entry point? Fail if not */
        if ( (vdhcp_entry[0] == 0) && (vdhcp_entry[1] == 0) ) return(0);

        /* Done */
        vdhcp_inited = 1;
    }

    /* Okey dokey */
    return(vdhcp_inited);
}

/* -----------------
   - vdhcp_callvxd -
   ----------------- */

/* This calls the DHCP VxD. If the entry point has not been obtained, then a
   protection fault will result. The return value is the contents of the EAX
   register. */

int vdhcp_callvxd (char *buffer, int buffersize)
{
    int result = -1;    /* Fail by default */

    __asm__ __volatile__ ( "pushl   %%es            \n\
                            pushl   %%eax           \n\
                            popl    %%es            \n\
                            lcall   _vdhcp_entry    \n\
                            popl    %%es"
                          : "=a" (result)
                          : "a" (_my_ds()), "b" (buffer), "c" (buffersize)
                          : "%eax", "%ebx", "%ecx", "%edx", "memory", "cc");

    return(result);
}

/* -----------------
   - vdhcp_getdata -
   ----------------- */

/* This gets a block of data from the DHCP VxD that contains the DNS IP data,
   but needs parsing (see vdhcp_getdnsaddrs()). */

char *vdhcp_getdata (void)
{
    char *buffer = NULL;
    int buffersize = 0;     /* Desired buffer size  */
    int ret;                /* Return from VxD call */

    /* Init first */
    if (!vdhcp_init()) return(NULL);

    /* Find out large the buffer has to be first */
    buffer = malloc(sizeof(char) * 32);
    if (buffer == NULL) return(NULL);
    memset(buffer, 0, sizeof(char) * 32);

    ret = vdhcp_callvxd(buffer, buffersize);

    if (ret == 111) {   /* Win32 error: Buffer overflow */
        /* Find out how much memory is needed */
        buffersize = * ((int *) buffer);

        /* Done with this memory now */
        free(buffer);

        /* Now get *all* the data */
        buffer = malloc(sizeof(char) * buffersize);
        if (buffer == NULL) return(NULL);
        memset(buffer, 0, sizeof(char) * buffersize);

        ret = vdhcp_callvxd(buffer, buffersize);
    } else {
        /* 0 = success, 111 = need more data => anything else = error? */
        if (ret != 0) {
            free(buffer);
            return(NULL);
        }
    }

    /* Done */
    return(buffer);
}

/* ---------------------
   - vdhcp_getdnsaddrs -
   --------------------- */

/* This parses the data returned by vdhcp_getdata() and converts it into a list
   of strings. If the function fails, NULL is returned. */

char **vdhcp_getdnsaddrs (void)
{
    char *buffer = NULL;
    long ip_addr = 0;           /* Temp. DNS IP address          */
    struct in_addr ip_in_addr;  /* Temp. DNS IP address struct   */
    int ip_count = 0;           /* No. DNS IP addresses          */
    int ip_offset = 0;          /* Offset in buffer to IP addr's */
    int i;                      /* Loop variable                 */

    char **p;

    /* Init first */
    if (!vdhcp_init()) return(NULL);

    /* Get the DHCP data */
    buffer = vdhcp_getdata();
    if (buffer == NULL) return(NULL);

    /* Get the number of DNS IP addresses & their offset in buffer */
    ip_count = (* ((short *) buffer + 0x20)) / 4;
    ip_offset = * ((short *) buffer + 0x24);

    /* Allocate the array of pointers */
    p = malloc(sizeof(char *) * (ip_count + 1));
    if (p == NULL) return(NULL);
    memset(p, 0, sizeof(char *) * (ip_count + 1));

    /* QUERY: Are the IP addresses returned by the DHCP VxD in host order?
       This would seem the most likely, so I have assumed this is the case. */

    /* Convert each of the addresses - First get the raw data, then convert
       it from host to network byte order and then get the dotted-quad
       version, i.e. xxx.xxx.xxx.xxx */
    for (i = 0; i < ip_count; i++) {
        ip_addr = * ((long *) buffer + ip_offset + (i * 4) );
        ip_in_addr.s_addr = htons(ip_addr);
        p[i] = strdup(inet_ntoa(ip_in_addr));
    }

    /* Free the buffer */
    free(buffer);

    /* Done */
    return(p);
}
