/*
 * This file is part of Palantir.
 * Version : 0.4.1
 * General purpose FIFO queue functions set.
 * Copyright (c) 1997 Dim Zegebart, Moscow Russia.
 * zager@post.comstar.ru
 * http://www.geocities.com/siliconvalley/pines/7817
 * file : dzqueue.c
 */

#include "palantir.h"
#include "internal.h"

#define QH(q) q->queue+q->head*q->dsize
#define QT(q) q->queue+q->tail*q->dsize

//------------------------- FIFO QUEUE -----------------------------
// 'first in first out' queue functions

inline void queue_reset(fifo_queue *q)
{
  q->tail=0;
  q->head=0;
  q->empt=EMPTY;
}
END_OF_FUNCTION(queue_reset);

int queue_resize(fifo_queue *q,uint new_size)
{ int *tmp_mem;
  int tmp;

  lwp_disable;
  if ((tmp_mem=(int*)realloc(q->queue,sizeof(int)*new_size))==NULL) return(0);

  if (new_size>q->size) q->resize_counter++;
  else if (new_size<=q->initial_size) q->resize_counter=0;
  else q->resize_counter--;

  q->queue=tmp_mem;
  q->size=new_size;
  q->fill_level=3*(q->size>>2); // 3/4

  lwp_unlock(&q->status);
  lwp_enable;
  return(1);
}

void queue_delete(fifo_queue *q)
{ int tmp;
  lwp_disable;
  if (q==NULL)
   { lwp_enable;
     return;
   }

  free(q->queue);
  free(q);
  lwp_enable;
  return;
}

//---------------------- QUEUE EMPTY ---------------------------

inline int queue_empty(fifo_queue *q)
{ int tmp;

  lwp_disable;
  if (q==NULL)
   { lwp_enable;
     return(1);
   }
  if (q->tail&&q->head==q->tail)
   { if (q->empty_handler!=NULL)
      { lwp_unlock(&q->status);
        lwp_enable;
        return(-1);
      }
     lwp_unlock(&q->status);
     lwp_enable;
     return(1);
   }
  lwp_unlock(&q->status);
  q->empt=EMPTY;
  lwp_enable;
  return(0);
}
END_OF_FUNCTION(queue_empty);

//---------------------- QUEUE PUT ----------------------------

inline int queue_put(fifo_queue *q,int c)
{// int t;
  int n=0;
  int tmp;

//  putch(c);
  lwp_disable;
  lwp_lock(&q->status);
  DISABLE();

  if (q->tail==q->head) q->empt=EMPTY;//queue_reset(q);
  memcpy(QT(q),&c,q->dsize);
  q->tail+=q->dsize;
  q->empt=NOT_EMPTY;
  if(q->tail>=q->size)
   { //q->resize_counter=0;
     q->tail=0;
//     t=0;
//     queue_reset(q);
   }
  lwp_unlock(&q->status);
  ENABLE();

  if (abs(q->head-q->tail)>=q->fill_level) n=1;
  return(n);
}
END_OF_FUNCTION(queue_put);

//---------------------- QUEUE GET ---------------------------

inline int queue_get(fifo_queue *q)
{ //int h;
  int c=0;
  int tmp;

  lwp_disable;
  lwp_lock(&q->status);
  DISABLE();
//  h=q->head;
//  q->head++;
  memcpy(&c,QH(q),q->dsize);
  q->head+=q->dsize;
  lwp_unlock(&q->status);
  if (q->tail==q->head) q->empt=EMPTY;//
//  if (q->head==q->tail) queue_reset(q);
//  else if (q->head>=q->size) q->head=0;
  if (q->head>=q->size) q->head=0;
  ENABLE();
  lwp_enable;
  return(c);
}
END_OF_FUNCTION(queue_get);

/*
inline void queue_get_n(fifo_queue *q,int *mem_to,byte n,get_mode m)
{ DISABLE();
  if (n!=0) memcpy((void*)mem_to,(void*)&(q->queue[q->head]),sizeof(int)*n);
  else memcpy((void*)mem_to,(void*)(&(q->queue[q->head])),sizeof(int)*(q->tail-q->head));
//  if (m==NORMAL)
   { q->head+=n;
     if (queue_empty(q)) queue_reset(q);
   }
  ENABLE();
}

inline int queue_put_n(fifo_queue *q,int *mem_from,byte n)
{ int k=0;
  DISABLE();
  if(q->tail>=(q->size-n-1))
   { q->resize_counter=0;
     #ifdef __DEVICE__
     system_msg_put(SYS_QUEUE_PUT_N_RESET,q->size,q->head);
     #endif
     queue_reset(q);
   }

  memcpy((void*)(&(q->queue[q->tail])),(void*)mem_from,sizeof(int)*n);
  q->tail+=n;
  if (q->tail>=q->fill_level) k=1; //more the 7/16 of queue
  ENABLE();
  return(k);
}
*/

void queue_init()
{
  LOCK_FUNCTION(queue_empty);
  LOCK_FUNCTION(queue_put);
  LOCK_FUNCTION(queue_get);
  LOCK_FUNCTION(queue_reset);
}

//------------------------------- QUEUE NEW --------------------------

fifo_queue* queue_new(uint size)
{
  return(queue_new_(size,4));
}

//------------------------- QUEUE_NEW_ --------------------------

fifo_queue* queue_new_(uint size,uint dsize)
{ fifo_queue *q=NULL; //=(fifo_queue*)malloc(sizeof(fifo_queue));
//  static n=0;

  if (dsize<=0||dsize>4) return(NULL);
  if ((q=(fifo_queue*)malloc(sizeof(fifo_queue)))==NULL) return(NULL);

  if (!size) size=1024; //if illegal size, turn it to default size
  if ((q->queue=malloc(sizeof(int)*size*dsize))==NULL) return(NULL);

  q->dsize=dsize;
  q->size=size*dsize;
  q->initial_size=size*dsize;//size;
  q->resize_counter=0;
  q->fill_level=3*q->size/4; // 3/4
  q->empty_handler=NULL;
  queue_reset(q);
  _go32_dpmi_lock_data(q->queue,sizeof(int)*size*dsize);
  _go32_dpmi_lock_data(q,sizeof(fifo_queue));

  return(q);
}
