/* lzo_swd.c -- sliding window dictionary

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1997 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with the LZO library; see the file COPYING.
   If not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */



/***********************************************************************
// This file is inspired by the sliding dictionary routines used
// in HA, a general purpose file archiver.
************************************************************************/

#if (LZO_UINT_MAX < 0xffffffffL)
#  error LZO_UINT_MAX
#endif

/* unsigned type for dictionary access - don't waste memory here */
#if (N + F + F < USHRT_MAX)
   typedef unsigned short	swd_uint;
#  define SWD_UINT_MAX		USHRT_MAX
#else
   typedef lzo_uint			swd_uint;
#  define SWD_UINT_MAX		LZO_UINT_MAX
#endif
#define SWD_UINT(x)			((swd_uint)(x))


#ifndef HSIZE
#define HSIZE		16384
#endif

#if 1
#define HEAD3(b,p) \
	(((0x9f5f*(((((lzo_uint32)b[p]<<5)^b[p+1])<<5)^b[p+2]))>>5) & (HSIZE-1))
#else
#define HEAD3(b,p) \
	(((0x9f5f*(((((lzo_uint32)b[p+2]<<5)^b[p+1])<<5)^b[p]))>>5) & (HSIZE-1))
#endif

#if (THRESHOLD == 1) || defined(NEED_HEAD2)
#  define HEAD2(b,p)	(b[p] | ((unsigned)b[p+1]<<8))
#  define NIL2			SWD_UINT_MAX
#endif

#ifndef SWD_MAXL3
#define SWD_MAXL3	2048
#endif


typedef struct
{
/* public: */
	lzo_uint n;
	lzo_uint f;
	lzo_uint threshold;
	lzo_uint maxl3;

/* public: */
	lzo_uint m_len;
	lzo_uint m_pos;
	lzo_uint look;
	int b_char;
	LZO_COMPRESS_T *c;

/* private: */
	lzo_uint b_size;
	lzo_uint iptr;
	lzo_uint bptr;
	lzo_uint node_count;

	unsigned char b [ N + F + F - 1];

	swd_uint head3 [ HSIZE ];
	swd_uint succ3 [ N + F ];
	swd_uint best3 [ N + F ];
	swd_uint llen3 [ HSIZE ];
#ifdef HEAD2
	swd_uint head2 [ 65536L ];
#endif
}
lzo_swd_t;



#if 0 && defined(LZO_UNALIGNED_OK_2)
#  if !defined(SWD_UNALIGNED_OK_2)
#    define SWD_UNALIGNED_OK_2
#  endif
#endif

#if 0 && defined(LZO_UNALIGNED_OK_4)
#  if !defined(SWD_UNALIGNED_OK_4)
#    define SWD_UNALIGNED_OK_4
#  endif
#endif


/***********************************************************************
//
************************************************************************/

static
int swd_init(lzo_swd_t *s)
{
	lzo_uint i;
	int c;

	s->n = N;
	s->f = F;
	s->threshold = THRESHOLD;
	if (s->maxl3 <= 0)
		s->maxl3 = SWD_MAXL3;

	s->b_size = s->n + s->f;
	if (s->b_size + s->f >= SWD_UINT_MAX)
		return LZO_E_ERROR;

	s->node_count = s->n;
	for (i = 0; i < HSIZE; i++)
		s->llen3[i] = 0;
#ifdef HEAD2
	for (i = 0; i < 65536L; i++)
		s->head2[i] = NIL2;
#endif
	s->bptr = s->iptr = s->look = 0;
	while (s->look < s->f)
	{
		if ((c = getbyte(*(s->c))) < 0)
			break;
		s->b[s->iptr++] = LZO_BYTE(c);
		s->look++;
	}
	s->m_len = s->threshold;
	return LZO_E_OK;
}


static
void swd_getbyte(lzo_swd_t *s)
{
	int c;

	if ((c = getbyte(*(s->c))) < 0)
	{
		if (s->look > 0)
			--s->look;
	}
	else if (s->iptr < s->f - 1)
	{
		s->b[s->iptr + s->b_size] = s->b[s->iptr] = LZO_BYTE(c);
	}
	else
	{
		s->b[s->iptr] = LZO_BYTE(c);
	}
	if (++s->iptr == s->b_size)
		s->iptr = 0;
	if (++s->bptr == s->b_size)
		s->bptr = 0;
}


/***********************************************************************
// remove node from lists
************************************************************************/

static
void _swd_remove_node(lzo_swd_t *s, lzo_uint node)
{
	if (s->node_count == 0)
	{
		lzo_uint key;

		key = HEAD3(s->b,node);
		assert(s->llen3[key] > 0);
		--s->llen3[key];

#ifdef HEAD2
		key = HEAD2(s->b,node);
		assert(s->head2[key] != NIL2);
		if (s->head2[key] == s->iptr)
			s->head2[key] = NIL2;
#endif
	}
	else
		--s->node_count;
}


/***********************************************************************
//
************************************************************************/

static
void swd_accept(lzo_swd_t *s, lzo_uint j)
{
	while (j--)
	{
		lzo_uint key;

		_swd_remove_node(s,s->iptr);

		/* add bptr into HEAD3 */
		key = HEAD3(s->b,s->bptr);
		s->succ3[s->bptr] = s->head3[key];
		s->head3[key] = SWD_UINT(s->bptr);
		s->best3[s->bptr] = SWD_UINT(s->f + 1);
		s->llen3[key]++;
		assert(s->llen3[key] < SWD_UINT_MAX);

#ifdef HEAD2
		/* add bptr into HEAD2 */
		key = HEAD2(s->b,s->bptr);
		s->head2[key] = SWD_UINT(s->bptr);
#endif

		swd_getbyte(s);
	}

	s->m_len = s->threshold;
}


/***********************************************************************
//
************************************************************************/

static lzo_uint
_swd_findbest(const lzo_swd_t *s, lzo_uint cnt, lzo_uint node, lzo_uint *m_pos)
{
	const unsigned char * const b = s->b;
	const unsigned char *pbptr = b + s->bptr;
	lzo_uint m_len = s->m_len;
	const unsigned char * const pxx = pbptr + s->look;
	lzo_uint i;
#if defined(SWD_UNALIGNED_OK_2)
	unsigned short ref2;
#else
	unsigned char ref1;
#endif


#ifdef HEAD2
	i = s->head2[ HEAD2(b,s->bptr) ];
	if (i == NIL2)
		return m_len;
	assert(memcmp(&b[s->bptr],&b[i],2) == 0);
	if (m_len <= 2)
	{
		m_len = 2;
		*m_pos = i;
		if (m_len == s->look)
			return m_len;
	}
#endif


	if (cnt == 0)
		return m_len;


#if defined(SWD_UNALIGNED_OK_2)
	ref2 = * (lzo_ushortp) &pbptr[m_len - 1];
#else
	ref1 = pbptr[m_len - 1];
#endif

	do {
		assert(m_len < s->look);

#if defined(SWD_UNALIGNED_OK_2)
		if (* (lzo_ushortp) &b[node + m_len - 1] == ref2 &&
		    * (lzo_ushortp) &b[node] == * (lzo_ushortp) &pbptr[0])
#else
		if (b[node + m_len - 1] == ref1 && b[node + m_len] == pbptr[m_len] &&
		    b[node] == pbptr[0] && b[node + 1] == pbptr[1])
#endif
		{
			const unsigned char *p1 = pbptr + 3;
			const unsigned char *p2 = b + node + 3;
			const unsigned char *px = pxx;

			while (p1 < px && *p1 == *p2)
				p1++, p2++;

			i = p1 - pbptr;
			if (i > m_len)
			{
				*m_pos = node;
				if ((m_len = i) == s->look || s->best3[node] < i)
					return m_len;
#if defined(SWD_UNALIGNED_OK_2)
				ref2 = * (lzo_ushortp) &pbptr[m_len - 1];
#else
				ref1 = pbptr[m_len - 1];
#endif
#if defined(SWD_UNALIGNED_OK_4)
				if (m_len >= 4)
				{
					--cnt;
					break;
				}
#endif
			}
		}
		node = s->succ3[node];
	} while (--cnt > 0);


#if defined(SWD_UNALIGNED_OK_4)
	if (cnt == 0)
		return m_len;

	assert(m_len >= 4); assert(s->look > 4);
	node = s->succ3[node];
	do {
		assert(m_len < s->look);

#if defined(SWD_UNALIGNED_OK_2)
		if (* (lzo_ushortp) &b[node + m_len - 1] == ref2 &&
		    * (lzo_uint32p) &b[node] == * (lzo_uint32p) &pbptr[0])
#else
		if (b[node + m_len - 1] == ref1 && b[node + m_len] == pbptr[m_len] &&
		    * (lzo_uint32p) &b[node] == * (lzo_uint32p) &pbptr[0])
#endif
		{
			const unsigned char *p1 = pbptr + 4;
			const unsigned char *p2 = b + node + 4;
			const unsigned char *px = pxx - 4;

			while (p1 <= px && * (lzo_uint32p) p1 == * (lzo_uint32p) p2)
				p1 += 4, p2 += 4;

			px += 4;
			while (p1 < px && *p1 == *p2)
				p1++, p2++;

			i = p1 - pbptr;
			if (i > m_len)
			{
				*m_pos = node;
				if ((m_len = i) == s->look || s->best3[node] < i)
					return m_len;
#if defined(SWD_UNALIGNED_OK_2)
				ref2 = * (lzo_ushortp) &pbptr[m_len - 1];
#else
				ref1 = pbptr[m_len - 1];
#endif
			}
		}
		node = s->succ3[node];
	} while (--cnt > 0);
#endif

	return m_len;
}


/***********************************************************************
//
************************************************************************/

static
void swd_findbest(lzo_swd_t *s)
{
	lzo_uint key;
	lzo_uint cnt, node, start_len;

	key = HEAD3(s->b,s->bptr);
	if ((cnt = s->llen3[key]++) > s->maxl3)
		cnt = s->maxl3;
	assert(s->llen3[key] < SWD_UINT_MAX);
	node = s->succ3[s->bptr] = s->head3[key];
	/* add bptr into HEAD3 */
	s->head3[key] = SWD_UINT(s->bptr);

	s->b_char = s->b[s->bptr];
	s->m_pos = s->n + 1;
	if ((start_len = s->m_len) >= s->look)
	{
		if (s->look == 0)
			s->b_char = -1;
		s->best3[s->bptr] = SWD_UINT(s->f + 1);
	}
	else
	{
		s->m_len = _swd_findbest(s,cnt,node,&s->m_pos);
		s->best3[s->bptr] = SWD_UINT(s->m_len);
		if (s->m_len > start_len)
		{
			if (s->m_pos < s->bptr)
				s->m_pos = s->bptr - s->m_pos - 1;
			else
				s->m_pos = s->b_size - 1 - s->m_pos + s->bptr;
		}
	}

	_swd_remove_node(s,s->iptr);

#ifdef HEAD2
	/* add bptr into HEAD2 */
	key = HEAD2(s->b,s->bptr);
	s->head2[key] = SWD_UINT(s->bptr);
#endif

	swd_getbyte(s);
}


/*
vi:ts=4
*/

