#include <ctype.h>
#include <setjmp.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>

#if defined (__DJGPP__)
#include <fcntl.h>
#include <io.h>
#endif

#if defined (__HUMAN68K__)
#include <fcntl.h>
#if !defined (__DOS_INLINE__)
#define __DOS_INLINE__
#endif
#include <sys/dos.h>
#include <sys/xunistd.h>
#endif

#include "general.h"
#include "variables.h"
#include "builtins/common.h"
#include "dosutil.h"

#define CURRENT_PID ((pid_t) 1)
#if defined (__MSDOS__)
#define SYSROOT_COMPARE strnicmp
#else
#define SYSROOT_COMPARE strncmp
#endif
#define SAVESTRING(p)  ((p != NULL) ? savestring (p) : NULL)
#define FREE_AND_CLEAR(p)  (free (p), p = NULL)

int dosutil_path_expand = 0;
int dosutil_path_separator = ';';
int dosutil_path_slash = '/';
char dosutil_sysroot[PATH_MAX + 1] = { '\0' };
int dosutil_sysroot_len = -1;
int dosutil_wait_status = -1;
pid_t dosutil_current_pid = CURRENT_PID;

static char *need_fix_variables[] =
{
  "PWD",
  "OLDPWD",
  "HOME",
  "PATH",
  "CDPATH",
  "MAILPATH",
  NULL,
};

#define DOSUTIL_VERSION "r3"
char *
dosutil_version_string (void)
{
  static char version_string[128];

  sprintf (version_string, " %s", DOSUTIL_VERSION);
#if defined (USE_MULTIBYTE)
  strcat (version_string, " w/multibyte extension");
#endif

  return version_string;
}

static void
envvar_delsysroot (char *name)
{
  char *p, *newenv;
  int len;

  len = strlen (name) + 1;
  p = getenv (name);
  if (p != NULL)
    {
      newenv = malloc (strlen (p) + len + 1);
      if (newenv != NULL)
	{
	  sprintf (newenv, "%s=%s", name, p);
#if defined (__DJGPP__)
	  /* lower conversion */
	  for (p = newenv + len; *p; p++)
	    if (isupper (*p))
	      *p = tolower (*p);
#endif
	  dosutil_delsysroot (newenv + len);
	  putenv (newenv);
	  free (newenv);
	}
    }
}

static void
envvar_import_path (char *name)
{
  char *p, *fixp, *newbuf;

  p = getenv (name);
  if (p != NULL)
    {
      fixp = dosutil_make_import_path (p);
      if (fixp != NULL)
	{
	  newbuf = malloc (strlen (fixp) + strlen (name) + 2);
	  if (newbuf != NULL)
	    {
	      sprintf (newbuf, "%s=%s", name, fixp);
	      putenv (newbuf);
	      free (newbuf);
	    }
	  free (fixp);
	}
    }
}

char *
dosutil_make_import_path (char *value)
{
  char *p, *newenv;
  int newlen, pos;

  if (value == NULL)
    return NULL;

  /* fix each element */
  newlen = strlen (value) + 1;
  newenv = malloc (newlen);
  if (newenv != NULL)
    {
      /* copy to buffer */
      strcpy (newenv, value);
      /* convert back-slash to slash */
      dosutil_toslash (newenv);

      /* may delete SYSROOT */
      for (p = newenv; p != NULL && *p; p = strchr (p + 1, ';'))
	{
	  int drive_len = 0;

	  /* replace delimiter */
	  while (*p == ';')
	    *p++ = dosutil_path_separator;
	  if (*p == '\0')
	    break;

	  /* check drive name */
	  if (isalpha (p[0]) && p[1] == ':')
	    {
	      if (dosutil_sysroot_len > 0 && isalpha (dosutil_sysroot[0])
		  && dosutil_sysroot[1] == ':'
		  && tolower (p[0]) == dosutil_sysroot[0])
		drive_len = 2;
	      else if (dosutil_path_separator == ':')
		{
		  newlen++;
		  pos = p - newenv;
		  newenv = realloc (newenv, newlen);
		  if (newenv == NULL)
		    return NULL;
		  p = newenv + pos;
		  memmove (p + 1, p, (newlen - 1) - (p - newenv));
		  p[2] = p[0];
		  p[1] = p[0] = '/';
		}
	    }

	  /* delete `SYSROOT' */
	  if (dosutil_sysroot_len > 0
	      && SYSROOT_COMPARE (dosutil_sysroot + drive_len, p + drive_len,
				  dosutil_sysroot_len - drive_len) == 0)
	    {
	      strcpy (p, p + dosutil_sysroot_len);
	      newlen -= dosutil_sysroot_len;
	    }
	}

      /* may need to convert slash to back-slash */
      if (dosutil_path_slash == '\\')
	dosutil_tobslash (newenv);
    }

  return newenv;
}

char *
dosutil_make_export_path (char *value)
{
  char *p, *newenv;
  int newlen, pos;

  if (value == NULL)
    return NULL;

  /* fix each element */
  newlen = strlen (value) + 1;
  newenv = malloc (newlen);
  if (newenv != NULL)
    {
      /* copy to buffer */
      strcpy (newenv, value);
      /* convert back-slash to slash */
      dosutil_toslash (newenv);

      /* may add SYSROOT */
      for (p = newenv; p != NULL && *p; p = strchr (p + 1, dosutil_path_separator))
	{
	  /* replace delimiter */
	  while (*p == dosutil_path_separator)
	    *p++ = ';';
	  if (*p == '\0')
	    break;

	  if (p[0] == '/' && p[1] != '/' && dosutil_sysroot_len > 0)
	    {
	      /* expand SYSROOT */
	      pos = p - newenv;
	      newenv = realloc (newenv, newlen + dosutil_sysroot_len);
	      if (newenv == NULL)
		return NULL;
	      p = newenv + pos;
	      memmove (p + dosutil_sysroot_len, p, newlen - (p - newenv));
	      newlen += dosutil_sysroot_len;
	      memcpy (p, dosutil_sysroot, dosutil_sysroot_len);
	      p += dosutil_sysroot_len;
	    }
	  else if (p[0] == '/' && p[1] == '/' && isalpha(p[2])
		   && (p[3] == '/' || p[3] == '\0'))
	    {
	      /* expand drive name */
	      p[0] = p[2];
	      p[1] = ':';
	      memcpy (p + 2, p + 3, newlen - (p + 3 - newenv));
	      newlen--;
	      pos = p - newenv;
	      newenv = realloc (newenv, newlen);
	      if (newenv == NULL)
		return NULL;
	      p = newenv + pos;
	      p += 2;
	    }
	}

      /* convert slash to back-slash */
      dosutil_tobslash (newenv);
    }

  return newenv;
}

void
dosutil_import_envvar (void)
{
  char *p;
  int i;

  p = getenv ("PATH_EXPAND");
  if (p != NULL)
    {
      if (*p == 'y' || *p == 'Y')
	dosutil_path_expand = 1;
    }
  p = getenv ("PATH_SEPARATOR");
  if (p != NULL)
    {
      if (*p != '\0' && *p != '/' && *p != '\\' && *p < 0x7f)
	dosutil_path_separator = (int) *(unsigned char *) p;
    }
  p = getenv ("PATH_SLASH");
  if (p != NULL)
    {
      if (*p != '\0' && (*p == '/' || *p == '\\') && *p < 0x7f)
	dosutil_path_slash = (int) *(unsigned char *) p;
    }

  p = getenv ("SYSROOT");
  if (p != NULL)
    {
      if (p[0] == '/' && p[1] == '/' && isalpha (p[2]) && (p[3] == '\0' || p[3] == '/'))
	{
	  dosutil_sysroot[0] = p[2];
	  dosutil_sysroot[1] = ':';
	  strcpy (dosutil_sysroot + 2, p + 3);
	}
      else
	strcpy (dosutil_sysroot, p);
      dosutil_toslash (dosutil_sysroot);
      if (dosutil_sysroot[0] != '/' || dosutil_sysroot[1] != '\0')
	dosutil_delslash (dosutil_sysroot);
      if (isalpha (dosutil_sysroot[0]) && dosutil_sysroot[1] == ':')
	dosutil_sysroot[0] = tolower (dosutil_sysroot[0]);
      dosutil_sysroot_len = strlen (dosutil_sysroot);
    }

  for (i = 0; (p = need_fix_variables[i]) != NULL; i++)
    envvar_import_path (p);
}

char *
dosutil_addslash (char *path)
{
  char *top = path;

  path += strlen (path);
  if (path != top && path[-1] != '/')
    {
      *path++ = '/';
      *path++ = '\0';
    }

  return top;
}

char *
dosutil_delslash (char *path)
{
  char *top = path;

  path += strlen (path);
  if (path != top && path[-1] == '/')
    path[-1] = '\0';

  return top;
}

char *
dosutil_toslash (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (*p == '\\')
	*p = '/';
#else
      if (*p == '\\')
	*p = '/';
#endif
    }

  return path;
}

char *
dosutil_tobslash (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (*p == '/')
	*p = '\\';
#else
      if (*p == '/')
	*p = '\\';
#endif
    }

  return path;
}

char *
dosutil_tolower (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (isupper (*p))
	*p = tolower (*p);
#else
      if (isupper (*p))
	*p = tolower (*p);
#endif
    }

  return path;
}

char *
dosutil_toupper (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (islower (*p))
	*p = toupper (*p);
#else
      if (islower (*p))
	*p = toupper (*p);
#endif
    }

  return path;
}

char *
dosutil_delsysroot (char *path)
{
  int drive_len = 0;

  /* convert back-slash to slash */
  dosutil_toslash (path);

  if (dosutil_sysroot_len > 0)
    {
      /* check drive name */
      if (isalpha (path[0]) && path[1] == ':'
	  && isalpha (dosutil_sysroot[0]) && dosutil_sysroot[1] == ':')
	{
	  if (tolower (path[0]) != dosutil_sysroot[0])
	    return path;
	  drive_len = 2;
	}

      /* delete `SYSROOT' */
      if (SYSROOT_COMPARE (dosutil_sysroot + drive_len, path + drive_len,
			   dosutil_sysroot_len - drive_len) == 0)
	strcpy (path, path + dosutil_sysroot_len);
    }

  return path;
}

char *
dosutil_expand_path (char *path)
{
  char *p;

  dosutil_toslash (path);
  if (path[0] == '/' && path[1] != '/' && dosutil_sysroot_len > 0)
    {
      /* expand SYSROOT */
      p = malloc (strlen (path) + dosutil_sysroot_len + 1);
      if (p != NULL)
	{
	  strcpy (p, dosutil_sysroot);
	  strcpy (p + dosutil_sysroot_len, path);
	  free (path);
	  path = p;
	}
    }
  else if (path[0] == '/' && path[1] == '/' && isalpha (path[2])
	   && (path[3] == '/' || path[3] == '\0'))
    {
      /* expand drive name */
      p = malloc (2 + strlen (path) - 3 + 1);
      if (p != NULL)
	{
	  p[0] = path[2];
	  p[1] = ':';
	  strcpy (p + 2, path + 3);
	  free (path);
	  path = p;
	}
    }

  return path;
}

char *
dosutil_fix_variable (char *vp)
{
  char *p, *fixp, *newbuf;
  int i;

  if (vp == NULL)
    return NULL;

  for (i = 0; (p = need_fix_variables[i]) != NULL; i++)
    if (strncmp (vp, p, strlen (p)) == 0
	&& vp[strlen (p)] == '=')
      {
	fixp = dosutil_make_export_path (vp + strlen (p) + 1);
	if (fixp != NULL)
	  {
	    newbuf = malloc (strlen (fixp) + strlen (p) + 2);
	    if (newbuf != NULL)
	      {
#if defined (__MSDOS__)
		/* convert upper case */
		if (strcmp (p, "PATH") == 0)
		  dosutil_toupper (fixp);
#endif
		sprintf (newbuf, "%s=%s", p, fixp);
		free (vp);
		vp = newbuf;
	      }
	    free (fixp);
	  }
	break;
      }

  return vp;
}

char **
dosutil_get_import_variables (void)
{
  char **p;
  int i;

  /* count need size */
  for (i = 0; need_fix_variables[i] != NULL; i++)
    ;
  p = malloc (i * sizeof (char*));
  if (p != NULL)
    {
      for (i = 0; need_fix_variables[i] != NULL; i++)
	p[i] = dosutil_make_export_path (get_string_value (need_fix_variables[i]));
    }

  return p;
}

void
dosutil_put_export_variables (char **vp)
{
  SHELL_VAR *var;
  int i;

  if (vp == NULL)
    return;

  for (i = 0; need_fix_variables[i] != NULL; i++)
    {
      var = find_variable (need_fix_variables[i]);
      if (var != NULL)
	{
	  if (var->value != NULL)
	    free (var->value);
	  var->value = dosutil_make_import_path (vp[i]);
	}
      if (vp[i] != NULL)
	free (vp[i]);
    }
  free (vp);
  set_working_directory (get_string_value ("PWD"));
}

static dosutil_fd5 = -1;
static dosutil_fd6 = -1;

#if defined (__HUMAN68K__)
static void
dosutil_fdunset (struct _psp *pp, int fd)
{
  pp->handle[(fd / 8)] &= ~(1 << (fd % 8));
}

static void
dosutil_fdset (struct _psp *pp, int fd)
{
  pp->handle[(fd / 8)] |= (1 << (fd % 8));
}

static void
dosutil_fdreset (void)
{
  struct _psp *pp;

  pp = _dos_getpdb ();
  pp->handle[0];

  if (dosutil_fd5 >= 0)
    {
      close (5);
      dup2 (dosutil_fd5, 5);
      _fddb[5].inuse = _fddb[dosutil_fd5].inuse;
      dosutil_fdunset (pp, 5);
      close (dosutil_fd5);
    }
  if (dosutil_fd6 >= 0)
    {
      close (6);
      dup2 (dosutil_fd6, 6);
      _fddb[6].inuse = _fddb[dosutil_fd6].inuse;
      dosutil_fdunset (pp, 6);
      close (dosutil_fd6);
    }
}

void
dosutil_reserve_fds (void)
{
  int fd;
  int devmod;

  /* add clean up routine */
  onexit (dosutil_fdreset);

  /* don't use fd 5 and 6. these are used by configure script. */
  if ((devmod = _dos_ioctrlgt (5)) < 0
      || (devmod & (_DEV_CDEV|_DEV_NUL)) != (_DEV_CDEV|_DEV_NUL))
    {
      if (devmod >= 0)
	{
	  dosutil_fd5 = fcntl (5, F_DUPFD, 7);
	  if (dosutil_fd5 > 0)
	    {
	      _fddb[_fd5].inuse = _fddb[5].inuse;
	      close (5);
	    }
	}
      fd = open ("NUL", O_RDWR);
      if (fd != 5)
	{
	  dup2 (fd, 5);
	  close (fd);
	}
    }

  if ((devmod = _dos_ioctrlgt (6)) < 0
      || (devmod & (_DEV_CDEV|_DEV_NUL)) != (_DEV_CDEV|_DEV_NUL))
    {
      if (devmod >= 0)
	{
	  dosutil_fd6 = fcntl (6, F_DUPFD, 7);
	  if (dosutil_fd6 > 0)
	    {
	      _fddb[_fd6].inuse = _fddb[6].inuse;
	      close (6);
	    }
	}
      fd = open ("NUL", O_RDWR);
      if (fd != 6)
	{
	  dup2 (fd, 6);
	  close (fd);
	}
    }
}
#elif defined (__DJGPP__)
#define _DEV_STDIN  0x0001
#define _DEV_STDOUT 0x0002
#define _DEV_NUL    0x0004
#define _DEV_CLOCK  0x0008
#define _DEV_RAW    0x0020
#define _DEV_CDEV   0x0080
#define _DEV_IOCTRL 0x4000

void
dosutil_reserve_fds (void)
{
  int fd;
  short devmod;

  /* don't use fd 5 and 6. these are used by configure script. */
  if ((devmod = _get_dev_info (5)) == -1)
    {
      fd = open ("NUL", O_RDWR);
      if (fd != 5)
	{
	  dup2 (fd, 5);
	  close (fd);
	}
    }

  if ((devmod = _get_dev_info (6)) == -1)
    {
      fd = open ("NUL", O_RDWR);
      if (fd != 6)
	{
	  dup2 (fd, 6);
	  close (fd);
	}
    }
}
#else
void
dosutil_reserve_fds (void)
{
}
#endif

#if defined (__DJGPP__)
static char dosutil_oldcwd[PATH_MAX + 1] = { '\0' };

static void
dosutil_restore_cwd (void)
{
  if (dosutil_oldcwd[0] != '\0')
    (void) (chdir) (dosutil_oldcwd);
}

void
dosutil_save_cwd (void)
{
  (void) (getcwd) (dosutil_oldcwd, sizeof (dosutil_oldcwd));
  atexit (dosutil_restore_cwd);
}
#else
void
dosutil_save_cwd (void)
{
}
#endif

void
dosutil_save_std_fds (int fds[3])
{
  int i;

  /* save stdin/out/err */
  for (i = 0; i < 3; i++)
#if defined (__DJGPP__)
    if ((fds[i] = fcntl (i, F_DUPFD, 20)) < 0)
#else
    if ((fds[i] = dup (i)) < 0)
#endif
      internal_error ("Cannot duplicate fd %d: %s",
		      i, strerror (errno));
}

void
dosutil_restore_std_fds (int fds[3])
{
  int i;

  /* restore stdin/out/err */
  for (i = 0; i < 3; i++)
    if (fds[i] >= 0)
      {
#if defined (__DJGPP__)
	if (dosutil_dup2_wrapper (fds[i], i) < 0)
#else
	if (dup2 (fds[i], i) < 0)
#endif
	  internal_error ("cannot duplicate fd %d to fd %d: %s",
			  fds[i], i, strerror (errno));
	close (fds[i]);
	fds[i] = -1;
      }
}

static void
save_jmp_buf (jmp_buf save, jmp_buf now)
{
  memcpy ((char *) save, (char *) now, sizeof (jmp_buf));
}

static void
restore_jmp_buf (jmp_buf save, jmp_buf now)
{
  memcpy ((char *) now, (char *) save, sizeof (jmp_buf));
}

extern jmp_buf top_level, subshell_top_level;
extern int interactive, interactive_shell, login_shell;
extern int subshell_environment;
extern int subshell_exit_builtin, exit_immediately_on_error;
extern int dollar_dollar_pid;
extern int array_needs_making;
extern char **temporary_env, **function_env, **builtin_env;

static void
save_current_directory (OLDENVBUF *envp)
{
  char *p;

  envp->pwd    = SAVESTRING (get_string_value ("PWD"));
#if 0
  envp->oldpwd = SAVESTRING (get_string_value ("OLDPWD"));
#endif
  envp->curdir = dosutil_getcwd (NULL, PATH_MAX);
}

static void
restore_current_directory (OLDENVBUF *envp)
{
  /* change old directory */
  if (envp->curdir != NULL)
    {
      dosutil_chdir (envp->curdir);
      if (envp->pwd != NULL)
	set_working_directory (envp->pwd);
      else
	set_working_directory (envp->curdir);
      FREE_AND_CLEAR (envp->curdir);
    }
  else if (envp->pwd != NULL)
    {
      dosutil_chdir (envp->pwd);
      set_working_directory (envp->pwd);
    }

#if 0
  /* replace variables */
  unbind_variable ("PWD");
  if (envp->pwd)
    {
      bind_variable ("PWD", envp->pwd);
      FREE_AND_CLEAR (envp->pwd);
    }

  unbind_variable ("OLDPWD");
  if (envp->oldpwd)
    {
      bind_variable ("OLDPWD", envp->oldpwd);
      FREE_AND_CLEAR (envp->oldpwd);
    }
#else
  if (envp->pwd)
    FREE_AND_CLEAR (envp->pwd);
#endif
}

static void
save_global_variables (OLDENVBUF *envp)
{
  envp->interactive               = interactive;
  envp->interactive_shell         = interactive_shell;
  envp->login_shell               = login_shell;
  envp->subshell_environment      = subshell_environment;
  envp->subshell_exit_builtin     = subshell_exit_builtin;
  envp->exit_immediately_on_error = exit_immediately_on_error;
  envp->variable_context          = variable_context;
  envp->dollar_dollar_pid         = dollar_dollar_pid;
}

static void
restore_global_variables (OLDENVBUF *envp)
{
  interactive               = envp->interactive;
  interactive_shell         = envp->interactive_shell;
  login_shell               = envp->login_shell;
  subshell_environment      = envp->subshell_environment;
  subshell_exit_builtin     = envp->subshell_exit_builtin;
  exit_immediately_on_error = envp->exit_immediately_on_error;
  variable_context          = envp->variable_context;
  dollar_dollar_pid         = envp->dollar_dollar_pid;
}

static HASH_TABLE *
copy_shell_variables_hash_table (HASH_TABLE *table)
{
  int i;
  HASH_TABLE *new_table;
  BUCKET_CONTENTS *new_item, *old_item;
  SHELL_VAR *new_var, *old_var;

  if (!table)
    return (HASH_TABLE *) NULL;

  new_table = make_hash_table (table->nbuckets);
  new_table->nentries = table->nentries;
  for (i = 0; i < table->nbuckets; i++)
    {
      old_item = table->bucket_array[i];
      new_item = (BUCKET_CONTENTS *) NULL;
      while (old_item)
	{
	  if (new_item)
	    {
	      new_item->next = (BUCKET_CONTENTS *) xmalloc (sizeof (BUCKET_CONTENTS));
	      new_item = new_item->next;
	    }
	  else
	    {
	      new_table->bucket_array[i] =
		(BUCKET_CONTENTS *) xmalloc (sizeof (BUCKET_CONTENTS));
	      new_item = new_table->bucket_array[i];
	    }
	  new_item->data = (char *) copy_variable ((SHELL_VAR *) old_item->data);
	  old_var = (SHELL_VAR *)old_item->data;
	  new_var = (SHELL_VAR *)new_item->data;
	  while (old_var->prev_context)
	    {
	      new_var->prev_context = copy_variable (old_var->prev_context);
	      new_var = new_var->prev_context;
	      old_var = old_var->prev_context;
	    }
	  new_item->next = (BUCKET_CONTENTS *)NULL;
	  new_item->key = SAVESTRING (old_item->key);
	  new_item->times_found = old_item->times_found;
	  old_item = old_item->next;
	}
    }

  return new_table;
}

static void
dispose_shell_variables_hash_table (HASH_TABLE *table)
{
  int i;
  BUCKET_CONTENTS *item, *next;
  SHELL_VAR *var, *prev;

  if (!table)
    return;

  for (i = 0; i < table->nbuckets; i++)
    {
      item = table->bucket_array[i];
      while (item)
	{
	  var = (SHELL_VAR *) item->data;
	  while (var)
	    {
	      prev = var->prev_context;
	      dispose_variable (var);
	      var = prev;
	    }
	  if (item->key)
	    free (item->key);
	  next = item->next;
	  free (item);
	  item = next;
	}
    }
  free (table->bucket_array);
  free (table);
}

static void
save_shell_variables (OLDENVBUF *envp)
{
  int i;

  maybe_make_export_env ();

  envp->shell_variables = shell_variables;
  shell_variables = copy_shell_variables_hash_table (shell_variables);
  envp->shell_functions = shell_functions;
  shell_functions = copy_shell_variables_hash_table (shell_functions);

  envp->temporary_env = temporary_env;
  if (temporary_env != NULL)
    temporary_env = copy_array (temporary_env);
  envp->function_env = function_env;
  if (function_env != NULL)
    function_env = copy_array (function_env);
  envp->builtin_env = builtin_env;
  if (builtin_env != NULL)
    builtin_env = copy_array (builtin_env);
  envp->export_env = export_env;
  if (export_env != NULL)
    export_env = copy_array (export_env);
  envp->non_unsettable_vars = non_unsettable_vars;
  if (non_unsettable_vars != NULL)
    non_unsettable_vars = copy_array (non_unsettable_vars);

  envp->rest_of_args = list_rest_of_args ();
}

static void
restore_shell_variables (OLDENVBUF *envp)
{
  int i;

  dispose_shell_variables_hash_table (shell_variables);
  shell_variables = envp->shell_variables;
  dispose_shell_variables_hash_table (shell_functions);
  shell_functions = envp->shell_functions;

  free_array (temporary_env);
  temporary_env = envp->temporary_env;
  free_array (function_env);
  function_env = envp->function_env;
  free_array (builtin_env);
  builtin_env = envp->builtin_env;
  free_array (export_env);
  export_env = envp->export_env;
  free_array (non_unsettable_vars);
  non_unsettable_vars = envp->non_unsettable_vars;

  remember_args (envp->rest_of_args, 1);

  array_needs_making = 0;
}

int
dosutil_save_all_environment (OLDENVBUF *envp)
{
  save_jmp_buf (envp->top_level, top_level);
  save_jmp_buf (envp->subshell_top_level, subshell_top_level);
  dosutil_save_std_fds (envp->fds);
  save_current_directory (envp);
  save_global_variables (envp);
  save_shell_variables (envp);
}

int
dosutil_restore_all_environment (OLDENVBUF *envp)
{
  restore_shell_variables (envp);
  restore_global_variables (envp);
  restore_current_directory (envp);
  dosutil_restore_std_fds (envp->fds);
  restore_jmp_buf (envp->top_level, top_level);
  restore_jmp_buf (envp->subshell_top_level, subshell_top_level);
}

/* pipe emulation */

struct pipe_t
{
  int fds[2];
  char *name;
};

#if !defined (OPEN_MAX)
#define OPEN_MAX 256
#endif
static struct pipe_t pipes[OPEN_MAX];
static int pipes_max = 0;

void
dosutil_pipe_clean (int no)
{
  int i;

  for (i = no; i < pipes_max; i++)
    {
      remove (pipes[i].name);
      free (pipes[i].name);
      pipes[i].fds[0] = pipes[i].fds[1] = 0;
      pipes[i].name = NULL;
    }

  pipes_max = no;
}

int
dosutil_pipe_no (void)
{
  return pipes_max;
}

int
dosutil_pipe (int fds[2])
{
  int ifd;
  int ofd;
  char *tmpdir;
  char *tname;

  if (pipes_max >= OPEN_MAX)
    {
      errno = EMFILE;
      return -1;
    }

  tmpdir = getenv ("TMPDIR");
  if (tmpdir == NULL)
    tmpdir = "/tmp";

  tname = malloc (strlen (tmpdir) + sizeof ("/ppXXXXXX"));
  if (tname == NULL)
    return -1;
  sprintf (tname, "%s/ppXXXXXX", tmpdir);
  tname = mktemp (dosutil_expand_path (tname));
  if (tname == NULL)
    return -1;

  ofd = open (tname, O_RDWR|O_CREAT|O_TRUNC, 0666);
  if (ofd < 0)
    {
      free (tname);
      return -1;
    }
#if defined (__DJGPP__)
  if (ofd < 20)
    {
      int tfd;

      tfd = fcntl (ofd, F_DUPFD, 20);
      close (ofd);
      if (tfd < 0)
	{
	  remove (tname);
	  free (tname);
	  return -1;
	}
      ofd = tfd;
    }
  ifd = fcntl (ofd, F_DUPFD, 20);
#else
  ifd = dup (ofd);
#endif
  if (ifd < 0)
    {
      close (ofd);
      remove (tname);
      free (tname);
      return -1;
    }

  pipes[pipes_max].name = tname;
  fds[0] = pipes[pipes_max].fds[0] = ifd;
  fds[1] = pipes[pipes_max].fds[1] = ofd;

  pipes_max++;

  return 0;
}

#include <sys/param.h>
#if !defined (MAXPID) 
#define MAXPID ((pid_t) 30000)
#endif

static pid_t last_pid = CURRENT_PID;

pid_t
dosutil_make_child_pid (void)
{
  if (++last_pid > MAXPID)
    last_pid = 1;

  return last_pid;
}

/* `SYSROOT' extention functions */
int
dosutil_chdir (char *path)
{
  char *fixdirp;
  int rc;

  if (path[0] == '/' && path[1] != '/' && dosutil_sysroot_len > 0)
    {
      fixdirp = malloc (strlen (path) + dosutil_sysroot_len + 1);
      if (fixdirp != NULL)
	{
	  sprintf (fixdirp, "%s%s", dosutil_sysroot, path);
	  rc = (chdir) (fixdirp);
	  free (fixdirp);
	  return rc;
	}
    }
  else if (path[0] == '/' && path[1] == '/' && isalpha(path[2])
	   && (path[3] == '/' || path[3] == '\0'))
    {
      fixdirp = malloc (2 + strlen (path) - 3 + 1);
      if (fixdirp != NULL)
	{
	  fixdirp[0] = path[2];
	  fixdirp[1] = ':';
	  strcpy (fixdirp + 2, path + 3);
	  rc = (chdir) (fixdirp);
	  free (fixdirp);
	  return rc;
	}
    }

  return (chdir) (path);
}

static char *
encode_drivename (char *path)
{
  int len = strlen (path);

  memmove (path + 1, path, len + 1);
  path[2] = path[1];
  path[0] = '/';
  path[1] = '/';

  return path;
}

char *
dosutil_getcwd (char *buffer, int length)
{
  char *p;

  p = (getcwd) (buffer, length);
  if (p != NULL)
    {
      dosutil_delsysroot (p);
      if (isalpha (p[0]) && p[1] == ':' && dosutil_path_separator == ':'
	  && strlen (p) < (length - 1))
	encode_drivename (p);
    }

  return p;
}

char *
dosutil_getwd (char *buffer)
{
  char *p;

  p = (getwd) (buffer);
  if (p != NULL)
    {
      dosutil_delsysroot (p);
      if (isalpha (p[0]) && p[1] == ':' && dosutil_path_separator == ':')
	encode_drivename (p);
    }

  return p;
}

#if defined (__DJGPP__)
#include <dos.h>
#include <dpmi.h>
#include <go32.h>
#include <pc.h>
#include <sys/exceptn.h>

static int dosutil_cbrk_vector = -1;
static char *known_suffixes[] =
{
  "com",
  "exe",
  "bat",
  "btm",
  "sh",
  "ksh",
  "pl",
  "sed",
  NULL,
};

char *
dosutil_add_executable_suffix (char *name, char *ext)
{
  int i;

  if (*ext != '.')
    {
      *ext = '.';
      strcpy (ext + 1, known_suffixes[0]);
      return name;
    }

  for (i = 0; known_suffixes[i] != NULL; i++)
    {
      if (strcmp (ext + 1, known_suffixes[i]) == 0)
	{
	  if (known_suffixes[i + 1] == NULL)
	    break;
	  strcpy (ext + 1, known_suffixes[i + 1]);
	  return name;
	}
    }

  return NULL;
}

void
dosutil_save_exceptions (OLDEXCEPTBUF *exceptionp)
{
  int i;

  if (dosutil_cbrk_vector < 0)
    {
      if (ScreenPrimary != 0xa0000)
	dosutil_cbrk_vector = 0x1b;
      else
	dosutil_cbrk_vector = 0x06;
    }

  for (i = 0; i < DOSUTIL_EXCEPTION_COUNT; i++)
    __dpmi_get_processor_exception_handler_vector (i, &exceptionp->except_original[i]);
  __dpmi_get_protected_mode_interrupt_vector (9, &exceptionp->kbd_original);
  __dpmi_get_protected_mode_interrupt_vector (0x75, &exceptionp->npx_original);
  __dpmi_get_real_mode_interrupt_vector (dosutil_cbrk_vector, &exceptionp->cbrk_original);
}

void
dosutil_restore_exceptions (OLDEXCEPTBUF *exceptionp)
{
  int i;

  for (i = 0; i < DOSUTIL_EXCEPTION_COUNT; i++)
    __dpmi_set_processor_exception_handler_vector (i, &exceptionp->except_original[i]);
  __dpmi_set_protected_mode_interrupt_vector (9, &exceptionp->kbd_original);
  __dpmi_set_protected_mode_interrupt_vector (0x75, &exceptionp->npx_original);
  __dpmi_set_real_mode_interrupt_vector (dosutil_cbrk_vector, &exceptionp->cbrk_original);
}

#if 0
int
dosutil_print_exception (void)
{
  __dpmi_paddr except;
  __dpmi_raddr rexcept;
  int i;

  for (i = 0; i < DOSUTIL_EXCEPTION_COUNT; i++)
    {
      __dpmi_get_processor_exception_handler_vector(i, &except);
      printf ("pe%d offset32=%8.8lx, selector=%4.4x\n", i, except.offset32, except.selector);
    }
  __dpmi_get_protected_mode_interrupt_vector(0x75, &except);
  printf ("0x75 offset32=%8.8lx, selector=%4.4x\n", except.offset32, except.selector);
  __dpmi_get_protected_mode_interrupt_vector(9, &except);
  printf ("9 offset32=%8.8lx, selector=%4.4x\n", except.offset32, except.selector);
  __dpmi_get_real_mode_interrupt_vector(0x1b, &rexcept);
  printf ("0x1b offset16=%4.4x, segment=%4.4x\n", rexcept.offset16, rexcept.segment);

  return 0;
}
#endif /* 0 */

#define CHECK_SELS 12
int *
dosutil_check_dpmi_selectors (void)
{
  int *p;
  int i;

  p = malloc (sizeof (int) * CHECK_SELS);
  if (p != NULL)
    {
      for (i = 0; i < CHECK_SELS; i++)
	p[i] = __dpmi_allocate_ldt_descriptors (1);
      for (i = 0; i < CHECK_SELS; i++)
	__dpmi_free_ldt_descriptor (p[i]);
    }

  return p;
}

void
dosutil_free_dpmi_selectors (int *selectors)
{
  int sel1;
  int i;

  if (selectors != NULL)
    {
      sel1 = __dpmi_allocate_ldt_descriptors (1);
      __dpmi_free_ldt_descriptor (sel1);
#if 0
      for (i = 0; i < CHECK_SELS; i++)
	printf ("%4.4x ", selectors[i]);
      printf ("%4.4x\n", sel1);
#endif
      if (sel1 != selectors[0])
	{
	  for (i = 0; i < CHECK_SELS; i++)
	    __dpmi_free_ldt_descriptor (selectors[i]);
	}
      free (selectors);
    }
}

int
dosutil_dup2_wrapper (int from, int to)
{
  int rc;
  int old_enable, old_ctrl_c;

  old_enable = __dpmi_get_and_disable_virtual_interrupt_state ();
  old_ctrl_c = __djgpp_set_ctrl_c (0);

  rc = dup2 (from, to);

  (void) __djgpp_set_ctrl_c (old_ctrl_c);
  if (old_enable)
    (void) __dpmi_get_and_enable_virtual_interrupt_state ();

  return rc;
}

char *
dosutil_make_response_file (char **argv)
{
  char *tmpdir, *tname;

  tmpdir = getenv ("TMPDIR");
  if (tmpdir == NULL)
    tmpdir = "/tmp";

  tname = malloc (strlen (tmpdir) + sizeof ("/rsXXXXXX"));
  if (tname != NULL)
    {
      sprintf (tname, "%s/rsXXXXXX", tmpdir);
      tname = mktemp (dosutil_expand_path (tname));
      if (tname != NULL)
	{
	  FILE *fp;
	  char *p;
	  int i;

	  fp = fopen (tname, "wt");
	  if (fp == NULL)
	    {
	      free (tname);
	      return NULL;
	    }

	  for (i = 1; (p = argv[i]) != NULL; i++)
	    {
	      if (i != 1)
		fputc (' ', fp);
	      if (strchr (p, '"') == NULL)
		{
		  fputc ('"', fp);
		  fputs (p, fp);
		  fputc ('"', fp);
		}
	      else
		{
		  fputc ('"', fp);
		  while (*p)
		    {
		      if (*p == '"')
			fputc ('\\', fp);
		      fputc (*p, fp);
		      p++;
		    }
		  fputc ('"', fp);
		}
	    }
	  fclose (fp);
	}
    }

  return tname;
}
#endif /* !__DJGPP__ */
