;;; po-mode.el -- A mode to help GNU gettext lovers edit PO files.
;;; Copyright (C) 1995, 1996, 1997 Free Software Foundation, Inc.

;;; Author: Franois Pinard <pinard@iro.umontreal.ca>, 1995.
;;; Helped by Greg McGary <gkm@magilla.cichlid.com>.
;;; Some patching and rewriting by Tim Van Holder <zastai@hotmail.com>
;;; Keywords: languages i18n

;; This file is part of GNU gettext.

;; GNU gettext is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU gettext is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, 59 Temple Place - Suite 330, Boston,
;; MA 02111-1307, USA.

;;; Commentary:

;; This package provides the tools meant to help editing PO files,
;; as documented in the GNU gettext user's manual.  See this manual
;; for user documentation, which is not repeated here.

;; To install, merely put this file somewhere GNU Emacs will find it,
;; then add the following lines to your .emacs file:
;;
;;   (autoload 'po-mode "po-mode")
;;   (setq auto-mode-alist (cons '("\\.po[tx]?\\'\\|\\.po\\." . po-mode)
;;				  auto-mode-alist))
;;
;; To automatically use proper fonts under Emacs 20, also add:
;;
;;   (autoload 'po-find-file-coding-system "po-mode")
;;   (modify-coding-system-alist 'file "\\.po[tx]?\\'\\|\\.po\\."
;;				  'po-find-file-coding-system)
;;
;; You may omit the autoload statements if `update-file-autoloads' has been
;;  run on this file.
;;
;; You may also adjust some variables, below, by defining them in your
;; `.emacs' file, either directly or through command `M-x customize'.

;;; Code:

(eval-and-compile
  (condition-case ()
      (require 'custom)
    (error nil))
  (if (and (featurep 'custom) (fboundp 'custom-declare-variable))
      nil
    (defmacro defgroup (&rest args)
      nil)
    (defmacro defcustom (var value doc &rest args)
      (` (defvar (, var) (, value) (, doc))))))

(defgroup po nil
  "Major mode for editing PO files"
  :group 'i18n)

(defcustom po-auto-edit-with-msgid nil
  "*Automatically use msgid when editing untranslated entries."
  :type 'boolean
  :group 'po)

(defcustom po-auto-fuzzy-on-edit nil
  "*Automatically mark entries fuzzy when being edited."
  :type 'boolean
  :group 'po)

(defcustom po-auto-select-on-unfuzzy nil
  "*Automatically select some new entry while make an entry not fuzzy."
  :type 'boolean
  :group 'po)

(defcustom po-auto-replace-revision-date 'ask
  "*Automatically revise date in headers.  Value is nil, t, or ask."
  :type '(choice (const nil)
		 (const t)
		 (const ask))
  :group 'po)

(defcustom po-default-file-header "\
# SOME DESCRIPTIVE TITLE.
# Copyright (C) YEAR Free Software Foundation, Inc.
# FIRST AUTHOR <EMAIL@ADDRESS>, YEAR.
#
#, fuzzy
msgid \"\"
msgstr \"\"
\"Project-Id-Version: PACKAGE VERSION\\n\"
\"PO-Revision-Date: YEAR-MO-DA HO:MI +ZONE\\n\"
\"Last-Translator: FULL NAME <EMAIL@ADDRESS>\\n\"
\"Language-Team: LANGUAGE <LL@li.org>\\n\"
\"MIME-Version: 1.0\\n\"
\"Content-Type: text/plain; charset=CHARSET\\n\"
\"Content-Transfer-Encoding: ENCODING\\n\"
"
  "*Default PO file header."
  :type 'string
  :group 'po)

(defcustom po-highlighting t
  "*Highlight text to be marked, when non-nil."
  :type 'boolean
  :group 'po)

(defcustom po-highlight-face 'highlight
  "*The face used for PO mode highlighting.  For Emacses with overlays.
Possible values are `highlight', `modeline', `secondary-selection',
`region', and `underline'.
This variable can be set by the user to whatever face they desire.
It's most convenient if the cursor color and highlight color are
slightly different."
  :type 'face
  :group 'po)

(defcustom po-gzip-uuencode-command "gzip -9 | uuencode -m"
  "*The filter to use for preparing a mail invoice of the PO file.
Normally \"gzip -9 | uuencode -m\", remove the -9 for lesser compression,
or remove the -m if you are not using the GNU version of `uuencode'."
  :type 'string
  :group 'po)

;;; The following block of declarations has the main purpose of avoiding
;;; byte compiler warnings.  It also introduces some documentation for
;;; each of these variables, all meant to be local to PO mode buffers.

;; Flag telling that MODE-LINE-STRING should be displayed.  See `Window'
;; page below.  Exceptionally, this variable is local to *all* buffers.

(defvar po-mode-flag)

;; The current entry extends from START-OF-ENTRY to END-OF-ENTRY, it
;; includes preceding whitespace and excludes following whitespace.  The
;; start of keyword lines are START-OF-MSGID and START-OF-MSGSTR.
;; ENTRY-TYPE classifies the entry.

(defvar po-start-of-entry)
(defvar po-start-of-msgid)
(defvar po-start-of-msgstr)
(defvar po-end-of-entry)
(defvar po-entry-type)

;; A few counters are usefully shown in the Emacs mode line.

(defvar po-translated-counter)
(defvar po-fuzzy-counter)
(defvar po-untranslated-counter)
(defvar po-obsolete-counter)
(defvar po-mode-line-string)

;; A WORK-BUFFER is associated with this PO file, for edition and other
;; various tasks.  WORK-BUFFER-LOCK indicates that the work buffer
;; is already in use, most probably editing some string through Emacs
;; recursive edit.  In this case, one cannot modify the buffer.

(defvar po-work-buffer)
(defvar po-work-buffer-lock)

;; We maintain a set of movable pointers for returning to entries.

(defvar po-marker-stack)

;; SEARCH path contains a list of directories where files may be found,
;; in a format suitable for read completion.  Each directory includes
;; its trailing slash.  PO mode starts with "./" and "../".

(defvar po-search-path)

;; The following variables are meaningful only when REFERENCE-CHECK
;; is identical to START-OF-ENTRY, else they should be recomputed.
;; REFERENCE-ALIST contains all known references for the current
;; entry, each list element is (PROMPT FILE LINE), where PROMPT may
;; be used for completing read, FILE is a string and LINE is a number.
;; REFERENCE-CURSOR is a cycling cursor into REFERENCE-ALIST.

(defvar po-reference-alist)
(defvar po-reference-cursor)
(defvar po-reference-check)

;; The following variables are for marking translatable strings in
;; program sources.  KEYWORDS is the list of keywords for marking
;; translatable strings, kept in a format suitable for reading with
;; completion.  NEXT-FILE-LIST is the list of source files to visit,
;; gotten from the tags table.  STRING-START is the position for
;; the beginning of the last found string, or nil if the string is
;; invalidated.  STRING-END is the position for the end of the string
;; and indicates where the search should be resumed, or nil for the
;; beginning of the current file.

(defvar po-keywords)
(defvar po-next-file-list)
(defvar po-string-start)
(defvar po-string-end)

;;; PO mode variables and constants (usually not to customize).

;; -- Their way (doesn't byte-compile (why?)) --
;(eval-and-compile
;  (or (fboundp 'set-translation-domain)
;      (defsubst set-translation-domain (string) nil))
;  (or (fboundp 'translate-string)
;      (defsubst translate-string (string) string)))
;(defsubst _ (string) (translate-string string))
;(defsubst N_ (string) string)

;;; The textdomain should really be "gettext", only trying it for now.
;;; All this requires more thinking, we cannot just do this like that.
;(set-translation-domain "po-mode")

;; My Way
(defun _ (string) "Shorthand for (translate-string-domain STRING \"po-mode\")"
  (if (fboundp 'translate-string-domain)
      (translate-string-domain string "po-mode")
    string))
(defsubst N_ (string) string)

(defun po-mode-version ()
  "Show Emacs PO mode version."
  (interactive)
  (message (_"Emacs PO mode, version %s") (substring "$Revision: 1.40 $" 11 -2)))

(defconst po-help-display-string
  (_"\
PO Mode Summary           Next Previous            Miscellaneous
*: Later, /: Docum        n    p    Any type       .     Redisplay
                          /t   /M-t Translated     /v    Version info
Moving around             f    M-f  Fuzzy          ?, h  This help
<    First if any         o    M-o  Obsolete       =     Current index
>    Last if any          u    M-u  Untranslated   O     Other window
/SPC Auto select                                   V     Validate
                        Msgstr Comments            M     Mail officially
Modifying entries         RET  #    Call editor    U     Undo
TAB   Remove fuzzy mark   k    K    Kill to        E     Edit out full
/DEL  Fuzzy or fade out   w    W    Copy to        Q     Forceful quit
LFD   Init with msgid     y    Y    Yank from      q     Confirm and quit

gettext Keyword Marking                            Position Stack
,    Find next string     Compendiums              m  Mark and push current
M-,  Mark translatable    *c    To compendium      r  Pop and return
M-.  Change mark, mark    *M-C  Select, save       x  Exchange current/top

Program Sources           Auxiliary Files          Lexicography
s    Cycle reference      *a    Cycle file         *l    Lookup translation
M-s  Select reference     *M-a  Select file        *M-l  Add/edit translation
S    Consider path        *A    Consider file      *L    Consider lexicon
M-S  Ignore path          *M-A  Ignore file        *M-L  Ignore lexicon
")
  "Help page for PO mode.")

(defconst po-mode-menu-layout
  '("PO"
    ("Moving around"
     ["Auto select" po-auto-select-entry t]
     "---"
     "Forward"
     ["Any next" po-next-entry t]
     ["Next translated" po-next-translated-entry t]
     ["Next fuzzy" po-next-fuzzy-entry t]
     ["Next obsolete" po-next-obsolete-entry t]
     ["Next untranslated" po-next-untranslated-entry t]
     ["Last file entry" po-last-entry t]
     "---"
     "Backward"
     ["Any previous" po-previous-entry t]
     ["Previous translated" po-previous-translated-entry t]
     ["Previous fuzzy" po-previous-fuzzy-entry t]
     ["Previous obsolete" po-previous-obsolete-entry t]
     ["Previous untranslated" po-previous-untranslated-entry t]
     ["First file entry" po-first-entry t]
     "---"
     "Position stack"
     ["Mark and push current" po-push-location t]
     ["Pop and return" po-pop-location t]
     ["Exchange current/top" po-exchange-location t]
     "---"
     ["Redisplay" po-current-entry t]
     ["Current index" po-statistics t])
    ("Modifying entries"
     ["Undo" po-undo t]
     "---"
     "Msgstr"
     ["Edit msgstr" po-edit-msgstr t]
     ["Kill msgstr" po-kill-msgstr t]
     ["Save msgstr" po-kill-ring-save-msgstr t]
     ["Yank msgstr" po-yank-msgstr t]
     "---"
     "Comments"
     ["Edit comment" po-edit-comment t]
     ["Kill comment" po-kill-comment t]
     ["Save comment" po-kill-ring-save-comment t]
     ["Yank comment" po-yank-comment t]
     "---"
     ["Remove fuzzy mark" po-unfuzzy t]
     ["Fuzzy or fade out" po-fade-out-entry t]
     ["Init with msgid" po-msgid-to-msgstr t])
    ("Other files"
     ["Other window" po-other-window t]
     "---"
     "Program sources"
     ["Cycle reference" po-cycle-source-reference t]
     ["Select reference" po-select-source-reference t]
     ["Consider path" po-consider-source-path t]
     ["Ignore path" po-ignore-source-path t]
     "---"
     "Compendiums"
     ["To compendium" po-save-entry nil]
     ["Select, save" po-select-and-save-entry nil]
     "---"
     "Auxiliary files"
     ["Cycle file" po-cycle-auxiliary nil]
     ["Select file" po-select-auxiliary nil]
     ["Consider file" po-consider-auxiliary-file nil]
     ["Ignore file" po-ignore-auxiliary-file nil]
     "---"
     "Lexicography"
     ["Lookup translation" po-lookup-lexicons nil]
     ["Add/edit translation" po-edit-lexicon-entry nil]
     ["Consider lexicon" po-consider-lexicon-file nil]
     ["Ignore lexicon" po-ignore-lexicon-file nil])
    "---"
    "Source marking"
    ["Find first string" (po-tags-search '(nil)) t]
    ["Prefer keyword" (po-select-mark-and-mark '(nil)) t]
    ["Find next string" po-tags-search t]
    ["Mark preferred" po-mark-translatable t]
    ["Mark with keyword" po-select-mark-and-mark t]
    "---"
    ["Version info" po-mode-version t]
    ["Help page" po-help t]
    ["Validate" po-validate t]
    ["Mail officially" po-send-mail t]
    ["Edit out full" po-edit-out-full t]
    "---"
    ["Forceful quit" po-quit t]
    ["Soft quit" po-confirm-and-quit t])
  "Menu layout for PO mode.")

(defvar po-translation-project-address
  "translation@iro.umontreal.ca"
  "Electronic mail address of the Translation Project.")

(defvar po-compose-mail-function
  (let ((functions '(compose-mail-other-window
		     message-mail-other-window
		     compose-mail
		     message-mail))
	result)
    (while (and (not result) functions)
      (if (fboundp (car functions))
	  (setq result (car functions))
	(setq functions (cdr functions))))
    (cond (result)
	  ((fboundp 'mail-other-window)
	   (function (lambda (to subject)
		       (mail-other-window nil to subject))))
	  ((fboundp 'mail)
	   (function (lambda (to subject)
		       (mail nil to subject))))
	  (t (function (lambda (to subject)
			 (error (_"I do not know how to mail to `%s'") to))))))
  "Function to start composing an electronic message.")

(defconst po-content-type-charset-alist
  '(("EUC" . japanese-iso-8bit)
    ("ISO-8859-1" . iso-latin-1)
    ("koi8-r" . cyrillic-koi8))
  "How to convert Content-Type into a Mule coding system.")

(defvar po-any-msgid-regexp
  "^\\(#~?[ \t]*\\)?msgid.*\n\\(\\(#~?[ \t]*\\)?\".*\n\\)*"
  "Regexp matching a whole msgid field, whether obsolete or not.")

(defvar po-any-msgstr-regexp
  "^\\(#~?[ \t]*\\)?msgstr.*\n\\(\\(#~?[ \t]*\\)?\".*\n\\)*"
  "Regexp matching a whole msgstr field, whether obsolete or not.")

(defvar po-msgfmt-program "msgfmt"
  "Path to msgfmt program from GNU gettext package.")

;; Font lock based highlighting code.
(defconst po-font-lock-keywords
  '(
    ("^\\(msgid \\|msgstr \\|domain \\)" . font-lock-keyword-face)
    ("\\\\.\\|%[-.0-9]*[a-zA-Z]" . font-lock-variable-name-face)
    ("^#\\([^:,\n].*\\|\\)$" 0 font-lock-comment-face t)
    ("^#: " (0 font-lock-comment-face)
     ("\\([^:]+\\):\\([0-9]+\\)" nil nil (1 font-lock-function-name-face)
      (2 font-lock-reference-face)))
    ("^#, " (0 font-lock-comment-face)
     ("\\([^,]+\\)\\(, \\|$\\)" nil nil (1 font-lock-function-name-face)))
    )
  "Additional expressions to highlight in PO mode.")

(defvar po-overlay nil "Saved information about current highlighting.")


;;; Mode activation.

;;;###autoload
(defun po-find-file-coding-system (arg-list)
  "Return a Mule (DECODING . ENCODING) pair, according to PO file charset.
Called through file-coding-system-alist, before the file is visited for real."
  (with-temp-buffer
    (let* ((coding-system-for-read 'no-conversion)
	   (slot nil))
      (insert-file-contents (nth 1 arg-list) nil 0 2048) ; is 2048 enough?
      (if (and (re-search-forward "^\"Content-Type: text/plain;" nil t)
	       (looking-at "[ \t]*charset=\\([^\\]+\\)")
	       (let ((charset (buffer-substring (match-beginning 1)
						(match-end 1))))
		 (setq slot (assoc-ignore-case charset
					       po-content-type-charset-alist))))
	  (cons (cdr slot) nil)
	'(no-conversion . nil)))))

(defvar po-mode-map nil
  "Keymap for PO mode.")
(if po-mode-map ()
  ;; The following line because (make-sparse-keymap) does not work on Demacs.
  (setq po-mode-map (make-keymap))
  (suppress-keymap po-mode-map)
  (define-key po-mode-map "\C-i" 'po-unfuzzy)
  (define-key po-mode-map "\C-j" 'po-msgid-to-msgstr)
  (define-key po-mode-map "\C-m" 'po-edit-msgstr)
  (define-key po-mode-map " " 'po-auto-select-entry)
  (define-key po-mode-map "?" 'po-help)
  (define-key po-mode-map "#" 'po-edit-comment)
  (define-key po-mode-map "," 'po-tags-search)
  (define-key po-mode-map "." 'po-current-entry)
  (define-key po-mode-map "<" 'po-first-entry)
  (define-key po-mode-map "=" 'po-statistics)
  (define-key po-mode-map ">" 'po-last-entry)
;;;;  (define-key po-mode-map "a" 'po-cycle-auxiliary)
;;;;  (define-key po-mode-map "c" 'po-save-entry)
  (define-key po-mode-map "f" 'po-next-fuzzy-entry)
  (define-key po-mode-map "h" 'po-help)
  (define-key po-mode-map "k" 'po-kill-msgstr)
;;;;  (define-key po-mode-map "l" 'po-lookup-lexicons)
  (define-key po-mode-map "m" 'po-push-location)
  (define-key po-mode-map "n" 'po-next-entry)
  (define-key po-mode-map "o" 'po-next-obsolete-entry)
  (define-key po-mode-map "p" 'po-previous-entry)
  (define-key po-mode-map "q" 'po-confirm-and-quit)
  (define-key po-mode-map "r" 'po-pop-location)
  (define-key po-mode-map "s" 'po-cycle-source-reference)
  (define-key po-mode-map "t" 'po-next-translated-entry)
  (define-key po-mode-map "u" 'po-next-untranslated-entry)
  (define-key po-mode-map "v" 'po-mode-version)
  (define-key po-mode-map "w" 'po-kill-ring-save-msgstr)
  (define-key po-mode-map "x" 'po-exchange-location)
  (define-key po-mode-map "y" 'po-yank-msgstr)
;;;;  (define-key po-mode-map "A" 'po-consider-auxiliary-file)
  (define-key po-mode-map "E" 'po-edit-out-full)
  (define-key po-mode-map "K" 'po-kill-comment)
;;;;  (define-key po-mode-map "L" 'po-consider-lexicon-file)
  (define-key po-mode-map "M" 'po-send-mail)
  (define-key po-mode-map "O" 'po-other-window)
  (define-key po-mode-map "Q" 'po-quit)
  (define-key po-mode-map "S" 'po-consider-source-path)
  (define-key po-mode-map "U" 'po-undo)
  (define-key po-mode-map "V" 'po-validate)
  (define-key po-mode-map "W" 'po-kill-ring-save-comment)
  (define-key po-mode-map "Y" 'po-yank-comment)
  (define-key po-mode-map "\177" 'po-fade-out-entry)
  (define-key po-mode-map "\M-," 'po-mark-translatable)
  (define-key po-mode-map "\M-." 'po-select-mark-and-mark)
;;;;  (define-key po-mode-map "\M-a" 'po-select-auxiliary)
;;;;  (define-key po-mode-map "\M-c" 'po-select-and-save-entry)
  (define-key po-mode-map "\M-f" 'po-previous-fuzzy-entry)
;;;;  (define-key po-mode-map "\M-l" 'po-edit-lexicon-entry)
  (define-key po-mode-map "\M-o" 'po-previous-obsolete-entry)
  (define-key po-mode-map "\M-t" 'po-previous-translated-entry)
  (define-key po-mode-map "\M-u" 'po-previous-untranslated-entry)
  (define-key po-mode-map "\M-s" 'po-select-source-reference)
;;;;  (define-key po-mode-map "\M-A" 'po-ignore-auxiliary-file)
;;;;  (define-key po-mode-map "\M-L" 'po-ignore-lexicon-file)
  (define-key po-mode-map "\M-S" 'po-ignore-source-path)
  )

(defvar po-subedit-mode-map nil
  "Keymap while editing a PO mode entry (or the full PO file).")
(if po-subedit-mode-map
    ()
  (setq po-subedit-mode-map (make-sparse-keymap))
  (define-key po-subedit-mode-map "\C-c\C-c" 'exit-recursive-edit)
  (define-key po-subedit-mode-map "\C-c\C-k" 'abort-recursive-edit))

;;;###autoload
(defun po-mode ()
  "Major mode for translators when they edit PO files.
Special commands:\\{po-mode-map}
Turning on PO mode calls the value of the variable `po-mode-hook',
if that value is non-nil.  Behaviour may be adjusted with
variables po-auto-edit-with-msgid, po-auto-fuzzy-on-edit,
po-auto-select-on-unfuzzy and po-auto-replace-revision-date."

  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'po-mode
	mode-name "PO")
  (use-local-map po-mode-map)
  (setq buffer-read-only t)
  (if (fboundp 'easy-menu-define)
      (easy-menu-define po-mode-menu po-mode-map "" po-mode-menu-layout))

  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(po-font-lock-keywords nil nil))
  (font-lock-mode)
  (make-local-variable 'po-start-of-entry)
  (make-local-variable 'po-start-of-msgid)
  (make-local-variable 'po-start-of-msgstr)
  (make-local-variable 'po-end-of-entry)
  (make-local-variable 'po-entry-type)

  (make-local-variable 'po-translated-counter)
  (make-local-variable 'po-fuzzy-counter)
  (make-local-variable 'po-untranslated-counter)
  (make-local-variable 'po-obsolete-counter)
  (make-local-variable 'po-mode-line-string)

  (setq po-mode-flag t)

  (po-check-file-header)
  (po-compute-counters nil)

  (make-local-variable 'po-work-buffer)
  (make-local-variable 'po-work-buffer-lock)
  (setq po-work-buffer nil
	po-work-buffer-lock nil)

  (make-local-variable 'po-marker-stack)
  (setq po-marker-stack nil)

  (make-local-variable 'po-search-path)
  (setq po-search-path '(("./") ("../")))

  (make-local-variable 'po-reference-alist)
  (make-local-variable 'po-reference-cursor)
  (make-local-variable 'po-reference-check)
  (setq po-reference-alist nil
	po-reference-cursor nil
	po-reference-check 0)

  (make-local-variable 'po-keywords)
  (make-local-variable 'po-next-file-list)
  (make-local-variable 'po-string-start)
  (make-local-variable 'po-string-end)
  (setq po-keywords '(("gettext") ("gettext_noop") ("_") ("N_")
		      ("Gettext") ("gettext__") ("dgettext") ("dcgettext"))
	po-next-file-list nil
	po-string-start nil
	po-string-end nil)

  (message (_"You may type `h' or `?' for a short PO mode reminder."))
  (run-hooks 'po-mode-hook))

(defun po-mode-destroy-cleanup ()
  "When destroying a PO mode buffer, kill editing buffer as well."
  (and (string-equal mode-name "PO")
       (bufferp po-work-buffer)
       (kill-buffer po-work-buffer)))

(if (boundp 'kill-buffer-hook)
    (add-hook 'kill-buffer-hook 'po-mode-destroy-cleanup))

;;; Highlighting, adapted (OK, OK... stolen! :-) from `ispell.el'.

(eval-and-compile
  (cond
   ((string-match "Lucid" emacs-version)

    (defun po-highlight (start end)
      "Highlight the string from START to END using `isearch-highlight'."
      (and po-overlay (isearch-dehighlight t))
      (isearch-highlight start end)
      (setq po-overlay t))

    (defun po-dehighlight ()
      "Display normally the last string which was highlighted."
      (and po-overlay (isearch-dehighlight t))
      (setq po-overlay nil)))

   ((and (string-lessp "19" emacs-version) (featurep 'faces) window-system)

    (defun po-highlight (start end)
      "Highlight the string from START to END using overlays.
The variable `po-highlight-face' selects the face to use for highlighting."
      (if po-overlay
	  (move-overlay po-overlay start end (current-buffer))
	(setq po-overlay (make-overlay start end))
	(overlay-put po-overlay 'face po-highlight-face)))

    (defun po-dehighlight ()
      "Display normally the last string which was highlighted."
      (delete-overlay po-overlay)
      (setq po-overlay nil)))

   (t

    (defun po-highlight (start end)
      "Highlight the string from START to END using `inverse-video'.
We all agree, no doubt about it, this is a kludge."
      (and po-overlay (po-dehighlight))
      (save-excursion
	(let ((modified (buffer-modified-p))
	      (buffer-read-only nil)
	      (inhibit-quit t)
	      (buffer-undo-list t)
	      (text (buffer-substring start end)))
	  (goto-char start)
	  (delete-region start end)
	  (insert-char ?  (- end start))
	  (sit-for 0)
	  (setq inverse-video (not inverse-video))
	  (delete-region start end)
	  (insert text)
	  (sit-for 0)
	  (setq inverse-video (not inverse-video))
	  (set-buffer-modified-p modified)
	  (setq po-overlay (cons (make-marker) (make-marker)))
	  (set-marker (car po-overlay) start)
	  (set-marker (cdr po-overlay) end))))

    (defun po-dehighlight ()
      "Display normally the last string which was highlighted."
      (let ((buffer (and po-overlay (marker-buffer (car po-overlay)))))
	(if buffer
	    (save-excursion
	      (set-buffer buffer)
	      (let ((start (marker-position (car po-overlay)))
		    (end (marker-position (cdr po-overlay)))
		    (modified (buffer-modified-p))
		    (buffer-read-only nil)
		    (inhibit-quit t)
		    (buffer-undo-list t))
		(let ((text (buffer-substring start end)))
		  (goto-char start)
		  (delete-region start end)
		  (insert-char ?  (- end start))
		  (sit-for 0)
		  (delete-region start end)
		  (insert text)
		  (sit-for 0)
		  (set-buffer-modified-p modified))))))
      (setq po-overlay nil)))))

;;; Window management.

(make-variable-buffer-local 'po-mode-flag)

(defvar po-mode-line-entry '(po-mode-flag ("  " po-mode-line-string))
  "Mode line format entry displaying MODE-LINE-STRING.")

;; Insert MODE-LINE-ENTRY in mode line, but on first load only.
(or (member po-mode-line-entry mode-line-format)
    (let ((entry (member 'global-mode-string mode-line-format)))
      (setcdr entry (cons po-mode-line-entry (cdr entry)))))

(eval-and-compile
  (if (fboundp 'force-mode-line-update)

      (fset 'po-force-mode-line-update
	    (symbol-function 'force-mode-line-update))

    (defun po-force-mode-line-update ()
      "Force the mode-line of the current buffer to be redisplayed."
      (set-buffer-modified-p (buffer-modified-p)))))

(defun po-update-mode-line-string ()
  "Compute a new statistics string to display in mode line."
  (setq po-mode-line-string
	(concat (format "%dt" po-translated-counter)
		(if (> po-fuzzy-counter 0)
		    (format "+%df" po-fuzzy-counter))
		(if (> po-untranslated-counter 0)
		    (format "+%du" po-untranslated-counter))
		(if (> po-obsolete-counter 0)
		    (format "+%do" po-obsolete-counter))))
  (po-force-mode-line-update))

(defun po-type-counter ()
  "Return the symbol name of the counter appropriate for the current entry."
  (cond ((eq po-entry-type 'obsolete) 'po-obsolete-counter)
	((eq po-entry-type 'fuzzy) 'po-fuzzy-counter)
	((eq po-entry-type 'translated) 'po-translated-counter)
	((eq po-entry-type 'untranslated) 'po-untranslated-counter)
	(t (error (_"Unknown entry type")))))

(defun po-decrease-type-counter ()
  "Decrease the counter corresponding to the nature of the current entry."
  (let ((counter (po-type-counter)))
    (set counter (1- (eval counter)))))

(defun po-increase-type-counter ()
  "Increase the counter corresponding to the nature of the current entry.
Then, update the mode line counters."
  (let ((counter (po-type-counter)))
    (set counter (1+ (eval counter))))
  (po-update-mode-line-string))

;; Avoid byte compiler warnings.
(defvar po-fuzzy-regexp)
(defvar po-untranslated-regexp)

(defun po-compute-counters (flag)
  "Prepare counters for mode line display.  If FLAG, also echo entry position."
  (and flag (po-find-span-of-entry))
  (setq po-translated-counter 0
	po-fuzzy-counter 0
	po-untranslated-counter 0
	po-obsolete-counter 0)
  (let ((position 0) (total 0) here)
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward po-any-msgstr-regexp nil t)
	(and (= (% total 20) 0)
	     (if flag
		 (message (_"Position %d/%d") position total)
	       (message (_"Position %d") total)))
	(setq here (point))
	(goto-char (match-beginning 0))
	(setq total (1+ total))
	(and flag (eq (point) po-start-of-msgstr) (setq position total))
	(cond ((eq (following-char) ?#)
	       (setq po-obsolete-counter (1+ po-obsolete-counter)))
	      ((looking-at po-untranslated-regexp)
	       (setq po-untranslated-counter (1+ po-untranslated-counter)))
	      (t (setq po-translated-counter (1+ po-translated-counter))))
	(goto-char here))

      ;; Make another pass just for the fuzzy entries, kind of kludgey.
      ;; FIXME: Counts will be wrong if untranslated entries are fuzzy, yet
      ;; this should not normally happen.
      (goto-char (point-min))
      (while (re-search-forward po-fuzzy-regexp nil t)
	(setq po-fuzzy-counter (1+ po-fuzzy-counter)))
      (setq po-translated-counter (- po-translated-counter po-fuzzy-counter)))

    ;; Push the results out.
    (if flag
	(message (_"\
Position %d/%d; %d translated, %d fuzzy, %d untranslated, %d obsolete")
		 position total po-translated-counter po-fuzzy-counter
		 po-untranslated-counter po-obsolete-counter)
      (message "")))
  (po-update-mode-line-string))

(defun po-redisplay ()
  "Redisplay the current entry."
  ;; FIXME: Should try to fit the whole entry on the window.  If this is not
  ;; possible, should try to fit the comment and the msgid.  Otherwise,
  ;; should try to fit the msgid.  Else, the first line of the msgid should
  ;; be at the top of the window.
  (goto-char po-start-of-msgid))

(defun po-other-window ()
  "Get the cursor into another window, out of PO mode."
  (interactive)
  (if (one-window-p t)
      (progn
	(split-window)
	(switch-to-buffer (other-buffer)))
    (other-window 1)))

(defun po-check-lock ()
  "Ensure that GNU Emacs is not currently in recursive edit for PO mode."
  (cond ((or (not po-work-buffer) (not (bufferp po-work-buffer)))
	 ;; The first time here, the work buffer has never been initialised.
	 ;; But it might also have been killed, then also release the lock.
	 (setq po-work-buffer (generate-new-buffer
			       (concat "*Edit " (buffer-name nil) "*"))
	       po-work-buffer-lock nil))
	(po-work-buffer-lock
	 ;; The work buffer exists and is already used for edition.
	 (pop-to-buffer po-work-buffer)
	 (if (y-or-n-p
	      (_"Here is your current edit.  Do you wish to abort it? "))
	     (abort-recursive-edit)
	   (error (_"Type `C-c C-c' once done, or `C-c C-k' to abort edit"))))))

(defun po-edit-out-full ()
  "Go into recursive edit for editing the PO file in fundamental mode."
  (interactive)
  (if (yes-or-no-p (_"Should I let you edit the whole PO file? "))
      (progn (let ((buffer-read-only nil))
	       (fundamental-mode)
	       (make-local-variable 'font-lock-defaults)
	       (setq font-lock-defaults '(po-font-lock-keywords))
	       (use-local-map po-subedit-mode-map)
	       (font-lock-mode)
	       (setq mode-name "PO Edit")
	       (message (_"Type `C-c C-c' once done"))
	       (recursive-edit)
	       (use-local-map nil)
	       (po-mode)))))

;;; Processing the PO file header entry.

(defun po-check-file-header ()
  "Create a missing PO mode file header, or replace an oldish one."
  (save-excursion
    (let ((buffer-read-only nil)
	  insert-flag end-of-header)
      (goto-char (point-min))
      (if (re-search-forward po-any-msgstr-regexp nil t)
	  (progn

	    ;; There is at least one entry.
	    (goto-char (match-beginning 0))
	    (previous-line 1)
	    (setq end-of-header (match-end 0))
	    (if (looking-at "msgid \"\"\n")

		;; There is indeed a PO file header.
		(if (re-search-forward "\n\"PO-Revision-Date: "
				       end-of-header t)
		    nil

		  ;; This is an oldish header.  Replace it all.
		  (goto-char end-of-header)
		  (while (> (point) (point-min))
		    (previous-line 1)
		    (insert "#~ ")
		    (beginning-of-line))
		  (beginning-of-line)
		  (setq insert-flag t))

	      ;; The first entry is not a PO file header, insert one.
	      (setq insert-flag t)))

	;; Not a single entry found.
	(setq insert-flag t))

      (goto-char (point-min))
      (and insert-flag (insert po-default-file-header "\n")))))

(defun po-replace-revision-date ()
  "Replace the revision date by current time in the PO file header."
  (if (fboundp 'format-time-string)
      (if (or (eq po-auto-replace-revision-date t)
	      (and (eq po-auto-replace-revision-date 'ask)
		   (y-or-n-p (_"May I set PO-Revision-Date? "))))
	  (save-excursion
	    (goto-char (point-min))
	    (if (re-search-forward "^\"PO-Revision-Date:.*" nil t)
		(let* ((buffer-read-only nil)
		       (time (current-time))
		       (seconds (or (car (current-time-zone time)) 0))
		       (minutes (/ (abs seconds) 60))
		       (zone (format "%c%02d:%02d"
				     (if (< seconds 0) ?- ?+)
				     (/ minutes 60)
				     (% minutes 60))))
		  (replace-match
		       (concat "\"PO-Revision-Date: "
			       (format-time-string "%Y-%m-%d %H:%M" time)
			       zone "\\n\"")
		       t t))))
	(message ""))
    (message (_"PO-Revision-Date should be adjusted..."))))

;;; Handling span of entry, entry type and entry attributes.

(defun po-find-span-of-entry ()
  "Find the extent of the PO file entry where the cursor is.  Set variables
PO-START-OF-ENTRY, PO-START-OF-MSGID, PO-START-OF-MSGSTR, PO-END-OF-ENTRY
and PO-ENTRY-TYPE to meaningful values.  Decreasing priority of type
interpretation is: obsolete, fuzzy, untranslated or translated."
  (let ((here (point)))
    (if (re-search-backward po-any-msgstr-regexp nil t)
	(progn

	  ;; After a backward match, (match-end 0) will not extend
	  ;; beyond point, in case point was *inside* the regexp.  We
	  ;; need a dependable (match-end 0), so we redo the match in
	  ;; the forward direction.
	  (re-search-forward po-any-msgstr-regexp)
	  (if (<= (match-end 0) here)
	      (progn

		;; We most probably found the msgstr of the previous
		;; entry.  The current entry then starts just after
		;; its end, save this information just in case.
		(setq po-start-of-entry (match-end 0))

		;; However, it is also possible that we are located in
		;; the crumb after the last entry in the file.  If
		;; yes, we know the middle and end of last PO entry.
		(setq po-start-of-msgstr (match-beginning 0)
		      po-end-of-entry (match-end 0))

		(if (re-search-forward po-any-msgstr-regexp nil t)
		    (progn

		      ;; We definitely were not in the crumb.
		      (setq po-start-of-msgstr (match-beginning 0)
			    po-end-of-entry (match-end 0)))

		  ;; We were in the crumb.  The start of the last PO
		  ;; file entry is the end of the previous msgstr if
		  ;; any, or else, the beginning of the file.
		  (goto-char po-start-of-msgstr)
		  (setq po-start-of-entry
			(if (re-search-backward po-any-msgstr-regexp nil t)
			    (match-end 0)
			  (point-min)))))

	    ;; The cursor was inside msgstr of the current entry.
	    (setq po-start-of-msgstr (match-beginning 0)
		  po-end-of-entry (match-end 0))
	    ;; The start of this entry is the end of the previous
	    ;; msgstr if any, or else, the beginning of the file.
	    (goto-char po-start-of-msgstr)
	    (setq po-start-of-entry
		  (if (re-search-backward po-any-msgstr-regexp nil t)
		      (match-end 0)
		    (point-min)))))

      ;; The cursor was before msgstr in the first entry in the file.
      (setq po-start-of-entry (point-min))
      (goto-char po-start-of-entry)
      ;; There is at least the PO file header, so this should match.
      (re-search-forward po-any-msgstr-regexp)
      (setq po-start-of-msgstr (match-beginning 0)
	    po-end-of-entry (match-end 0)))

    ;; Find start of msgid.
    (goto-char po-start-of-entry)
    (re-search-forward po-any-msgid-regexp)
    (setq po-start-of-msgid (match-beginning 0))

    ;; Classify the entry.
    (setq po-entry-type
	  (if (eq (following-char) ?#)
	      'obsolete
	    (goto-char po-start-of-entry)
	    (if (re-search-forward po-fuzzy-regexp po-start-of-msgid t)
		'fuzzy
	      (goto-char po-start-of-msgstr)
	      (if (looking-at po-untranslated-regexp)
		  'untranslated
		'translated))))

    ;; Put the cursor back where it was.
    (goto-char here)))

(defun po-add-attribute (name)
  "Add attribute NAME to the current entry, unless it is already there."
  (save-excursion
    (let ((buffer-read-only nil))
      (goto-char po-start-of-entry)
      (if (re-search-forward "\n#[,!] .*" po-start-of-msgid t)
	  (save-restriction
	    (narrow-to-region (match-beginning 0) (match-end 0))
	    (goto-char (point-min))
	    (if (re-search-forward (concat "\\b" name "\\b") nil t)
		nil
	      (goto-char (point-max))
	      (insert ", " name)))
	(skip-chars-forward "\n")
	(while (eq (following-char) ?#)
	  (next-line 1))
	(insert "#, " name "\n")))))

(defun po-delete-attribute (name)
  "Delete attribute NAME from the current entry, if any."
  (save-excursion
    (let ((buffer-read-only nil))
      (goto-char po-start-of-entry)
      (if (re-search-forward "\n#[,!] .*" po-start-of-msgid t)
	  (save-restriction
	    (narrow-to-region (match-beginning 0) (match-end 0))
	    (goto-char (point-min))
	    (if (re-search-forward
		 (concat "\\(\n#[,!] " name "$\\|, " name "$\\| " name ",\\)")
		 nil t)
		(replace-match "" t t)))))))

;;; Entry positionning.

(defun po-say-location-depth ()
  "Tell how many entries in the entry location stack."
  (let ((depth (length po-marker-stack)))
    (cond ((= depth 0) (message (_"Empty location stack")))
	  ((= depth 1) (message (_"One entry in location stack")))
	  (t (message (_"%d entries in location stack") depth)))))

(defun po-push-location ()
  "Stack the location of the current entry, for later return."
  (interactive)
  (po-find-span-of-entry)
  (save-excursion
    (goto-char po-start-of-msgid)
    (setq po-marker-stack (cons (point-marker) po-marker-stack)))
  (po-say-location-depth))

(defun po-pop-location ()
  "Unstack a saved location, and return to the corresponding entry."
  (interactive)
  (if po-marker-stack
      (progn
	(goto-char (car po-marker-stack))
	(setq po-marker-stack (cdr po-marker-stack))
	(po-current-entry)
	(po-say-location-depth))
    (error (_"The entry location stack is empty"))))

(defun po-exchange-location ()
  "Exchange the location of the current entry with the top of stack."
  (interactive)
  (if po-marker-stack
      (progn
	(po-find-span-of-entry)
	(goto-char po-start-of-msgid)
	(let ((location (point-marker)))
	  (goto-char (car po-marker-stack))
	  (setq po-marker-stack (cons location (cdr po-marker-stack))))
	(po-current-entry)
	(po-say-location-depth))
    (error (_"The entry location stack is empty"))))

(defun po-current-entry ()
  "Display the current entry."
  (interactive)
  (po-find-span-of-entry)
  (po-redisplay))

(defun po-first-entry-with-regexp (regexp)
  "Display the first entry in the file which msgstr matches REGEXP."
  (let ((here (point)))
    (goto-char (point-min))
    (if (re-search-forward regexp nil t)
	(progn
	  (goto-char (match-beginning 0))
	  (po-current-entry))
      (goto-char here)
      (error (_"There is no such entry")))))

(defun po-last-entry-with-regexp (regexp)
  "Display the last entry in the file which msgstr matches REGEXP."
  (let ((here (point)))
    (goto-char (point-max))
    (if (re-search-backward regexp nil t)
	(po-current-entry)
      (goto-char here)
      (error (_"There is no such entry")))))

(defun po-next-entry-with-regexp (regexp wrap)
  "Display the entry following the current entry which msgstr matches REGEXP.
If WRAP is not nil, the search may wrap around the buffer."
  (po-find-span-of-entry)
  (let ((here (point)))
    (goto-char po-end-of-entry)
    (if (re-search-forward regexp nil t)
	(progn
	  (goto-char (match-beginning 0))
	  (po-current-entry))
      (if (and wrap
	       (progn
		 (goto-char (point-min))
		 (re-search-forward regexp po-start-of-entry t)))
	  (progn
	    (goto-char (match-beginning 0))
	    (po-current-entry)
	    (message (_"Wrapping around the buffer")))
	(goto-char here)
	(error (_"There is no such entry"))))))

(defun po-previous-entry-with-regexp (regexp wrap)
  "Redisplay the entry preceding the current entry which msgstr matches REGEXP.
If WRAP is not nil, the search may wrap around the buffer."
  (po-find-span-of-entry)
  (let ((here (point)))
    (goto-char po-start-of-entry)
    (if (re-search-backward regexp nil t)
	(po-current-entry)
      (if (and wrap
	       (progn
		 (goto-char (point-max))
		 (re-search-backward regexp po-end-of-entry t)))
	  (progn
	    (po-current-entry)
	    (message (_"Wrapping around the buffer")))
	(goto-char here)
	(error (_"There is no such entry"))))))

;; Any entries.

(defun po-first-entry ()
  "Display the first entry."
  (interactive)
  (po-first-entry-with-regexp po-any-msgstr-regexp))

(defun po-last-entry ()
  "Display the last entry."
  (interactive)
  (po-last-entry-with-regexp po-any-msgstr-regexp))

(defun po-next-entry ()
  "Display the entry following the current entry."
  (interactive)
  (po-next-entry-with-regexp po-any-msgstr-regexp nil))

(defun po-previous-entry ()
  "Display the entry preceding the current entry."
  (interactive)
  (po-previous-entry-with-regexp po-any-msgstr-regexp nil))

;; Untranslated entries.

(defvar po-after-entry-regexp
  "\\(\\'\\|\\(#[ \t]*\\)?[^\"]\\)"
  "Regexp which should be true after a full msgstr string matched.")

(defvar po-untranslated-regexp
  (concat "^msgstr[ \t]*\"\"\n" po-after-entry-regexp)
  "Regexp matching a whole msgstr field, but only if active and empty.")

(defun po-next-untranslated-entry ()
  "Find the next untranslated entry, wrapping around if necessary."
  (interactive)
  (po-next-entry-with-regexp po-untranslated-regexp t))

(defun po-previous-untranslated-entry ()
  "Find the previous untranslated entry, wrapping around if necessary."
  (interactive)
  (po-previous-entry-with-regexp po-untranslated-regexp t))

(defun po-msgid-to-msgstr ()
  "Use another window to edit msgstr reinitialized with msgid."
  (interactive)
  (po-find-span-of-entry)
  (if (or (eq po-entry-type 'untranslated)
	  (eq po-entry-type 'obsolete)
	  (y-or-n-p (_"Really loose previous translation? ")))
      (progn
	(po-decrease-type-counter)
	(po-set-field nil (po-get-field t nil))
	(po-current-entry)
	(po-increase-type-counter)))
  (message ""))

;; Obsolete entries.

(defvar po-obsolete-msgstr-regexp
  "^#~?[ \t]*msgstr.*\n\\(#~?[ \t]*\".*\n\\)*"
  "Regexp matching a whole msgstr field of an obsolete entry.")

(defun po-next-obsolete-entry ()
  "Find the next obsolete entry, wrapping around if necessary."
  (interactive)
  (po-next-entry-with-regexp po-obsolete-msgstr-regexp t))

(defun po-previous-obsolete-entry ()
  "Find the previous obsolete entry, wrapping around if necessary."
  (interactive)
  (po-previous-entry-with-regexp po-obsolete-msgstr-regexp t))

;; Fuzzy entries.

(defvar po-fuzzy-regexp "^#[,!] .*fuzzy"
  "Regexp matching the string inserted by msgmerge for translations
which does not match exactly.")

(defun po-next-fuzzy-entry ()
  "Find the next fuzzy entry, wrapping around if necessary."
  (interactive)
  (po-next-entry-with-regexp po-fuzzy-regexp t))

(defun po-previous-fuzzy-entry ()
  "Find the next fuzzy entry, wrapping around if necessary."
  (interactive)
  (po-previous-entry-with-regexp po-fuzzy-regexp t))

(defun po-unfuzzy ()
  "Remove the fuzzy attribute for the current entry."
  (interactive)
  (po-find-span-of-entry)
  (cond ((eq po-entry-type 'fuzzy)
	 (po-decrease-type-counter)
	 (po-delete-attribute "fuzzy")
	 (po-current-entry)
	 (po-increase-type-counter)))
  (if po-auto-select-on-unfuzzy
      (po-auto-select-entry))
  (po-update-mode-line-string))

;; Translated entries.

(defun po-next-translated-entry ()
  "Find the next untranslated entry, wrapping around if necessary."
  (interactive)
  (if (= po-translated-counter 0)
      (error (_"There is no such entry"))
    (po-next-entry-with-regexp po-untranslated-regexp t)
    (po-find-span-of-entry)
    (while (not (eq po-entry-type 'translated))
      (po-next-entry-with-regexp po-any-msgstr-regexp t)
      (po-find-span-of-entry))))

(defun po-previous-translated-entry ()
  "Find the previous untranslated entry, wrapping around if necessary."
  (interactive)
  (if (= po-translated-counter 0)
      (error (_"There is no such entry"))
    (po-previous-entry-with-regexp po-any-msgstr-regexp t)
    (po-find-span-of-entry)
    (while (not (eq po-entry-type 'translated))
      (po-previous-entry-with-regexp po-untranslated-regexp t)
    (po-find-span-of-entry))))

;; Auto-selection feature.

(defun po-auto-select-entry ()
  "Select the next entry having the same type as the current one.
If none, wrap from the beginning of the buffer with another type,
going from untranslated to fuzzy, and from fuzzy to obsolete.
Plain translated entries are always disregarded unless there are
no entries of the other types."
  (interactive)
  (po-find-span-of-entry)
  (goto-char po-end-of-entry)
  (if (and (= po-untranslated-counter 0)
	   (= po-fuzzy-counter 0)
	   (= po-obsolete-counter 0))

      ;; All entries are plain translated.  Next entry will do, or
      ;; wrap around if there is none.
      (if (re-search-forward po-any-msgstr-regexp nil t)
	  (goto-char (match-beginning 0))
	(goto-char (point-min)))

    ;; If over a translated entry, look for an untranslated one first.
    ;; Else, look for an entry of the same type first.
    (let ((goal (if (eq po-entry-type 'translated)
		    'untranslated
		  po-entry-type)))
      (while goal

	;; Find an untranslated entry, or wrap up for a fuzzy entry.
	(if (eq goal 'untranslated)
	    (if (and (> po-untranslated-counter 0)
		     (re-search-forward po-untranslated-regexp nil t))
		(progn
		  (goto-char (match-beginning 0))
		  (setq goal nil))
	      (goto-char (point-min))
	      (setq goal 'fuzzy)))

	;; Find a fuzzy entry, or wrap up for an obsolete entry.
	(if (eq goal 'fuzzy)
	    (if (and (> po-fuzzy-counter 0)
		     (re-search-forward po-fuzzy-regexp nil t))
		(progn
		  (goto-char (match-beginning 0))
		  (setq goal nil))
	      (goto-char (point-min))
	      (setq goal 'obsolete)))

	;; Find an obsolete entry, or wrap up for an untranslated entry.
	(if (eq goal 'obsolete)
	    (if (and (> po-obsolete-counter 0)
		     (re-search-forward po-obsolete-msgstr-regexp nil t))
		(progn
		  (goto-char (match-beginning 0))
		  (setq goal nil))
	      (goto-char (point-min))
	      (setq goal 'untranslated))))))

  ;; Display this entry nicely.
  (po-current-entry))

;;; Killing and yanking fields.

(eval-and-compile
  (fset 'po-buffer-substring
	(if (fboundp 'buffer-substring-no-properties)
	    (symbol-function 'buffer-substring-no-properties)
	  (symbol-function 'buffer-substring))))

(eval-and-compile
  (if (fboundp 'kill-new)

      (fset 'po-kill-new (symbol-function 'kill-new))

    (defun po-kill-new (string)
      "Push STRING onto the kill ring, for Emacs 18 where kill-new is missing."
      (po-check-lock)
      (save-excursion
	(set-buffer po-work-buffer)
	(erase-buffer)
	(insert string)
	(kill-region (point-min) (point-max))))))

(defun po-extract-unquoted (buffer start end)
  "Extract and return the unquoted string in BUFFER going from START to END.
Crumb preceding or following the quoted string is ignored."
  (po-check-lock)
  (save-excursion
    (set-buffer po-work-buffer)
    (erase-buffer)
    (insert-buffer-substring buffer start end)
    ;; Remove preceding crumb.
    (goto-char (point-min))
    (search-forward "\"")
    (delete-region (point-min) (point))
    ;; Remove following crumb.
    (goto-char (point-max))
    (search-backward "\"")
    (delete-region (point) (point-max))
    ;; Glue concatenated strings.
    (goto-char (point-min))
    (while (re-search-forward "\"[ \t]*\\\\?\n\\(#~?\\)?[ \t]*\"" nil t)
      (replace-match "" t t))
    ;; Remove escaped newlines.
    (goto-char (point-min))
    (while (re-search-forward "\\\\[ \t]*\n" nil t)
      (replace-match "" t t))
    ;; Unquote individual characters.
    (goto-char (point-min))
    (while (re-search-forward "\\\\[\"abfnt\\0-7]" nil t)
      (cond ((eq (preceding-char) ?\") (replace-match "\"" t t))
	    ((eq (preceding-char) ?a) (replace-match "\a" t t))
	    ((eq (preceding-char) ?b) (replace-match "\b" t t))
	    ((eq (preceding-char) ?f) (replace-match "\f" t t))
	    ((eq (preceding-char) ?n) (replace-match "\n" t t))
	    ((eq (preceding-char) ?t) (replace-match "\t" t t))
	    ((eq (preceding-char) ?\\) (replace-match "\\" t t))
	    (t (let ((value (- (preceding-char) ?0)))
		 (replace-match "" t t)
		 (while (looking-at "[0-7]")
		   (setq value (+ (* 8 value) (- (following-char) ?0)))
		   (replace-match "" t t))
		 (insert value)))))
    (buffer-string)))

(defun po-eval-requoted (form prefix obsolete)
  "Eval FORM, which inserts a string, and return the string fully requoted.
If PREFIX, precede the result with its contents.  If OBSOLETE, comment all
generated lines in the returned string.  Evaluating FORM should insert the
wanted string in the buffer which is current at the time of evaluation.
If FORM is itself a string, then this string is used for insertion."
  (po-check-lock)
  (save-excursion
    (set-buffer po-work-buffer)
    (erase-buffer)
    (if (stringp form)
	(insert form)
      (push-mark)
      (eval form))
    (goto-char (point-min))
    (let ((multi-line (re-search-forward "[^\n]\n+[^\n]" nil t)))
      (goto-char (point-min))
      (while (re-search-forward "[\"\a\b\f\n\t\\]" nil t)
	(cond ((eq (preceding-char) ?\") (replace-match "\\\"" t t))
	      ((eq (preceding-char) ?\a) (replace-match "\\a" t t))
	      ((eq (preceding-char) ?\b) (replace-match "\\b" t t))
	      ((eq (preceding-char) ?\f) (replace-match "\\f" t t))
	      ((eq (preceding-char) ?\n)
	       (replace-match (if (or (not multi-line) (eobp))
				  "\\n"
				"\\n\"\n\"")
			      t t))
	      ((eq (preceding-char) ?\t) (replace-match "\\t" t t))
	      ((eq (preceding-char) ?\\) (replace-match "\\\\" t t))))
      (goto-char (point-min))
      (if prefix (insert prefix " "))
      (insert (if multi-line "\"\"\n\"" "\""))
      (goto-char (point-max))
      (insert "\"")
      (if prefix (insert "\n"))
      (if obsolete
	  (progn
	    (goto-char (point-min))
	    (while (not (eobp))
	      (or (eq (following-char) ?\n) (insert "#~ "))
	      (search-forward "\n"))))
      (buffer-string))))

(defun po-get-field (msgid kill)
  "Extract and return the unquoted msgstr string, unless MSGID selects msgid.
If KILL, then add the unquoted string to the kill ring."
  (let ((string (if msgid
		    (po-extract-unquoted (current-buffer)
					 po-start-of-msgid po-start-of-msgstr)
		  (po-extract-unquoted (current-buffer)
				       po-start-of-msgstr po-end-of-entry))))
    (if kill (po-kill-new string))
    string))

(defun po-set-field (msgid form)
  "Replace the current msgstr, unless MSGID, using FORM to get a string.
If MSGID is true, replace the current msgid instead.  In either case,
evaluating FORM should insert the wanted string in the current buffer.
If FORM is itself a string, then this string is used for insertion.
The string is properly requoted before the replacement occurs.

Returns `nil' if the buffer has not been modified, for if the new field
described by FORM is merely identical to the field already in place."
  (let ((string (po-eval-requoted form (if msgid "msgid" "msgstr")
				  (eq po-entry-type 'obsolete))))
    (save-excursion
      (goto-char po-start-of-entry)
      (re-search-forward (if msgid po-any-msgid-regexp po-any-msgstr-regexp)
			 po-end-of-entry)
      (and (not (string-equal (po-buffer-substring (match-beginning 0)
						   (match-end 0))
			      string))
	   (let ((buffer-read-only nil))
	     (or msgid (po-decrease-type-counter))
	     (replace-match string t t)
	     (goto-char po-start-of-msgid)
	     (po-current-entry)
	     (or msgid (po-increase-type-counter))
	     t)))))

(defun po-kill-ring-save-msgstr ()
  "Push the msgstr string from current entry on the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-get-field nil t))

(defun po-kill-msgstr ()
  "Empty the msgstr string from current entry, pushing it on the kill ring."
  (interactive)
  (po-kill-ring-save-msgstr)
  (po-decrease-type-counter)
  (po-set-field nil "")
  (po-current-entry)
  (po-increase-type-counter))

(defun po-yank-msgstr ()
  "Replace the current msgstr string by the top of the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-decrease-type-counter)
  (po-set-field nil (if (eq last-command 'yank) '(yank-pop 1) '(yank)))
  (po-current-entry)
  (po-increase-type-counter)
  (setq this-command 'yank))

(defun po-fade-out-entry ()
  "Mark an active entry as fuzzy; obsolete a fuzzy or untranslated entry;
or completely delete an obsolete entry, saving its msgstr on the kill ring."
  (interactive)
  (po-check-lock)
  (po-find-span-of-entry)

  (cond ((eq po-entry-type 'translated)
	 (po-decrease-type-counter)
	 (po-add-attribute "fuzzy")
	 (po-current-entry)
	 (po-increase-type-counter))

	((or (eq po-entry-type 'fuzzy)
	     (eq po-entry-type 'untranslated))
	 (if (yes-or-no-p (_"Should I really obsolete this entry? "))
	     (progn
	       (po-decrease-type-counter)
	       (save-excursion
		 (save-restriction
		   (narrow-to-region po-start-of-entry po-end-of-entry)
		   (let ((buffer-read-only nil))
		     (goto-char (point-min))
		     (skip-chars-forward "\n")
		     (while (not (eobp))
		       (insert "#~ ")
		       (search-forward "\n")))))
	       (po-current-entry)
	       (po-increase-type-counter)))
	 (message ""))

	((eq po-entry-type 'obsolete)
	 (po-decrease-type-counter)
	 (po-update-mode-line-string)
	 (po-get-field nil t)
	 (let ((buffer-read-only nil))
	   (delete-region po-start-of-entry po-end-of-entry))
	 (goto-char po-start-of-entry)
	 (if (re-search-forward po-any-msgstr-regexp nil t)
	     (goto-char (match-beginning 0))
	   (re-search-backward po-any-msgstr-regexp nil t))
	 (po-current-entry)
	 (message ""))))

;;; Killing and yanking comments.

(defvar po-active-comment-regexp
  "^\\(#\n\\|# .*\n\\)+"
  "Regexp matching the whole editable comment part of an active entry.")

(defvar po-obsolete-comment-regexp
  "^\\(#~? #\n\\|#~? # .*\n\\)+"
  "Regexp matching the whole editable comment part of an obsolete entry.")

(defun po-get-comment (kill-flag)
  "Extract and return the editable comment string, uncommented.
If KILL-FLAG, then add the unquoted comment to the kill ring."
  (po-check-lock)
  (let ((buffer (current-buffer))
	(obsolete (eq po-entry-type 'obsolete)))
    (save-excursion
      (goto-char po-start-of-entry)
      (if (re-search-forward (if obsolete po-obsolete-comment-regexp
			         po-active-comment-regexp)
			     po-end-of-entry t)
	  (progn
	    (set-buffer po-work-buffer)
	    (erase-buffer)
	    (insert-buffer-substring buffer (match-beginning 0) (match-end 0))
	    (goto-char (point-min))
	    (while (not (eobp))
	      (if (looking-at (if obsolete "#~? # ?" "#~? ?"))
		  (replace-match "" t t))
	      (forward-line 1))
	    (and kill-flag (copy-region-as-kill (point-min) (point-max)))
	    (buffer-string))
	""))))

(defun po-set-comment (form)
  "Using FORM to get a string, replace the current editable comment.
Evaluating FORM should insert the wanted string in the current buffer.
If FORM is itself a string, then this string is used for insertion.
The string is properly recommented before the replacement occurs."
  (po-check-lock)
  (let ((obsolete (eq po-entry-type 'obsolete))
	string)
    (save-excursion
      (set-buffer po-work-buffer)
      (erase-buffer)
      (if (stringp form)
	  (insert form)
	(push-mark)
	(eval form))
      (if (not (or (bobp) (= (preceding-char) ?\n)))
	  (insert "\n"))
      (goto-char (point-min))
      (while (not (eobp))
	(insert (if (= (following-char) ?\n)
		    (if obsolete "#~ #" "#")
		  (if obsolete "#~ # " "# ")))
	(search-forward "\n"))
      (setq string (buffer-string)))
    (goto-char po-start-of-entry)
    (if (re-search-forward
	 (if obsolete po-obsolete-comment-regexp po-active-comment-regexp)
	 po-end-of-entry t)
	(if (not (string-equal (po-buffer-substring (match-beginning 0)
						    (match-end 0))
			       string))
	    (let ((buffer-read-only nil))
	      (replace-match string t t)))
      (skip-chars-forward " \t\n")
      (let ((buffer-read-only nil))
	(insert string))))
  (po-current-entry))

(defun po-kill-ring-save-comment ()
  "Push the msgstr string from current entry on the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-get-comment t))

(defun po-kill-comment ()
  "Empty the msgstr string from current entry, pushing it on the kill ring."
  (interactive)
  (po-kill-ring-save-comment)
  (po-set-comment "")
  (po-redisplay))

(defun po-yank-comment ()
  "Replace the current comment string by the top of the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-set-comment (if (eq last-command 'yank) '(yank-pop 1) '(yank)))
  (setq this-command 'yank)
  (po-redisplay))

;;; Editing translations.

(defun po-edit-string (string expand-tabs)
  "Edit STRING recursively in a pop-up buffer, return the edited string.
Run po-subedit-mode-hook first prior to beginning edition.  If recursive edit
is aborted, return nil instead.  If EXPAND-TABS, expand tabs to spaces."

  (po-check-lock)
  (let ((po-work-buffer-lock t))
    (prog1
	(save-window-excursion
	  (pop-to-buffer po-work-buffer)
	  (erase-buffer)
	  (run-hooks 'po-subedit-mode-hook)
	  (insert string "<")
	  (goto-char (point-min))
	  (condition-case nil
	      (let ((indent-tabs-mode (if expand-tabs nil indent-tabs-mode)))
		(use-local-map po-subedit-mode-map)
		(message
		 (_"Type `C-c C-c' once done, or `C-c C-k' to abort edit"))
		(recursive-edit)
		(goto-char (point-max))
		(skip-chars-backward " \t\n")
		(if (eq (preceding-char) ?<)
		    (delete-region (1- (point)) (point-max)))
		(buffer-string))
	    (quit nil)))
      (bury-buffer po-work-buffer))))

(defun po-edit-comment ()
  "Use another window to edit the current translator comment."
  (interactive)
  (po-find-span-of-entry)
;  ;; Try showing all of msgid in the upper window while editing.
;  (goto-char po-start-of-entry)
;  (re-search-forward po-any-msgid-regexp)
;  (backward-char 1)
;  (recenter -1)
  (let ((string (po-edit-string (po-get-comment nil) nil)))
    (and string (po-set-comment string))
    (po-redisplay)))

(defun po-edit-msgstr ()
  "Use another window to edit the current msgstr."
  (interactive)
  (po-find-span-of-entry)
  (and po-auto-edit-with-msgid
       (eq po-entry-type 'untranslated)
       (po-msgid-to-msgstr))
  (if po-highlighting
      (save-excursion
	(goto-char po-start-of-entry)
	(re-search-forward po-any-msgid-regexp nil t)
	(let ((end (1- (match-end 0))))
	  (goto-char (match-beginning 0))
	  (re-search-forward "msgid +" nil t)
	  (po-highlight (point) end))))
  (let ((string (po-edit-string (po-get-field nil nil) t)))
    (and po-highlighting (po-dehighlight))
    (if string
	(let (replaced)
	  (po-decrease-type-counter)
	  (setq replaced (po-set-field nil string))
	  (po-current-entry)
	  (if (and replaced
		   po-auto-fuzzy-on-edit
		   (eq po-entry-type 'translated))
	      (progn
		 (po-add-attribute "fuzzy")
		 (po-current-entry)))
	  (po-increase-type-counter)))))

;;; String normalization and searching.

(defun po-normalize-old-style (explain)
  "Normalize old gettext style fields using K&R C multiline string syntax.
To minibuffer messages sent while normalizing, add the EXPLAIN string."
  (let ((here (point-marker))
	(counter 0)
	(buffer-read-only nil))
    (goto-char (point-min))
    (message (_"Normalizing %d, %s") counter explain)
    (while (re-search-forward
	    "\\(^#?[ \t]*msg\\(id\\|str\\)[ \t]*\"\\|[^\" \t][ \t]*\\)\\\\\n"
	    nil t)
      (if (= (% counter 10) 0)
	  (message (_"Normalizing %d, %s") counter explain))
      (replace-match "\\1\"\n\"" t nil)
      (setq counter (1+ counter)))
    (goto-char here)
    (message (_"Normalizing %d...done") counter)))

(defun po-normalize-field (msgid explain)
  "Normalize all msgstr's, or msgid's if MSGID.
To minibuffer messages sent while normalizing, add the EXPLAIN string."
  (let ((here (point-marker))
	(counter 0))
    (goto-char (point-min))
    (while (re-search-forward po-any-msgstr-regexp nil t)
      (if (= (% counter 10) 0)
	  (message (_"Normalizing %d, %s") counter explain))
      (goto-char (match-beginning 0))
      (po-find-span-of-entry)
      (po-set-field msgid (po-get-field msgid nil))
      (goto-char po-end-of-entry)
      (setq counter (1+ counter)))
    (goto-char here)
    (message (_"Normalizing %d...done") counter)))

(defun po-normalize ()
  "Normalize all entries in the PO file."
  (interactive)
  (po-normalize-old-style (_"pass 1/3"))
  (po-normalize-field t   (_"pass 2/3"))
  (po-normalize-field nil (_"pass 3/3"))
  ;; The last PO file entry has just been processed.
  (if (not (= po-end-of-entry (point-max)))
      (let ((buffer-read-only nil))
	(kill-region po-end-of-entry (point-max))))
  ;; A bizarre format might have fooled the counters, so recompute
  ;; them to make sure their value is dependable.
  (po-compute-counters nil))

;;; Original program sources as context.

(defun po-show-source-path ()
  "Echo the current source search path in the message area."
  (let ((path po-search-path)
	(string (_"Path is:")))
    (while path
      (setq string (concat string " " (car (car path)))
	    path (cdr path)))
    (message string)))

(defun po-consider-source-path (directory)
  "Add a given DIRECTORY, requested interactively, to the source search path."
  (interactive (_"DDirectory for search path: "))
  (setq po-search-path (cons (list (if (string-match "/$" directory)
					 directory
				       (concat directory "/")))
			     po-search-path))
  (setq po-reference-check 0)
  (po-show-source-path))

(defun po-ignore-source-path ()
  "Delete a directory, selected with completion, from the source search path."
  (interactive)
  (setq po-search-path
	(delete (list (completing-read (_"Directory to remove? ")
				       po-search-path nil t))
		po-search-path))
  (setq po-reference-check 0)
  (po-show-source-path))

(defun po-ensure-source-references ()
  "Extract all references into a list, with paths resolved, if necessary."
  (po-find-span-of-entry)
  (if (= po-start-of-entry po-reference-check)
      ()
    (setq po-reference-alist nil)
    (save-excursion
      (goto-char po-start-of-entry)
      (if (re-search-forward "^#:" po-start-of-msgid t)
	  (while (looking-at "\\(\n#:\\)? *\\([^: ]+\\):\\([0-9]+\\)")
	    (goto-char (match-end 0))
	    (let* ((name (po-buffer-substring (match-beginning 2)
					      (match-end 2)))
		   (line (po-buffer-substring (match-beginning 3)
					      (match-end 3)))
		   (path po-search-path)
		   file)
	      (while (and (progn (setq file (concat (car (car path)) name))
				 (not (file-exists-p file)))
			  path)
		(setq path (cdr path)))
	      (if path
		  (setq po-reference-alist
			(cons (list (concat file ":" line)
				    file
				    (string-to-int line))
			      po-reference-alist)))))))
    (setq po-reference-alist (nreverse po-reference-alist)
	  po-reference-cursor po-reference-alist
	  po-reference-check po-start-of-entry)))

(defun po-show-source-context (triplet)
  "Show the source context given a TRIPLET which is (PROMPT FILE LINE)."
  (find-file-other-window (car (cdr triplet)))
  (goto-line (car (cdr (cdr triplet))))
  (other-window 1)
  (let ((maximum 0)
	position
	(cursor po-reference-alist))
    (while (not (eq triplet (car cursor)))
      (setq maximum (1+ maximum)
	    cursor (cdr cursor)))
    (setq position (1+ maximum)
	  po-reference-cursor cursor)
    (while cursor
      (setq maximum (1+ maximum)
	    cursor (cdr cursor)))
    (message (_"Displaying %d/%d: \"%s\"") position maximum (car triplet))))

(defun po-cycle-source-reference ()
  "Display some source context for the current entry.
If the command is repeated many times in a row, cycle through contexts."
  (interactive)
  (po-ensure-source-references)
  (if po-reference-cursor
      (po-show-source-context
       (car (if (eq last-command 'po-cycle-source-reference)
		(or (cdr po-reference-cursor) po-reference-alist)
	      po-reference-cursor)))
    (error (_"No resolved source references"))))

(defun po-select-source-reference ()
  "Select one of the available source contexts for the current entry."
  (interactive)
  (po-ensure-source-references)
  (if po-reference-alist
      (po-show-source-context
       (assoc
	(completing-read (_"Which source context? ") po-reference-alist nil t)
	po-reference-alist))
    (error (_"No resolved source references"))))

;;; Program sources strings though tags table.

;;; C mode.

;;; A few long string cases (submitted by Ben Pfaff).

;; #define string "This is a long string " \
;; "that is continued across several lines " \
;; "in a macro in order to test \\ quoting\\" \
;; "\\ with goofy strings.\\"

;; char *x = "This is just an ordinary string "
;; "continued across several lines without needing "
;; "to use \\ characters at end-of-line.";

;; char *y = "Here is a string continued across \
;; several lines in the manner that was sanctioned \
;; in K&R C compilers and still works today, \
;; even though the method used above is more esthetic.";

;;; End of long string cases.

(defun po-find-c-string (keywords)
  "Find the next C string, excluding those marked by any of KEYWORDS.
Returns (START . END) for the found string, or (nil . nil) if none found."
  (let (start end)
    (while (and (not start)
		(re-search-forward "\\([\"']\\|/\\*\\)" nil t))
      (cond ((= (preceding-char) ?*)
	     ;; Disregard comments.
	     (search-forward "*/"))

	    ((= (preceding-char) ?\')
	     ;; Disregard character constants.
	     (forward-char (if (= (following-char) ?\\) 3 2)))

	    ((save-excursion
	       (beginning-of-line)
	       (looking-at "^# *\\(include\\|line\\)"))
	     ;; Disregard lines being #include or #line directives.
	     (end-of-line))

	    ;; Else, find the end of the (possibly concatenated) string.
	    (t (setq start (1- (point))
		     end nil)
	       (while (not end)
		 (cond ((= (following-char) ?\")
			(if (looking-at "\"[ \t\n\\\\]*\"")
			    (goto-char (match-end 0))
			  (forward-char 1)
			  (setq end (point))))
		       ((= (following-char) ?\\) (forward-char 2))
		       (t (skip-chars-forward "^\"\\\\"))))

	       ;; Check before string for keyword and opening parenthesis.
	       (goto-char start)
	       (skip-chars-backward " \n\t")
	       (if (= (preceding-char) ?\()
		   (progn
		     (backward-char 1)
		     (skip-chars-backward " \n\t")
		     (let ((end-keyword (point)))
		       (skip-chars-backward "_A-Za-z0-9")
		       (if (member (list (po-buffer-substring (point)
							      end-keyword))
				   keywords)

			   ;; Disregard already marked strings.
			   (progn
			     (goto-char end)
			     (setq start nil
				   end nil)))))))))

    ;; Return the found string, if any.
    (cons start end)))

(defun po-mark-c-string (start end keyword)
  "Mark the C string, from START to END, with KEYWORD.
Return the adjusted value for END."
  (goto-char end)
  (insert ")")
  (goto-char start)
  (insert keyword)
  (if (not (string-equal keyword "_"))
      (progn (insert " ") (setq end (1+ end))))
  (insert "(")
  (+ end 2 (length keyword)))

;;; Emacs LISP mode.

(defun po-find-emacs-lisp-string (keywords)
  "Find the next Emacs LISP string, excluding those marked by any of KEYWORDS.
Returns (START . END) for the found string, or (nil . nil) if none found."
  (let (start end)
    (while (and (not start)
		(re-search-forward "[;\"?]" nil t))

      (cond ((= (preceding-char) ?\;)
	     ;; Disregard comments.
	     (search-forward "\n"))

	    ((= (preceding-char) ?\?)
	     ;; Disregard character constants.
	     (forward-char (if (= (following-char) ?\\) 2 1)))

	    ;; Else, find the end of the string.
	    (t (setq start (1- (point)))
	       (while (not (= (following-char) ?\"))
		 (skip-chars-forward "^\"\\\\")
		 (if (= (following-char) ?\\) (forward-char 2)))
	       (forward-char 1)
	       (setq end (point))

	       ;; Check before string for keyword and opening parenthesis.
	       (goto-char start)
	       (skip-chars-backward " \n\t")
	       (let ((end-keyword (point)))
		 (skip-chars-backward "-_A-Za-z0-9")
		 (if (and (= (preceding-char) ?\()
			  (member (list (po-buffer-substring (point)
							     end-keyword))
				  keywords))

		     ;; Disregard already marked strings.
		     (progn
		       (goto-char end)
		       (setq start nil
			     end nil)))))))

    ;; Return the found string, if any.
    (cons start end)))

(defun po-mark-emacs-lisp-string (start end keyword)
  "Mark the Emacs LISP string, from START to END, with KEYWORD.
Return the adjusted value for END."
  (goto-char end)
  (insert ")")
  (goto-char start)
  (insert "(" keyword)
  (if (not (string-equal keyword "_"))
      (progn (insert " ") (setq end (1+ end))))
  (+ end 2 (length keyword)))

;;; Processing generic to all programming modes.

(eval-and-compile
  (autoload 'visit-tags-table-buffer "etags"))

; This is part of my highlighting code -- it may no longer be necessary, but
;  I'm keeping it around just in case.
;(defvar po-string-overlay (make-overlay 0 0 nil nil 'rear-advance))
;(overlay-put po-string-overlay 'detachable nil)
;(overlay-put po-string-overlay 'face 'region)

(defun po-tags-search (restart)
  (interactive "P")
  "Find an unmarked translatable string through all files in tags table.
Disregard some simple strings which are most probably non-translatable.
With prefix argument, restart search at first file."

  ;; Take care of restarting the search if necessary.
  (if restart (setq po-next-file-list nil))

  ;; Loop doing things until an interesting string is found.
  (let ((keywords po-keywords)
	found buffer start
	(end po-string-end))
    (while (not found)

      ;; Reinitialize the source file list if necessary.
      (if (not po-next-file-list)
	  (progn
	    (setq po-next-file-list
		  (save-excursion
		    (visit-tags-table-buffer)
		    (copy-sequence (tags-table-files))))
	    (or po-next-file-list (error (_"No files to process")))
	    (setq end nil)))

      ;; Try finding a string after resuming the search position.
      (message (_"Scanning %s...") (car po-next-file-list))
      (save-excursion
	(setq buffer (find-file-noselect (car po-next-file-list)))
	(set-buffer buffer)
	(goto-char (or end (point-min)))

	(cond ((or (string-equal mode-name "C") (string-equal mode-name "C++"))
	       (let ((pair (po-find-c-string keywords)))
		 (setq start (car pair)
		       end (cdr pair))))
	      ((string-equal mode-name "Emacs-Lisp")
	       (let ((pair (po-find-emacs-lisp-string keywords)))
		 (setq start (car pair)
		       end (cdr pair))))
	      (t (message (_"Unknown source mode for PO mode, skipping..."))
		 (setq start nil
		       end nil))))

; This is part of my highlighting code -- it may no longer be necessary, but
;  I'm keeping it around just in case.
;     (if (and start end) (move-overlay po-string-overlay start end buffer))

      ;; Advance to next file if no string was found.
      (if (not start)
	  (progn
	    (setq po-next-file-list (cdr po-next-file-list))
	    (if (not po-next-file-list) (error (_"All files processed")))
	    (setq end nil))

	;; Push the string just found string into the work buffer for study.
	(po-extract-unquoted buffer start end)
	(save-excursion
	  (set-buffer po-work-buffer)
	  (goto-char (point-min))

	  ;; Do not disregard if at least three letters in a row.
	  (if (re-search-forward "[A-Za-z][A-Za-z][A-Za-z]" nil t)
	      (setq found t)

	    ;; Disregard if two letters, and more punctuations than letters.
	    (if (re-search-forward "[A-Za-z][A-Za-z]" nil t)
		(let ((total (buffer-size)))
		  (goto-char (point-min))
		  (while (re-search-forward "[A-Za-z]+" nil t)
		    (replace-match "" t t))
		  (if (< (* 2 (buffer-size)) total)
		      (setq found t))))

	    ;; Disregard if single letters or no letters at all.
	    ))))

    ;; Ensure the string is being displayed.

    (if (one-window-p t) (split-window) (other-window 1))
    (switch-to-buffer buffer)
    (goto-char start)
    (or (pos-visible-in-window-p start) (recenter '(nil)))
    (if (pos-visible-in-window-p end)
	(goto-char end)
      (goto-char end)
      (recenter -1))
    (and po-highlighting (po-highlight start end))
    (other-window 1)

    ;; Save the string for later commands.
    (message (_"Scanning %s...done") (car po-next-file-list))
    (setq po-string-start start
	  po-string-end end)))

(defun po-mark-found-string (keyword)
  "Mark last found string in program sources as translatable, using KEYWORD."
  (and po-highlighting (po-dehighlight))
  (let ((buffer (find-file-noselect (car po-next-file-list)))
	(start po-string-start)
	(end po-string-end)
	line string)

    ;; Mark string in program sources.
    (setq string (po-extract-unquoted buffer start end))
    (save-excursion
      (set-buffer buffer)
      (setq line (count-lines (point-min) start)
	    end (cond ((or (string-equal mode-name "C")
			   (string-equal mode-name "C++"))
		       (po-mark-c-string start end keyword))
		      ((string-equal mode-name "Emacs-Lisp")
		       (po-mark-emacs-lisp-string start end keyword))
		      (t (error (_"Cannot mark in unknown source mode"))))))
    (setq po-string-end end)

    ;; Add PO file entry.
    (let ((buffer-read-only nil))
      (goto-char (point-max))
      (insert "\n"
	      (format "#: %s:%d\n" (car po-next-file-list) line)
	      (po-eval-requoted string "msgid" nil)
	      "msgstr \"\"\n")
      (previous-line 1)
      (setq po-untranslated-counter (1+ po-untranslated-counter))
      (po-update-mode-line-string))))

(defun po-mark-translatable ()
  (interactive)
  "Mark last found string in program sources as translatable, using `_'."
  (if (and po-string-start po-string-end)
      (progn
	(po-mark-found-string "_")
	(setq po-string-start nil))
    (error (_"No such string"))))

(defun po-select-mark-and-mark (arg)
  (interactive "P")
  "Mark last found string in program sources as translatable, ask for keywoard,
using completion.  With prefix argument, just ask the name of a preferred
keyword for subsequent commands, also added to possible completions."
  (if arg
      (let ((keyword (list (read-from-minibuffer (_"Keyword: ")))))
	(setq po-keywords (cons keyword (delete keyword po-keywords))))
    (if (and po-string-start po-string-end)
	(let* ((default (car (car po-keywords)))
	       (keyword (completing-read (format (_"Mark with keyword? [%s] ")
						 default)
					 po-keywords nil t )))
	  (if (string-equal keyword "") (setq keyword default))
	  (po-mark-found-string keyword)
	  (setq po-string-start nil))
      (error (_"No such string")))))

;;; Miscellaneous features.

(eval-and-compile
  (fset 'po-read-event
	(cond ((fboundp 'read-event) 'read-event) ; GNU Emacs
	      ((fboundp 'next-command-event) 'next-command-event) ; XEmacs
	      (t 'read-char))))		; older Emacses

(defun po-help ()
  "Provide an help window for PO mode."
  (interactive)
  (po-check-lock)
  (save-window-excursion
    (switch-to-buffer po-work-buffer)
    (erase-buffer)
    (insert po-help-display-string)
    (delete-other-windows)
    (goto-char (point-min))
    (message (_"Type any character to continue"))
    (po-read-event))
  (bury-buffer po-work-buffer))

(defun po-undo ()
  "Undo the last change to the PO file."
  (interactive)
  (let ((buffer-read-only nil))
    (undo))
  (po-compute-counters nil))

(defun po-statistics ()
  "Say how many entries in each category, and the current position."
  (interactive)
  (po-compute-counters t))

(defun po-validate ()
  "Use `msgfmt' for validating the current PO file contents."
  (interactive)

  ;; If modifications were done already, change the last revision date.
  (if (buffer-modified-p)
      (po-replace-revision-date))

  ;; This `let' is for protecting the previous value of compile-command.
  (let ((compile-command (concat po-msgfmt-program
				 " --statistics -c -v -o /dev/null "
				 buffer-file-name)))
    (compile compile-command)))

(defun po-guess-archive-name ()
  "Return the ideal file name for this PO file in the central archives."
  (let (start-of-header end-of-header package version team)
    (save-excursion
      ;; Find the PO file header entry.
      (goto-char (point-min))
      (re-search-forward po-any-msgstr-regexp)
      (setq start-of-header (match-beginning 0)
	    end-of-header (match-end 0))
      ;; Get the package and version.
      (goto-char start-of-header)
      (if (re-search-forward
	   "\n\"Project-Id-Version:\\( GNU\\)? \\([^\n ]+\\) \\([^\n ]+\\)\\\\n\"$"
	   end-of-header t)
	  (setq package (buffer-substring (match-beginning 2) (match-end 2))
		version (buffer-substring (match-beginning 3) (match-end 3))))
      (if (or (not package) (string-equal package "PACKAGE")
	      (not version) (string-equal version "VERSION"))
	  (error (_"Project-Id-Version field does not have a proper value")))
      ;; Get the team.
      (goto-char start-of-header)
      (if (re-search-forward "\n\"Language-Team:.*<\\(.*\\)@li.org>\\\\n\"$"
			     end-of-header t)
	  (setq team (buffer-substring (match-beginning 1) (match-end 1))))
      (if (or (not team) (string-equal team "LL"))
	  (error (_"Language-Team field does not have a proper value")))
      ;; Compose the name.
      (concat package "-" version "." team ".po"))))

(defun po-guess-team-address ()
  "Return the team address related to this PO file."
  (let (team)
    (save-excursion
      (goto-char (point-min))
      (re-search-forward po-any-msgstr-regexp)
      (goto-char (match-beginning 0))
      (if (re-search-forward
	   "\n\"Language-Team: +\\(.*<\\(.*\\)@li.org>\\)\\\\n\"$"
	   (match-end 0) t)
	  (setq team (buffer-substring (match-beginning 2) (match-end 2))))
      (if (or (not team) (string-equal team "LL"))
	  (error (_"Language-Team field does not have a proper value")))
      (buffer-substring (match-beginning 1) (match-end 1)))))

(defun po-send-mail ()
  "Start composing a letter, possibly including the current PO file."
  (interactive)
  (let* ((team-flag (y-or-n-p
		     (_"Write to team (instead of translation project)? ")))
	 (address (if team-flag
		      (po-guess-team-address)
		    po-translation-project-address)))
    (if (not (y-or-n-p (_"Include current PO file? ")))
	(apply po-compose-mail-function address
	       (read-string (_"Subject? ")) nil)
      (if (buffer-modified-p)
	  (error (_"The file is not even saved, you did not validate it.")))
      (if (and (y-or-n-p (_"You validated (`V') this file, didn't you? "))
	       (or (zerop po-untranslated-counter)
		   (y-or-n-p
		    (format (_"%d entries are untranslated, include anyway? ")
			    po-untranslated-counter)))
	       (or (zerop po-fuzzy-counter)
		   (y-or-n-p
		    (format (_"%d entries are still fuzzy, include anyway? ")
			    po-fuzzy-counter)))
	       (or (zerop po-obsolete-counter)
		   (y-or-n-p
		    (format (_"%d entries are obsolete, include anyway? ")
			    po-obsolete-counter))))
	  (let ((buffer (current-buffer))
		(name (po-guess-archive-name))
		(transient-mark-mode nil))
	    (apply po-compose-mail-function address
		   (if team-flag
		       (read-string (_"Subject? "))
		     (format "TP-Robot %s" name))
		   nil)
	    (goto-char (point-min))
	    (re-search-forward
	     (concat "^" (regexp-quote mail-header-separator) "\n"))
	    (save-excursion
	      (insert-buffer buffer)
	      (shell-command-on-region
	       (region-beginning) (region-end)
	       (concat po-gzip-uuencode-command " " name ".gz") t))))))
  (message ""))

(defun po-confirm-and-quit ()
  "Confirm if quit should be attempted and then, do it.
This is a failsafe.  Confirmation is asked if only the real quit would not."
  (interactive)
  (if (or (buffer-modified-p)
	  (> po-untranslated-counter 0)
	  (> po-fuzzy-counter 0)
	  (> po-obsolete-counter 0)
	  (y-or-n-p (_"Really quit editing this PO file? ")))
      (po-quit))
  (message ""))

(defun po-quit ()
  "Save the PO file and kill buffer.  However, offer validation if
appropriate and ask confirmation if untranslated strings remain."
  (interactive)
  (let ((quit t))

    ;; Offer validation of newly modified entries.
    (if (and (buffer-modified-p)
	     (not (y-or-n-p (_"File was modified; skip validation step? "))))
	(progn
	  (message "")
	  (po-validate)
	  ;; If we knew that the validation was all successful, we should
	  ;; just quit.  But since we do not know yet, as the validation
	  ;; might be asynchronous with PO mode commands, the safest is to
	  ;; stay within PO mode, even if this implies that another
	  ;; `po-quit' command will be later required to exit for true.
	  (setq quit nil)))

    ;; Offer to work on untranslated entries.
    (if (and quit
	     (or (> po-untranslated-counter 0)
		 (> po-fuzzy-counter 0)
		 (> po-obsolete-counter 0))
	     (not (y-or-n-p (_"Unprocessed entries remain; quit anyway? "))))
	(progn
	  (setq quit nil)
	  (po-auto-select-entry)))

    ;; Clear message area.
    (message "")

    ;; Or else, kill buffers and quit for true.
    (if quit
	(progn
	  (if (buffer-modified-p)
	      (po-replace-revision-date))
	  (save-buffer)
	  ;; kill-buffer-hook is used if it exists: avoid overkilling! :-)
	  (or (boundp 'kill-buffer-hook) (kill-buffer po-work-buffer))
	  (kill-buffer (current-buffer))))))

(provide 'po-mode)
;;; po-mode.el ends here
