/* Lisp functions for i18n through GNU gettext.
   Copyright (C) 1998 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#ifdef USE_NLS /* Encloses this entire file. */

#include <string.h>
#include "config.h"
#include "lisp.h"
#include "commands.h"
#include <libintl.h>
#ifndef _
#define _(s) gettext(s)
#endif

/*
 *  TODO: Merge both translate-* functions; possibly use dcgettext and
 *         support categories as well.
 *  TODO: Make a buffer-local domain variable; the global translation
 *         should be used by emacs internally (and is the default value for
 *         the buffer-local domain.
 *        Make set-translation-domain operate on this variable ONLY.
 *        Have translate-string use the buffer-local binding.
 */

DEFUN("translation-domain",
      Ftranslation_domain, Stranslation_domain, 0, 0, 0,
      "Returns the current translation domain, as used by `translate-string'.\
\n") ()
{
Lisp_Object Ldomain = Qnil;
char* s = NULL;
  s = textdomain(NULL);
  if(s) Ldomain = build_string(s);
  return Ldomain;
}

DEFUN("set-translation-domain",
      Fset_translation_domain, Sset_translation_domain, 1, 1, 0,
      "Sets the translation domain to DOMAIN and returns the previous value.\n\
If DOMAIN is nil, the domain is set to 'emacs'. Use `translation-domain' to\n\
 retrieve the current domain without changing it.\n\
Messages are fetched from a 'DOMAIN.mo' file in a <language>/LC_MESSAGES\n\
 subdirectory of the message directory (default is '/usr/local/share'). For\n\
 example, 'usr/local/share/nl/LC_MESSAGES/emacs.mo' would contain the Dutch\n\
 messages for Emacs.\n\
You can use `bind-translation-domain' to change the default directory to\n\
 some other location.\n\
The language dirs searched depend on the LANG or LANGUAGE environment vars.\n\
These can either specify a single language (ie 'fr', or 'dutch') or a list\n\
 (ie 'french:german:dutch:english').\n\
For more information, consult your gettext documentation.\n")
     (domain)
     Lisp_Object domain;
{
Lisp_Object Lolddomain = Qnil;
char* s = NULL;
  CHECK_STRING(domain, 0);
  if(NILP(domain)) domain = build_string("emacs");
  s = textdomain(XSTRING(domain)->data);
  if(s) Lolddomain = build_string(s);
  return Lolddomain;
}

DEFUN("bind-translation-domain",
      Fbind_translation_domain, Sbind_translation_domain, 2, 2, 0,
      "Sets the message directory for DOMAIN to DIRECTORY.\n\
If t, messages for DOMAIN will be subsequently retrieved from a subdirectory\n\
 'LANGUAGE/LC_MESSAGES' of DIRECTORY.\n\
If nil, the domain couldn't be bound.\n")
     (domain, directory)
     Lisp_Object domain, directory;
{
Lisp_Object Lold = Qnil;
char* s = NULL;
  CHECK_STRING(domain, 0);
  CHECK_STRING(directory, 0);
  if(NILP(domain)) domain = build_string("emacs");
  if(NILP(directory)) return Qnil;
  s = bindtextdomain(XSTRING(domain)->data, XSTRING(directory)->data);
  if(!s) return Qnil;
  Lold = build_string(s);
  return Qt;
}

DEFUN("translate-string",
      Ftranslate_string, Stranslate_string, 1, 1, 0,
      "Retrieves the translation of STRING from the current translation \
domain.\n\This is equivalent to (translate-string-domain STRING\
 (translation-domain)).\n")
     (string)
     Lisp_Object string;
{
Lisp_Object Lmessage = Qnil;
char* s = NULL;
  CHECK_STRING(string, 0);
  if(NILP(string)) return Qnil;
  s = _(XSTRING(string)->data);
  if(s) Lmessage = build_string(s);
  return Lmessage;
}

DEFUN("translate-string-domain",
      Ftranslate_string_domain, Stranslate_string_domain, 1, 2, 0,
      "Retrieves the translation of STRING from DOMAIN.\nIf DOMAIN is nil, \
the default translation domain is used instead.")
     (string, domain)
     Lisp_Object string, domain;
{
Lisp_Object Lmessage = Qnil;
char* s    = NULL;
  CHECK_STRING(string, 0);
  if(NILP(string)) return Qnil;
  CHECK_STRING(domain, 0);
  s = dgettext(NILP(domain) ? NULL : XSTRING(domain)->data,
	       XSTRING(string)->data);
  if(s) Lmessage = build_string(s);
  return Lmessage;
}

/* module initialization */
syms_of_gettext()
{
  textdomain("emacs");

  defsubr(&Stranslation_domain);
  defsubr(&Sset_translation_domain);
  defsubr(&Sbind_translation_domain);
  defsubr(&Stranslate_string);
  defsubr(&Stranslate_string_domain);
}

#else

syms_of_gettext() { }

#endif
