/*****************************************************************************
 *
 *  Keyboard routines, these routines are based on the Allegro routines.
 *  Things added by SET:
 *  - Better support for extended scan key codes (insert doesn't turn off
 *    Shift now.
 *  - Changed some entrys in the tables to get different values from some
 *  keyboard combinations.
 *  - Led support.
 *  - Caps Lock support.
 *  - Num Lock support.
 *  - Alt+Key pad sequence.
 *  - Portability to Borland C.
 *
 *  SET == Salvador Eduardo Tropea.
 *  Curapaligue 2124
 *  Caseros, Pdo. Tres de Febrero
 *  (1678) Buenos Aires
 *  Argentina.
 *
 *  e-mail: salvador@inti.edu.ar
 *
 *  Phone: +(541) 759 0013
 *
 *****************************************************************************
 *         ______   ___    ___
 *        /\  _  \ /\_ \  /\_ \
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      Keyboard routines.
 *
 * Allegro is swap-ware. You may use, modify, redistribute, and generally
 * hack it about in any way you like, but if you do you must send me
 * something in exchange. This could be a complimentary copy of a game, an
 * addition or improvement to Allegro, a bug report, some money (this is
 * particularly encouraged if you use Allegro in a commercial product), or
 * just a copy of your autoexec.bat if you don't have anything better. If
 * you redistribute parts of Allegro or make a game using it, it would be
 * nice if you mentioned me somewhere in the credits, but if you just want
 * to pinch a few routines that is OK too. I'll trust you not to rip me off.
 *
 */

// That's the first include because is used to configure the editor.
#include "ceditint.h"

#include <stdlib.h>
#include <dos.h>
#include <conio.h>
#include <sys/movedata.h>

#include "keyboard.h"
#include "internal.h"


#define KEYBOARD_INT    9

int three_finger_flag = TRUE;
int not_locked_yet = TRUE;

static int keyboard_installed = FALSE;

volatile char key[128];                   /* key pressed flags */


/* Modified to detect Alt F1 to F12 as diferent keys, so I modified the
   common value */
char key_ascii_table[128] =
{
/* 0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F             */
   0,   27,  '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=', 8,   9,       /* 0 */
   'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '[', ']', 13,  0,   'a', 's',     /* 1 */
   'd', 'f', 'g', 'h', 'j', 'k', 'l', ';', 39,  '`', 0,   92,  'z', 'x', 'c', 'v',     /* 2 */
   'b', 'n', 'm', ',', '.', '/', 0,   '*', 0,   ' ', 0,   3,   3,   3,   3,   8,       /* 3 */
   3,   3,   3,   3,   3,   0,   0,   0,   0,   0,   '-', 0,   0,   0,   '+', 0,       /* 4 */
   0,   0,   0,   127, 0,   0,   92,  3,   3,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   13,  0,   '/', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,     /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0        /* 7 */
};

char key_capslock_table[128] =
{
/* 0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F             */
   0,   27,  '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=', 8,   9,       /* 0 */
   'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '[', ']', 13,  0,   'A', 'S',     /* 1 */
   'D', 'F', 'G', 'H', 'J', 'K', 'L', ';', 39,  '`', 0,   92,  'Z', 'X', 'C', 'V',     /* 2 */
   'B', 'N', 'M', ',', '.', '/', 0,   '*', 0,   ' ', 0,   3,   3,   3,   3,   8,       /* 3 */
   3,   3,   3,   3,   3,   0,   0,   0,   0,   0,   '-', 0,   0,   0,   '+', 0,       /* 4 */
   0,   0,   0,   127, 0,   0,   92,  3,   3,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   13,  0,   '/', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,     /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0        /* 7 */
};


/* Modified to detect Shift F1 to F12 as diferent keys
   idem Tab,BackSpace */
char key_shift_table[128] =
{
/* 0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F             */
   0,   27,  '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+',0x7E,0x7E,       /* 0 */
   'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '{', '}',0x7E, 0,   'A', 'S',     /* 1 */
   'D', 'F', 'G', 'H', 'J', 'K', 'L', ':', 34,  '~', 0,   '|', 'Z', 'X', 'C', 'V',     /* 2 */
   'B', 'N', 'M', '<', '>', '?', 0,   '*', 0,   1,   0,   1,   1,   1,   1,   1,       /* 3 */
   1,   1,   1,   1,   1,   0,   0,   0,   0,   0,   '-', 0,   0,   0,   '+', 0,       /* 4 */
   0,   0,   1,   127, 0,   0,   0,   1,   1,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   13,  0,   '/', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,     /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0        /* 7 */
};

/* Modified to detect Ctrl F1 to F12 as diferent keys
   idem Tab,BackSpace */
char key_control_table[128] =
{
/* 0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F             */
   0,   0,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   0,   0,0x7F,0x7F,       /* 0 */
   17,  23,  5,   18,  20,  25,  21,  9,   15,  16,  2,   2,  10,   0,   1,   19,      /* 1 */
   4,   6,   7,   8,   10,  11,  12,  0,   0,   0,   0,   0,   26,  24,  3,   22,      /* 2 */
   2,   14,  13,  0,   0,   0,   0,   0,   0,   0,   0,   2,   2,   2,   2,   2,       /* 3 */
   2,   2,   2,   2,   2,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 4 */
   0,   0,   2,   0,   0,   0,   0,   2,   2,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0        /* 7 */
};

char key_numlock_table[128] =
{
/* 0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F             */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 0 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 1 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 2 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 3 */
   0,   0,   0,   0,   0,   0,   0, '7', '8', '9',   0, '4', '5', '6',   0, '1',       /* 4 */
 '2', '3', '0', '.',   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0        /* 7 */
};

/*
  Mapping:
  0 = Use the second value as scan char, just like before.
  1 = Ignore the key.
  2 = Use the scan but put the shift flags instead of the ASCII.

  Extended values:
  E0 1C = Enter
  E0 1D = RCtrl         => Ctrl
  E0 2A = ?????? generated in conjuntion with Insert!!
  E0 35 = \
  E0 38 = AltGr or RAlt => Alt
  E0 46 = Ctrl-Pause
  E0 47 = Home
  E0 48 = Up
  E0 4B = Left
  E0 4D = Right
  E0 4F = End
  E0 50 = Down
  E0 51 = Page-Down
  E0 52 = Insert
  E0 53 = Delete
*/
char key_extended_table[128] =
{
/* 0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F             */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 0 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   2,   0,   0,   0,       /* 1 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   0,   0,   0,   0,   0,       /* 2 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 3 */
   0,   0,   0,   0,   0,   0,   2,   2,   2,   2,   0,   2,   2,   2,   0,   2,       /* 4 */
   2,   2,   2,   2,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0        /* 7 */
};

/*
  Shift, Ctrl, etc.
*/
char key_special_table[128] =
{
/* 0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F             */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 0 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   2,   0,   0,       /* 1 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   0,   0,   0,   0,   0,       /* 2 */
   0,   0,   0,   0,   0,   0,   1,   0,   4,   0,  32,   0,   0,   0,   0,   0,       /* 3 */
   0,   0,   0,   0,   0,  16,   8,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 4 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,       /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0        /* 7 */
};

#define KEY_BUFFER_SIZE    256

static volatile int key_buffer[KEY_BUFFER_SIZE];
static volatile int key_buffer_start = 0;
static volatile int key_buffer_end = 0;
static volatile int _extended_key=0;
static volatile int kbNeedReply=0;
static volatile int kbAckResponse=0;
static volatile int kbResendResponse=0;
static volatile int KeyPadSeq;
volatile int _shifts_flags=0;



/* add_key:
 *  Helper function to add a keypress to the buffer.
 */
static Inline void add_key(int c)
{
   key_buffer[key_buffer_end] = c;

   key_buffer_end++;
   if (key_buffer_end >= KEY_BUFFER_SIZE)
      key_buffer_end = 0;
   if (key_buffer_end == key_buffer_start) {    /* buffer full */
      key_buffer_start++;
      if (key_buffer_start >= KEY_BUFFER_SIZE)
	 key_buffer_start = 0;
   }
}



/* clear_keybuf:
 *  Clears the keyboard buffer.
 */
void clear_keybuf()
{
   int c;

   DISABLE();

   key_buffer_start = 0;
   key_buffer_end = 0;

   for (c=0; c<128; c++)
      key[c] = FALSE;

   _extended_key=0;
   _shifts_flags=0;

   ENABLE();
}



/* keypressed:
 *  Returns TRUE if there are keypresses waiting in the keyboard buffer.
 */
int keypressed()
{
   if (key_buffer_start == key_buffer_end)
      return FALSE;
   else
      return TRUE;
}



/* readkey:
 *  Returns the next character code from the keyboard buffer. If the
 *  buffer is empty, it waits until a key is pressed. The low byte of
 *  the return value contains the ASCII code of the key, and the high
 *  byte the scan code.
 */
int readkey()
{
   int r;

   if (!keyboard_installed)
      return 0;

   do {
   } while (key_buffer_start == key_buffer_end);  /* wait for a press */

   DISABLE();

   r = key_buffer[key_buffer_start];
   key_buffer_start++;
   if (key_buffer_start >= KEY_BUFFER_SIZE)
      key_buffer_start = 0;

   ENABLE();

   return r;
}



/* simulate_keypress:
 *  Pushes a key into the keyboard buffer, as if it has just been pressed.
 */
void simulate_keypress(int key)
{
   DISABLE();

   add_key(key);

   ENABLE();
}

#ifdef DEBUG_KEYS
/*
 * Code to get all the scan codes reported by the keyboard, these routines
 * are very good for debuging
 */
int BufCircular[10];
int entra=0;
int sale=0;

int popBufC(void)
{
 int ret=-1;

 if (entra!=sale)
   {
    ret=BufCircular[sale];
    if (sale==9)
       sale=0;
    else
       sale++;
   }
 return ret;
}

void pushBufC(int v)
{
 BufCircular[entra]=v;
 if (entra==9)
    entra=0;
 else
    entra++;
}
#endif

#define MAX_TRYES_FOR_READY 1000000L

static void kbWaitForReady(void)
{
 volatile long i=MAX_TRYES_FOR_READY;

 while (i-- && (inportb(0x64) & 0x02));
}


#define MAX_TRYES_TO_SEND_DATA 4
#define MAX_TRYES_TO_RESPONSE  2000000L
/*
 * kbSendDataToKb:
 *  Sends a byte to the keyboard.
 *  Returns 1 if all OK.
 */
static int kbSendDataToKb(unsigned char data)
{
 volatile long i;
 int kbResends = MAX_TRYES_TO_SEND_DATA;

 do
  {
   kbWaitForReady();
   kbNeedReply = 1;
   kbAckResponse = kbResendResponse = 0;
   outportb(0x60,data);
   i=MAX_TRYES_TO_RESPONSE;
   while (--i)
      {
       inportb(0x64);
       if (kbAckResponse)
	  return 1;
       if (kbResendResponse)
	  break;
      }
  }
 while (kbResends-->0 && i);

 return 0;
}


#define UpdateLeds() \
{                                                       \
 if (!kbSendDataToKb(0xed) || !kbSendDataToKb(_shifts_flags>>3))  \
    kbSendDataToKb(0xf4);                                    \
}


#ifdef __DJGPP__
#define RetValFromHandler 0
#else
#define RetValFromHandler
#endif

/* my_keyint:
 *  Hardware level keyboard interrupt (int 9) handler.
 */
static fISR my_keyint(VarArgList)
{
   int t, temp, release;
   int flag;

   temp = inportb(0x60);         /* read keyboard byte */

   if (kbNeedReply)
     {
      /* 0xFA == ack. */
      /* 0xFE == resend */
      if (temp==0xFA)
        {
	 kbAckResponse = 1;
         kbNeedReply = 0;
	 goto Exit_Keyboard_Handle;
	}
      else
         if (temp==0xFE)
           {
	    kbResendResponse = 1;
            kbNeedReply = 0;
	    goto Exit_Keyboard_Handle;
	   }
     }

   #ifdef DEBUG_KEYS
   pushBufC(temp);
   #endif

   if (temp==0xE0)
      _extended_key=1; /* 0xE0 is the extended code prefix */
   else
     {
      release = temp & 0x80; /* bit7==1 is release */
      temp &= 0x7F;          /* b6-b0 is the scan code */

      if (_extended_key)
        { /* If is an extended code: */
         _extended_key=0;
         switch (key_extended_table[temp])
           {
            case 1: goto Exit_Keyboard_Handle; /* Ignore the key */
            case 2: if (release)
                       key[temp] = FALSE;
                    else
                      { /* Report the scan code + the shift state */
                       key[temp] = TRUE;
                       t = (temp<<8) | (_shifts_flags & kbSpecialMask);
                       add_key(t);
                      }
                    goto Exit_Keyboard_Handle;
           }
        } /* case 0 is: process as normal */

      if (release)             /* key was released */
        {
         key[temp] = FALSE;
         /* check for Shift or Ctrl or Alt or CapsLock or NumLock or ScrollLock */
         if ((flag=key_special_table[temp])!=0)
           {
             if (flag<kbAltFlg)
                _shifts_flags&=(~flag); /* Shift and Ctrl */
             else
                if (flag==kbAltFlg)
                  { /* Alt is special */
                   _shifts_flags&=(~flag);
                   if (_shifts_flags & kbInAltSeqFlg)
                     { /* Is a num-pad seq? */
                      _shifts_flags&=(~kbInAltSeqFlg);
                      add_key(KeyPadSeq & 0xFF);
                     }
                  }
            /* CapsLock,NumLock and ScrollLock are toggle keys */
           }
        }
      else
        {                        /* key was pressed */
         char newchar;
         key[temp] = TRUE;

         /* check for Shift or Ctrl or Alt or CapsLock or NumLock or ScrollLock */
         if ((flag=key_special_table[temp])!=0)
           {
            if (flag>kbAltFlg)
              { /* CapsLock,NumLock and ScrollLock are toggle keys */
               _shifts_flags^=flag;
               goto SetKeyboardLeds;
              }
            else
               _shifts_flags|=flag; /* The rest */
           }
         else
           { /* Nop is a common key */
            if (_shifts_flags & kbAltFlg)
              { /* Is Alt pressed? */
               if (temp>=0x47 && key_extended_table[temp]==2)
                 { /* It's an special case Alt+KeyPadNumber */
                  if (_shifts_flags & kbInAltSeqFlg)
                    { /* The sequence is allready on */
                     KeyPadSeq=KeyPadSeq*10+(key_numlock_table[temp]-'0');
                    }
                  else
                    { /* We start a sequence */
                     _shifts_flags|=kbInAltSeqFlg;
                     KeyPadSeq=key_numlock_table[temp]-'0';
                    }
                  goto Exit_Keyboard_Handle;
                 }
               else /* Alt but not keypad */
                  t = SCANCODE_TO_ALT(temp);
              }
            else if (_shifts_flags & kbCtrlFlg)
               t = SCANCODE_TO_CONTROL(temp);
            else if (_shifts_flags & kbShiftFlg)
	            t = SCANCODE_TO_SHIFT(temp);
            else if ((_shifts_flags & kbNumLockFlg) &&
                     (newchar=key_numlock_table[temp])!=0)
               t = (KEY_PAD<<8) | newchar; /* Replace the scan code to avoid confution */
            else if (_shifts_flags & kbCapsLockFlg)
               t = SCANCODE_TO_CAPS(temp);
            else
               t = SCANCODE_TO_KEY(temp);

            /* If we are in a key pad seq. destroy the sequence */
            _shifts_flags&=(~kbInAltSeqFlg);

	    add_key(t);

            #ifdef __DJGPP__
            /* only for DJGPP */
	    if ((three_finger_flag) && (key[KEY_CONTROL]) && (key[KEY_ALT]) &&
	        ((temp == KEY_DEL) || (temp == KEY_END))) {
	       asm (
	          "  movb $0x79, %%al ; "
	          "  call ___djgpp_hw_exception "
	       : : : "%eax", "%ebx", "%ecx", "%edx", "%esi", "%edi", "memory"
	       );
	    }
            #endif
         }
      }
     }

Exit_Keyboard_Handle:
   outportb(0x20,0x20);       /* ack. the interrupt */

   return RetValFromHandler;

SetKeyboardLeds:
   /* This part is for the led's update. We need responses from the keyboard
      so we need the interrupts enabled */
   outportb(0x20,0x20);
   enable();

   UpdateLeds();

   return RetValFromHandler;
}

static END_OF_FUNCTION(my_keyint);



/* install_keyboard:
 *  Installs Allegro's keyboard handler. You must call this before using
 *  any of the keyboard input routines. Note that Allegro completely takes
 *  over the keyboard, so the debugger will not work properly, and under
 *  DOS even ctrl-alt-del will have no effect. Returns -1 on failure.
 */
int install_keyboard()
{
   unsigned short shifts;

   if (keyboard_installed)
      return -1;

   if (not_locked_yet)
     {
      not_locked_yet=FALSE;
      LOCK_VARIABLE(key);
      LOCK_VARIABLE(key_ascii_table);
      LOCK_VARIABLE(key_shift_table);
      LOCK_VARIABLE(key_control_table);
      LOCK_VARIABLE(key_buffer);
      LOCK_VARIABLE(key_buffer_start);
      LOCK_VARIABLE(key_buffer_end);
      LOCK_VARIABLE(three_finger_flag);
      LOCK_VARIABLE(_extended_key);
      LOCK_VARIABLE(_shifts_flags);
      LOCK_VARIABLE(kbNeedReply);
      LOCK_VARIABLE(kbAckResponse);
      LOCK_VARIABLE(kbResendResponse);
      LOCK_VARIABLE(key_extended_table);
      LOCK_VARIABLE(key_special_table);
      LOCK_VARIABLE(KeyPadSeq);
      LOCK_FUNCTION(my_keyint);
     }

   clear_keybuf();

   /* transfer keys waiting in BIOS keyboard buffer */
   while ((kbhit()) && (key_buffer_end < KEY_BUFFER_SIZE-1))
      key_buffer[key_buffer_end++] = getch();

   /* Get the leds info from the BIOS */
   #ifdef __DJGPP__
   _dosmemgetw(0x417,2,&shifts);
   #else
   shifts=*((unsigned short *)MK_FP(0x40,0x17));
   #endif

   _shifts_flags=0;
   if (shifts & 64)
      _shifts_flags|=kbCapsLockFlg;
   if (shifts & 32)
      _shifts_flags|=kbNumLockFlg;
   if (shifts & 16)
      _shifts_flags|=kbScroLockFlg;

   _install_irq(KEYBOARD_INT, my_keyint);

   UpdateLeds();

   _add_exit_func(remove_keyboard);
   keyboard_installed = TRUE;
   return 0;
}



/* remove_keyboard:
 *  Removes the keyboard handler, returning control to the BIOS. You don't
 *  normally need to call this, because allegro_exit() will do it for you.
 */
void remove_keyboard()
{
   unsigned short shifts;

   if (!keyboard_installed)
      return;

   _remove_irq(KEYBOARD_INT);

   /* Transfer de leds info to the BIOS */
   shifts=0;
   if (_shifts_flags & kbCapsLockFlg)
      shifts|=64;
   if (_shifts_flags & kbNumLockFlg)
      shifts|=32;
   if (_shifts_flags & kbScroLockFlg)
      shifts|=16;

   #ifdef __DJGPP__
   _dosmemputw(&shifts,1,0x417);
   #else
   *((unsigned short *)MK_FP(0x40,0x17))=shifts;
   #endif

   clear_keybuf();

   _remove_exit_func(remove_keyboard);
   keyboard_installed = FALSE;
}


