/* util.c -- LZO packer

   This file is part of the LZO real-time data compression package.

   Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1997 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library and packer is free software; you can redistribute it
   and/or modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.
   If not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


#include "lzopack.h"


/*************************************************************************
// filename util
**************************************************************************/

static const char dir_sep[] = DIR_SEP;

#define fn_is_sep(c)		(strchr(dir_sep,c) != NULL)

#if defined(DOSISH)
#define fn_is_drive(n)		(n[0] && n[1] == ':')
#define fn_skip_drive(n)	(fn_is_drive(n) ? (n) + 2 : (n))
#else
#define fn_is_drive(n)		(0)
#define fn_skip_drive(n)	(n)
#endif


char *fn_basename(const char *name)
{
	const char *n, *nn;

	name = fn_skip_drive(name);
	for (nn = n = name; *nn; nn++)
		if (fn_is_sep(*nn))
			n = nn + 1;
	return (char *) n;
}


void fn_addslash(char *name, lzo_bool slash)
{
	char *p;

	name = fn_skip_drive(name);
	p = name + strlen(name);
	while (p > name && fn_is_sep(p[-1]))
		*p-- = 0;
	if (p > name)
	{
		if (slash)
			*p++ = dir_sep[0];
		*p = 0;
	}
}


char *fn_strlwr(char *n)
{
	char *p;
	for (p = n; *p; p++)
		*p = (char) fn_tolower(*p);
	return n;
}


int fn_strcmp(const char *n1, const char *n2)
{
	for (;;)
	{
		if (*n1 != *n2)
		{
			int c = fn_tolower(*n1) - fn_tolower(*n2);
			if (c)
				return c;
		}
		if (*n1 == 0)
			return 0;
		n1++; n2++;
    }
}


lzo_bool fn_is_same_file(const char *n1, const char *n2)
{
	/* very simple... */
	if (fn_strcmp(n1,n2) == 0)
		return 1;
	return 0;
}


int fn_has_suffix(const char *name)
{
	size_t l;
	size_t s;

	name = fn_skip_drive(name);
	l = strlen(name);
	if (l > 4 && name[l-4] == '.')
	{
		if (strcasecmp(&name[l-3],"lzo") == 0)
			return SUFF_LZO;
		if (strcasecmp(&name[l-3],"tzo") == 0)
			return SUFF_TZO;
		if (strcasecmp(&name[l-3],"tar") == 0)
			return SUFF_TAR;
	}

	if (l > 5 && name[l-5] == '.')
	{
		if (strcasecmp(&name[l-4],"lzop") == 0)
			return SUFF_LZOP;
	}

	s = strlen(opt_suffix);
	if (s > 0 && l > s)
	{
		if (strcasecmp(&name[l-s],opt_suffix) == 0)
			return SUFF_USER;
	}

	return SUFF_NONE;
}


/*************************************************************************
// time util
**************************************************************************/

#if defined(HAVE_LOCALTIME)
void tm2str(char *s, const struct tm *tmp)
{
	sprintf(s,"%04d-%02d-%02d %02d:%02d:%02d",
			(int) tmp->tm_year + 1900, (int) tmp->tm_mon + 1,
			(int) tmp->tm_mday,
		    (int) tmp->tm_hour, (int) tmp->tm_min, (int) tmp->tm_sec);
}
#endif


void time2str(char *s, const time_t *t)
{
#if defined(HAVE_LOCALTIME)
	tm2str(s,localtime(t));
#elif defined(HAVE_CTIME)
	const char *p = ctime(t);
	memset(s, ' ', 16);
	memcpy(s + 2, p + 4, 6);
	memcpy(s + 11, p + 11, 5);
	s[16] = 0;
#else
	s[0] = 0;
#endif
}


/*************************************************************************
//
**************************************************************************/

lzo_bool file_exists(const char *name)
{
	int fd;

	fd = open(name,O_RDONLY);
	if (fd < 0)
		return 0;
	(void) close(fd);
	return 1;
}


/* test if fd is connected to a file (or a pipe) */
lzo_bool isafile(int fd)
{
	if (isatty(fd))
		return 0;
#if defined(HAVE_FSTAT)
	{
		struct stat st;
		if (fstat(fd, &st) != 0)
			return 0;
		/* fprintf(stderr,"fstat(%d): %o\n", fd, st.st_mode); */
		if (S_ISDIR(st.st_mode))
			return 0;
	}
#endif
	return 1;
}


/*************************************************************************
// Some systems have very exotic mode values.
// Convert them to standard values for portable use in our header.
**************************************************************************/

lzo_uint32 fix_mode_for_header(lzo_uint32 mode)
{
	lzo_uint32 m = mode;

	if (mode == 0)
		return 0;

	/* This function can only deal with S_ISREG and S_ISDIR modes. */
	assert(S_ISREG(mode) || S_ISDIR(mode));

#if defined(__LZO_TOS) && defined(__PUREC__)
	m = 0444;
	if (mode & S_IWRITE)
		m |= 0200;
	if (mode & S_IEXEC)
		m |= 0111;
	if (S_ISREG(mode))
		m |= 0100000 | 0400;
	else if (S_ISDIR(mode))
		m |= 0040000 | 0700;
#elif defined(DOSISH)
	m &= 0777;
	if (S_ISREG(mode))
		m |= 0100000 | 0400;
	else if (S_ISDIR(mode))
		m |= 0040000 | 0700;
#else
	m &= 07777;
	if (S_ISREG(mode))
		m |= 0100000;
	else if (S_ISDIR(mode))
		m |= 0040000 | 0700;
#endif

	if ((m & 0777) == 0)
		m |= 0644;
	return m;
}


int fix_mode_for_chmod(lzo_uint32 mode)
{
	int m = (int)mode & 07777;
	if ((m & 0777) == 0)
		m |= 0644;
	return m;
}


int fix_mode_for_ls(lzo_uint32 mode)
{
	int m = (int)mode;
	if ((m & 0777) == 0)
		m |= 0644;
	return m;
}


int fix_mode_for_open(int mode)
{
	int m = mode & 0666;
#if defined(__LZO_TOS) && defined(__PUREC__)
	m = S_IWRITE | S_IREAD;
#else
	if ((m & 0777) == 0)
		m |= 0644;
	m |= 0600;
#endif
	return m;
}


/*************************************************************************
// adapted from GNU fileutils 3.16
**************************************************************************/

/* Return a character indicating the type of file described by
   file mode BITS:
   'd' for directories
   'b' for block special files
   'c' for character special files
   'm' for multiplexor files
   'l' for symbolic links
   's' for sockets
   'p' for fifos
   '-' for regular files
   '?' for any other file type.  */

static char ftypelet (unsigned bits)
{
#ifdef S_ISBLK
	if (S_ISBLK (bits))
		return 'b';
#endif
#ifdef S_ISCHR
	if (S_ISCHR (bits))
		return 'c';
#endif
	if (S_ISDIR (bits))
		return 'd';
	if (S_ISREG (bits))
		return '-';
#ifdef S_ISFIFO
	if (S_ISFIFO (bits))
		return 'p';
#endif
#ifdef S_ISLNK
	if (S_ISLNK (bits))
		return 'l';
#endif
#ifdef S_ISSOCK
	if (S_ISSOCK (bits))
		return 's';
#endif
#ifdef S_ISMPC
	if (S_ISMPC (bits))
		return 'm';
#endif
#ifdef S_ISNWK
	if (S_ISNWK (bits))
		return 'n';
#endif

#ifdef S_ISOFD
	/* Cray migrated dmf file.  */
	if (S_ISOFD (bits))
		return 'M';
#endif
#ifdef S_ISOFL
	/* Cray migrated dmf file.  */
	if (S_ISOFL (bits))
		return 'M';
#endif
	return '?';
}


/* Look at read, write, and execute bits in BITS and set
   flags in CHARS accordingly.  */

static void rwx (unsigned bits, char *chars)
{
	chars[0] = (bits & 0400) ? 'r' : '-';
	chars[1] = (bits & 0200) ? 'w' : '-';
	chars[2] = (bits & 0100) ? 'x' : '-';
}


/* Set the 's' and 't' flags in file attributes string CHARS,
   according to the file mode BITS.  */

static void setst (unsigned bits, char *chars)
{
#ifdef S_ISUID
	if (bits & S_ISUID)
	{
		if (chars[3] != 'x')
			/* Set-uid, but not executable by owner.  */
			chars[3] = 'S';
		else
			chars[3] = 's';
	}
#endif
#ifdef S_ISGID
	if (bits & S_ISGID)
	{
		if (chars[6] != 'x')
			/* Set-gid, but not executable by group.  */
			chars[6] = 'S';
		else
			chars[6] = 's';
	}
#endif
#ifdef S_ISVTX
	if (bits & S_ISVTX)
	{
		if (chars[9] != 'x')
			/* Sticky, but not executable by others.  */
			chars[9] = 'T';
		else
			chars[9] = 't';
	}
#endif
	UNUSED(bits);
	UNUSED(chars);
}


void mode_string (unsigned mode, char *str)
{
	str[0] = ftypelet ( mode);
	rwx ((mode & 0700) << 0, &str[1]);
	rwx ((mode & 0070) << 3, &str[4]);
	rwx ((mode & 0007) << 6, &str[7]);
	setst (mode, str);
}


/*************************************************************************
// adapted from the djgpp port of cpio 2.4.2 by Eli Zaretskii
**************************************************************************/

#if defined(DOSISH)

/* If the original file name includes characters illegal for MS-DOS and
   MS-Windows, massage it to make it suitable and return a pointer to
   static storage with a new name.  If the original name is legit,
   return it instead.  Return NULL if the rename failed (shouldn't happen).

   The file name changes are: (1) any name that is reserved by a DOS
   device driver (such as `prn.txt' or `aux.c') is prepended with a `_';
   and (2) illegal characters are replaced with `_' or `-' (well, almost;
   look at the code below for details).  */

#define is_slash(x)		fn_is_sep(x)

char *maybe_rename_file(const char *original_name)
{
	static char dosified_name[PATH_MAX+1];
	static const char illegal_chars_dos[] = ".+, ;=[]|<>\":?*";
	static const char * const illegal_chars_w95 = &illegal_chars_dos[8];
	const char *illegal_aliens = illegal_chars_dos;
	int idx, dot_idx;
	const char *s = original_name;
	char *d = dosified_name;
#if defined(__DJGPP__)
	int windows9x = _use_lfn (original_name);
#elif defined(__LZO_WIN32)
	int windows9x = 1;
#else
	int windows9x = 0;
#endif

	/* Support for Windows 9X VFAT systems, when available.  */
	if (windows9x)
		illegal_aliens = illegal_chars_w95;
	else
		illegal_aliens = illegal_chars_dos;

  /* Get past the drive letter, if any. */
	if (fn_is_drive(s))
	{
		*d++ = *s++;
		*d++ = *s++;
	}

	for (idx = 0, dot_idx = -1; *s; s++, d++)
	{
		if (strchr (illegal_aliens, *s))
		{
			/* Dots are special on DOS: it doesn't allow them as the leading
			 character, and a file name cannot have more than a single dot.
			 We leave the first non-leading dot alone, unless it comes too
			 close to the beginning of the name: we want sh.lex.c to become
			 sh_lex.c, not sh.lex-c.  */
			if (*s == '.')
			{
				if (idx == 0
					&& (is_slash(s[1]) || s[1] == '\0'
						|| (s[1] == '.' && (is_slash(s[2]) || s[2] == '\0'))))
				{
					/* Copy "./" and "../" verbatim.  */
					*d++ = *s++;
					if (*s == '.')
						*d++ = *s++;
					*d = *s;
				}
				else if (idx == 0)
					*d = '_';
				else if (dot_idx >= 0)
				{
					if (dot_idx < 5) /* 5 is merely a heuristic ad-hoc'ery */
					{
						d[dot_idx - idx] = '_'; /* replace previous dot */
						*d = '.';
					}
					else
						*d = '-';
				}
				else
					*d = '.';

				if (*s == '.')
					dot_idx = idx;
			}
			else if (*s == '+' && s[1] == '+')
			{
				if (idx - 2 == dot_idx) /* .c++, .h++ etc. */
				{
					*d++ = 'x';
					*d   = 'x';
				}
				else
				{
					/* libg++ etc.  */
					memcpy (d, "plus", 4);
					d += 3;
				}
				s++;
				idx++;
			}
			else
				*d = '_';
		}
		else
			*d = *s;
		if (is_slash(*s))
		{
			idx = 0;
			dot_idx = -1;
		}
		else
			idx++;
	}

	*d = '\0';

#if defined(S_ISCHR)
	/* We could have a file in an archive whose name is reserved
	 on MS-DOS by a device driver.  Trying to extract such a
	 file would fail at best and wedge us at worst.  We need to
	 rename such files.  */

	if (idx > 0)
	{
		struct stat st_buf;
		char *base = d - idx;
		int i = 0;

		/* The list of character devices is not constant: it depends on
		 what device drivers did they install in their CONFIG.SYS.
		 `stat' will tell us if the basename of the file name is a
		 characer device.  */
		while (stat (base, &st_buf) == 0 && S_ISCHR (st_buf.st_mode))
		{
			size_t blen = strlen (base);

			/* I don't believe any DOS character device names begin with a
			 `_'.  But in case they invent such a device, let us try twice.  */
			if (++i > 2)
				return (char *)0;

			/* Prepend a '_'.  */
			memmove (base + 1, base, blen + 1);
			base[0] = '_';
		}
	}
#endif

	return dosified_name;
}

#endif /* DOSISH */


/*
vi:ts=4
*/

