/* $Revision: 1.37 $ */
/*
 EPSHeader

   File: tags.c
   Author: J. Kercheval
   Created: Sun, 03/31/1991  14:59:48
*/
/*
 EPSRevision History

   J. Kercheval  Sun, 03/31/1991  16:24:01  creation
   J. Kercheval  Sun, 03/31/1991  16:43:18  allow @filename listfile syntax
   J. Kercheval  Tue, 05/14/1991  19:46:50  add tag_type
   J. Kercheval  Wed, 05/15/1991  18:19:22  add -m and -o parameters
   J. Kercheval  Wed, 05/15/1991  19:23:42  correctly parse for logfile
   J. Kercheval  Wed, 05/15/1991  22:22:33  add the sort module
   J. Kercheval  Thu, 05/16/1991  22:47:04  move file IO to fileio.c
   J. Kercheval  Wed, 06/26/1991  23:03:55  move back to standard IO
   J. Kercheval  Thu, 06/27/1991  21:43:11  create tags.h
   J. Kercheval  Fri, 07/12/1991  23:04:23  revise command line parsing
   J. Kercheval  Sat, 07/13/1991  11:29:02  update Usage()
   J. Kercheval  Sat, 07/13/1991  12:47:34  finish argf parsing
   J. Kercheval  Sat, 07/13/1991  13:06:16  move input routines to input.c
   J. Kercheval  Sun, 07/14/1991  19:18:36  finish DoTags
   J. Kercheval  Wed, 07/17/1991  22:18:58  allow append file logging
   J. Kercheval  Thu, 07/18/1991  18:53:31  use fully qualified input pathname
   J. Kercheval  Fri, 07/19/1991  20:15:43  allow sort only and non-sort options
   J. Kercheval  Sun, 07/21/1991  17:38:00  add post processing
   J. Kercheval  Mon, 07/22/1991  17:35:08  tweak log output and interface
   J. Kercheval  Sat, 07/27/1991  20:38:36  remove ASM public flag and post processing hooks
   J. Kercheval  Sat, 08/17/1991  23:03:47  enable c tagging
   J. Kercheval  Sun, 08/25/1991  22:49:55  fix bug in ASM flag parsing
   J. Kercheval  Thu, 08/29/1991  23:30:47  add CRC checking for virus check
   J. Kercheval  Sat, 08/31/1991  23:56:02  add prototype flag
   J. Kercheval  Thu, 09/05/1991  01:29:13  add input file name tracking
   J. Kercheval  Thu, 09/05/1991  01:29:42  add -t flag
   J. Kercheval  Thu, 09/05/1991  02:01:28  finish tag output logic
   J. Kercheval  Thu, 09/05/1991  20:07:45  move arglist processing to seperate module
   J. Kercheval  Thu, 09/05/1991  20:13:50  move MergeFile() to tagio.c
   J. Kercheval  Tue, 09/10/1991  23:24:27  add ctrl-c handler
   J. Kercheval  Wed, 09/11/1991  01:45:35  add usage comments for extern switch
   J. Kercheval  Tue, 09/17/1991  19:43:23  add support for case_sensitive flag
   J. Kercheval  Wed, 09/25/1991  13:42:28  support sorted arglists
   J. Kercheval  Wed, 09/25/1991  16:02:18  check for duplicate file names at the command line level
   J. Kercheval  Wed, 09/25/1991  22:47:37  supress tag file merge if not input files found
   J. Kercheval  Tue, 10/01/1991  19:30:26  close all open files in external_cleanup()
   J. Kercheval  Thu, 10/03/1991  13:55:09  add exclude file processing
   J. Kercheval  Thu, 10/03/1991  16:46:07  improve list file parsing
   J. Kercheval  Sat, 10/05/1991  10:56:18  add switch summary to usage
   J. Kercheval  Sat, 04/25/1992  01:36:19  add junk tagging for C
   Bradley M. Small  Thu, 03-04-1993  10:27:25 added #ifdef and #define to alias __cdecl
											to _cdecl.  for __TURBOC__
   Bradley M. Small  Thu, 03-04-1993  10:27:25 added #ifdef and to use __TIMESTAMP__
											instead of __TIME__ .  for __TURBOC__
   Bradley M. Small  Thu, 03-04-1993  10:27:25 added secondary environment search of TEMP
											if TMP is not found.
   J. Kercheval  Mon, 08/21/1995 01:00:24   The junk filter is now off be default.
   J. Kercheval  Mon, 08/21/1995 23:01:35   merge in changes by Dan McMullen
   J. Kercheval  Sun, 09/03/1995 22:10:21   Add support for Codewright output format
   J. Kercheval  Mon, 09/04/1995 15:20:28	Add native windows support
*/

/* disable inline compiler warning */
#pragma warning(disable:4001 4201 4209 4214 4711)

#if defined(WIN16) || defined(WIN32)
#include <windows.h>
#include <windowsx.h>
#include "resource.h"
#ifdef WIN32
	#define EXPORT      WINAPI
#else
	#define EXPORT      _export
#endif

static void checkMessages(void);
HWND hTagsDlg = NULL;
static BOOL exiting = FALSE;

#endif /* defined(WIN16) || defined(WIN32) */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <io.h>

#include "flags.h"
#include "log.h"
#include "wildfile.h"
#include "sort.h"
#include "ctag.h"
#include "asmtag.h"
#include "tagio.h"
#include "viruscrc.h"
#include "arglist.h"
#include "compile.h"


#define Author "J. Kercheval"
#define Description "Tags Generator V2.01"
#if defined(_CONSOLE)
#define Version Description " (Win32 Console)"
#elif defined(WIN32)
#define Version Description " (Win32)"
#elif defined(WIN16)
#define Version Description " (Win16)"
#else
#define Version Description " (DOS)"
#endif
#define CompileDate __TIMESTAMP__

#if defined(__TURBOC__) || defined(__BORLANDC__)
#define __cdecl _cdecl
unsigned int _stklen = 0x8000;
#undef CompileDate
#define CompileDate __DATE__ " " __TIME__
#endif


/* The following variables are pointers to the temporary filenames used as
 * an intermediary files.  These are globals for use in external_cleanup()
 * to delete any temporary files. */
char *tmp_filename = NULL;
char *tmp_tagfilename = NULL;


/*----------------------------------------------------------------------------
 *
 * Print Usage
 *
 ---------------------------------------------------------------------------*/


#if defined(WIN16) || defined(WIN32)
#define usage_exit() 
#else
#define usage_exit() exit(1)
#endif

#define print_usage(fname, usage) \
{ \
	char tmpstr[MAXPATH + 1]; \
    int i;                      /* index into help text array */ \
\
	log_stdout(""); \
    sprintf(tmpstr, "%s -- %s", Version, Author); \
	log_stdout(tmpstr); \
    sprintf(tmpstr, "Compiled: %s", CompileDate); \
	log_stdout(tmpstr); \
	log_stdout(""); \
    sprintf(tmpstr, "Usage: %s {[OPTIONS] [SOURCEFILE|@LISTFILE]}", \
            fname); \
	log_stdout(tmpstr); \
	log_stdout(""); \
\
    for (i = 0; usage[i][0]; i++) { \
        sprintf(tmpstr, "%s", usage[i]); \
		log_stdout(tmpstr); \
    } \
\
	usage_exit(); \
}

void Usage(char *fname)
{
    char usage[][85] =
    {
		"  -h,-? for extended usage output to stdout",
		"  @LISTFILE for list file of input file names",
		"  -x{EXCLUDEFILE|@LISTFILE} to exclude files",
		"  -tTAGFILE create a tag file",
		"  -TTAGFILE to merge to a tag file",
		"  -pCOMPILEFILE to compile output database (-oc,-t only)",
		"  -lLOGFILE create a log file",
		"  -LLOGFILE to append to a log file",
		"  -o[e5gsmc] for output format",
		"  -a[fdlmsu] to specify assembly tagging",
		"  -c[dmstekuvcfpxi] to specify C tagging",
		"  -d delete listfiles and exclude response files (ie. @delete.me)",
		"  -j will enable junk output filter",
		"  -q will suppress normal status output",
		"  -r use relative pathnames in output",
		"  -n do not sort the tag output",
		"  -i use a case sensitive sort",
#if defined(MSDOS) || defined(_CONSOLE)
		"  -m for merge sort of the existing sorted files",
		"  -s sort input files only",
#endif
        "\0"
    };

	print_usage(fname, usage);
}


/*----------------------------------------------------------------------------
 *
 * Print Extended Usage
 *
 ---------------------------------------------------------------------------*/

void Help(char *fname)
{
    char usage[][85] =
    {
        "  -h or -? to obtain this help screen",
		" ",
        "  @LISTFILE for list file of input file names.  A list file is in the form",
        "     of a response file (ie. a list of files seperated by some delimiter).",
        "     The allowed delimiters for file seperation are '+', ',', ';' and normal",
        "     whitespace.  This file allows commented lines by preceding any comment",
        "     with a pound sign (#).  A comment is from the '#' to the end of the",
        "     line and may start after any whitespace character",
		" ",
        "  -x{EXCLUDEFILE|@LISTFILE} excludes the files specified by EXCLUDEFILE",
        "     or exclude all files listed in LISTFILE.",
		" ",
        "  -tTAGFILE to output to a (possibly existing) tag file and will result in",
        "     previous tags for input files being removed from the output file.",
        "     This tagfile is assumed to be in one of this utilities output formats.",
        "     if -m or -s are used this switch is ignored (all output is to stdout).",
		" ",
        "     Behavior regarding existing files is determined by the case of the switch.",
        "     t  creates and outputs to a file overwriting any currently existing file",
        "     T  merges all output to the tagfile if there is an already existing file",
		" ",
		"  -pCOMPILEFILE to compile output database.  This is used only when an output",
		"     file and Codewright output format has been selected (-t and -oc).",
		" ",
        "  -lLOGFILE for output to a log file in a LISTFILE format suitable as input.",
        "     Behavior regarding existing files is determined by the case of the switch.",
        "     l  creates and outputs to a file overwriting any currently existing file",
        "     L  appends all output to the logfile if there is an already existing file",
		" ",
        "  -o[options] for output format to stdout or the tag file if -t switch is used",
        "     e  Epsilon >= V6.0  ( tokenString \\t fileName \\t characterOffset \\t line )",
        "     5  Epsilon <= V5.03 ( tokenString;fileName;characterOffset )",
        "     g  GNU tag          ( tokenString \\t fileName \\t /$line^/ )",
        "     s  Space-Delimited  ( tokenString fileName lineNumber )",
        "     m  MicroSoft Error  ( tokenString fileName(lineNumber) )",
        "     c  Codewright       ( tokenString fileName(lineNumber)tokenType )",
		" ",
        "  -a[options] to specify assembly tagging and to detail the token types.",
        "     Default tagging is -afdlmsu (80x86 assembly using MASM/TASM syntax)",
        "     f  procedure labels       ( token proc )( proc token )",
        "     d  definition labels      ( token equ const )( token db declaration )",
        "     l  local labels           ( token label )( label token )( token: )",
        "     m  macro labels           ( token macro )( macro token )",
        "     s  struc labels           ( token struc )( struc token )",
        "     u  union labels           ( token union )( union token )",
		" ",
        "  -c[options] to specify C tagging and to detail the token types. Default",
        "     tagging options are -cdmstekuvcfpxi (standard ANSI 2.0 C/C++). Note that",
        "     use of the -cx and the -ci switch are modifiers and will only be effective",
        "     when other options are used (ie. -cpx must be specified to obtain extern",
        "     prototypes, -cx alone yields nothing).  Note that the -cx and -ci modifier",
        "     has no effect for define and macro tags which are tagged only according",
        "     to the -cd and -cm switches respectively.  Additionally the -cx modifier",
        "     is ignored for function tags.  The order of all of these options is not",
        "     significant.",
        "     d  defines                ( #define token statement )",
        "     m  macro labels           ( #define token() statement )",
        "     s  struct globals         ( struct token {} )",
        "     t  typedef globals        ( typedef declaration token, token, ... )",
        "     e  enum globals           ( enum token {} )",
        "     k  enum konstants         ( enum { token, token, token, ...} )",
        "     u  union globals          ( union token {} )",
        "     v  global variable        ( declaration token, token = {}, token, ... )",
        "     c  global class           ( class token: {} )",
        "     f  function definitions   ( token() declaration {} )",
        "     p  prototypes             ( token(, )",
        "     i  static declarations    ( static declaration )",
        "     x  extern defines         ( extern declaration )",
        "                               ( extern \"C\" declaration )",
        "                               ( extern \"C\" { declaration; declaration; ... } )",
		" ",
		"  -d delete listfiles and exclude response files (ie. @delete.me)",
		"  -j will enable junk (C delimiters) filtering during token output",
        "  -q will suppress normal output to stderr and program version information",
        "  -r use relative pathnames in output rather than fully qualified path names",
        "  -n do not sort the tag output (Often used in conjunction with GNU style tags)",
        "  -i use a case sensitive sort (Normally a case insensitive sort is used)",
		" ",
        "  The following result only in sorting of the input files (no tagging is done).",
        "     Output is to stdout only (-t is ignored) when using these switches.",
#if defined(MSDOS) || defined(_CONSOLE)
        "     -m for merge sort of the specified existing files (assumed to be sorted)",
        "     -s sort input files only, all files are assumed to be in an unsorted state",
#endif
		" ",
        "  The TMP environment variable is used for temporary files.  The default for",
        "     tags is to use C style tagging, the Epsilon tag file format, to sort",
        "     the output before finally placing it in the output file (or stdout if -t",
        "     is not used) and to be verbose and log activity to stderr.",
        "  Each file specified on the command line or within a LISTFILE will be tagged",
        "     only once regardless of the number of times it appears on the command",
        "     line (This includes LISTFILEs as well as filenames and the files listed",
        "     within LISTFILEs).",
        "  All of the switches may be specified anywhere on the command line and with",
        "     the exception of the style switches (-a, -c) are not position dependent.",
        "     The style switches are active only for input files which fall after them",
        "     on the command line and allows the specification of different tagging",
        "     styles and types on a file by file basis.",
        "  Input file and LISTFILE specifications allow the use of *IX shell style",
        "     expressions (A subset of the standard UNIX regular expression syntax).",
        "     This allows input file names such as \"*\", \"*t?*.c\" and \"*[e-gxyz]*\".",
        "     Note that \"*\" in this case is completely equivalent to \"*.*\" in normal",
        "     DOS usage. The use of \"*.\" will obtain files without extensions.",
        "  This utility performs a CRC validation on itself to prevent corruption and",
        "     viral infection from outside influences.  Modification of this file in",
        "     any way will result in a failure of the internal CRC check.  On CRC",
        "     failure the program will exit with a warning message.",
        "\0"
    };

	print_usage(fname, usage);
}


/*----------------------------------------------------------------------------
 *
 * external_cleanup() removes the temporary file used here
 *
 ---------------------------------------------------------------------------*/

void external_cleanup(void)
{
	/* let the compile module clean itself up */
	CDB_cleanup();

	/* close down the log module */
	log_close();

	/* close any stray open files */
    fcloseall();

	/* remove tmp files */
    if (tmp_filename != NULL) {
        remove(tmp_filename);
		free(tmp_filename);
	}
    if (tmp_tagfilename != NULL) {
        remove(tmp_tagfilename);
		free(tmp_tagfilename);
	}
	
#if defined(WIN16) || defined(WIN32)
	/* we are shutting down */
	exiting = TRUE;
#endif
}


/*----------------------------------------------------------------------------
 *
 * CtrlCHandler() cleans up and aborts
 *
 ---------------------------------------------------------------------------*/

void __cdecl CtrlCHandler(int sig)
{
	/* Disallow CTRL+C during handler. */
    signal(SIGINT, SIG_IGN);

    /* cleanup and abort */
    external_cleanup();
    abort();
}


/*----------------------------------------------------------------------------
 *
 * nextfile takes the list_file parameter and finds the next filename from
 * the opened listfile.
 *
 ---------------------------------------------------------------------------*/

BOOLEAN nextfile(FILE * list_file, char *fname, int max_length)
{
    char white[] =
    {
        " \t\n\f\v\b\r,+;#"
    };                          /* valid whitespace characters */

    char delim[] =
    {
        " \t\n\f\v\b\r,+;"
    };                          /* end of filename characters */

    char *s;                    /* temporary char pointer */
    char c;                     /* temporary char variable */

    int length;                 /* current length of fname */

    /* init */
    s = fname;
    *s = '\0';
    length = 0;

    c = (char) fgetc(list_file);

    /* if end of file then we are done */
    if (feof(list_file))
        return FALSE;

    /* pass over whitespace */
    while (strchr(white, c)) {
        if (c == '#') {
            while (c != '\n') {
                c = (char) fgetc(list_file);
                if (feof(list_file))
                    return FALSE;
            }
        }
        c = (char) fgetc(list_file);
        if (feof(list_file))
            return FALSE;
    }

    /* get the filename */
    while (!strchr(delim, c)) {

        /* don't add to fname if maximum length reached */
        if (length < max_length - 1) {
            *s++ = c;
            length++;
        }
        c = (char) fgetc(list_file);
        if (feof(list_file)) {
            *s = '\0';
            return TRUE;
        }
    }

    /* success */
    *s = '\0';
    return TRUE;
}


/*----------------------------------------------------------------------------
 *
 * Process() tags the input stream from the input file and outputs to a
 * temporary file which is registered in argf.
 *
 ---------------------------------------------------------------------------*/

void Process(FILE * infile, char *infname, FILE * outfile, Flags * flags)
{
    /* log the input filename */
    log_message(infname);

    switch (flags->tag_type) {

        case C:         /* use C type parsing */
            CTags(infile, infname, outfile, flags);
            break;

        case ASM:               /* use ASM type parsing */
            ASMTags(infile, infname, outfile, flags);
            break;

        default:
            break;
    }
}


/*----------------------------------------------------------------------------
 *
 * SingleFileProcess() tags a single file and outputs to a temporary file
 * registered in argf.  Wildcards are allowed as file names.
 *
 ---------------------------------------------------------------------------*/

void SingleFileProcess(char *pathname, Flags * flags,
                        FILE * outfile, ArgList argInput,
                        ArgList argInputRaw)
{
    char fname[MAXPATH + 1];    /* the current file name */
    char full_pathname[MAXPATH + 1];    /* the full path and filename */
    char tmpstr[2 * MAXPATH + 1];       /* error string */

    struct file_info_struct ff; /* the file find structure */

    FILE *input_file;           /* the input file being worked on */

    /* initialize ff */
    strcpy(ff.file_pattern, pathname);
    ff.file_attributes = _FA_NORMAL | _FA_READONLY | _FA_ARCHIVE |
        _FA_HIDDEN | _FA_SYSTEM;

    /* find the initial file matching pattern */
    if (!find_firstfile(&ff)) {

        /* bad file name */
        sprintf(tmpstr, "# Could not find the file '%s'", pathname);
        log_message(tmpstr);
    }
    else {

        /* loop through all matching files in the parameter */
        do {

            /* make the file name */
            strcpy(fname, ff.file_path);
            strcat(fname, ff.file.name);

            /* get the fully qualified pathname if wanted */
            if (!flags->use_relative_pathnames) {
                _fullpath(full_pathname, fname, MAXPATH);
            }
            else {
                strcpy(full_pathname, fname);
            }

            /* convert pathname to lower case */
            strlwr(full_pathname);

            /* register the original input file name after verifying that we
             * have not previously processed this file */
            if (!ArgIsMember(argInputRaw, full_pathname)) {

                ArgRegisterName(argInput, full_pathname);
                ArgRegisterName(argInputRaw, full_pathname);

                /* Try to open the file */
                if ((input_file = fopen(fname, "r")) == (FILE *) NULL) {
                    sprintf(tmpstr,
                            "# Could not open %s for Tagging", fname);
                    log_message(tmpstr);
                }
                else {

                    /* perform the needed tagging function */
                    Process(input_file, full_pathname, outfile, flags);

                    /* close the input file */
                    fclose(input_file);

#if defined(WIN16) || defined(WIN32)
					/* check message stream */
					checkMessages();
#endif /* defined(WIN16) || defined(WIN32) */
                }
            }

        } while (find_nextfile(&ff));
    }
}


/*----------------------------------------------------------------------------
 *
 * BatchFileProcess() tags a list of files within the file given.  Wildcards
 * are allowed in the listing file and as the listing file name.
 *
 ---------------------------------------------------------------------------*/

void BatchFileProcess(char *pathname, Flags * flags,
                       FILE * outfile, ArgList argInput,
                       ArgList argInputRaw)
{
    FILE *list_file;            /* file with list of files */

    char list_filename[MAXPATH + 1];    /* the current file name */
    char input_filename[MAXPATH + 1];   /* the current file name */
    char tmpstr[2 * MAXPATH + 1];       /* error string */
    struct file_info_struct ff; /* the file find structure */

    /* initialize ff */
    strcpy(ff.file_pattern, pathname);
    ff.file_attributes = _FA_NORMAL | _FA_READONLY | _FA_ARCHIVE |
        _FA_HIDDEN | _FA_SYSTEM;

    /* find the initial file matching pattern */
    if (!find_firstfile(&ff)) {

        /* bad file name */
        sprintf(tmpstr,
                "# Could not find the listfile '%s'", pathname);
        log_message(tmpstr);
    }
    else {

        /* loop through all available list files with pathname */
        do {

            /* make the file name */
            strcpy(list_filename, ff.file_path);
            strcat(list_filename, ff.file.name);

            /* open the list file and parse the file */
            if ((list_file = fopen(list_filename, "r")) ==
                (FILE *) NULL) {
                sprintf(tmpstr,
                        "# Could not open %s as a listfile", list_filename);
                log_message(tmpstr);
            }
            else {

                /* output the file we are parsing */
                sprintf(tmpstr,
                        "# Opening listfile %s", list_filename);
                log_message(tmpstr);

                /* loop while there are more filenames in the file */
                while (nextfile(list_file, input_filename, MAXPATH + 1)) {

                    /* do not tag this file set if seen before */
                    if (!ArgIsMember(argInputRaw, input_filename)) {

                        /* tag the obtained file path */
                        SingleFileProcess(input_filename, flags,
                                          outfile, argInput, argInputRaw);

                        /* register the input file as seen in the list file */
                        ArgRegisterArg(argInputRaw, input_filename);
                    }
                }

                /* close the list file */
                fclose(list_file);
            }
        } while (find_nextfile(&ff));
    }
}


/*----------------------------------------------------------------------------
 *
 * SingleFileExclude() places a single file into argInputRaw ArgList to
 * exclude a file from further processing.  Wildcards are allowed as file
 * names.
 *
 ---------------------------------------------------------------------------*/

void SingleFileExclude(char *pathname, Flags * flags, ArgList argInputRaw)
{
    char fname[MAXPATH + 1];    /* the current file name */
    char full_pathname[MAXPATH + 1];    /* the full path and filename */
    struct file_info_struct ff; /* the file find structure */

    /* initialize ff */
    strcpy(ff.file_pattern, pathname);
    ff.file_attributes = _FA_NORMAL | _FA_READONLY | _FA_ARCHIVE |
        _FA_HIDDEN | _FA_SYSTEM;

    /* find the initial file matching pattern */
    if (find_firstfile(&ff)) {

        /* loop through all matching files in the parameter */
        do {

            /* make the file name */
            strcpy(fname, ff.file_path);
            strcat(fname, ff.file.name);

            /* get the fully qualified pathname if wanted */
            if (!flags->use_relative_pathnames) {
                _fullpath(full_pathname, fname, MAXPATH);
            }
            else {
                strcpy(full_pathname, fname);
            }

            /* convert pathname to lower case */
            strlwr(full_pathname);

            /* register the original input file name after verifying that we
             * have not previously processed this file */
            if (!ArgIsMember(argInputRaw, full_pathname)) {

                ArgRegisterName(argInputRaw, full_pathname);
            }

        } while (find_nextfile(&ff));
    }
}


/*----------------------------------------------------------------------------
 *
 * BatchFileExclude() places a list of files within the argInputRaw ArgList
 * to inhibit processing of those files.  Wildcards are allowed in the
 * listing file and as the listing file name.
 *
 ---------------------------------------------------------------------------*/

void BatchFileExclude(char *pathname, Flags * flags, ArgList argInputRaw)
{
    FILE *list_file;            /* file with list of files */

    char list_filename[MAXPATH + 1];    /* the current file name */
    char input_filename[MAXPATH + 1];   /* the current file name */
    char tmpstr[2 * MAXPATH + 1];       /* error string */

    struct file_info_struct ff; /* the file find structure */

    /* initialize ff */
    strcpy(ff.file_pattern, pathname);
    ff.file_attributes = _FA_NORMAL | _FA_READONLY | _FA_ARCHIVE |
        _FA_HIDDEN | _FA_SYSTEM;

    /* find the initial file matching pattern */
    if (!find_firstfile(&ff)) {

        /* bad file name */
        sprintf(tmpstr,
                "# Could not find the Exclude listfile '%s'", pathname);
		log_error(tmpstr);
		external_cleanup();
        exit(1);
    }
    else {

        /* loop through all available list files with pathname */
        do {

            /* make the file name */
            strcpy(list_filename, ff.file_path);
            strcat(list_filename, ff.file.name);

            /* open the list file and parse the file */
            if ((list_file = fopen(list_filename, "r")) ==
                (FILE *) NULL) {
                sprintf(tmpstr,
                        "# Could not open %s as an Exclude listfile",
                        list_filename);
				log_error(tmpstr);
				external_cleanup();
                exit(1);
            }
            else {

                /* make the file name */
                strcpy(list_filename, ff.file_path);
                strcat(list_filename, ff.file.name);

                /* loop while there are more filenames in the file */
                while (nextfile(list_file, input_filename, MAXPATH + 1)) {

                    /* do not register this file set if seen before */
                    if (!ArgIsMember(argInputRaw, input_filename)) {

                        /* exclude the obtained file path */
                        SingleFileExclude(input_filename, flags, argInputRaw);

                        /* register the input file as seen in the list file */
                        ArgRegisterArg(argInputRaw, input_filename);
                    }
                }

                /* close the list file */
                fclose(list_file);
            }

        } while (find_nextfile(&ff));
    }
}


/*----------------------------------------------------------------------------
 *
 * Validate() calls the CRC validation routines and exits with a warning if
 * there are problems.
 *
 ---------------------------------------------------------------------------*/

void Validate(char *filename)
{
    char tmpstr[2 * MAXPATH + 1];       /* error string */
	int strlength;

	/* add the extension if the OS did not do so */
	strcpy(tmpstr, filename);
	strlength = strlen(filename);
	if (strlength > 4 &&
		tmpstr[strlength - 4] != '.') {
		strcat(tmpstr, ".exe");
	}
	
    /* validate the executable against the internally stored CRC */
    switch (validatecrc(tmpstr)) {

	case CRC_VALID:

		break;

	case CRC_INVALID:
	case CRC_ISZERO:

		/* internal failure */
		sprintf(tmpstr, "%s%s%s",
				"This executable fails internal CRC checking due to ",
				"external modification. \nPlease validate this copy ",
				"and scan for virus infection.");
		log_error(tmpstr);
		external_cleanup();
		exit(1);
		break;

	case CRC_NOMEM:

		log_error("# Validate() - Unable to compute CRC, out of memory");
		external_cleanup();
		exit(1);
		break;

	case CRC_FILEERR:
		log_message("# Validate() - Unable to compute CRC, could not find program file");
		external_cleanup();
		exit(1);
		break;

	default:

		break;
    }
}


/*----------------------------------------------------------------------------
 *
 * main loops through the parameter list and calls the appropriate parsers
 *
 ---------------------------------------------------------------------------*/
#if defined(WIN16) || defined(WIN32)
int __cdecl tags_main(int argc, char *argv[])
#else
int __cdecl main(int argc, char *argv[])
#endif /* defined(WIN16) || defined(WIN32) */
{
    Flags flags;

    char tmpstr[2 * MAXPATH + 1];   /* error and temp string */

    char *environ_ptr;          /* holds pointer to TMP variable */
    int environ_len;            /* holds length of TMP variable */

    ArgList argSort;            /* the list of files sent to sort */
    ArgList argInput;           /* the list of input files */
    ArgList argInputRaw;        /* the list of command line input files and
                                 * files which are excluded */
    struct tm *now;             /* clock variables */
    time_t cur_clock;

    unsigned int i;             /* temp looping var */
    int switch_count;           /* the number of switches found */

    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];

    FILE *output_stream;        /* the output stream */
    FILE *tag_stream;           /* the stream for temporary output */

    /* Register CTRL+C handler. */
    signal(SIGINT, CtrlCHandler);

    /* validate the executable against the internally stored CRC */
    Validate(argv[0]);

    /* obtain parts of the startup path */
    _splitpath(argv[0], drive, dir, fname, ext);

    /* if not enough parameters than Usage() */
    if (argc < 2)
        Usage(fname);

    /* compute the current time */
    time(&cur_clock);
    now = localtime(&cur_clock);

    /* initialize flags structure */
    init_flags(&flags);

    /* the initial file stuff */
    flags.log_filename[0] = '\0';
    flags.output_filename[0] = '\0';
    flags.compile_filename[0] = '\0';
    output_stream = stdout;

    /* get the TMP environment variable */
    environ_ptr = getenv("TMP");
    if (!environ_ptr)
		environ_ptr = getenv("TEMP");
	if (!environ_ptr) {
		environ_len = 0;
	} else {
		environ_len = strlen(environ_ptr);
	}

    /* obtain the intermediate tag filename */
    if (environ_len &&
        environ_ptr[environ_len - 1] != '\\' &&
        environ_ptr[environ_len - 1] != '/')
        tmp_tagfilename = tempnam("\\", "tg");
    else
        tmp_tagfilename = tempnam("", "tg");

    /* verify filename allocation */
    if (tmp_tagfilename == NULL) {
        log_error("# Could not create temporary files for output");
		external_cleanup();
        exit(1);
    }

    /* init arglist variable */
    argSort = CreateArgList(ARGLIST_NORMAL);
    argInput = CreateArgList(ARGLIST_SORTED);
    argInputRaw = CreateArgList(ARGLIST_SORTED);

	/* preparse for response file deletion flag */
    for (i = 1; i < (unsigned) argc; i++) {
		if ((argv[i][0] == '/' ||
			 argv[i][0] == '-') &&
			(argv[i][1] == 'd' ||
			 argv[i][1] == 'D')) {
			
			/* found it */
			flags.delete_response_files = TRUE;
			break;
		}
	}
	
    /* preparse for global switches */
    switch_count = 0;
    for (i = 1; i < (unsigned) argc; i++) {

        switch (argv[i][0]) {

            case '/':
            case '-':
                switch_count++;
                switch (argv[i][1]) {
                    case 'l':   /* tag from file list */
                    case 'L':

                        /* filename must hug the switch */
                        if (strlen(argv[i] + 2)) {
                            strcpy(flags.log_filename, argv[i] + 2);
                        }
                        else {
							log_error("# -l/-L switch used incorrectly");
							external_cleanup();
                            exit(1);
                        }

                        /* set overwrite flag */
                        if (argv[i][1] == 'l') {
                            flags.log_overwrite = TRUE;
                        }
                        else {
                            flags.log_overwrite = FALSE;
                        }

                        break;

                    case 't':   /* send to tag file */
                    case 'T':

                        /* filename must hug the switch */
                        if (strlen(argv[i] + 2)) {
                            strcpy(flags.output_filename, argv[i] + 2);
                            if (environ_len &&
                                environ_ptr[environ_len - 1] != '\\' &&
                                environ_ptr[environ_len - 1] != '/')
                                tmp_filename = tempnam("\\", "tf");
                            else
                                tmp_filename = tempnam("", "tf");

                            /* verify filename allocation */
                            if (tmp_filename == NULL) {
								log_error("# Could not create temporary files for output");
								external_cleanup();
                                exit(1);
                            }
                        }
                        else {
							log_error("# -t/-T switch used incorrectly");
							external_cleanup();
                            exit(1);
                        }

                        /* set output flag */
                        flags.output_file = TRUE;

						/* set overwrite flag */
                        if (argv[i][1] == 't') {
							flags.output_overwrite = TRUE;
						}
						else {
							flags.output_overwrite = FALSE;
						}
                        break;

                    case 'p':   /* compiled file */
                    case 'P':

                        /* filename must hug the switch */
                        if (strlen(argv[i] + 2)) {
                            strcpy(flags.compile_filename, argv[i] + 2);

							/* set output flag */
							flags.convert_file = TRUE;

                        }
                        else {
							log_error("# -p switch used incorrectly");
							external_cleanup();
                            exit(1);
                        }
                        break;

                    case 'j':   /* use junk filters */
                    case 'J':

                        flags.filter_junk = TRUE;
                        break;

                    case 'n':   /* do not sort the output */
                    case 'N':

                        /* set tag_type */
                        flags.sort_tags = FALSE;
                        break;

#if defined(MSDOS) || defined(_CONSOLE)
                    case 'm':   /* merge files only */
                    case 'M':

                        /* set tag_type */
                        flags.tag_type = MERGE;
                        break;

                    case 's':   /* sort files only */
                    case 'S':

                        /* set tag_type */
                        flags.tag_type = SORT;
                        break;
#endif

                    case 'o':   /* output format specifier follows */
                    case 'O':   /* the last format found is the one */

                        /* set the option modifier */
                        parse_output_flags(argv[i], &flags);
                        break;

                    case 'r':   /* output format specifier follows */
                    case 'R':   /* the last format found is the one */

                        /* set to use relative pathnames */
                        flags.use_relative_pathnames = TRUE;
                        break;

                    case 'q':   /* use quiet mode (no logging to stderr) */
                    case 'Q':
                        flags.quiet = TRUE;
                        break;

                    case 'i':   /* use case sensitive sort */
                    case 'I':
                        flags.case_sensitive = TRUE;
                        break;

                    case 'x':   /* exclude this file or filelist */
                    case 'X':
                        /* filename must hug the switch */
                        if (!strlen(argv[i] + 2)) {
							log_error("# -x switch used incorrectly");
							external_cleanup();
                            exit(1);
                        }
                        else {

                            /* check if list file exclude or normal file
                             * exclude */

                            if (argv[i][2] != '@') {

                                /* this is a particular exclusion */
                                /* do not use this Exclude file if seen
                                 * previously */
                                if (!ArgIsMember(argInputRaw, argv[i] + 2)) {

                                    /* process the Exclude file */
                                    SingleFileExclude(argv[i] + 2, &flags,
                                                      argInputRaw);

                                    /* register the input list file as seen
                                     * on the command line */
                                    ArgRegisterArg(argInputRaw, argv[i] + 2);
                                }
                            }
                            else {

                                /* this is an exclusion list file */
                                if (strlen(argv[i] + 3)) {

                                    /* do not use this Exclude list file if
                                     * seen previously */
                                    if (!ArgIsMember(argInputRaw,
                                                     argv[i] + 3)) {

                                        /* process the list file */
                                        BatchFileExclude(argv[i] + 3, &flags,
                                                         argInputRaw);

										/* delete the response file if appropriate */
										if (flags.delete_response_files) {
											sprintf(tmpstr,
													"# Removing exclusion listfile %s", argv[i] + 3);
											log_message(tmpstr);
											remove(argv[i] + 3);
										}

                                        /* register the input list file as
                                         * seen on the command line */
                                        ArgRegisterArg(argInputRaw,
                                                       argv[i] + 3);
                                    }
                                }
                                else {
									log_error("# @ syntax error");
									external_cleanup();
                                    exit(1);
                                }
                            }
                        }
                        break;

                    case 'h':
                    case 'H':
					case '?':
                        Help(fname);
                        break;

                    default:
                        break;
                }
                break;

            default:
                break;
        }
    }

    /* open the log file */
    if (!log_open(flags.log_filename, flags.quiet, flags.log_overwrite)) {
        sprintf(tmpstr, "# Could not open %s as a logfile", flags.log_filename);
        log_error(tmpstr);
		external_cleanup();
        exit(1);
    }

    /* print initial messages */
    sprintf(tmpstr, "# %s -- %s", Version, Author);
    log_info(tmpstr);
    sprintf(tmpstr, "# Tagging performed: %s", asctime(now));
	*strchr(tmpstr, '\n') = '\0';
    log_info(tmpstr);
	log_info("#");

	/* output global flags data */
	log_flags(&flags);

#if defined(WIN16) || defined(WIN32)
	checkMessages();
#endif

    /* exit if no input files specified */
    if (switch_count == argc - 1) {
        log_error("# No input files specified");
		external_cleanup();
        exit(1);
    }

#if defined(WIN16) || defined(WIN32)
	/* check for output file spec (required for GUI versions) */
	if (!flags.output_file) {
		log_error("# No output file specified");
		external_cleanup();
		exit(1);
	}
#endif
	
    /* open the temporary tag file */
    if ((tag_stream = fopen(tmp_tagfilename, "w")) ==
        (FILE *) NULL) {
		log_error("# Internal error opening temporary files");
		external_cleanup();
        exit(1);
    }

    if (flags.tag_type == MERGE) {
        log_message("# Merging previously processed tag files");
    }

    if (flags.tag_type == SORT) {
        log_message("# Sorting input files only");
    }

    /* enter the main loop */
    for (argc--, argv++; argc; argc--, argv++) {

        /* parse the argument list */
        switch (argv[0][0]) {
            case '@':
                if (strlen(argv[0] + 1)) {

                    /* do not use this list file if seen previously */
                    if (!ArgIsMember(argInputRaw, argv[0] + 1)) {

                        /* process the list file */
                        BatchFileProcess(argv[0] + 1, &flags,
                                         tag_stream, argInput,
                                         argInputRaw);

						/* delete the response file if appropriate */
						if (flags.delete_response_files) {
							sprintf(tmpstr,
									"# Removing listfile %s", argv[0] + 1);
							log_message(tmpstr);
							remove(argv[0] + 1);
						}

                        /* register the input list file as seen on the
                         * command line */
                        ArgRegisterArg(argInputRaw, argv[0] + 1);
                    }
                    break;
                }
                else {
                    log_message("# @ syntax error -- continuing ...");
                }
                break;

            case '/':
            case '-':
                switch (argv[0][1]) {
                    case 'a':   /* use ASM tagging scheme */
                    case 'A':

                        /* ignore if merge or sort tag_types */
                        if (flags.tag_type != MERGE &&
                            flags.tag_type != SORT) {

                            /* set the option flags */
                            parse_ASM_flags(*argv, &flags);
                        }
                        break;

                    case 'c':   /* use C tagging scheme */
                    case 'C':

                        /* ignore if merge or sort tag_types */
                        if (flags.tag_type != MERGE &&
                            flags.tag_type != SORT) {

                            /* set the option flags */
                            parse_C_flags(*argv, &flags);
                        }
                        break;

                    default:
                        break;
                }
                break;

            default:

                /* this is a file parameter */
                /* do not tag this file set if seen previously */
                if (!ArgIsMember(argInputRaw, *argv)) {

                    /* process the file */
                    SingleFileProcess(*argv, &flags, tag_stream,
                                      argInput, argInputRaw);

                    /* register the input file as seen on the command line */
                    ArgRegisterArg(argInputRaw, *argv);
                }
                break;
        }
    }

    /* free the command line input arglist */
    DestroyArgList(argInputRaw);

    /* close the intermediate tag file */
    fclose(tag_stream);

    /* do the sorting and collating */
    if (argInput->num_args) {

        /* register elements needed for the sort module unless no sorting is
         * requested */
        if (flags.sort_tags) {

            /* register the file name in argf */
            ArgRegisterArg(argSort, fname);

            /* add the sort parameters to beginning of the name register
             * array */
            ArgRegisterArg(argSort, "-u");      /* delete duplicate lines */

            /* if not a case sensitive sort then register case fold */
            if (!flags.case_sensitive)
                ArgRegisterArg(argSort, "-f");  /* fold to lower case */

            /* if merge then add argument to sort parameters */
            if (flags.tag_type == MERGE)
                ArgRegisterArg(argSort, "-m");  /* merge sort files */

            /* if output file specified then tmp_filename as output */
            if (flags.output_file &&
                flags.tag_type != MERGE &&
                flags.tag_type != SORT) {
                sprintf(tmpstr, "-o%s", tmp_filename);
                ArgRegisterArg(argSort, tmpstr);
            }

            if (flags.tag_type == MERGE ||
                flags.tag_type == SORT) {

                /* copy the names from ArgInput to ArgSort */
                ArgCopy(argSort, argInput);
            }
            else {
                /* register the intermediate tag file to be sorted */
                ArgRegisterName(argSort, tmp_tagfilename);
            }

            /* log the activity */
            if (flags.tag_type == MERGE) {
                log_message("# Merging Input Files...");
            }
            else {
                log_message("# Sorting ...");
            }

            /* perform the sort */
            sort_main(argSort->num_args, argSort->argv);
        }
        else {

			if (flags.output_file) {
				/* no need to copy the file twice */
				free(tmp_filename);
				tmp_filename = tmp_tagfilename;
				tmp_tagfilename = NULL;
			}
			else {

				/* register the intermediate tag file */
				ArgRegisterName(argSort, tmp_tagfilename);

				/* output the results */
				ArgToOutputStream(output_stream, argSort);

				/* close the output file if necessary */
				if (flags.output_file) {
					fclose(output_stream);
				}
			}
        }
    }

    /* merge the temporary output file and the current tag file */
    if (flags.output_file &&
        argInput->num_args) {
		if (flags.output_overwrite) {
			/* remove the old file */
			if (!access(flags.output_filename, 00) &&
				remove(flags.output_filename)) {
				sprintf(tmpstr, "# Error removing %s", flags.output_filename);
				log_message(tmpstr);
			}
			else {
				log_message("# Copying into tag file...");
				MergeFile(tmp_filename, flags.output_filename, argInput, &flags);
			}
		}
		else {
	        log_message("# Merging into tag file ...");
	        MergeFile(tmp_filename, flags.output_filename, argInput, &flags);
		}
    }

    /* delete the temporary file if used and free its memory */
    if (flags.output_file) {
        remove(tmp_filename);
        free(tmp_filename);
    }

    /* delete the intermediate tags file and free its memory */
	if (tmp_tagfilename) {
		remove(tmp_tagfilename);
		free(tmp_tagfilename);
	}

	/* Generate the Codewright compiled file if appropriate */
	if (flags.output_file &&
		flags.oc &&
		flags.convert_file &&
		!access(flags.output_filename, 00)) {
		log_message( "# Converting to Codewright Database format ...");

		if (compileTagFile(flags.output_filename, flags.compile_filename)) {
			log_message("# Codewright conversion complete");
		} else {
			log_message("# Failed to generate converted file");
		}
	}

    /* list the number of files tagged and closing message */
    sprintf(tmpstr,
            "# Tagging operation complete with %u file(s) tagged",
            argInput->num_args);
    log_message(tmpstr);

    /* free the arglist structures */
    DestroyArgList(argSort);
    DestroyArgList(argInput);

    /* close the log file */
    log_close();

    /* successful exit */
    return (0);
}


#if defined(WIN16) || defined(WIN32)

extern int __argc;
extern char **__argv;
extern char *__environ[];

/*----------------------------------------------------------------------------
 *
 * exit_func is the windows exit check
 *
 ---------------------------------------------------------------------------*/
void
exit_func(void)
{
	if (exiting == FALSE) {
		MessageBox(0, "Unexpected TAGS error, exiting.", "Error",
				   MB_SYSTEMMODAL | MB_OK | MB_ICONSTOP);
	}
	if (hTagsDlg) {
		DestroyWindow(hTagsDlg);
		hTagsDlg = (HWND) 0;
	}
}


/*----------------------------------------------------------------------------
 *
 * checkMessages is the peek message loop responsible for dialog dispatching
 *
 ---------------------------------------------------------------------------*/
static void
checkMessages(void)
{
	MSG msg;

	while (PeekMessage(&msg, (HWND) 0, (UINT) 0, (UINT) 0, PM_NOREMOVE)) {
		if (GetMessage(&msg, (HWND) 0, (UINT) 0, (UINT) 0)) {
			if (!IsDialogMessage(hTagsDlg, &msg)) {
				TranslateMessage(&msg);
				DispatchMessage (&msg);
			}
		}
	}
	if (exiting) {
		exit(1);
	}
}


/*----------------------------------------------------------------------------
 *
 * centerWindow centers hChild on hParent
 *
 ---------------------------------------------------------------------------*/
static void
centerWindow(
			 HWND hParent,		// The parent window
			 HWND hChild		// The child window
			 )
{
	RECT parentRect;			// rectangle of parent
	RECT childRect;				// rectangle of child
	int childX;					// horizontal child size
	int childY;					// vertical child size
	int screenX;				// horizontal screen resolution
	int screenY;				// vertical screen resolution

	/* get the screen coordinates */
	screenX = GetSystemMetrics(SM_CXSCREEN);
	screenY = GetSystemMetrics(SM_CYSCREEN);

	/* get the child dimensions */
	GetWindowRect(hChild, &childRect);
	childX = childRect.right - childRect.left;
	childY = childRect.bottom - childRect.top;

	/* obtain coordinates for parent */
	if (IsWindow(hParent) &&
		!IsIconic(hParent)) {
		GetWindowRect(hParent, &parentRect);
		/* clip the parent to the current screen */
		if (parentRect.left < 0) {
			parentRect.left = 0;
		}
		if (parentRect.top < 0) {
			parentRect.top = 0;
		}
		if (parentRect.right > screenX) {
			parentRect.right = screenX;
		}
		if (parentRect.bottom > screenY) {
			parentRect.bottom = screenY;
		}
	} else {
		parentRect.left = 0;
		parentRect.top = 0;
		parentRect.right = screenX;
		parentRect.bottom = screenY;
	}

	/* compute new coordinates */
	childRect.left = (parentRect.left +
					  ((parentRect.right - parentRect.left) - childX) / 2);
	childRect.top  = (parentRect.top +
					  ((parentRect.bottom - parentRect.top) - childY) / 2);

	/* make sure that the coordinates are at least within the parents
	 * client window. */
	if (childRect.left < parentRect.left + GetSystemMetrics(SM_CXBORDER)) {
		childRect.left = parentRect.left + GetSystemMetrics(SM_CXBORDER);
	}
	if (childRect.top < parentRect.top + GetSystemMetrics(SM_CYCAPTION)) {
		childRect.top = parentRect.top + GetSystemMetrics(SM_CYCAPTION);
	}
	/* make sure that the coordinates are within the screen edges */
	if (childRect.left < 0) {
		childRect.left = 0;
	}
	if (childRect.top < 0) {
		childRect.top = 0;
	}
	if (childRect.right > screenX) {
		if (childX < screenX) {
			childRect.left = screenX - childX;
		} else {
			childRect.left = 0;
		}
	}
	if (childRect.bottom > screenY) {
		if (childY < screenY) {
			childRect.top = screenY - childY;
		} else {
			childRect.top = 0;
		}
	}
	SetWindowPos(hChild, NULL, childRect.left, childRect.top, 0, 0,
			 SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE | SWP_NOREDRAW);
}


/*----------------------------------------------------------------------------
 *
 * TagsUsageProc is the message proc for the usage dialog
 *
 ---------------------------------------------------------------------------*/
DWORD EXPORT WINAPI
TagsUsageProc(
			  HWND hDlg,
			  UINT message,
			  WPARAM wParam,
			  LPARAM lParam
			  )
{
	static cxDialogOld;
	static cyDialogOld;
	
	switch (message) {
    case WM_INITDIALOG:
	{
		HWND hCtlButton = GetDlgItem(hDlg, IDOK);
		RECT tmpRect;
		POINT tmpPoint;
		int cx;
	
		/* set global HWND */
		hTagsDlg = hDlg;
		
		/* obtain last known size */
		GetClientRect(hDlg, &tmpRect);
		cxDialogOld = tmpRect.right - tmpRect.left;
		cyDialogOld = tmpRect.bottom - tmpRect.top;
		
		/* obtain current rectangle for button and set its initial position */
		GetWindowRect(hCtlButton, &tmpRect);
		cx = tmpRect.right - tmpRect.left;
		tmpPoint.x = tmpRect.left;
		tmpPoint.y = tmpRect.top;
		ScreenToClient(hDlg, &tmpPoint);
		
		SetWindowPos(hCtlButton, (HWND) 0, (cxDialogOld - cx) >> 1,
					 tmpPoint.y, 0, 0, SWP_NOSIZE | SWP_NOZORDER);

		/* init the dialog location and controls */
		centerWindow(NULL, hDlg);
		SetWindowText(hDlg, Version);
#ifdef WIN32
		Help("TagsW32");
#else
		Help("TagsW16");
#endif
        break;
	}
		
	case WM_SIZE:
	{
		HWND hCtlList = GetDlgItem(hDlg, IDC_LIST);
		HWND hCtlButton = GetDlgItem(hDlg, IDOK);
		
		RECT tmpRect;
		POINT tmpPoint;
		
		int cxDialogNew = LOWORD(lParam);
		int cyDialogNew = HIWORD(lParam);
		int cx;
		int cy;
		
		/* obtain current rectangle for list */
		GetWindowRect(hCtlList, &tmpRect);
		cx = tmpRect.right - tmpRect.left;
		cy = tmpRect.bottom - tmpRect.top;
		
		/* compute new size */
		cx += cxDialogNew - cxDialogOld;
		cy += cyDialogNew - cyDialogOld;

		/* set new list size */
		SetWindowPos(hCtlList, (HWND) 0, 0, 0, cx, cy, 
					 SWP_NOMOVE | SWP_NOZORDER);
		
		/* obtain current rectangle for button and set its new position */
		GetWindowRect(hCtlButton, &tmpRect);
		cx = tmpRect.right - tmpRect.left;
		tmpPoint.x = tmpRect.left;
		tmpPoint.y = tmpRect.top;
		ScreenToClient(hDlg, &tmpPoint);
		
		SetWindowPos(hCtlButton, (HWND) 0, 
					 (cxDialogNew - cx) >> 1,
					 tmpPoint.y + (cyDialogNew - cyDialogOld),
					 0, 0, SWP_NOSIZE | SWP_NOZORDER);

		/* update saved dialog size */
		cxDialogOld = cxDialogNew;
		cyDialogOld = cyDialogNew;
		
		break;
	}
		
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
		case IDCANCEL:
			/* exit */
			hTagsDlg = NULL;
			EndDialog(hDlg, 0);
			break;

		default:
			return(FALSE);
		}
		break;

	case WM_CLOSE:
		/* exit */
		hTagsDlg = NULL;
		EndDialog(hDlg, 0);
		break;

	default:
		return(FALSE);
	}
	return(TRUE);
}


/*----------------------------------------------------------------------------
 *
 * TagsProc is the message proc for the status dialog
 *
 ---------------------------------------------------------------------------*/
DWORD EXPORT WINAPI
TagsProc(
		 HWND hDlg,
		 UINT message,
		 WPARAM wParam,
		 LPARAM lParam
		 )
{
	switch (message) {
    case WM_INITDIALOG:
		/* init the dialog location and controls */
		centerWindow(NULL, hDlg);
		SetWindowText(hDlg, Version);
        break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
		case IDCANCEL:
			/* exit */
			exiting = TRUE;
			break;

		default:
			return(FALSE);
		}
		break;

	case WM_CLOSE:
		/* exit */
		exiting = TRUE;
		break;

	default:
		return(FALSE);
	}
	return(TRUE);
}


/*----------------------------------------------------------------------------
 *
 * Windows main creates the status dialog and lets loose
 *
 ---------------------------------------------------------------------------*/
int FAR PASCAL
WinMain(
		HINSTANCE hInst,
		HINSTANCE hPrevInstance,
		LPSTR lpCmdLine,
		int nCmdShow)
{
	int		argc = __argc;
	LPSTR	*argv = NULL;
	LPSTR	buf;

	UINT	dialogId = IDD_TAGS_DIALOG;
	int		rval = 0;
	int		i;

	if (argc < 2) {
		dialogId = IDD_USAGE_DIALOG;
	} 
	
	/* copy over the argc and argv parameters */
	argv = (LPSTR *) malloc(argc * sizeof(LPVOID));
	for (i = 0; i < argc; i++) {
		buf = (LPSTR) malloc(strlen(__argv[i]) + 1);
		strcpy(buf, __argv[i]);
		argv[i] = buf;

		/* check for usage dialog */
		if ((argv[i][0] == '-' ||
			 argv[i][0] == '/') &&
			(argv[i][1] == 'h' ||
			 argv[i][1] == 'H' ||
			 argv[i][1] == '?')) {
			dialogId = IDD_USAGE_DIALOG;
		}
	}

	if (dialogId == IDD_USAGE_DIALOG) {
		Validate(argv[0]);
		DialogBox(hInst, MAKEINTRESOURCE(IDD_USAGE_DIALOG),
				  NULL, (FARPROC) TagsUsageProc);
	} 
	else {
		/* prep the dialog */
		hTagsDlg = CreateDialog(hInst, MAKEINTRESOURCE(IDD_TAGS_DIALOG),
								NULL, (FARPROC) TagsProc);

		/* insert an exit handler for strange cases */
		_onexit((_onexit_t) exit_func);

		/* flush the message queue */
		checkMessages();

		/* launch */
		rval = tags_main(argc, argv);

		/* completed appropriately */
		exiting = TRUE;
		if (hTagsDlg) {
			DestroyWindow(hTagsDlg);
			hTagsDlg = (HWND) 0;
		}
	}

	/* free memory allocated for argv */
	for (i = 0; i < argc; i++) {
		free(argv[i]);
	}
	free(argv);

	return rval;
}

#endif /* defined(WIN16) || defined(WIN32) */
