//
// NAME: RayTracer3.Cpp
//
// DESCRIPTION:
//
//  This demo is the third in the ray tracer demo set. The first one
//  demonstrated how to build a scene programatically, then render it to
//  disk. The second used the scene description language support, but was
//  still single threaded.
//
//  This versions creates an array object to hold the whole image, then it
//  kicks of the requested number of rendering threads. The threads run
//  through the pixels of the image, updating the array. The main thread
//  waits for the other threads to finish, then it saves the data in memory
//  out to disk.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/18/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  This program uses a very simplistic threading architecture in order
//      to make it an incremental improvement over its predecessor. As such
//      it uses a lot of global data for the threads to all access. In
//      subsequent versions, this will be much improved through the use of
//      a specialized thread derivative. This will encapsulate what the
//      threads do and the data they manipulate. It will be much, much
//      cleaner than this version. But, this one does show the raw
//      capabilities of the system so its worth studying.
//
//  2)  It also uses a lot of memory and keeps the whole image in memory
//      until its done. This makes it go fast, but its not a practical
//      architecture because large images would take a lot of memory and
//      because it would lose any work done if something goes wrong.
//
//      Later reincarnations will update this architecture to make it more
//      like a practical implementation.
//
// MODIFICATION LOG:
//


// ----------------------------------------------------------------------------
//  CIDLib includes. CIDTracer is relatively high level, so including it gets
//  everything that we need.
// ----------------------------------------------------------------------------
#include    "CIDTracer.Hpp"


// ----------------------------------------------------------------------------
//  Include our own headers
// ----------------------------------------------------------------------------
#include    "Tracer3_ErrorIds.Hpp"
#include    "Tracer3_MessageIds.Hpp"


// ----------------------------------------------------------------------------
//  Local types
//
//  TLineBuffer
//      This is an instantiation of the TBaseArray template, in terms of a
//      TCard1 type. We use this to create a raw byte array for the image
//      data in memory.
// ----------------------------------------------------------------------------
typedef TBaseArray<tCIDLib::TCard1> TLineBuffer;


// ----------------------------------------------------------------------------
//  Local function prototypes
//
//  __eMainThead
//      This is the thread function for the main thread that provides the
//      execution for this program.
//
//  __eRenderThread
//      This is the thread function for the rendering threads. There are
//      as many instances of this as the user indicates.
// ----------------------------------------------------------------------------
static tCIDLib::EExitCodes  __eMainThread(TThread&, tCIDLib::TVoid*);
static tCIDLib::EExitCodes  __eRenderThread(TThread&, tCIDLib::TVoid*);
static tCIDLib::TVoid       __ShowUsage();



// ----------------------------------------------------------------------------
//  Local constant data
//
//  __c4MinSize
//  __c4MaxSize
//      These are the min/max sizes of the image in pixels.
//
//  __c4MaxThreads
//      This is the maximum number of rendering threads that the program
//      supports.
//
//  __c4RequiredParms
//      The count of required command line parameters.
// ----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4MinSize         = 16;
static const tCIDLib::TCard4    __c4MaxSize         = 4096;
static const tCIDLib::TCard4    __c4MaxThreads      = 16;
static const tCIDLib::TCard4    __c4RequiredParms   = 5;


// ----------------------------------------------------------------------------
//  Local data declarations
//
//  __apthrList
//      This is the list of pointers to threads. We create new threads
//      objects for each thread that the user asks us to use. Note that
//      we cannot have an array of thread objects themselves, because
//      that would cause a run of the default constructor which is not
//      available. In subsequent versions, where we have our own derived
//      thread class, we will be able to do this.
//
//  __baImageBuf
//      An array of bytes for the colors of the pels in the image. We
//      don't know how large the image is yet, so we just create it
//      with a single element. It will be reallocated later for the
//      needed size.
//
//  __c4CurColumn
//  __c4CurRow
//      These are used by the rendering threads to keep up with what's the
//      next pixel to render. They cooperate via a mutex to synchronize
//      access to these values.
//
//  __c4HorzSize
//  __c4VertSize
//      These are set to the user's desired pixel image size and provide
//      the limits for the rendering loop. They are set via the command
//      line (required parameters.)
//
//  __c4Threads
//      This is the number of threads to use when rendering. Defaults to
//      1 if not overridden on the command line. The /Threads= parameter
//      sets it.
//
//  __c4ThreadsRunning
//      This is used by the rendering threads to know when the last of them
//      dies, so it can wake up the main thread. It is initialized to the
//      number of threads before the rendering starts. Each one bumps it
//      down as it dies because of no more work todo.
//
//  __conOut
//      A console stream for our program input and output. The console
//      class provides nice editing and command recall support. It is
//      redirectable by default. We don't use the interactive support and
//      the default constructor disables it.
//
//  __crsSync
//      This is the critical section that is used by the rendering threads
//      to control access to the line buffer and current column values.
//
//  __evWaitMain
//      This event semaphore is used by the main thread to wait for the
//      rendering threads to complete the image. The last rendering
//      thread to complete releases the main thread.
//
//  __facTracer3
//      This is the facility object for the Exe. Be sure that its before
//      __strTitle, below, because its used by that object! We don't have
//      any particular needs, so we just create a generic TFacility object.
//
//  __pathFileName
//  __pathOutFileName
//      These are used to hold the path to the initial scene file and the
//      output file. They are required parameters.
//
//  __pathInclude
//      This is the scene file include path. It is set by the /Include
//      command line parameter. It is used to search for imported files
//      in the scene description language files.
//
//  __strTitle
//      The title string for the program. This method of construction will
//      load the string from a resource, add 32 extra bytes for replacement
//      parameters, and do the replacement with the maj/min version values.
//      Note that this is somewhat dangerous because it will cause a trap
//      during global construction if something goes awray, but that's just
//      the nature of global objects.
//
//  __viewScene
//      This is the view object that represents the scene as a whole.
// ----------------------------------------------------------------------------
static TThread*         __apthrList[__c4MaxThreads];
static TLineBuffer      __baImageBuf(1);
static tCIDLib::TCard4  __c4CurColumn;
static tCIDLib::TCard4  __c4CurRow;
static tCIDLib::TCard4  __c4HorzSize;
static tCIDLib::TCard4  __c4Threads(1);
static tCIDLib::TCard4  __c4ThreadsRunning(1);
static tCIDLib::TCard4  __c4VertSize;
static TConsole         __conOut;
static TCriticalSection __crsSync;
static TEvent           __evWaitMain(tCIDLib::EEventState_Reset);
static TFacility        __facTracer3
                        (
                            L"Tracer3"
                            , tCIDLib::EModType_Exe
                            , kCIDLib::c4MajVersion
                            , kCIDLib::c4MinVersion
                        );
static TPathStr         __pathFileName;
static TPathStr         __pathOutFileName;
static TPathStr         __pathInclude;
static TString          __strTitle
                        (
                            kTraceMsgs::midProgramTitle
                            , __facTracer3
                            , 32
                            , TCardinal(kCIDLib::c4MajVersion)
                            , TCardinal(kCIDLib::c4MinVersion)
                        );
static TViewGeometry    __viewScene;


// ----------------------------------------------------------------------------
//  Include magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"MainThread", __eMainThread))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __GetParms
//
// DESCRIPTION:
//
//  This method checks the command line parms for all the required and
//  optional parameters. It finds them, confirms them, and sets global
//  variables as needed to reflect the parameters.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __GetParms()
{
    tCIDLib::TCard4     c4Val;
    tCIDLib::TFloat8    f8Val;
    tCIDLib::TInt4      i4Val;
    TString             strTmp(kCIDLib::pszEmptyZStr, 128);

    //
    //  Check out the command line parms and see if we have at least
    //  the required number of them.
    //
    if (TSysInfo::c4CmdLineArgCount() < __c4RequiredParms)
    {
        __ShowUsage();
        facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
    }

    //
    //  Get the image size parameters. If they don't resolve to values
    //  between the min and max, then issue an error.
    //
    try
    {
        TSysInfo::CmdLineArg(3, strTmp);
        __c4HorzSize = strTmp.c4Val();

        TSysInfo::CmdLineArg(4, strTmp);
        __c4VertSize = strTmp.c4Val();
    }

    catch(...)
    {
        __ShowUsage();
        facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
    }

    if (!bInRange(__c4HorzSize, __c4MinSize, __c4MaxSize)
    ||  !bInRange(__c4VertSize, __c4MinSize, __c4MaxSize))
    {
        __ShowUsage();
        facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
    }

    // They look ok, so set the scene dimensions on the view
    __viewScene.SetSceneDimensions(__c4HorzSize, __c4VertSize);

    // Get the input and output file names
    TSysInfo::CmdLineArg(1, __pathFileName);
    TSysInfo::CmdLineArg(2, __pathOutFileName);

    //
    //  Get the file name of the scene. Insure that it has no path
    //  component because this program does not want any.
    //
    if (__pathFileName.bQueryPath(strTmp))
    {
        __conOut << L"The scene file should not have any path component"
                 << NewLn;
        facCIDLib.ExitProcess(tCIDLib::EExit_BadParameters);
    }

    // Add the output file extension if needed
    if (!__pathOutFileName.bHasExt())
        __pathOutFileName.Append(L".Dis");


    // Loop through the command line args, start past the required ones
    const tCIDLib::TCard4 c4ArgCount = TSysInfo::c4CmdLineArgCount();
    for (tCIDLib::TCard4 c4Tmp = __c4RequiredParms; c4Tmp < c4ArgCount; c4Tmp++)
    {
        TSysInfo::CmdLineArg(c4Tmp, strTmp);

        if (strTmp.eCompareN(L"/Quality=", 9) == tCIDLib::ESort_Equal)
        {
            strTmp.Cut(0, 9);
            i4Val = strTmp.i4Val();
            if ((i4Val < tCIDTracer::EQualityLevels_Min)
            ||  (i4Val > tCIDTracer::EQualityLevels_Max))
            {
                __conOut << L"Quality option must have a value of "
                         << tCIDTracer::EQualityLevels_Min << L" to "
                         << tCIDTracer::EQualityLevels_Max << DNewLn;
                facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
            }
            __viewScene.eQuality(tCIDTracer::EQualityLevels(i4Val));
        }
         else if (strTmp.eCompareN(L"/Sample=", 8) == tCIDLib::ESort_Equal)
        {
            strTmp.Cut(0, 8);
            if (strTmp == L"Normal")
                __viewScene.eSample(tCIDTracer::ESampleMode_Normal);
            if (strTmp == L"Stochastic")
                __viewScene.eSample(tCIDTracer::ESampleMode_Stochastic);
            else if (strTmp == L"Statistical")
                __viewScene.eSample(tCIDTracer::ESampleMode_Statistical);
        }
         else if (strTmp.eCompareN(L"/Threshold=", 11) == tCIDLib::ESort_Equal)
        {
            strTmp.Cut(0, 11);
            f8Val = strTmp.f8Val();
            if ((f8Val < 0.0) || (f8Val > 3.0))
            {
                __conOut   << L"Sampling threshold must have a value "
                                    L"of 0.0 to 3.0" << DNewLn;
                facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
            }
            __viewScene.f8StatSampleThreshold(f8Val);
        }
         else if (strTmp.eCompareN(L"/Threads=", 9) == tCIDLib::ESort_Equal)
        {
            strTmp.Cut(0, 9);
            c4Val = strTmp.c4Val();
            if ((c4Val < 1) || (c4Val > __c4MaxThreads))
            {
                __conOut << L"Thread count must have a value of 1 to "
                         << __c4MaxThreads << DNewLn;
                facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
            }
            __c4Threads = c4Val;
            __c4ThreadsRunning = c4Val;
        }
         else if (strTmp.eCompareN(L"/Include=", 9) == tCIDLib::ESort_Equal)
        {
            strTmp.Cut(0, 9);
            __pathInclude = strTmp;
        }
    }
}   


//
// FUNCTION/METHOD NAME: __ShowUsage
//
// DESCRIPTION:
//
//  Shows the parameter usage for the program.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid __ShowUsage()
{
    __conOut
    <<  L"Usage: RayTracer3 SceneFile OutputFile hhh vvv [/option=val]" << DNewLn
    <<  L"   If not extension for the output file, it will be .Dis." << NewLn
    <<  L"   hhh and vvv are the horz/vertical size in pixels" << NewLn
    <<  L"        which must be 16 <= x <= 4096." << NewLn
    <<  L"   Options are:" << NewLn
    <<  L"        /Include=X   - where X is the scene include path" << NewLn
    <<  L"        /Quality=X   - where X is 0 to 5" << NewLn
    <<  L"        /Sample=X    - where X is Normal, Stocastic, Statistical" << NewLn
    <<  L"        /Threshold=X - where X is the statistical threshold" << NewLn
    <<  L"        /Threads=X   - where X is the thread count, default is 1" << DNewLn;
}


//
// FUNCTION/METHOD NAME: __eMainThread
//
// DESCRIPTION:
//
//  This is the main program thread. It sets up the scene, then uses the
//  ray tracer engine to trace it line by line, outputting each line to 
//  the output file.
// -------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread
//          pData is a pointer to the optional data buffer passed by our
//              creator.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values
//
tCIDLib::EExitCodes __eMainThread(TThread& thrThis, tCIDLib::TVoid* const)
{
    // Let our caller go
    thrThis.Sync();

    // Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    // Log our arrival
    __facTracer3.LogMsg
    (
        __FILE__
        , __LINE__
        , L"Tracer3 starting up..."
        , tCIDLib::ESev_Status
    );

    // Announce ourselves.
    __conOut    << NewLn << __strTitle << NewLn << L"Compiled: "
                << TString(__DATE__) << DNewLn;

    //
    //  Get all of the command line parms. This function will set up all
    //  of the global data to reflect the parameters we get. If any of
    //  them are not right, then we don't come back from this call. If any
    //  exceptions occur, we catch them and show the error.
    //
    try
    {
        __GetParms();
    }

    catch(const TError& errToCatch)
    {
        __conOut
                << L"Error occured while parsing optional parameters" << DNewLn
                << L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_BadParameters;
    }

    //
    //  Create the parser information object. This guy will do the parsing
    //  for us. We give it the view geometry object that we want it to
    //  fill in. If the parsing goes well, then we have our scene set up
    //  and ready. If not, it throws an exception.
    //
    //  We put it in a faux block so that the parse info object gets cleaned
    //  up naturally when we are done parsing.
    //
    {
        TParseInfo  prsiScene(__viewScene);

        //
        //  Ok, since this likely to fail in many cases, we go ahead and do
        //  the parsing now before event attempting to deal with the output
        //  file.
        //
        try
        {
            //
            //  Ask the scene parser to parse the input file. Pass the include
            //  path that we were given.
            //
            __conOut << L"Compiling file: " << __pathFileName << NewLn;
            prsiScene.ParseFile(__pathFileName, __pathInclude);
        }

        catch(const TError& errToCatch)
        {
            __conOut
                << L"An error occured while parsing scene description\r\n";

            // Display the error information
            __conOut << L"  Error: " << DNewLn << errToCatch << NewLn;
            return tCIDLib::EExit_RuntimeError;
        }

        catch(const TRTParseErr& rterrToCatch)
        {
            __conOut
                << L"An error occured while parsing scene description" << NewLn;

            // Display the error information
            __conOut
                    << L"     File: " << rterrToCatch.strFile << NewLn
                    << L"   Object: " << rterrToCatch.strObjFail << NewLn
                    << L" Line Num: " << rterrToCatch.c4LineNum << NewLn
                    << L" Err Text: " << rterrToCatch.strMsg << DNewLn;
            return tCIDLib::EExit_FatalError;
        }

        __conOut << L"Compilation successful" << NewLn;
    }

    //
    //  Now we need to create a binary file stream to stream the data out
    //  to. In this demo, we create a binary file object and open/create
    //  it. Then we ask it to create a binary file stream for us that
    //  is set up to stream on itself.
    //
    TBinaryFile bflTarget(__pathOutFileName);

    try
    {
        bflTarget.Open
        (
            tCIDLib::EAccess_Excl_Write
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );
    }

    catch(const TError& errToCatch)
    {
        __conOut
                << L"Error occured while opening file "
                << __pathOutFileName << DNewLn
                << L"Error: " << errToCatch.strErrText()
                << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Create the stream object for this file. We can ask the file
    //  to create a stream for us, which is a convenient way to get
    //  a binary stream for a file. By default, the new stream object
    //  does not adopt the binary file object. This is what we want
    //  in this case since the file object is a stack based object.
    //
    TBinaryStream* pstrmTarget = bflTarget.pstrmMakeNew();

    //
    //  Output the major attributes of the scene, so that they user can
    //  confirm it visually.
    //
    __conOut
            << L"       Threads: " << __c4Threads << NewLn
            << L"       Include: " << __pathInclude << NewLn
            << L" Image Quality: " << __viewScene.eQuality() << NewLn
            << L"   Max Reflect: " << __viewScene.c4MaxReflections() << NewLn
            << L"      Sampling: " << __viewScene.eSample();

    if (__viewScene.eSample() == tCIDTracer::ESampleMode_Statistical)
    {
        __conOut   << L", Threshold="
                        << TFloat(__viewScene.f8StatSampleThreshold());
    }
    __conOut << DNewLn;

    //
    //  Allocate the rendering thread objects. We have an array of
    //  pointers to thread objects, so we are allocating new threads
    //  for each pointer. Each thread in a process must have a unique
    //  name, so we use the index to create it.
    //
    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4Threads; c4Index++)
    {
        TString strName(L"RenderThread%(1)");
        strName.ReplaceToken(c4Index, L'1');

        __apthrList[c4Index] = new TThread(strName, __eRenderThread);
    }

    //
    //  We need to reallocate the image buffer now to be large enough for
    //  the image. We did not know how big it needed to be when it was
    //  constructed. We need 3 bytes per pixel, and we need two extra
    //  bytes per row to hold the row number.
    //
    //  Also, tell it not to stream its element count when it streams.
    //  Normally we want that, in order to stream it back later, but this
    //  is streaming out a third party format so it wouldn't be right.
    //
    __baImageBuf.Reallocate
    (
        ((__c4HorzSize * 3) * __c4VertSize)
        + (__c4VertSize * 2)
    );
    __baImageBuf.bStoreCount(kCIDLib::False);

    // Get the starting time
    TCurrentTime tmStartTime;

    //
    //  Now install a try block and do the rendering, writing out the
    //  rows as they are done.
    //
    __conOut << L"Rendering..." << NewLn;
    try
    {
        //
        //  First, we need to write out the file size header, which is
        //  required by the .Dis format. These are words in the file so
        //  cast them. The data type passed to Write determines the size
        //  of the value written.
        //
        *pstrmTarget << tCIDLib::TCard2(__c4HorzSize);
        *pstrmTarget << tCIDLib::TCard2(__c4VertSize);
    }

    catch(const TError& errToCatch)
    {
        __conOut
                << L"An error occured while writing file header."
                << L" File will not be properly formatted" << NewLn
                << L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Start up the threads. They will all work until there is nothing
    //  more to do. The last of them to finish will release our event
    //  semaphore.
    //
    for (c4Index = 0; c4Index < __c4Threads; c4Index++)
        __apthrList[c4Index]->Start();

    // Now wait for them to finish the image
    __evWaitMain.WaitFor();

    //
    //  Stream out the image buffer. It is in the required format already,
    //  so we just stream it straight out all at once.
    //
    *pstrmTarget << __baImageBuf;

    //
    //  Display the rendering statistics. We get another current time stamp
    //  and subtract the start time from it. Then we set the default format
    //  on the rendering time object so that it will format how we want
    //  it to.
    //
    TCurrentTime tmRendering;
    tmRendering -= tmStartTime;
    tmRendering.strDefaultFormat(L"%(R,2,0):%(h,2,0):%(s,2,0)");

    //
    //  Now we can delete the stream object, which will close the file and
    //  get all the data flushed out.
    //
    delete pstrmTarget;

    // Clean up the rendering thread objects.
    for (c4Index = 0; c4Index < __c4Threads; c4Index++)
        delete __apthrList[c4Index];

    __conOut
            << L"\r\nRendering completed successfully" << NewLn
            << L"Rendering Time: " << tmRendering << L"  (hh:mm:ss)" << NewLn
            << L"     Rays Cast: " << __viewScene.c4LightRaysCast() << NewLn
            << L"  Initial Hits: " << __viewScene.c4InitRayHits() << NewLn
            << L"Pels Processed: " << __viewScene.c4PelsProcessed() << DNewLn;

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    // Log our death
    __facTracer3.LogMsg
    (
        __FILE__
        , __LINE__
        , L"Tracer3 completed"
        , tCIDLib::ESev_Status
    );

    return tCIDLib::EExit_Normal;
}



//
// FUNCTION/METHOD NAME: __eRenderThread
//
// DESCRIPTION:
//
//  This is the thread function for the rendering threads. These guys
//  are started by the main thread to do the rendering work. All the
//  instances access global data, rendering pixels into the image buffer.
//
//  The instances all cooperate on access to the current row/column
//  data values, playing leapfrog through the pixels until they are all
//  done.
// -------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread
//          pData is a pointer to the optional data buffer passed by our
//              creator. Not used here.
//
//  OUTPUT: None
//
//  RETURN: Just returns tCIDLib::EExitCodes_Normal.
//
tCIDLib::EExitCodes __eRenderThread(TThread& thrThis, tCIDLib::TVoid* const)
{
    // Let our caller go
    thrThis.Sync();

    // Bump down our priority so we don't eat up the system
    thrThis.SetPriority(tCIDLib::EPrioLevel_BelowNormal);

    // A pixel color value to pass to the tracing method
    TRGBClr rgbPelClr;

    while (1)
    {
        tCIDLib::TCard4 c4MyCol;
        tCIDLib::TCard4 c4MyRow;

        //
        //  Lock the sync mutex and get the next pixel, bumping it up for
        //  the next thread. We do this in a faux block so that the mutex
        //  will get released no matter what.
        //
        {
            TCritSecLock crslCritSec(&__crsSync);

            //
            //  If the row is at the end, then we are done and can
            //  exit. If we are the last man out, then let the main
            //  thread go.
            //
            if (__c4CurRow == __c4VertSize)
            {
                __c4ThreadsRunning--;
                if (!__c4ThreadsRunning)
                    __evWaitMain.Pulse();
                return tCIDLib::EExit_Normal;
            }

            //
            //  If the current column is the horizontal size, then we are
            //  done with the current row. So we move up to the next row
            //  and reset the column.
            //
            if (__c4CurColumn == __c4HorzSize)
            {
                // Bump up the row since we've finished this one
                __c4CurRow++;

                // If we are done, then jump back to the top
                if (__c4CurRow == __c4VertSize)
                    continue;

                //
                //  Put the row number into the image buffer. Each row
                //  is prefixed by its row number. Its a 16 bit value,
                //  so we have to put it in as two bytes because we are
                //  using a raw byte array.
                //
                //  Note that this might happen multiple times at the end
                //  of the image because each thread will come through
                //  here and die.
                //
                tCIDLib::TCard4 c4RowOfs = ((3 * __c4HorzSize) * __c4CurRow)
                                            + (__c4CurRow * 2);
                __baImageBuf[c4RowOfs] = tCIDLib::TCard1(__c4CurRow & 0xFF);
                __baImageBuf[c4RowOfs+1] = tCIDLib::TCard1((__c4CurRow & 0xFF00) >> 8);

                // Every ten rows, say what row we are doing
                if (__c4CurRow && !(__c4CurRow % 10))
                    __conOut << L"Doing row # " << __c4CurRow << NewLn;

                // Reset the column counter
                __c4CurColumn = 0;
            }

            //
            //  There is still work to do, so get the next column out
            //  and bump it up for the next guy.
            //
            c4MyCol = __c4CurColumn++;
            c4MyRow = __c4CurRow;
        }

        // Trace the current pixel
        __viewScene.TracePel(c4MyCol, c4MyRow, rgbPelClr);

        //
        //  Put this pel into the line buffer. The reds, greens, and
        //  blues are in separate sections. Lock the access mutex while
        //  we do it.
        //
        //  Each row starts with its row number (a 2 byte value) so we add
        //  2 to the base address.
        //
        const tCIDLib::TCard4 c4Base = ((3 * __c4HorzSize) * c4MyRow)
                                        + (c4MyRow * 2)
                                        + 2;
        __baImageBuf[c4Base + c4MyCol] = rgbPelClr.c1Red();
        __baImageBuf[c4Base + c4MyCol + __c4HorzSize] = rgbPelClr.c1Green();
        __baImageBuf[c4Base + c4MyCol + (__c4HorzSize * 2)] = rgbPelClr.c1Blue();
    }
    return tCIDLib::EExit_Normal;
}
