//
// NAME: RayTracer2.Cpp
//
// DESCRIPTION:
//
//  This demo is the second in the ray tracer demo set. The first one
//  demonstrated how to build a scene programatically, then render it to
//  disk.
//
//  This demo shows how to use the scene parsing system to read in a
//  text description of a scene, and then render it to file. This one is
//  almost trivial because the parsing is all built into the ray tracer
//  system.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 10/28/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  This guy just hard codes the path to the scene description files,
//      which it passes to the scene parser. Later versions will make this
//      settable via the environment or command line. This version though
//      will not work if moved to another directory.
//
// MODIFICATION LOG:
//


// ----------------------------------------------------------------------------
//  CIDLib includes. CIDTracer is relatively high level, so including it gets
//  everything that we need.
// ----------------------------------------------------------------------------
#include    "CIDTracer.Hpp"


// ----------------------------------------------------------------------------
//  Local function prototypes
//
//  __eMainThead
//      This is the thread function for the main thread, thrMain, that
//      provides the execution for this program.
// ----------------------------------------------------------------------------
static tCIDLib::EExitCodes  __eMainThread(TThread&, tCIDLib::TVoid*);


// ----------------------------------------------------------------------------
//  Local constant data
//
//  __c4MinSize
//  __c4MaxSize
//      These are the min/max sizes of the image in pixels.
// ----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4MinSize = 16;
static const tCIDLib::TCard4    __c4MaxSize = 4096;


// ----------------------------------------------------------------------------
//  Local data declarations
//
//  __conOut
//      We create a console stream for our input and output.  It is redirectable
//      by default.
//
//  __strTitle
//      The title string for the program. The version is formatted into them
//      during startup. This is one way to do it, another being via a
//      resource. The resource method is used in subsequent versions.
// ----------------------------------------------------------------------------
static TConsole     __conOut;
static TString      __strTitle
                    (
                        L"CIDLib Ray Tracer Demo #2, Version %(1).%(2,3,0)"
                    );


// ----------------------------------------------------------------------------
//  Include magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"MainThread", __eMainThread))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __ShowUsage
//
// DESCRIPTION:
//
//  Shows the parameter usage for the program.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid __ShowUsage()
{
    __conOut
    <<  L"Usage: RayTracer2 SceneFile OutputFile hhh vvv [/option=val]" << DNewLn
    <<  L"   If not extension for the output file, it will be .Dis." << NewLn
    <<  L"   hhh and vvv are the horz/vertical size in pixels" << NewLn
    <<  L"        which must be 16 <= x <= 4096." << NewLn
    <<  L"   Options are:" << NewLn
    <<  L"        /Quality=X   - where X is 0 to 5" << NewLn
    <<  L"        /Sample=X    - where X is Normal, Stocastic, Statistical" << NewLn
    <<  L"        /Threshold=X - where X is the statistical threshold"
    <<  DNewLn;
}


// ----------------------------------------------------------------------------
//  Global functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThread
//
// DESCRIPTION:
//
//  This is the main program thread. It sets up the scene, then uses the
//  ray tracer engine to trace it line by line, outputting each line to 
//  the output file.
// -------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread
//          pData is a pointer to the optional data buffer passed by our
//              creator.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThread(TThread& thrThis, tCIDLib::TVoid* const)
{
    // Let our caller go
    thrThis.Sync();

    // Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Announce ourselves. We replace the tokens in the title string with
    //  the version info from the build environment.
    //
    __strTitle.ReplaceToken(kCIDLib::c4MajVersion, L'1');
    __strTitle.ReplaceToken(kCIDLib::c4MinVersion, L'2');
    __conOut    << L"\r\n" << __strTitle << NewLn << L"Compiled: "
                << TString(__DATE__) << DNewLn;

    // Check out the command line parms
    if (TSysInfo::c4CmdLineArgCount() < 5)
    {
        __ShowUsage();
        return tCIDLib::EExit_BadParameters;
    }

    //
    //  Get the image size parameters. If they don't resolve to values
    //  between 16 and 4096, then issue an error.
    //
    tCIDLib::TCard4 c4HorzSize, c4VertSize;
    try
    {
        TString strTmp(kCIDLib::pszEmptyZStr, 64);

        TSysInfo::CmdLineArg(3, strTmp);
        c4HorzSize = strTmp.c4Val();

        TSysInfo::CmdLineArg(4, strTmp);
        c4VertSize = strTmp.c4Val();
    }

    catch(...)
    {
        __ShowUsage();
        return tCIDLib::EExit_SystemException;
    }

    if (!bInRange(c4HorzSize, __c4MinSize, __c4MaxSize)
    ||  !bInRange(c4VertSize, __c4MinSize, __c4MaxSize))
    {
        __ShowUsage();
        return tCIDLib::EExit_BadParameters;
    }

    // Create a path string to read the file name into
    TPathStr strFileName;

    //
    //  Create the parser information object. This guy will do the parsing
    //  for us. we give it the view geometry object that we want it to
    //  fill in. If the parsing goes well, then all we need to do is to
    //  parse it. If not, it throws an exception.
    //
    //  We put it in a faux block so that the parse info object gets cleaned
    //  up naturally.
    //
    TViewGeometry   viewScene;
    viewScene.SetSceneDimensions(c4HorzSize, c4VertSize);
    {
        TParseInfo  prsiScene(viewScene);

        //
        //  Ok, since this likely to fail in many cases, we go ahead and do
        //  the parsing now before event attempting to deal with the output
        //  file.
        //
        try
        {
            //
            //  Get the file name of the scene. Insure that it has no path
            //  component because this program does not want any.
            //
            TSysInfo::CmdLineArg(1, strFileName);
            TString strTmp;
            if (strFileName.bQueryPath(strTmp))
            {
                __conOut << L"The scene file should not have any path component"
                         << NewLn;
                return tCIDLib::EExit_BadParameters;
            }

            //
            //  Ask the scene parser to parse the input file. Pass the include
            //  path that we know ahead of time because the installation
            //  program put them there.
            //
            __conOut << L"Compiling file: " << strFileName << NewLn;
            prsiScene.ParseFile(strFileName, L"..\\..\\SceneFiles");
        }

        catch(const TError& errToCatch)
        {
            __conOut
                << L"An error occured while parsing scene description"
                << NewLn;

            // Display the error information
            __conOut << L"Err Text: \r\n\n" << errToCatch << NewLn;
            return tCIDLib::EExit_RuntimeError;
        }

        catch(const TRTParseErr& rterrToCatch)
        {
            __conOut
                << L"An error occured while parsing scene description" << NewLn;

            // Display the error information
            __conOut
                    << L"     File: " << rterrToCatch.strFile << NewLn
                    << L"   Object: " << rterrToCatch.strObjFail << NewLn
                    << L" Line Num: " << rterrToCatch.c4LineNum << NewLn
                    << L" Err Text: " << rterrToCatch.strMsg << NewLn;
            return tCIDLib::EExit_FatalError;
        }
        __conOut << L"Compilation successful" << NewLn;
    }

    // Get the output file name parameter and attempt to create it.
    TSysInfo::CmdLineArg(2, strFileName);

    //
    //  Add the extension if needed and truncate the file, giving it an
    //  initial size that we calculated.
    //
    if (!strFileName.bHasExt())
        strFileName.Append(L".Dis");

    //
    //  Now we need to create a binary file straem to stream out the data
    //  to the file. For this demo we first create a file stream
    //  implementation object, giving it a binary stream object to adopt.
    //  We then create the binary stream and give it the implememtation
    //  object, which it adopts.
    //
    //  The other option is to create the binary file and then ask it to
    //  give you a stream for itself.
    //
    //  In this case, the stream implementation object adopts the binary
    //  file object, and the stream object adopts the implementation
    //  object, so everyone is taken care of.
    //
    TBinaryFileStream strmTarget;
    try
    {
        strmTarget.Open
        (
            strFileName
            , tCIDLib::EAccess_Excl_Write
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );
    }

    catch(const TError& errToCatch)
    {
        __conOut    << L"Error occured while opening file "
                    << strFileName << DNewLn
                    << L"Error: " << errToCatch.strErrText()
                    << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Get any trailing options. These are for overriding the overall
    //  scene attributes set in the file. On each one, the leading chars
    //  are compared. If they match, they are cut out, leaving only the
    //  value in the temp string.
    //
    try
    {
        tCIDLib::TCard4 c4Tmp;
        TString         strTmp(kCIDLib::pszEmptyZStr, 128);
        for (c4Tmp = 5; c4Tmp < TSysInfo::c4CmdLineArgCount(); c4Tmp++)
        {
            tCIDLib::TInt4      i4Val;
            tCIDLib::TFloat8    f8Val;

            TSysInfo::CmdLineArg(c4Tmp, strTmp);

            if (strTmp.eCompareN(L"/Quality=", 9) == tCIDLib::ESort_Equal)
            {
                strTmp.Cut(0, 9);
                i4Val = strTmp.i4Val();
                if ((i4Val < 0) || (i4Val > 5))
                {
                    __conOut
                        << L"Quality option must have a value of 0 to 5"
                        << DNewLn;
                    facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
                }
                viewScene.eQuality(tCIDTracer::EQualityLevels(i4Val));
            }
             else if (strTmp.eCompareN(L"/Sample=", 8) == tCIDLib::ESort_Equal)
            {
                strTmp.Cut(0, 8);
                if (strTmp == L"Normal")
                    viewScene.eSample(tCIDTracer::ESampleMode_Normal);
                if (strTmp == L"Stochastic")
                    viewScene.eSample(tCIDTracer::ESampleMode_Stochastic);
                else if (strTmp == L"Statistical")
                    viewScene.eSample(tCIDTracer::ESampleMode_Statistical);
            }
             else if (strTmp.eCompareN(L"/Threshold=", 11) == tCIDLib::ESort_Equal)
            {
                strTmp.Cut(0, 11);
                f8Val = strTmp.f8Val();
                if ((f8Val < 0.0) || (f8Val > 3.0))
                {
                    __conOut   << L"Sampling threshold must have a value "
                                        L"of 0.0 to 3.0" << DNewLn;
                    facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
                }
                viewScene.f8StatSampleThreshold(f8Val);
            }
        }
    }

    catch(const TError& errToCatch)
    {
        __conOut
                << L"Error occured while parsing optional parameters" << DNewLn
                << L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Output the major attributes of the scene, so that they use will have
    //  some idea how long its gonna take.
    //
    __conOut
            << L" Image Quality: " << viewScene.eQuality() << NewLn
            << L"   Max Reflect: " << viewScene.c4MaxReflections() << NewLn
            << L"      Sampling: " << viewScene.eSample();

    if (viewScene.eSample() == tCIDTracer::ESampleMode_Statistical)
    {
        __conOut  << L", Threshold="
                  << TFloat(viewScene.f8StatSampleThreshold());
    }
    __conOut << DNewLn;

    //
    //  Ok, we are ready to start tracing the image. SO we need variables
    //  to track the current row and column and a temp color
    //
    tCIDLib::TCard4 c4CurRow;
    tCIDLib::TCard4 c4CurCol;
    TRGBClr         rgbPelClr;

    //
    //  Create an array of bytes for the colors in a row of pels in the
    //  image. We are dealing with a third party file format here, so tell
    //  him not store his element count in the flattened format. Normally he
    //  will do this so that he can stream back in a flattened array. But
    //  that would cause an incorrect file format in this case.
    //
    TBaseArray<tCIDLib::TCard1> baLineBuf(c4HorzSize*3);
    baLineBuf.bStoreCount(kCIDLib::False);

    // Get the starting time
    TCurrentTime tmStartTime;

    //
    //  Now install a try block and do the rendering, writing out the
    //  rows as they are done.
    //
    __conOut << L"Rendering..." << NewLn;
    try
    {
        //
        //  Crank down our priority a little or we will eat up everything
        //  and make the machine unusable. We use a priority janitor to
        //  do it, so our priority will go back to normal when we exit
        //  this rendering scope block.
        //
        TPrioJanitor  janPriority(tCIDLib::EPrioLevel_BelowNormal);

        //
        //  First, we need to write out the file size header, which is
        //  required by the .Dis format. These are words in the file so
        //  cast them. The data type passed to Write determines the size
        //  of the value written.
        //
        strmTarget << tCIDLib::TCard2(c4HorzSize);
        strmTarget << tCIDLib::TCard2(c4VertSize);

        //
        //  Now start looping through the rows/columns of the image.
        //
        for (c4CurRow = 0; c4CurRow < c4VertSize; c4CurRow++)
        {
            // Every ten rows, say what row we are doing
            if (!(c4CurRow % 10))
                __conOut << L"Doing row # " << c4CurRow << NewLn;

            //
            //  Write out the row number. Here again, we must cast it to
            //  a 2 byte value because that is what the file format wants.
            //
            strmTarget << tCIDLib::TCard2(c4CurRow);

            for (c4CurCol = 0; c4CurCol < c4HorzSize; c4CurCol++)
            {
                // Trace the current pixel
                viewScene.TracePel(c4CurCol, c4CurRow, rgbPelClr);
                
                //
                //  Put this pel into the line buffer. The reds, greens,
                //  and blues are in separate sections.
                //
                baLineBuf[c4CurCol] = rgbPelClr.c1Red();
                baLineBuf[c4CurCol + c4HorzSize] = rgbPelClr.c1Green();
                baLineBuf[c4CurCol + (c4HorzSize * 2)] = rgbPelClr.c1Blue();
            }
            strmTarget << baLineBuf;
        }
    }

    catch(const TError& errToCatch)
    {
        __conOut
                << L"An error occured while tracing row: "
                << c4CurRow << L". File will not be properly formatted" << NewLn
                << L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Display the rendering statistics. We get another current time stamp
    //  and subtract the start time from it. Then we set the default format
    //  on the rendering time object so that it will format how we want
    //  it to.
    //
    TCurrentTime tmRendering;
    tmRendering -= tmStartTime;
    tmRendering.strDefaultFormat(L"%(R,2,0):%(h,2,0):%(s,2,0)");

    __conOut
            << L"\r\nRendering completed successfully" << NewLn
            << L"Rendering Time: " << tmRendering << L"  (hh:mm:ss)" << NewLn
            << L"     Rays Cast: " << viewScene.c4LightRaysCast() << NewLn
            << L"  Initial Hits: " << viewScene.c4InitRayHits() << NewLn
            << L"Pels Processed: " << viewScene.c4PelsProcessed() << DNewLn;

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
