//
//  FILE NAME: Tracer1.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the first ray tracer testing program. This
//  one is a simple, TTY output style program. It builds a simple scene
//  programatically and traces it. The user provides the output file name,
//  and the size of the image in pixels.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "Tracer1.Hpp"



// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// -----------------------------------------------------------------------------
//  Local constant data
//
//  __c4MinSize
//  __c4MaxSize
//      These are the min/max sizes of the image in pixels.
// -----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4MinSize = 16;
static const tCIDLib::TCard4    __c4MaxSize = 4096;


// ----------------------------------------------------------------------------
//  Local, static data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. It is redirectable by default.
//
//  __facTracer1
//      This is the facility object for the Exe facility. Since we have no
//      special needs, we just create an object of the basic TFacility
//      class. We indicate its name and that its an exe type facility. All
//      of the CIDLib code uses the standard release versions as their
//      facility versions, but you would use something appropriate for you.
// ----------------------------------------------------------------------------
TConsole        __conOut;
TFacility       __facTracer1
                (
                    L"Tracer1"
                    , tCIDLib::EModType_Exe
                    , kCIDLib::c4MajVersion
                    , kCIDLib::c4MinVersion
                );


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Tracer1MainThread", __eMainThreadFunc))



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __Announce
//
// DESCRIPTION:
//
//  This guy just outputs a program blurb to announce the program and
//  its version and compile date.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __Announce()
{
    __conOut    << L"Tracer1.Exe" << NewLn
                << L"CIDLib Ray Tracer Demo #1" << NewLn
                << L"Compiled on:" << TString(__DATE__)
                << DNewLn;
}


//
// FUNCTION/METHOD NAME: __ShowUsage
//
// DESCRIPTION:
//
//  Shows the parameter usage for the program.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ShowUsage()
{
    __conOut    <<  L"Usage: Tracer1 OutputFile hhh vvv" << DNewLn
                <<  L"     Don't give an extension, it will be .Dis." << NewLn
                <<  L"     hhh and vvv are the horz/vertical size in pixels"
                    L" which must be 16 <= x <= 4096." << DNewLn;
}


//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the EExitCodes values
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    {
        // Log our arrival
        __facTracer1.LogMsg
        (
            __FILE__
            , __LINE__
            , L"Tracer1 starting up..."
            , tCIDLib::ESev_Status
        );

        // Announce ourself
        __Announce();

        // Check out the command line parms
        if (TSysInfo::c4CmdLineArgCount() != 4)
        {
            __ShowUsage();
            return tCIDLib::EExit_BadParameters;
        }

        //
        //  Get the image size parameters. If they don't resolve to values
        //  between 16 and 4096, then issue an error.
        //
        tCIDLib::TCard4 c4HorzSize, c4VertSize;

        try
        {
            TString strTmp;

            TSysInfo::CmdLineArg(2, strTmp);
            c4HorzSize = strTmp.c4Val();

            TSysInfo::CmdLineArg(3, strTmp);
            c4VertSize = strTmp.c4Val();
        }

        catch(...)
        {
            __ShowUsage();
            return tCIDLib::EExit_BadParameters;
        }

        //
        //  The values were valid numbers. So make sure that they are within
        //  our valid range.
        //
        if (!bInRange(c4HorzSize, __c4MinSize, __c4MaxSize)
        ||  !bInRange(c4VertSize, __c4MinSize, __c4MaxSize))
        {
            __ShowUsage();
            return tCIDLib::EExit_BadParameters;
        }

        //
        //  Get the file name parameter and attempt to create it. Get read it
        //  into a path string, which provides us with nice path interpretation
        //  support.
        //
        TPathStr  strFileName;
        TSysInfo::CmdLineArg(1, strFileName);

        // if it has no extension, then add the default one
        if (!strFileName.bHasExt())
            strFileName.Append(L".Dis");

        //
        //  Create a file stream implementation object for a binary file. This
        //  is the manual way to do it, just to demonstrate. You can also 
        //  just create a TBinaryFileStream object which will do this work
        //  for you.
        //
        TFileBinStreamImpl* pstrmiTarget = new TFileBinStreamImpl
        (
            new TBinaryFile(strFileName)
            , tCIDLib::EAdoptOpt_Adopt
        );

        try
        {
            //
            //  Now lets open the file that this stream has adopted. We do it
            //  via the file stream implementation object. We replace it if a
            //  previous version exists and indicate that our access is going
            //  to be totally sequential.
            //
            pstrmiTarget->Open
            (
                tCIDLib::EAccess_Excl_Write
                , tCIDLib::ECreateAct_ReplaceIfExists
                , tCIDLib::EFileAttr_None
                , tCIDLib::EFileFlag_SequentialScan
            );
        }

        catch(const TError& errToCatch)
        {
            __conOut    << L"Error occured while opening file "
                        << strFileName << DNewLn
                        << L"Error: " << errToCatch.strErrText()
                        << DNewLn;
            return tCIDLib::EExit_RuntimeError;
        }

        //
        //  Create the binary stream for this file. This lets us stream data
        //  out to the file, using convenient, type safe insertion operators.
        //  It will be closed when we go out of this scope block.
        //
        TBinaryStream   strmTarget(pstrmiTarget);

        //
        //  Now we can set up the canned scene that we trace in this demoo
        //  program. First we set up the view geometry attributes that were
        //  not set correctly for us by the default constructor.
        //
        //  The scene is a box, which has half a sphere and a cylinder
        //  subtracted from it. This allows you to look through the object.
        //
        TViewGeometry   viewScene;

        // Set the scene dimensions in pixels
        viewScene.SetSceneDimensions(c4HorzSize, c4VertSize);

        //
        //  Put the eye up 8 and back along the z axis 5. Instead of setting
        //  the eye direction manually, use the 'look at' attribute to tell
        //  it to look at the origin. So we are looking straight down at
        //  the origin.
        //
        viewScene.vecEyePos(T3DVector(0.0, 8.0, -5.0));
        viewScene.LookAt(T3DVector(0.0, 0.0, 0.0));

        //
        //  The max reflections and trace quality affect the realism of the
        //  scene. These are both high so it will look good at the expense of
        //  rendering time.
        //
        viewScene.c4MaxReflections(4);
        viewScene.eQuality(tCIDTracer::eQuality_5);

        //
        //  Insert a light source object into the scene. Its over the right
        //  and up a little and back 10.
        //
        viewScene.InsertObj
        (
            new TLightSource(T3DVector(10, 20.0, -10.0), facCIDTracer.frgbWhite)
        );

        //
        //  First we need to do the floor. It will be the usual black
        //  and white checkerboard. This constructor creates a plane
        //  'pointing' upwards, i.e. its 'on the floor'. We then apply a
        //  checker board texture to it. The texture is made semi glossy
        //  so it will look slick. We scale it up a little to make the
        //  checkers larger, though it only needs scaling in the x & z
        //  directions since it is flat in the y dimension.
        //
        TRTPlane*   prtobjFloor = new TRTPlane(T3DVector(0.0, 1.0, 0.0), 0);
        TTxtrChecker* ptxtrChecker = new TTxtrChecker
        (
            facCIDTracer.frgbWhite
            , facCIDTracer.frgbBlack
        );
        ptxtrChecker->f8Roughness(0.1);
        ptxtrChecker->f8PhongSize(70.0);
        ptxtrChecker->f8Phong(0.75);
        ptxtrChecker->f8Reflection(0.15);
        ptxtrChecker->f8Brilliance(2.0);
        prtobjFloor->AppendTexture(ptxtrChecker);
        prtobjFloor->Scale(2, 1, 2);
        viewScene.InsertObj(prtobjFloor);

        //
        //  Now add a sphere with a wood texture. Scale it by 4 and translate
        //  up by 2 so that it sits on the floor. Note that the scaling is
        //  done before the adoption of the texture so it will not affect it.
        //
        TRTSphere* prtobjSphere = new TRTSphere;
        TTxtrWood* ptxtrWood = new TTxtrWood;
        ptxtrWood->Rotate(0, 0, 25);
        ptxtrWood->f8Specular(0.25);
        ptxtrWood->f8Roughness(0.5);
        ptxtrWood->f8PhongSize(70.0);
        ptxtrWood->f8Phong(0.75);
        ptxtrWood->f8Reflection(0.15);
        ptxtrWood->f8Turbulence(0.2);
        ptxtrWood->AppendBand
        (
            0.0
            , 0.8
            , TFRGBClr(0.666, 0.312, 0.2)
            , TFRGBClr(0.666, 0.312, 0.2)
        );

        ptxtrWood->AppendBand
        (
            0.8
            , 1.0
            , TFRGBClr(0.4, 0.133, 0.06)
            , TFRGBClr(0.2, 0.06, 0.4)
        );
        prtobjSphere->Scale(4, 4, 4);
        prtobjSphere->Translate(0.0, 2.0, 0.0);
        prtobjSphere->AppendTexture(ptxtrWood);
        viewScene.InsertObj(prtobjSphere);

        //
        //  Ok, we are ready to start tracing the image.
        //
        tCIDLib::TCard4 c4CurRow;
        tCIDLib::TCard4 c4CurCol;
        TRGBClr         rgbPelClr;

        //
        //  Create an array of bytes for the colors in a row of pels in the
        //  image. We need 3 bytes for each pixel in the row.
        //
        //  We are dealing with a third party file format here, so tell him
        //  not to store his element count in the flattened format.
        //
        TBaseArray<tCIDLib::TCard1> ac1LineBuf(c4HorzSize*3);
        ac1LineBuf.bStoreCount(kCIDLib::False);

        //
        //  Now install a try block and do the rendering, writing out the
        //  rows as they are done.
        //
        try
        {
            //
            //  Crank down our priority a little or we will eat up everything.
            //  We use a priority janitor, which will restore the priority
            //  when we exit this block. We set it to below normal so that it
            //  will get as much CPU as possible without interfering with
            //  normal threads.
            //
            TPrioJanitor janPriority(tCIDLib::EPrioLevel_BelowNormal);

            //
            //  First, we need to write out the file size header, which is
            //  required by the .Dis format. These are words in the file so
            //  cast them. The data type being streamed determines the size
            //  of the value written (i.e. the version of the streaming
            //  operator called.)
            //
            strmTarget << tCIDLib::TCard2(c4HorzSize);
            strmTarget << tCIDLib::TCard2(c4VertSize);

            //
            //  Now start looping through the rows/columns of the image.
            //
            for (c4CurRow = 0; c4CurRow < c4VertSize; c4CurRow++)
            {
                // Output what row we are on every 10 rows
                if (!(c4CurRow % 10))
                    __conOut << L"Doing row # " << c4CurRow << NewLn;

                //
                //  Write out the row number. Here again, we must cast it to
                //  a 2 byte value because that is what the file format wants.
                //
                strmTarget << tCIDLib::TCard2(c4CurRow);

                for (c4CurCol = 0; c4CurCol < c4HorzSize; c4CurCol++)
                {
                    // Trace the current pixel
                    viewScene.TracePel(c4CurCol, c4CurRow, rgbPelClr);
                
                    //
                    //  Put this pel into the line buffer. The reds, greens,
                    //  and blues are in separate sections.
                    //
                    ac1LineBuf[c4CurCol] = rgbPelClr.c1Red();
                    ac1LineBuf[c4CurCol + c4HorzSize] = rgbPelClr.c1Green();
                    ac1LineBuf[c4CurCol + (c4HorzSize * 2)] = rgbPelClr.c1Blue();
                }

                // Write this line of pixels
                strmTarget << ac1LineBuf;
            }
        }

        catch(const TError& errToCatch)
        {
            __conOut    << L"An error occured while tracing row: " << c4CurRow
                        << L". " << NewLn << L"   Error: "
                        << errToCatch.strErrText() << NewLn;
            return tCIDLib::EExit_RuntimeError;
        }

        catch(const TKrnlError& kerrToCatch)
        {
            __conOut    << L"A kernel error occured while tracing row: " << c4CurRow
                        << L". " << NewLn << L"   Error: "
                        << kerrToCatch.errcId() << NewLn;
            return tCIDLib::EExit_FatalError;
        }

        catch(...)
        {
            __conOut    << L"A system exception occured while tracing row: "
                        << c4CurRow << L"." << NewLn;
            return tCIDLib::EExit_SystemException;
        }
    }

    // Log our exit
    __facTracer1.LogMsg
    (
        __FILE__
        , __LINE__
        , L"Tracer1 terminating..."
        , tCIDLib::ESev_Status
    );

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
