//
//  FILE NAME: TestKernel_Time.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/22/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module tests the TKrnlTimeStamp class, which provides access to
//  core system time services.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __TestXXXXX
//
// DESCRIPTION:
//
//  These methods are all called from TestTime() below to test out various
//  aspects of the time services classes.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestConversions()
{
    TKrnlTimeStamp ktmsTest;

    try
    {
        ktmsTest.FromDetails(1996, tCIDLib::EMonth_December, 5);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error setting time from details. "
             << kerrToCatch << "\r\n";
        return;
    }

    if (ktmsTest.eDayOfWeek() != tCIDLib::EWeekDay_Thursday)
    {
        cout << _CurLn_ << "Time from details not == to thursday\r\n";
        return;
    }

}

static tCIDLib::TVoid __TestDefaultConstruction()
{
    // This should create a time at the start of our time universe
    TKrnlTimeStamp  ktmsDefault;

    // Query out the date info and see if its what whe think it is
    tCIDLib::TCard4     c4Year;
    tCIDLib::EMonths    eMonth;
    tCIDLib::TCard4     c4Day;
    tCIDLib::EWeekDays  eWeekDay;

    try
    {
        eWeekDay = ktmsDefault.eAsDateInfo(c4Year, eMonth, c4Day);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error getting as date info. "
             << kerrToCatch << "\r\n";
    }

    if (c4Year != 1601)
    {
        cout << _CurLn_ << "Default time stamp not == to base year (Actual="
                        << c4Year << ")\r\n";
    }

    if (c4Day != 1)
    {
        cout << _CurLn_ << "Default time stamp not set to 1st day (Actual="
                        << c4Day << ")\r\n";
    }

    if (eMonth != tCIDLib::EMonth_January)
        cout << _CurLn_ << "Default time stamp not set to January\r\n";


    // Now make sure its time stamp is at midnight on that day
    tCIDLib::TCard4 c4Hour;
    tCIDLib::TCard4 c4Minute;
    tCIDLib::TCard4 c4Second;
    tCIDLib::TCard4 c4MilliSecs;

    try
    {
        c4MilliSecs = ktmsDefault.c4AsTimeInfo(c4Hour, c4Minute, c4Second);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error getting as time info. "
             << kerrToCatch << " \r\n";
    }

    if (c4Hour || c4Minute || c4Second || c4MilliSecs)
    {
        cout << _CurLn_ << "Default time stamp not at midnight (Actual="
                        << c4Hour << "." << c4Minute << "."
                        << c4Second << "." << c4MilliSecs << "\r\n";
    }
}

static tCIDLib::TVoid __TestExplicitConstruction()
{
    TKrnlTimeStamp  ktmsTest(kCIDLib::i8OneHour);

    tCIDLib::TCard4 c4Hour;
    tCIDLib::TCard4 c4Minute;
    tCIDLib::TCard4 c4Second;
    tCIDLib::TCard4 c4MilliSecs;

    try
    {
        c4MilliSecs = ktmsTest.c4AsTimeInfo(c4Hour, c4Minute, c4Second);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error getting as time info. "
             << kerrToCatch << "\r\n";
    }

    if (c4Hour != 1)
    {
        cout << _CurLn_ << "Construct to 1 hour failed. Actual="
                        << c4Hour << "." << c4Minute << "."
                        << c4Second << "." << c4MilliSecs << "\r\n";
    }
}

static tCIDLib::TVoid __TestFromDetails()
{
    TKrnlTimeStamp  ktmsTest;

    // Set it to what we know should create a 0 value
    try
    {
        ktmsTest.FromDetails(1601, tCIDLib::EMonth_January, 1);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error setting from details. "
             << kerrToCatch << "\r\n";
    }

    if (ktmsTest.i8Time())
        cout << _CurLn_ << "Setting from details was not 0\r\n";
}

static tCIDLib::TVoid __TestMathAndOperators()
{
    TKrnlTimeStamp  ktmsTest1(kCIDLib::i8OneHour);
    TKrnlTimeStamp  ktmsTest2(kCIDLib::i8OneDay);

    if (ktmsTest1.i8Time() > ktmsTest2.i8Time())
        cout << _CurLn_ << "Operator > reported incorrect order\r\n";

    // Now make it equal and test again by adding 23 hours
    ktmsTest1.Add(kCIDLib::i8OneHour * 23);
    if (ktmsTest1.i8Time() != ktmsTest2.i8Time())
        cout << _CurLn_ << "Objects not equal.\r\n";

    // Copy construct one and make sure its equal
    TKrnlTimeStamp  ktmsCopy(ktmsTest1);
    if (ktmsCopy.i8Time() != ktmsTest1.i8Time())
        cout << _CurLn_ << "Copy constructed time != original\r\n";

    // Subtract an hour from the second value and make sure they aren't equal
    ktmsTest2.Subtract(kCIDLib::i8OneDay);
    if (ktmsTest2.i8Time() >= ktmsTest1.i8Time())
        cout << _CurLn_ << "Relative magnitude bad after subtraction\r\n";
}

static tCIDLib::TVoid __TestFileTimes()
{
    
}


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestTime
//
// DESCRIPTION:
//
//  This method is called from the main module to test the time stamp
//  class.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TestTime()
{
    __TestDefaultConstruction();
    __TestExplicitConstruction();
    __TestFromDetails();
    __TestMathAndOperators();
    __TestConversions();
    __TestFileTimes();
}
