//
//  FILE NAME: TestKernel_Threads.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/12/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module tests the 
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"

// ----------------------------------------------------------------------------
//  Local data types
// ----------------------------------------------------------------------------
struct  TThreadInfo
{
    tCIDLib::TCard4     c4Frequency;
    tCIDLib::TCard4     c4Duration;
    tCIDLib::TCard4     c4SleepTime;
    tCIDLib::TCard4     c4Count;
};


// ----------------------------------------------------------------------------
//  Local static data
//
//  __kevWakeup
//      This is an event semaphore that the last thread to die will trigger,
//      to allow the main thread to continue. We want it to be nonshared.
//
//  __scntCounter
//      This is a global counter used by the thread instances. They bump it
//      up when they start and down when they end. The last thread will then
//      release the main thread again, which is blocked on __kevWakeup.
// ----------------------------------------------------------------------------
static  TKrnlEvent          __kevWakeup(tCIDLib::EShareState_Unshared);
static  TKrnlSafeCounter    __scntCounter;


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eThreadFunc
//
// DESCRIPTION:
//
//  This is the thread function for the test thread. Multiple instances of
//  it are created and use semaphores to coordinate their activities (which
//  consists of manipulating counters.)
// -------------------------------------
//   INPUT: pData is the optional data pointers. In our case it is the
//              pointer to a TThreadInfo structure.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::EExitCodes THREADCALL __eThreadFunc(tCIDLib::TVoid* pData)
{
    // Bump up the counter
    __scntCounter.bInc();

    TThreadInfo& Info = *(TThreadInfo*)pData;
    for  (tCIDLib::TCard4 c4Counter = 0; c4Counter < Info.c4Count; c4Counter++)
    {
        TKrnlThread::BeepSpeaker(Info.c4Frequency, Info.c4Duration);
        TKrnlThread::Sleep(Info.c4SleepTime);
    }

    // Dec the counter and release main thread if 0
    if (!__scntCounter.bDec())
        __kevWakeup.Trigger();

    return tCIDLib::EExit_Normal;
}


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid TestThreads()
{
    // Create the event semaphore in a reset state so we will block
    try
    {
        __kevWakeup.Create
        (
            tCIDLib::EEventState_Reset
            , tCIDLib::EAutoMode_Manual
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error creating wakeup event. "
                << kerrToCatch << "\r\n";
        return;
    }

    //
    //  Set up a thread info structure for each of the threads and kick
    //  them off. They are all started blocked, then let go at the same
    //  time.
    //
    TThreadInfo Info1 = { 440,  75,  500, 10 };
    TThreadInfo Info2 = { 660, 100,  700, 10 };
    TThreadInfo Info3 = { 880, 200, 1500,  6 };

    TKrnlThread kthr1;
    TKrnlThread kthr2;
    TKrnlThread kthr3;

    tCIDLib::TCard4         c4StartCount = 0;
    tCIDLib::EThreadFlags   eFlags = tCIDLib::EThreadFlag_None;

    //
    //  Flush standard out before we start so that preceeding messages
    //  will be visible before we potentially freak out here.
    //
    cout.flush();

    // Now kick off the threads
    try
    {
        kthr1.BeginThread(__eThreadFunc, 80 * 1024, &Info1, eFlags);
        c4StartCount++;

        kthr2.BeginThread(__eThreadFunc, 80 * 1024, &Info2, eFlags);
        c4StartCount++;

        kthr3.BeginThread(__eThreadFunc, 80 * 1024, &Info3, eFlags);
        c4StartCount++;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error starting test thread #" << c4StartCount
                << L" " << kerrToCatch << "\r\n";
    }

    if (!c4StartCount)
    {
        cout << _CurLn_ << "No threads started up, bailing out\r\n";
        return;
    }

    //
    //  Now we just block on the wakeup event. When the last thread dies
    //  we should be awakened.
    //
    try
    {
        __kevWakeup.WaitFor(30 Seconds);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if (kerrToCatch.errcId() == kKrnlErrors::errcTimeout)
        {
            cout << _CurLn_ << "Test threads did not awaken main thread\r\n";
        }
         else
        {
            cout    << _CurLn_ << "Error " << kerrToCatch.errcId()
                    << " waiting on test threads to wake us up\r\n";
        }
    }
}
