//
//  FILE NAME: TestKernel_RawStrings.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/12/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module tests the raw string manipulation APIs.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::TVoid __TestConversions()
{
    const tCIDLib::Tch* const pszUNIString  = L"This is a string";
    const tCIDLib::Tsch* const pszANSIString= "This is a string";

    //
    //  Convert the UNICode string to an ANSI string and do some comparison
    //  tests on them.
    //
    tCIDLib::Tsch* pszANSITest = TRawStr::pszConvert(pszUNIString);

    if (TRawStr::eCompareStr(pszANSITest, pszANSIString))
        cout << _CurLn_ << "Converted ANSI string != to original\r\n";

    //
    //  Change a byte of the test string and then compare them again.
    //  It should not be equal anymore.
    //
    pszANSITest[0] = '1';

    if (!TRawStr::eCompareStr(pszANSITest, pszANSIString))
        cout << _CurLn_ << "Modified ANSI string still == to original\r\n";

    //
    //  Convert the ANSI string to the UNICode string and do some comparison
    //  tests on them.
    //
    tCIDLib::Tch* pszUNITest = TRawStr::pszConvert(pszANSIString);

    if (TRawStr::eCompareStr(pszUNITest, pszUNIString))
        cout << _CurLn_ << "Converted UNICode string != to original\r\n";

    //
    //  Change a byte of the test string and then compare them again.
    //  It should not be equal anymore.
    //
    pszUNITest[0] = L'1';

    if (!TRawStr::eCompareStr(pszUNITest, pszUNIString))
        cout << _CurLn_ << "Modified UNICode string still == to original\r\n";


    //
    //  Now test the versions that convert into a buffer. We test that they
    //  treat the buffer length correctly.
    //
    //  Put marks beyond the nul to test for overwrites
    //
    tCIDLib::Tch  szUNIString[6];
    tCIDLib::Tsch szANSIString[6];
    szUNIString[5] = 0xFFFE;
    szANSIString[5] = 0x7E;
    
    // Convert the UNI to ANSI and test the results
    TRawStr::pszConvert(pszUNIString, szANSIString, 4);

    if (szANSIString[5] != 0x7E)
        cout << _CurLn_ << "ANSI target string was overwritten\r\n";

    if (szANSIString[4] != 0)
        cout << _CurLn_ << "ANSI target string was not terminated\r\n";

    if (TRawStr::eNCompareStr(szANSIString, pszANSIString, 4))
        cout << _CurLn_ << "Target ANSI string != to start of original\r\n";


    // Now go the other way, from ANSI to UNI
    TRawStr::pszConvert(pszANSIString, szUNIString, 4);

    if (szUNIString[5] != 0xFFFE)
        cout << _CurLn_ << "UNICode target string was overwritten\r\n";

    if (szUNIString[4] != 0)
        cout << _CurLn_ << "UNICode target string was not terminated\r\n";

    if (TRawStr::eNCompareStr(szUNIString, pszUNIString, 4))
        cout << _CurLn_ << "Target UNICode string != to start of original\r\n";


    delete pszANSITest;
    delete pszUNITest;
}


static tCIDLib::TVoid __TestFormatting()
{
    //
    //  Try to format each of the possible fundamental data types into a
    //  raw string.
    //
    tCIDLib::TZStr128   szTmp;

    // Try to format a string into the string, left justified
    TRawStr::FormatStr
    (
        L"This is a test"
        , szTmp
        , 15
        , L' '
        , tCIDLib::EHJustify_Left
    );
    if (TRawStr::eCompareStr(szTmp, L"This is a test "))
        cout << _CurLn_ << "Left justified string format failed\r\n";

    // Try to format a string into the string, right justified
    TRawStr::FormatStr
    (
        L"This is a test"
        , szTmp
        , 15
        , L' '
        , tCIDLib::EHJustify_Right
    );
    if (TRawStr::eCompareStr(szTmp, L" This is a test"))
        cout << _CurLn_ << "Right justified string format failed\r\n";


    // Format all the cardinal types
    TRawStr::FormatVal(tCIDLib::TCard1(0xAC), szTmp, 128, tCIDLib::ERadix_Hex);
    if (TRawStr::eCompareStr(szTmp, L"AC"))
        cout << _CurLn_ << "TCard1 format failed\r\n";

    TRawStr::FormatVal(tCIDLib::TCard2(0xFACE), szTmp, 128, tCIDLib::ERadix_Hex);
    if (TRawStr::eCompareStr(szTmp, L"FACE"))
        cout << _CurLn_ << "TCard2 format failed\r\n";

    TRawStr::FormatVal(tCIDLib::TCard4(0xACDB9875), szTmp, 128, tCIDLib::ERadix_Hex);
    if (TRawStr::eCompareStr(szTmp, L"ACDB9875"))
        cout << _CurLn_ << "TCard4 format failed\r\n";


    // Format all the integer types
    TRawStr::FormatVal(tCIDLib::TInt1(-1), szTmp, 128);
    if (TRawStr::eCompareStr(szTmp, L"-1"))
        cout << _CurLn_ << "TInt1 format failed\r\n";

    TRawStr::FormatVal(tCIDLib::TInt2(-1), szTmp, 128);
    if (TRawStr::eCompareStr(szTmp, L"-1"))
        cout << _CurLn_ << "TInt2 format failed\r\n";

    TRawStr::FormatVal(tCIDLib::TInt4(-1), szTmp, 128);
    if (TRawStr::eCompareStr(szTmp, L"-1"))
        cout << _CurLn_ << "TInt4 format failed\r\n";


    // Format the unsigned int type
    TRawStr::FormatVal(tCIDLib::TUInt(12345), szTmp, 128);
    if (TRawStr::eCompareStr(szTmp, L"12345"))
        cout << _CurLn_ << "TUInt format failed\r\n";


    // Format the 64 bit integer type
    TRawStr::FormatVal(tCIDLib::TInt8(-1), szTmp, 128);
    if (TRawStr::eCompareStr(szTmp, L"-1"))
        cout << _CurLn_ << "TInt8 format failed\r\n";


    //
    //  Format the floating point types. Use a couple of different scenarios
    //  with the last digit of the ingoing value, to test for bad rounding.
    //
    //  We only need to test the TFloat8 version, since the TFloat4 version
    //  calls it internallly.
    //
    TRawStr::FormatVal
    (
        tCIDLib::TFloat8(12345.9876)
        , szTmp
        , 3
        , c4MaxBufChars(szTmp)
        , tCIDLib::ETrail_Spaces
    );
    if (TRawStr::eCompareStr(szTmp, L"12345.987"))
        cout << _CurLn_ << "TFloat8 format failed\r\n";

    TRawStr::FormatVal
    (
        tCIDLib::TFloat8(12345.9878)
        , szTmp
        , 3
        , c4MaxBufChars(szTmp)
        , tCIDLib::ETrail_Spaces
    );
    if (TRawStr::eCompareStr(szTmp, L"12345.987"))
        cout << _CurLn_ << "TFloat8 format failed\r\n";

    TRawStr::FormatVal
    (
        tCIDLib::TFloat8(12345.987)
        , szTmp
        , 3
        , c4MaxBufChars(szTmp)
        , tCIDLib::ETrail_Spaces
    );
    if (TRawStr::eCompareStr(szTmp, L"12345.987"))
        cout << _CurLn_ << "TFloat8 format failed\r\n";

    TRawStr::FormatVal
    (
        tCIDLib::TFloat8(12345.98)
        , szTmp
        , 3
        , c4MaxBufChars(szTmp)
        , tCIDLib::ETrail_Spaces
    );
    if (TRawStr::eCompareStr(szTmp, L"12345.98 "))
        cout << _CurLn_ << "TFloat8 format failed\r\n";
}

//
//  In all of these tests the memory debug alloc/delete stuff will catch
//  overflows of the string buffer as long as we use allocated strings then
//  delete them immediately afterwards. It allocates extra bytes and puts a
//  guard pattern at the end.
//
static tCIDLib::TVoid __TestOverflows()
{
    //
    //  Test out the basic copy and cat operations to make sure that they
    //  don't have any boundary conditions. Do both the UNICode and the
    //  short character versions.
    //
    {
        // Do a string same size as target
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , L"12345678"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy did not result in len of 8\r\n";
        delete pszTest;

        pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , "12345678"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy did not result in len of 8\r\n";
        delete pszTest;
    }

    {
        // Do a string longer than the target to insure clipping
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , L"12345678901234"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy did not result in len of 8\r\n";
        delete pszTest;

        pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , "12345678901234"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy did not result in len of 8\r\n";
        delete pszTest;
    }


    {
        // Cat a string same size as target
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[9];
        pszTest[0] = kCIDLib::chNull;
        TRawStr::CatStr
        (
            pszTest
            , L"12345678"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Cat did not result in len of 8\r\n";
        delete pszTest;

        pszTest = new tCIDLib::Tch[9];
        pszTest[0] = kCIDLib::chNull;
        TRawStr::CatStr
        (
            pszTest
            , "12345678"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Cat did not result in len of 8\r\n";
        delete pszTest;
    }

    {
        // Do a copy then cat a string that should exactly fit
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , L"1234"
            , 8
        );
        TRawStr::CatStr
        (
            pszTest
            , L"5678"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy and Cat did not result in len of 8\r\n";
        delete pszTest;

        pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , "1234"
            , 8
        );
        TRawStr::CatStr
        (
            pszTest
            , "5678"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy and Cat did not result in len of 8\r\n";
        delete pszTest;
    }

    {
        // Do a copy then cat a string that should be clipped
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , L"1234"
            , 8
        );
        TRawStr::CatStr
        (
            pszTest
            , L"5678901234"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy and Cat did not result in len of 8\r\n";
        delete pszTest;

        pszTest = new tCIDLib::Tch[9];
        TRawStr::CopyStr
        (
            pszTest
            , "1234"
            , 8
        );
        TRawStr::CatStr
        (
            pszTest
            , "5678901234"
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Copy and Cat did not result in len of 8\r\n";
        delete pszTest;
    }

    {
        //
        //  Test the character fill operation, on both short and long
        //  strings.
        //
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[9];
        TRawStr::FillString
        (
            pszTest
            , L'1'
            , 8
        );
        if (TRawStr::c4StrLen(pszTest) != 8)
            cout << _CurLn_ << "Fill did not result in len of 8\r\n";
        delete pszTest;

        tCIDLib::Tsch* pszTest2 = new tCIDLib::Tsch[9];
        TRawStr::FillString
        (
            pszTest2
            , '1'
            , 8
        );
        if (TRawStr::c4StrLen(pszTest2) != 8)
            cout << _CurLn_ << "Fill did not result in len of 8\r\n";
        delete pszTest2;
    }


    //
    //  Check out the string formatting function (which is at the core of
    //  all of the other type formatting functions, so we can get a lot of
    //  mileage by just checking it.
    //
    {
        // Test a string string shorter than the target, right justified
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[17];
        TRawStr::FormatStr
        (
            L"This is a test"
            , pszTest
            , 16
            , L' '
            , tCIDLib::EHJustify_Right
        );
        if (TRawStr::c4StrLen(pszTest) != 16)
            cout << _CurLn_ << "Format did not result in len of 16\r\n";
        delete pszTest;

        // Test a string string shorter than the target, centered
        pszTest = new tCIDLib::Tch[17];
        TRawStr::FormatStr
        (
            L"This is a test"
            , pszTest
            , 16
            , L' '
            , tCIDLib::EHJustify_Center
        );
        if (TRawStr::c4StrLen(pszTest) != 16)
            cout << _CurLn_ << "Format did not result in len of 16\r\n";
        delete pszTest;
    }

    {
        // Test a string longer than the target, right justified
        tCIDLib::Tch* pszTest = new tCIDLib::Tch[17];
        TRawStr::FormatStr
        (
            L"This is a test of a string longer than the target"
            , pszTest
            , 16
            , L' '
            , tCIDLib::EHJustify_Right
        );
        if (TRawStr::c4StrLen(pszTest) != 16)
            cout << _CurLn_ << "Format did not result in len of 16\r\n";
        delete pszTest;

        // Test a string longer than the target, centered
        pszTest = new tCIDLib::Tch[17];
        TRawStr::FormatStr
        (
            L"This is a test of a string longer than the target"
            , pszTest
            , 16
            , L' '
            , tCIDLib::EHJustify_Center
        );
        if (TRawStr::c4StrLen(pszTest) != 16)
            cout << _CurLn_ << "Format did not result in len of 16\r\n";
        delete pszTest;
    }
}


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid TestRawStrings()
{
    __TestOverflows();
    __TestConversions();
    __TestFormatting();
}
