//
//  FILE NAME: TestKernel_Mutex.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/12/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module tests the TKrnlMutex class.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"


// ----------------------------------------------------------------------------
//  Local data types
// ----------------------------------------------------------------------------
struct  TThreadInfo
{
    tCIDLib::TCard4     c4SleepTime;
    tCIDLib::TBoolean   bShutdown;
    tCIDLib::Tch*       pszError;
    TKrnlMutex*         pkmtxTest;
};


// ----------------------------------------------------------------------------
//  Local data
//
//  __bGone
//      The main thread sets this when it thinks all the test threads are
//      shut down. If any of the threads see this set, they know that they
//      are still alive even though the main thread thinks they are dead.
//
//  __tidCurrent
//      This is set to the id of the current thread that has the lock. If
//      another thread sees this set to a valid id when it gets control, then
//      it knows something happened bad. Each thread zeros it before letting
//      go of the value.
// ----------------------------------------------------------------------------
static tCIDLib::TBoolean        __bGone = kCIDLib::False;
static tCIDLib::TThreadId       __tidCurrent = kCIDLib::tidInvalid;


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::EExitCodes THREADCALL __eThreadFunc(tCIDLib::TVoid* pData)
{
    // Get a pointer to our data, which is a TThreadInfo structure
    TThreadInfo& Info = *(TThreadInfo*)pData;

    //
    //  Loop until our shutdown flag gets set. We sleep for our sleep
    //  time on each round.
    //
    while (!Info.bShutdown)
    {
        // See if the gone flag is set
        if (__bGone)
        {
            cout << _CurLn_
                 << "Main thread exited while test thread is running\n";
            return tCIDLib::EExit_FatalError;
        }

        // Sleep for our time
        TKrnlThread::Sleep(Info.c4SleepTime);

        // Get control of the mutex in a faux block
        {
            TKrnlMutexLocker kmtxlAccess(Info.pkmtxTest);

            // Check the global id. It should be invalid
            if (__tidCurrent != kCIDLib::tidInvalid)
            {
                Info.pszError = L"Thread id was valid upon getting control";
                return tCIDLib::EExit_FatalError;
            }

            // Set it to our id
            __tidCurrent = TKrnlThread::tidCaller();

            //
            //  And we should be able to lock the mutex again because its
            //  a counting mutex.
            //
            {
                TKrnlMutexLocker kmtxlAccess2(Info.pkmtxTest);
            }

            // And sleep for half of our time
            TKrnlThread::Sleep(Info.c4SleepTime / 2);

            // Set the thread id back to invalid
            __tidCurrent = kCIDLib::tidInvalid;
        }
    }
    return tCIDLib::EExit_Normal;
}


static tCIDLib::TVoid __BasicMutexTests()
{
    tCIDLib::Tch szTmpName[1024];

    // Test the constructors and make sure they set the right values
    TKrnlMutex  kmtxTest1;

    if (kmtxTest1.eShareState() != tCIDLib::EShareState_Unshared)
        cout << _CurLn_ << "Init share state was not unshared\r\n";

    if (kmtxTest1.bNamed())
        cout << _CurLn_ << "Default constructor did not create unnamed\r\n";

    if (kmtxTest1.bValid())
        cout << _CurLn_ << "Handle was valid before creation\r\n";

    try
    {
        kmtxTest1.Create(tCIDLib::ELockState_Unlocked);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error creating test mutex 1. "
             << kerrToCatch << "\r\n";
    }

    if (!kmtxTest1.bValid())
        cout << _CurLn_ << "Created mutex was not valid\r\n";

    // Create a name object for the mutex
    TKrnlRscName krsnTest(L"TestKernel", L"Test2", L"Mutex");

    // Build the real mutex name
    krsnTest.BuildFullName
    (
        szTmpName
        , c4MaxBufChars(szTmpName)
        , tCIDLib::ENamedRsc_Mutex
    );

    // Create a named semaphore now and test it out
    TKrnlMutex   kmtxTest2(szTmpName);
    if (kmtxTest2.eShareState() != tCIDLib::EShareState_Shared)
        cout << _CurLn_ << "Init share state was not shared\r\n";

    if (!kmtxTest2.bNamed())
        cout << _CurLn_ << "Constructor did not create named\r\n";

    if (TRawStr::eCompareStr(kmtxTest2.pszName(), szTmpName))
        cout << _CurLn_ << "Mutex name was not same as original\r\n";

    if (kmtxTest2.bValid())
        cout << _CurLn_ << "Handle was valid before creation\r\n";

    try
    {
        kmtxTest2.Create(tCIDLib::ELockState_Unlocked);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error creating test mutex 2. "
             << kerrToCatch << "\r\n";
    }

    if (!kmtxTest2.bValid())
        cout << _CurLn_ << "Created mutex was not valid\r\n";

    //
    //  Now try to create it again. This should fail because it already
    //  exists.
    //
    tCIDLib::TBoolean bCaught = kCIDLib::False;
    TKrnlMutex kmtxDuplicate(szTmpName);
    try
    {
        kmtxDuplicate.Create(tCIDLib::ELockState_Unlocked);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        // Make sure its the expected error
        if (kerrToCatch.errcId() == kKrnlErrors::errcAlreadyExists)
            bCaught = kCIDLib::True;
    }

    if (!bCaught)
        cout << _CurLn_ << "Failed to catch duplicate mutex name\r\n";
}


static tCIDLib::TVoid __ThreadedMutexTests()
{
    // Create an array of thread objects and thread info structures
    const tCIDLib::TCard4 c4ThreadCnt = 8;
    TKrnlThread* apkthrTest[c4ThreadCnt];
    TThreadInfo aThreadInfo[c4ThreadCnt];

    // Create the mutex that we want to test with
    TKrnlMutex  kmtxTest;
    kmtxTest.Create(tCIDLib::ELockState_Unlocked);

    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4ThreadCnt; c4Index++)
    {
        apkthrTest[c4Index] = new TKrnlThread;

        aThreadInfo[c4Index].c4SleepTime = c4Index * 250;
        aThreadInfo[c4Index].bShutdown = kCIDLib::False;
        aThreadInfo[c4Index].pszError = 0;
        aThreadInfo[c4Index].pkmtxTest = &kmtxTest;
    }

    // Kick off the threads
    for (c4Index = 0; c4Index < c4ThreadCnt; c4Index++)
    {
        apkthrTest[c4Index]->BeginThread
        (
            __eThreadFunc
            , 0
            , &aThreadInfo[c4Index]
            , tCIDLib::EThreadFlag_None
        );
    }

    //
    //  Sleep for a while while they work. Then wake up and ask them all
    //  to shut down.
    //
    TKrnlThread::Sleep(10000);

    for (c4Index = 0; c4Index < c4ThreadCnt; c4Index++)
        aThreadInfo[c4Index].bShutdown = kCIDLib::True;


    for (c4Index = 0; c4Index < c4ThreadCnt; c4Index++)
    {
        if (apkthrTest[c4Index]->eWaitForDeath(5000) != tCIDLib::EExit_Normal)
        {
            cout << "Thread #" << c4Index << " had an error: Text: \n    "
                 << aThreadInfo[c4Index].pszError << "\n";
        }
    }

    // Indicate that we think everyone is down
    __bGone = kCIDLib::True;

    // Delete the thread objects
    for (c4Index = 0; c4Index < c4ThreadCnt; c4Index++)
        delete apkthrTest[c4Index];
}



// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestMutexes
//
// DESCRIPTION:
//
//  This method called from the main module to test out the mutex classes.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TestMutexes()
{
    __BasicMutexTests();
    __ThreadedMutexTests();
}
