//
//  FILE NAME: TestKernel_Helper.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements some simple helper stuff that we just want to
//  segregate out from the actual testing stuff.
//
//  CAVEATS/GOTCHAS:
//
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"



// ----------------------------------------------------------------------------
//  Public functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: operator<<
//
// DESCRIPTION:
//
//  This is a streaming operator to let us get out UNICode strings out to
//  the standard output stream. It basically forces a flush on the stream,
//  so that our output will syncrhonize, then it uses the wide string
//  file write functions to output them.
//
//  There is also one to output current line objects and one for kernel
//  errors.
// -------------------------------------
//   INPUT: OutStream is the stream to write to
//          ustrToStream is the unicode string object to stream
//
//  OUTPUT: None
//
//  RETURN: 0 if no errors, else 1.
//
ostream& operator<<(ostream& OutStream, const TUNIStr& ustrToStream)
{
    tCIDLib::Tsch* pszAscii;
    if (ustrToStream.__pszStr)
        pszAscii = TRawStr::pszConvert(ustrToStream.__pszStr);
    else
        pszAscii = TRawStr::pszConvert(ustrToStream.__szStr);

    OutStream << pszAscii;
    delete pszAscii;
    return OutStream;
}


ostream& operator<<(ostream& OutStream, const TCurLn& clnToStream)
{
    OutStream << "(";
    OutStream << TUNIStr(clnToStream.__pszFile);
    OutStream << "." << clnToStream.__c4Line << ") ";
    return OutStream;
}


ostream& operator<<(ostream& OutStream, const TKrnlError& kerrToStream)
{
    OutStream << "Error: " << kerrToStream.errcId();
    if (kerrToStream.errcHostId())
        OutStream << ", Host Error: " << kerrToStream.errcHostId();
    return OutStream;
}


// ----------------------------------------------------------------------------
//   CLASS: TCurLn
//  PREFIX: cln
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TCurLn
//                       ~TCurLn
//
// DESCRIPTION:
//
//  This guy will make a copy of the root part of the file name, in an
//  allocated buffer. The line is just stored away. The destructor cleans
//  up the buffer.
// -------------------------------------
//   INPUT: pszFile is the file name. It is a short character type string
//              because that's what the __FILE__ macro provides.
//          c4Line is the line number
//
//  OUTPUT: None
//
//  RETURN: None
//
TCurLn::TCurLn( const   tCIDLib::Tsch* const    pszFile
                , const tCIDLib::TCard4         c4Line) :
    __c4Line(c4Line)
    , __pszFile(0)
{
    // Sanity check
    if (!pszFile || !*pszFile)
    {
        __pszFile = new tCIDLib::Tch[1];
        *__pszFile = 0;
    }
     else
    {
        //
        //  Find the root part of the file. We go to the end of the string
        //  and work back to the first slash.
        //
        const tCIDLib::Tsch* pszCur = &pszFile[TRawStr::c4StrLen(pszFile)];
        while ((*pszCur != '\\') && (pszCur > pszFile))
            pszCur--;

        if (pszCur == pszFile)
        {
            // There was no slash
            const tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszFile);
            __pszFile = new tCIDLib::Tch[c4Len+1];
            TRawStr::pszConvert(pszFile, __pszFile, c4Len);
        }
         else
        {
            // Copy from the next char
            const tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszCur);
            pszCur++;
            __pszFile = new tCIDLib::Tch[c4Len+1];
            TRawStr::pszConvert(pszCur, __pszFile, c4Len);
        }
    }
}

TCurLn::~TCurLn()
{
    if (__pszFile)
    {
        delete __pszFile;
        __pszFile = 0;
    }
}
