//
//  FILE NAME: TestKernel_HashMap.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 08/13/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module tests the TKrnlHashMap class. This class is a very simple
//  hashed map for raw strings for use within the kernel.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::TVoid __BasicTests()
{
    static tCIDLib::Tch* apszKeys[] = 
    {
        L"PATH"
        , L"LIBPATH"
        , L"BUILDSOURCE"
        , L"USERNAME"
        , L"TMP"
        , L"CIDDRIVE"
    };

    static tCIDLib::Tch* apszValues[] = 
    {
        L"C:\\TEMP;D:\\SOMEDIR;"
        , L"C:\\DLL;D:\\APP\\DLL;"
        , L"J:\\CID_DEV\\Source"
        , L"Dean Roddey"
        , L"C:\\Tmp"
        , L"J:"
    };

    const tCIDLib::TCard4 c4Count = c4ArrayElems(apszKeys);

    // Create a test hash map and give it the desired modulus.
    TKrnlHashMap  khshmTest(23);

    // There should be 0 elements
    if (khshmTest.c4ElemCount())
    {
        cout    << _CurLn_
                << "New hash map has non-zero element count\n";
        return;
    }

    //
    //  Add some values to it. They are just statically defined lists
    //  of strings above.
    //
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
        khshmTest.Add(apszKeys[c4Index], apszValues[c4Index]);

    // There should be c4Count elements now
    if (khshmTest.c4ElemCount() != c4Count)
    {
        cout    << _CurLn_
                << "Hash map has " << khshmTest.c4ElemCount()
                << "elements instead of " << c4Count << "\n";
        return;
    }

    //
    //  We should be able to find each key in the list and gets its
    //  value and compare it to the original.
    //
    for (c4Index = 0; c4Index < c4Count; c4Index++)
    {
        const tCIDLib::Tch* pszTmp = khshmTest.pszValueForKey(apszKeys[c4Index]);

        if (TRawStr::eCompareStr(pszTmp, apszValues[c4Index]))
        {
            cout    << _CurLn_
                    << "Expected value: '" << apszValues[c4Index]
                    << "' for key: '" << apszKeys[c4Index]
                    << "' but got value: '" << pszTmp
                    << "'\r\n";
            return;
        }
    }

    //
    //  Now lets update every element and set its value to the values
    //  in reverse of the original order.
    //
    for (c4Index = 0; c4Index < c4Count; c4Index++)
    {
        khshmTest.UpdateKey
        (
            apszKeys[c4Index]
            , apszValues[c4Count - (c4Index+1)]
        );
    }

    // And test the again
    for (c4Index = 0; c4Index < c4Count; c4Index++)
    {
        const tCIDLib::Tch* pszTmp = khshmTest.pszValueForKey(apszKeys[c4Index]);

        if (TRawStr::eCompareStr(pszTmp, apszValues[c4Count - (c4Index+1)]))
        {
            cout    << _CurLn_
                    << "Expected value: '" << apszValues[c4Count - (c4Index+1)]
                    << "' for key: '" << apszKeys[c4Index]
                    << "' but got value: '" << pszTmp
                    << "'\r\n";
            return;
        }
    }

    //
    //  And now we should be able to flush the map and not find any elements
    //  in it.
    //
    khshmTest.Flush();

    // There should be 0 elements again
    if (khshmTest.c4ElemCount())
    {
        cout    << _CurLn_
                << "Flushed hash map has non-zero element count\n";
        return;
    }

    // Now try to find each element, all of which should fail
    for (c4Index = 0; c4Index < c4Count; c4Index++)
    {
        tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
        try
        {
            const tCIDLib::Tch* pszTmp = khshmTest.pszValueForKey
            (
                apszKeys[c4Index]
            );
        }

        catch(const TKrnlError& kerrToCatch)
        {
            if (kerrToCatch.errcId() == kKrnlErrors::errcNotFound)
                bCaughtIt = kCIDLib::True;
        }

        if (!bCaughtIt)
        {
            cout    << _CurLn_
                    << L"Found an element in a flushed hash map\r\n";
            return;
        }
    }

    //
    //  Re-add the elements back and do the search again
    //
    for (c4Index = 0; c4Index < c4Count; c4Index++)
        khshmTest.Add(apszKeys[c4Index], apszValues[c4Index]);

    // There should be c4Count elements again now
    if (khshmTest.c4ElemCount() != c4Count)
    {
        cout    << _CurLn_
                << "Hash map has " << khshmTest.c4ElemCount()
                << "elements instead of " << c4Count << "\n";
        return;
    }

    for (c4Index = 0; c4Index < c4Count; c4Index++)
    {
        const tCIDLib::Tch* pszTmp = khshmTest.pszValueForKey(apszKeys[c4Index]);

        if (TRawStr::eCompareStr(pszTmp, apszValues[c4Index]))
        {
            cout    << _CurLn_
                    << "Expected value: '" << apszValues[c4Index]
                    << "' for key: '" << apszKeys[c4Index]
                    << "' but got value: '" << pszTmp
                    << "'\r\n";
            return;
        }
    }

    //
    //  And now remove each element by key. This should also result in an
    //  empty hash map.
    //
    for (c4Index = 0; c4Index < c4Count; c4Index++)
        khshmTest.RemoveKey(apszKeys[c4Index]);

    // There should be 0 elements again
    if (khshmTest.c4ElemCount())
    {
        cout    << _CurLn_
                << "Emptied hash map has non-zero element count\n";
        return;
    }
}


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestHashMap
//
// DESCRIPTION:
//
//  This method is called from the main module to test the TKrnlHashMap class.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TestHashMap()
{
    // And do our tests
    __BasicTests();
}
