//
//  FILE NAME: TestKernel_File.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/10/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module tests the TKrnlFile, TKrnlDirIter, and xxxx classes. These
//  are all file system oriented class so we do them all here.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"
#include    <windows.h>



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::TVoid __BasicFileTest()
{
    // Create a file object and open it
    TKrnlFile   kflTest(L"Basic.TestFile");

    // Make sure the file name is correct
    if (wcscmp(kflTest.pszName(), L"Basic.TestFile"))
    {
        cout    << _CurLn_
                << "File not correct. It was: " << kflTest.pszName()
                << "\r\n";
        return;
    }

    // Make sure the handle is 0
    if (kflTest.bValid())
    {
        cout << _CurLn_ << "Handle should not be valid yet\r\n";
        return;
    }

    try
    {
        kflTest.Open
        (
            tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_System
            , tCIDLib::EFileFlag_None
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error opening file. "
                << kerrToCatch << "\r\n";
    }

    //
    //  Write out a buffer to the file. We just create a buffer and
    //  set each value to its index for testing.
    //
    const tCIDLib::TCard4 c4TableSize = 4096;
    tCIDLib::TCard2 ac2ToWrite[c4TableSize];
    for (tCIDLib::TCard2 c2Cur = 0; c2Cur < c4TableSize; c2Cur++)
        ac2ToWrite[c2Cur] = c2Cur;

    // Now write it out
    tCIDLib::TCard4 c4Actual;
    try
    {
        kflTest.WriteBuffer(ac2ToWrite, sizeof(ac2ToWrite), c4Actual);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error writing to file. "
                << kerrToCatch << "\r\n";
    }

    if (c4Actual != sizeof(ac2ToWrite))
        cout << _CurLn_ << "Only wrote " << c4Actual << " bytes to file\r\n";


    // Check the current file position and see if its at the expected pos
    tCIDLib::TFilePos fposCurPos;
    try
    {
        fposCurPos = kflTest.fposFilePtr();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error getting file position. "
                << kerrToCatch << "\r\n";
    }

    if (fposCurPos != sizeof(ac2ToWrite))
    {
        cout << _CurLn_
             << "Current position was " << tCIDLib::TInt4(fposCurPos) << "\r\n";
    }

    // Seek back to the start and check the position again
    try
    {
        kflTest.Seek(0, tCIDLib::ESeekFrom_Start, fposCurPos);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error seeking back to 0. "
                << kerrToCatch << "\r\n";
    }

    if (fposCurPos != 0)
    {
        cout << _CurLn_
             << "Current position was " << tCIDLib::TInt4(fposCurPos) << "\r\n";
    }

    //
    //  Zero out the buffer and read back into again now. Check that we got
    //  the expected data back.
    //
    for (c2Cur = 0; c2Cur < c4TableSize; c2Cur++)
        ac2ToWrite[c2Cur] = 0;

    try
    {
        kflTest.ReadBuffer(ac2ToWrite, sizeof(ac2ToWrite), c4Actual);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error reading in data. "
                << kerrToCatch << "\r\n";
    }

    if (c4Actual != sizeof(ac2ToWrite))
        cout << _CurLn_ << "Only read " << c4Actual << " bytes from file\r\n";

    for (c2Cur = 0; c2Cur < c4TableSize; c2Cur++)
    {
        if (ac2ToWrite[c2Cur] != c2Cur)
        {
            cout << _CurLn_ << "Expected " << c2Cur
                 << " but got " << ac2ToWrite[c2Cur] << "\r\n";
            break;
        }
    }

    // Truncate the file at 16 bytes
    try
    {
        kflTest.TruncateAt(16);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error truncating file. "
                << kerrToCatch << "\r\n";
    }

    // The file pointer should now be at 16
    try
    {
        fposCurPos = kflTest.fposFilePtr();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout    << _CurLn_ << "Error getting file position. "
                << kerrToCatch << "\r\n";
    }

    if (fposCurPos != 16)
    {
        cout << _CurLn_
             << "Current position was "
             << tCIDLib::TInt4(fposCurPos) << "\r\n";
    }

    //
    //  Close the file now. After we close it, the handle should no longer
    //  be valid.
    //
    try
    {
        kflTest.Close();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error closing file. "
             << kerrToCatch << "\r\n";
        return;
    }

    if (kflTest.bValid())
        cout << _CurLn_ << "Handle should not be valid now\r\n";
}


static tCIDLib::TVoid __FileFindTest()
{
    const tCIDLib::Tch* const   pszFile = L"TestKernel_File.Cpp";
    TKrnlDirSearch              kdsTest;
    TKrnlFileSys::TRawFileFind  fndbToFill;

    if (!kdsTest.bFindFirst(pszFile
                            , tCIDLib::EFileAttr_None
                            , tCIDLib::EFileAttr_None
                            , fndbToFill))
    {
        cout << _CurLn_ << "Could not find test file\r\n";
    }

    // Compare the name and make sure its reasonable
    if (TRawStr::eCompareStr(fndbToFill.szName, pszFile))
        cout << _CurLn_ << "Found name was not the same as original\r\n";

    //
    //  Do a raw Win32 lookup of the same file and lets compare the results
    //  to make sure that they agree.
    //
    WIN32_FIND_DATA FindData;
    HANDLE hDir = FindFirstFileW(pszFile, &FindData);
    FindClose(hDir);

    if (TRawStr::eCompareStr(fndbToFill.szName, FindData.cFileName))
        cout << _CurLn_ << "Found name does not agree with Win32 search\r\n";

    if (fndbToFill.fposFileSize != TRawBits::i8From32
    (
        FindData.nFileSizeLow
        , FindData.nFileSizeHigh))
    {
        cout << _CurLn_ << "Found size does not agree with Win32 search\r\n";
    }
}


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestFiles
//
// DESCRIPTION:
//
//  This method is called from the main module to test files.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TestFiles()
{
    __BasicFileTest();
    __FileFindTest();
}
