//
//  FILE NAME: TestKernel_Events.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/12/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module tests the TKrnlEvent class.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"


// ----------------------------------------------------------------------------
//  Local static data
//
//  __c4ThreadSet
//      A test flag that the test thread sets to indicate that it did what
//      it was supposed to do.
//
//  __pkevTest1
//  __pkevTest2
//      Two events that are created and destroyed as required to do testing.
//      They are global to make it easy for the spun off test threads to
//      see.
// ----------------------------------------------------------------------------
static tCIDLib::TCard4  __c4ThreadSet = 0;
static TKrnlEvent*      __pkevTest1 = 0;
static TKrnlEvent*      __pkevTest2 = 0;


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::EExitCodes THREADCALL __eThreadFunc(tCIDLib::TVoid*)
{
    // Sleep for a little while
    TKrnlThread::Sleep(2000);

    // Bump up the global flag
    __c4ThreadSet++;

    // And post the first test event
    __pkevTest1->Pulse();

    return tCIDLib::EExit_Normal;
}

static tCIDLib::EExitCodes THREADCALL __eThreadFunc2(tCIDLib::TVoid*)
{
    // Bump up the global flag
    __c4ThreadSet++;

    // Signal that we are up and waiting
    __pkevTest1->Trigger();

    // Now wait on the other one
    __pkevTest2->WaitFor();

    // Bump up the global flag down
    __c4ThreadSet++;

    // And signal that we are done
    __pkevTest1->Trigger();

    return tCIDLib::EExit_Normal;
}



static tCIDLib::TVoid __BasicEventTests()
{
    tCIDLib::Tch szTmpName[1024];

    // Test the constructors and make sure they set the right values
    TKrnlEvent  kevTest1;

    if (kevTest1.eShareState() != tCIDLib::EShareState_Unshared)
        cout << _CurLn_ << "Init share state was not unshared\r\n";

    if (kevTest1.bNamed())
        cout << _CurLn_ << "Default constructor did create unnamed\r\n";

    if (kevTest1.bValid())
        cout << _CurLn_ << "Handle was valid before creation\r\n";

    try
    {
        kevTest1.Create(tCIDLib::EEventState_Triggered);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error creating test event 1. "
             << kerrToCatch << "\r\n";
    }

    if (!kevTest1.bValid())
        cout << _CurLn_ << "Created event was not valid\r\n";

    // Create a name object for our test event
    TKrnlRscName krsnTest(L"TestKernel", L"Test2", L"Event");

    //
    //  Build the full name to the event and construct an event object
    //
    krsnTest.BuildFullName
    (
        szTmpName
        , c4MaxBufChars(szTmpName)
        , tCIDLib::ENamedRsc_Event
    );
    TKrnlEvent kevTest2(szTmpName);

    if (kevTest2.eShareState() != tCIDLib::EShareState_Shared)
        cout << _CurLn_ << "Init share state was not shared\r\n";

    if (!kevTest2.bNamed())
        cout << _CurLn_ << "Constructor did not create named\r\n";

    if (TRawStr::eCompareStr(kevTest2.pszName(), szTmpName))
        cout << _CurLn_ << "Event name was not same as original\r\n";

    if (kevTest2.bValid())
        cout << _CurLn_ << "Handle was valid before creation\r\n";

    try
    {
        kevTest2.Create(tCIDLib::EEventState_Triggered);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << _CurLn_ << "Error creating test event 2. "
             << kerrToCatch << "\r\n";
    }

    if (!kevTest2.bValid())
        cout << _CurLn_ << "Created event was not valid\r\n";

    //
    //  Now try to create it again. This should fail because it already
    //  exists.
    //
    tCIDLib::TBoolean bCaught = kCIDLib::False;
    TKrnlEvent kevDuplicate(szTmpName);
    try
    {
        kevDuplicate.Create(tCIDLib::EEventState_Triggered);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        // Make sure its the expected error
        if (kerrToCatch.errcId() == kKrnlErrors::errcAlreadyExists)
            bCaught = kCIDLib::True;
    }

    if (!bCaught)
        cout << _CurLn_ << "Failed to catch duplicate event name\r\n";
}


static tCIDLib::TVoid __ThreadedEventTests()
{
    // Create the test event. Create it reset
    __pkevTest1 = new TKrnlEvent;
    __pkevTest1->Create(tCIDLib::EEventState_Reset);

    TKrnlThread kthrTest;
    kthrTest.BeginThread
    (
        __eThreadFunc
        , 0
        , 0
        , tCIDLib::EThreadFlag_None
    );

    // Wait on the event, up to 5 seconds
    __pkevTest1->WaitFor(5 Seconds);

    // Make sure that the global flag is set to 1
    if (__c4ThreadSet != 1)
        cout << _CurLn_ << "Test thread did not set global flag\r\n";

    // Delete the test event
    delete __pkevTest1;
}


static tCIDLib::TVoid __ThreadedEventTests2()
{
    //
    //  Create the test events. Create them as auto mode events that are
    //  initially triggered.
    //
    __pkevTest1 = new TKrnlEvent;
    __pkevTest1->Create(tCIDLib::EEventState_Reset, tCIDLib::EAutoMode_Automatic);

    __pkevTest2 = new TKrnlEvent;
    __pkevTest2->Create(tCIDLib::EEventState_Reset, tCIDLib::EAutoMode_Automatic);

    //
    //  Create an array of thread objects.
    //
    const tCIDLib::TCard4 c4ThreadCount = 8;
    TKrnlThread* apkthrTest[c4ThreadCount];
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        apkthrTest[c4Index] = new TKrnlThread;

    //
    //  Now kick each one off and wait for it to signal that it up, which it
    //  does on the first event. It should bump up the global flag before
    //  it signals us.
    //
    for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
    {
        // Zero the global flag
        __c4ThreadSet = 0;

        // And start the thread
        apkthrTest[c4Index]->BeginThread
        (
            __eThreadFunc2
            , 0
            , 0
            , tCIDLib::EThreadFlag_None
        );

        // Wait for the thread to get started
        __pkevTest1->WaitFor(2000);

        // The global flag should be 1
        if (__c4ThreadSet != 1)
        {
            cout << _CurLn_ << "Thread " << c4Index
                 << " failed to set global flag\r\n";
        }
    }

    // All of the threads should still be running, waiting on the 2nd event
    for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
    {
        if (!apkthrTest[c4Index]->bIsRunning())
            cout << _CurLn_ << "Thread " << c4Index
                 << " did not block on event\r\n";
    }

    //
    //  Now we pulse the second event for each thread. Each thread should
    //  bump up the global counter, so it should stay in sync with the index
    //  value.
    //
    __c4ThreadSet = 0;
    for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
    {
        // Trigger one thread
        __pkevTest2->Pulse();

        // Sleep a little
        TKrnlThread::Sleep(1000);

        if (__c4ThreadSet != c4Index+1)
            cout << _CurLn_ << "The pulse did not create desired effect\n";
    }

    // Make sure that they are all dead
    for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        apkthrTest[c4Index]->eWaitForDeath();

    // Delete the thread objecst
    for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        delete apkthrTest[c4Index];

    // Delete the test events
    delete __pkevTest1;
    delete __pkevTest2;
}



// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestEvents
//
// DESCRIPTION:
//
//  This method called from the main module to test out the event classes.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TestEvents()
{
    __BasicEventTests();
    __ThreadedEventTests();
    __ThreadedEventTests2();
}
