//
//  FILE NAME: TestKernel.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/07/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the program. This program tests the
//  services of the CIDKernel.Dll facility.
//
//  CAVEATS/GOTCHAS:
//
//  1)  This program is a terrible example of a CIDLib program because of its
//      special needs, to test out the lowest level stuff. This means it
//      makes some use of C/C++ RTL stuff in places for sanity checking.
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestKernel.Hpp"



// ----------------------------------------------------------------------------
//  Externs for the testing functions
// ----------------------------------------------------------------------------
extern tCIDLib::TVoid TestEvents();
extern tCIDLib::TVoid TestEnvironment();
extern tCIDLib::TVoid TestFiles();
extern tCIDLib::TVoid TestHashMap();
extern tCIDLib::TVoid TestModules();
extern tCIDLib::TVoid TestMutexes();
extern tCIDLib::TVoid TestRawStrings();
extern tCIDLib::TVoid TestRawMemory();
extern tCIDLib::TVoid TestSearchAndSort();
extern tCIDLib::TVoid TestSemaphores();
extern tCIDLib::TVoid TestSysExcepts();
extern tCIDLib::TVoid TestTime();
extern tCIDLib::TVoid TestThreads();


// ----------------------------------------------------------------------------
//  Global data
//
//  kmodTestKernel
//      We don't have facilities at this level, however the kernel class
//      that encapsulates the Exe/Dll module concept is what the facility
//      class is built on, so we test it out here.
// ----------------------------------------------------------------------------
TKrnlModule     kmodTestKernel(L"TestKernel", tCIDLib::EModType_Exe);


// ----------------------------------------------------------------------------
//  Local, static data
//
//  __aTestFunctions
//      An array of test function structure that are used by the main code
//      below to invoke the testing functions.
//
//  __c4TestFuncCount
//      The number of entries in the test function array.
// ----------------------------------------------------------------------------
static TTestFuncRecord      __aTestFunctions[] =
{
        { TestRawStrings        , L"Strings"    , kCIDLib::False }
    ,   { TestRawMemory         , L"Memory"     , kCIDLib::False }
    ,   { TestHashMap           , L"HashMap"    , kCIDLib::False }
    ,   { TestEnvironment       , L"Environment", kCIDLib::False }
    ,   { TestEvents            , L"Events"     , kCIDLib::False }
    ,   { TestMutexes           , L"Mutexes"    , kCIDLib::False }
    ,   { TestSemaphores        , L"Semaphores" , kCIDLib::False }
    ,   { TestTime              , L"Time"       , kCIDLib::False }
    ,   { TestSearchAndSort     , L"SearchNSort", kCIDLib::False }
    ,   { TestFiles             , L"Files"      , kCIDLib::False }
    ,   { TestThreads           , L"Threads"    , kCIDLib::False }
    ,   { TestModules           , L"Modules"    , kCIDLib::False }
    ,   { TestSysExcepts        , L"SysExcepts" , kCIDLib::False }
};
static const tCIDLib::TCard4 __c4TestFuncCount = c4ArrayElems(__aTestFunctions);


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __Announce
//
// DESCRIPTION:
//
//  This method is called to do the standard program announcement. Its just
//  here to keep it separated out.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __Announce()
{
    cout    << "\nTestKernel.Exe, Version: "
            << TUNIStr(kCIDLib::pszVersion) << "\n"
            << "A Testing Suite for the CIDKernel Facility\n"
            << "Compiled On: " << __DATE__ << "\n\n";
}


//
// FUNCTION/METHOD NAME: __ShowSysInfo
//
// DESCRIPTION:
//
//  This method is called to dump out all of the general system information
//  gatherable via the kernel facility.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ShowSysInfo()
{
    TKrnlSysInfo                ksysiTest;

    tCIDLib::TCard4 c4OSMaj, c4OSMin, c4OSBuild;
    ksysiTest.QueryOSInfo(c4OSMaj, c4OSMin, c4OSBuild);

    tCIDLib::TZStr128 szUser;
    ksysiTest.QueryUserName(szUser, c4MaxBufChars(szUser));

    cout    << "General System Information\n"
            << "-------------------------------------------------\n"
            << "        Node Name: " << TUNIStr(ksysiTest.pszNodeName()) << "\n"
            << "   CIDLib Version: " << TUNIStr(kCIDLib::pszVersion) << "\n"
            << "         CPU Type: " << tCIDLib::TInt4(ksysiTest.eCPUType()) << "\n"
            << "        CPU Count: " << ksysiTest.c4CPUCount() << "\n"
            << "     Total Memory: " << ksysiTest.c4TotalPhysicalMem() << "\n"
            << "       OS Version: " << c4OSMaj << "." << c4OSMin 
            <<                  "  Build:" << c4OSBuild << "\n"
            << "     Current User: " << TUNIStr(szUser) << "\n"
            << "\n";


    cout    << "Process/Thread Information\n"
            << "-------------------------------------------------\n"
            << "       Process Id: " << ksysiTest.pidThis() << "\n"
            << "        Thread Id: " << ksysiTest.tidCurrent() << "\n"
            << "   Process Handle: " << ksysiTest.hprocThis() << "\n"
            << "\n";


    TKrnlLocale::TLocaleInfo lociDisplay;
    TKrnlLocale::QueryLocaleInfo(lociDisplay);
    cout    << "Locale Information\n"
            << "-------------------------------------------------\n"
            << "           Locale: "
            <<          TUNIStr(lociDisplay.szLocale) << "\n"
            << "    Decimal Point: "
            <<          TUNIStr(lociDisplay.chDecimalSym) << "\n"
            << "    Thousands Sep: "
            <<          TUNIStr(lociDisplay.chThousandsSep) << "\n"
            << "  Currency Symbol: "
            <<          TUNIStr(lociDisplay.chCurrencySymbol) << "\n"
            << "    Positive Sign: "
            <<          TUNIStr(lociDisplay.chPosSign) << "\n"
            << "    Negative Sign: "
            <<          TUNIStr(lociDisplay.chNegSign) << "\n"
            << "\n";

    cout.flush();
}


// ----------------------------------------------------------------------------
//  Public functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: main
//
// DESCRIPTION:
//
//  This is the program entry point. It will do some overall setup and
//  announcement, then call each of the testing modules in the correct order.
// -------------------------------------
//   INPUT: i4ArgC, apszArgs are the standard command line parms.
//
//  OUTPUT: None
//
//  RETURN: 0 if no errors, else 1.
//
extern "C" tCIDLib::TInt4 wmain(const   tCIDLib::TInt4  i4ArgC
                                ,       tCIDLib::Tch*   apszArgs[]
                                ,       tCIDLib::Tch*   apszEnv[])
{
    // Turn off buffer for cout
    cout.setf(ios::unitbuf, ios::unitbuf);

    //  Announce ourself
    __Announce();

    //
    //  See if there was a command line parm, which indicates the test
    //  to run. If not, then assume all threads.
    //
    tCIDLib::TBoolean   bRunAll = kCIDLib::False;
    tCIDLib::TCard4     c4FnInd;
    if (i4ArgC < 2)
    {
        bRunAll = kCIDLib::True;
    }
     else
    {
        //
        //  Search the list of test functions and turn on each one that
        //  has a match in the list.
        //
        tCIDLib::TCard4 c4ArgInd;
        for (c4ArgInd = 1; c4ArgInd < tCIDLib::TCard4(i4ArgC); c4ArgInd++)
        {
            for (c4FnInd = 0; c4FnInd < __c4TestFuncCount; c4FnInd++)
            {
                if (!TRawStr::eICompareStr
                (
                    apszArgs[c4ArgInd]
                    , __aTestFunctions[c4FnInd].pszName))
                {
                    __aTestFunctions[c4FnInd].bRun = kCIDLib::True;
                }
            }
        }
    }

    //
    //  Put an overall exception handler around the calls to the testing
    //  methods so that we can catch any unhandled exceptions.
    //
    try
    {
        //
        //  First of all just dump the system information values. This is
        //  just for examining by eye. Its a local function.
        //
        __ShowSysInfo();

        cout << "\n\nStarting Tests...\n\n";

        for (c4FnInd = 0; c4FnInd < __c4TestFuncCount; c4FnInd++)
        {
            if (__aTestFunctions[c4FnInd].bRun || bRunAll)
            {
                cout    << "Testing "
                        << TUNIStr(__aTestFunctions[c4FnInd].pszName)
                        << "\n";

                // Create a faux block to force destruction of heap checker
                {
                    // Create a heap check tester
                    TKrnlMemCheck kmchkTest;

                    __aTestFunctions[c4FnInd].pfnTester();
                }

                // Validate the heap
                TKrnlMemCheck::ValidateHeap();
            }
        }

        cout << "\nTests complete\n\n";
    }

    catch(const TKrnlError& kerrToCatch)
    {
        cout << "Caught unhandled kernel exception. "
             << kerrToCatch << "\n";
        return 1;
    }

    catch(...)
    {
        cout << "Caught unknown exception\n";
        return 1;
    }

    // We made it
    return 0;
}
