//
// NAME: TestCIDLib_String.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program. This module is called
//  from the program's main function. The functions in this module test the
//  string classes to make sure that they are functioning correctly.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/05/92
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_ResourceIds.H"


// ----------------------------------------------------------------------------
//  Functions for local use
// ----------------------------------------------------------------------------

static tCIDLib::TVoid __TestConstructors(TTextStream& strmOut)
{
    // Create some variables to get back the broken down time into
    tCIDLib::TCard4     c4Hour;
    tCIDLib::TCard4     c4Minute;
    tCIDLib::TCard4     c4Second;
    tCIDLib::TCard4     c4MilliSecs;

    tCIDLib::TCard4     c4Year;
    tCIDLib::TCard4     c4Day;
    tCIDLib::EMonths    eMonth;
    tCIDLib::EWeekDays  eWeekDay;

    //
    //  First test the constructor that uses an absolute number of
    //  100 nano-second intervals from 01/01/1601. We create one that
    //  is one hour from the base time.
    //
    {
        TTime tmTest(kCIDLib::i8OneHour);

        // Make sure that the raw time interval comes back correct
        if (tmTest.i8Time() != kCIDLib::i8OneHour)
        {
            strmOut << _CurLn_
                    << L"Raw time was not what was used in constructor" << NewLn;
        }

        // Get the individual time values
        c4MilliSecs = tmTest.c4AsTimeInfo(c4Hour, c4Minute, c4Second);

        // Make sure that the values are what they should be
        if (c4Hour != 1)
        {
            strmOut << _CurLn_ << L"Hour was not equal to 1" << NewLn;
            return;
        }

        if (c4Minute || c4Second || c4MilliSecs)
        {
            strmOut << _CurLn_
                    << L"Minutes, seconds, and milliseconds should have been 0"
                    << NewLn;
            return;
        }

        // And test the individual date values
        eWeekDay = tmTest.eAsDateInfo(c4Year, eMonth, c4Day);

        if ((c4Year != 1601)
        ||  (eMonth != tCIDLib::EMonth_January)
        ||  (c4Day != 1))
        {
            strmOut << _CurLn_
                    << L"Date values were not correct after construction"
                    << NewLn;
            return;
        }

        // Copy construct and compare
        TTime tmCopy(tmTest);

        if (tmCopy.i8Time() != tmTest.i8Time())
        {
            strmOut << _CurLn_
                    << L"Copied time != to original"
                    << NewLn;
        }
    }

    //
    //  Now test the other way around. We build one up from all of the
    //  individual values, for the base time of midnight on 01/01/1601. Then
    //  we get back out the 100-nanosecond raw value and it should be 0.
    //
    {
        TTime tmTest
        (
            1601
            , tCIDLib::EMonth_January
            , 1
            , 0, 0, 0, 0
        );

        if (tmTest.i8Time() != 0)
        {
            strmOut << _CurLn_
                    << L"Raw time was not correct for component construction"
                    << NewLn;
        }
    }


    //
    //  Now lets do a particular date that we know is a certain day of the
    //  week and check it out.
    //
    {
        TTime tmTest(1997, tCIDLib::EMonth_March, 1);

        if (tmTest.eDayOfWeek() != tCIDLib::EWeekDay_Saturday)
        {
            strmOut << _CurLn_
                    << L"Day of the week was incorrect" << NewLn;
        }
    }
}


static tCIDLib::TVoid __TestEquality(TTextStream& strmOut)
{
    // Create some time objects that are equal and inequal
    TTime   tmSmall(kCIDLib::i8OneHour);
    TTime   tmLarge;
    tmLarge.SetToNow();
    TTime   tmSameAsLarge(tmLarge);

    // Make sure that small is less than large
    if (!(tmSmall < tmLarge))
    {
        strmOut << _CurLn_ << L"Small time >= large time" << NewLn;
        return;
    }

    // Make sure that large is greater than small
    if (!(tmLarge > tmSmall))
    {
        strmOut << _CurLn_ << L"Large time <= small time" << NewLn;
        return;
    }

    // Make sure that large and same as large are equal
    if (tmLarge != tmSameAsLarge)
    {
        strmOut << _CurLn_ << L"Copy constructed times not equal" << NewLn;
        return;
    }

    // Check <= and >= versions
    if (!(tmSmall <= tmLarge))
    {
        strmOut << _CurLn_ << L"Small time > large time" << NewLn;
        return;
    }

    if (!(tmLarge >= tmSmall))
    {
        strmOut << _CurLn_ << L"Large time < small time" << NewLn;
        return;
    }

    if (!(tmLarge >= tmSameAsLarge))
    {
        strmOut << _CurLn_ << L"Copy constructed times not equal" << NewLn;
        return;
    }

    if (!(tmLarge <= tmSameAsLarge))
    {
        strmOut << _CurLn_ << L"Copy constructed times not equal" << NewLn;
        return;
    }

    // Make sure non-equal objects are not equal
    if (tmLarge == tmSmall)
    {
        strmOut << _CurLn_ << L"Inequal objects showed up equal" << NewLn;
        return;
    }
}


static tCIDLib::TVoid __TestMath(TTextStream& strmOut)
{
    //
    //  Create a timestamp for 1 hour and one for two hours, then subtract
    //  one from the other.
    //
    TTime   tmTest1(kCIDLib::i8OneHour);
    TTime   tmTest2(kCIDLib::i8OneHour * 2);

    //
    //  First do it via the i8Difference() method, which is basically like
    //  the subtraction operator, but for folks who don't like operators.
    //
    if (tmTest2.i8Difference(tmTest1) != kCIDLib::i8OneHour)
    {
        strmOut << _CurLn_
                << L"Difference between times was not 1 hour" << NewLn;
    }

    // Add them together via the friend operator
    if (tmTest1 + tmTest2 != TTime(kCIDLib::i8OneHour * 3))
    {
        strmOut << _CurLn_
                << L"Friend addition of times was incorrect" << NewLn;
    }

    // Add one to the other via the member operator
    tmTest1 += tmTest2;
    if (tmTest1 != kCIDLib::i8OneHour * 3)
    {
        strmOut << _CurLn_
                << L"Member addition of times was incorrect" << NewLn;
    }

    // Subtract it back out and make sure its back to the original
    if (tmTest1 - tmTest2 != TTime(kCIDLib::i8OneHour))
    {
        strmOut << _CurLn_
                << L"Friend subtraction of times was incorrect" << NewLn;
    }

    // And via the member operator
    tmTest1 -= tmTest2;
    if (tmTest1 != kCIDLib::i8OneHour)
    {
        strmOut << _CurLn_
                << L"Member subtraction of times was incorrect" << NewLn;
    }

    //
    //  Subtracting the other way should cause an exception, because that
    //  would create a negative time.
    //
    tCIDLib::TBoolean bGotIt = kCIDLib::False;
    try
    {
        tmTest1 -= tmTest2;
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcTime_NegativeTime))
        {
            strmOut << _CurLn_
                    << L"Got the wrong exception for negative time" << NewLn;
        }
         else
        {
            bGotIt = kCIDLib::True;
        }
    }

    if (!bGotIt)
    {
        strmOut << _CurLn_
                << L"Negative time result did not cause exception" << NewLn;
    }

    // Make sure that the 100-nano second and millisecond views agree
    if (tmTest1.i8Time() / 10000 != tmTest1.i8MilliSeconds())
        strmOut << _CurLn_ << L"Different time views don't agree" << NewLn;
}


// ----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestTime
//
// DESCRIPTION:
//
//  This method is one of the testing methods of the facility class. It tests
//  out the TTime class, which is a timestamp and time manipulation class.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestTime()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"constructor";
        __TestConstructors(strmOut());

        pszCurTest = L"equality";
        __TestEquality(strmOut());

        pszCurTest = L"math";
        __TestMath(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"An exception occured during the "
                    << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
