//
// NAME: TestCIDLib_TextStreams.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  text stream classes.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/29/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"


// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
//  This method provides some common tests for text streams. Streams are
//  created with various implementation objects and passed here for testing.
//  These should all create the same results.
//
static tCIDLib::TVoid
__CommonTextTests(          TTextStream&        strmOut
                    ,       TTextStream&        strmToTest
                    , const tCIDLib::TFilePos   fposCurLogical
                    , const tCIDLib::TFilePos   fposPhysical)
{
    //
    //  Insure that the logical and physical ends are what they are claimed
    //  to be by the caller.
    //
    // See if the stream shows the correct logical and physical ends
    if (strmToTest.fposLogicalEnd() != fposCurLogical)
    {
        strmOut << _CurLn_ << L"Initial logical end was "
                << strmToTest.fposLogicalEnd() << L" but should have been "
                << fposCurLogical << NewLn;
        return;
    }

    if (strmToTest.fposPhysicalEnd() != fposPhysical)
    {
        strmOut << _CurLn_ << L"Physical end was "
                << strmToTest.fposPhysicalEnd() << L" but should have been "
                << fposPhysical << NewLn;
        return;
    }

    // Reset the stream and make sure the position goes to zero
    strmToTest.Reset();
    if (strmToTest.fposCurPos() != 0)
    {
        strmOut << _CurLn_ << L"Reset did not result in position of zero"
                << NewLn;
        return;
    }

    //
    //  The caller insures that there is no initial data in the stream
    //  so we should have an 'end of stream' state now.
    //
    if (!strmToTest.bEndOfStream())
    {
        strmOut << _CurLn_ << L"Initial stream is not at end of data" << NewLn;
        return;
    }

    //
    //  Lets stream out a block of text. Then we check that the position is
    //  what it should be, given what we just streamed out.
    //
    static tCIDLib::Tch* szTestText = 
    {
        L"Frapit, scand'lous moppet. Freezing wonder, plunders, blunders "
        L"and thunderous flop. Flowing poets, mowed low by bullets. Singing "
        L"rings scream, knowing no hits. Zits and flotsom, billious ransom "
        L"of kinds of minds, who crossed no transom of flowing bits of seasons "
        L"reasoned not by youth "
        L"Who reigns, games with sane names never lied to lowered truth."
    };

    //
    //  Stream out the text to the stream, then check that it adjusted the
    //  size appropriately.
    //
    strmToTest << TString(szTestText);
    if (strmToTest.fposCurPos() != TRawStr::c4StrLen(szTestText))
    {
        strmOut << _CurLn_ << L"Stream out of text did not adjust position as "
                              L"expected." << NewLn;
        return;
    }

    //
    //  Now create a string object and stream the text back into it. It should
    //  be the same text as what we streamed out.
    //
    strmToTest.Reset();
    if (strmToTest.fposCurPos() != 0)
    {
        strmOut << _CurLn_ << L"Reset did not result in position of zero"
                << NewLn;
        return;
    }

    TString strTmp;
    strmToTest.c4GetLine(strTmp);
    if (strTmp != szTestText)
    {
        strmOut << _CurLn_ << L"Streamed in text != to original text" << NewLn;
        return;
    }

    // This should have put the position back where it was before
    if (strmToTest.fposCurPos() != TRawStr::c4StrLen(szTestText))
    {
        strmOut << _CurLn_ << L"Stream in of text did not adjust position as "
                              L"expected." << NewLn;
        return;
    }

    //
    //  The logical end should not also be equal to the current position.
    //
    if (strmToTest.fposLogicalEnd() != strmToTest.fposCurPos())
    {
        strmOut << _CurLn_ << L"Logical end != current position" << NewLn;
        return;
    }

    // Truncate the stream
    strmToTest.TruncateAtZero();

    // Now both the logical end and current position should be zero.
    if (strmToTest.fposLogicalEnd() != 0)
    {
        strmOut << _CurLn_ << L"Truncate did not result in logical end of zero"
                << NewLn;
        return;
    }

    if (strmToTest.fposCurPos() != 0)
    {
        strmOut << _CurLn_ << L"Truncate did not result in current pos of zero"
                << NewLn;
        return;
    }

    //
    //  Now create another text buffer that has multiple lines in it so that
    //  we can test some line reading stuff.
    //
    static tCIDLib::Tch* szTestTextLines = 
    {
        L"This is line #1\r\n"
        L"This is line #2\r\n"
        L"1892.9821\r\n"
        L"10\r\n"
        L"False\r\n"
    };

    strmToTest << TString(szTestTextLines);
    if (strmToTest.fposCurPos() != TRawStr::c4StrLen(szTestTextLines))
    {
        strmOut << _CurLn_ << L"Stream out of text did not adjust position as "
                              L"expected." << NewLn;
        return;
    }

    // Reset the stream and read it back now
    strmToTest.Reset();

    //
    //  Get the first two lines in, and compare them to what we think that
    //  they should be.
    //
    strmToTest.c4GetLine(strTmp);
    if (strTmp != L"This is line #1")
    {
        strmOut << _CurLn_ << L"Streamed in text != to original text" << NewLn;
        return;
    }

    strmToTest.c4GetLine(strTmp);
    if (strTmp != L"This is line #2")
    {
        strmOut << _CurLn_ << L"Streamed in text != to original text" << NewLn;
        return;
    }

    tCIDLib::TBoolean   bTmp;
    tCIDLib::TFloat8    f8Tmp;
    tCIDLib::TInt4      i4Tmp;

    strmToTest >> f8Tmp;
    if (f8Tmp != 1892.9821)
    {
        strmOut << _CurLn_ << L"Streamed in value was: " << f8Tmp
                << L" but should have been 1892.9821" << NewLn;
        return;
    }

    strmToTest >> i4Tmp;
    if (i4Tmp != 10)
    {
        strmOut << _CurLn_ << L"Streamed in value was: " << i4Tmp
                << L" but should have been 10" << NewLn;
        return;
    }

    strmToTest >> bTmp;
    if (bTmp)
    {
        strmOut << _CurLn_ << L"Streamed in value was != False" << NewLn;
        return;
    }
}


static tCIDLib::TVoid __BasicStreamTests(TTextStream& strmOut)
{
    //
    //  We create text streams with each of the available implementations
    //  and then pass it to a common testing function. These should all
    //  create the same results.
    //
    {
        TTextFileStream strmToTest
        (
            L"TestTextFileStream.Dat"
            , tCIDLib::ETextFmt_UNICode
            , tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );

        //
        //  The physical end is the max file size divided by the size of
        //  character, which is kCIDLib::c4CharBytes for a UNICode stream,
        //  which this one is.
        //
        __CommonTextTests
        (
            strmOut
            , strmToTest
            , 0
            , kCIDLib::i8MaxInt / kCIDLib::c4CharBytes
        );
    }

    // Now try a memory buffer based text stream
    {
        TTextMBufStream strmToTest
        (
            4096
            , tCIDLib::ETextFmt_UNICode
            , 8192
            , tCIDLib::EMemAcc_ReadWrite
        );

        __CommonTextTests(strmOut, strmToTest, 0, 8192 / kCIDLib::c4CharBytes);
    }

    // And try a string based text stream
    {
        TStringStream strmToTest(0, 1024);
        __CommonTextTests(strmOut, strmToTest, 0, 1024);
    }
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestTextStreams
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various
//  aspects of the text stream classes.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestTextStreams()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        c4CurCount = facCIDLib.c4ObjectCount();

        pszCurTest = L"basic";
        __BasicStreamTests(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Text stream test leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" text streams test" << NewLn;
        throw;
    }
}
