//
// NAME: TestCIDLib_String.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program. This module is called
//  from the program's main function. The functions in this module test the
//  string classes to make sure that they are functioning correctly.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/05/92
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Functions for local use
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __TestBufferGrowth
//
// DESCRIPTION:
//
//  This function will test the string buffer growth system.
// ---------------------------------------
//   INPUT: strmOut is the stream to output messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestBufferGrowth(TTextStream& strmOut)
{
    TString strTmp(kCIDLib::pszEmptyZStr, 16, 32);

    if (strTmp.c4BufChars() != 16)
    {
        strmOut << _CurLn_ << L"Buffer size not initially 16" << NewLn;
        return;
    }

    if (strTmp.c4MaxChars() != 32)
    {
        strmOut << _CurLn_ << L"Max buffer size not initially 32" << NewLn;
        return;
    }

    // Append more than can fit
    strTmp.Append(L"1234567890123456789012345678901234");

    // Make sure it clipped correctly
    if (strTmp != L"12345678901234567890123456789012")
    {
        strmOut << _CurLn_
                << L"Appended to string not equal to expected value" << NewLn;
        return;
    }

    // Make sure the buffer expanded to 32
    if (strTmp.c4BufChars() != 32)
    {
        strmOut << _CurLn_ << L"String buffer did not grow to 32 bytes" << NewLn;
        return;
    }

    // Make sure the current length is 32
    if (strTmp.c4Length() != 32)
    {
        strmOut << _CurLn_ << L"String buffer length was not 32 bytes << NewLn";
        return;
    }

    //
    //  make the string so that does not tolerate overflows and cause
    //  it to overflow to make sure it works. We left it full above,
    //  so any append should do it.
    //
    tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
    strTmp.bErrOnOverflow(kCIDLib::True);
    try
    {
        strTmp.Append(L"Overflow");
    }

    catch(...)
    {
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
    {
        strmOut << _CurLn_ << L"Did not catch an overflow of the buffer"
                << NewLn;
        return;
    }
}


//
// FUNCTION/METHOD NAME: __TestClipping
//
// DESCRIPTION:
//
//  This function will test some string clipping functionality. We make sure
//  that putting too large values into strings and make sure that they get
//  clipped.
// ---------------------------------------
//   INPUT: strmOut is the stream to send messages to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestClipping(TTextStream& strmOut)
{
    // Create a big string that will be used for the initializations.
    tCIDLib::Tch    szBig[301];

    szBig[0] = 0;
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 30; c4Ind++)
        TRawStr::CatStr(szBig, L"0123456789", c4MaxBufChars(szBig));
    szBig[300] = 0;

    //
    //  Test initial value clipping by creating some strings with values that
    //  are too long.
    //
    TString  str8Test(szBig, 8, 8);
    TString  str16Test(szBig, 16, 16);
    TString  str32Test(szBig, 32, 32);
    TString  str64Test(szBig, 64, 64);
    TString  str128Test(szBig, 128, 128);
    TString  str256Test(szBig, 256, 256);

    //
    //  Test clipping of initial values. We init the strings with values that
    //  are too long and then check the value, which should be clipped to the
    //  maximum number of chars that will fit.
    //
    szBig[256] = 0;
    if (str256Test != szBig)
    {
        strmOut << _CurLn_
                << L"String constructor did not clip initial valuer" << NewLn;
    }

    if ((str256Test.c4Length() != 256)
    ||  (str256Test.c4BufChars() != 256))
    {
        strmOut << _CurLn_
                << L"String length " << str256Test.c4Length()
                << L" and buf size " << str256Test.c4BufChars()
                << L" should be 256" << NewLn;
        return;
    }

    szBig[128] = 0;
    if (str128Test != szBig)
    {
        strmOut << _CurLn_
                << L"String constructor did not clip initial value" << NewLn;
    }

    szBig[64] = 0;
    if (str64Test != szBig)
    {
        strmOut << _CurLn_
                << L"String constructor did not clip initial value" << NewLn;
    }

    szBig[32] = 0;
    if (str32Test != szBig)
    {
        strmOut << _CurLn_
                << L"String constructor did not clip initial value" << NewLn;
    }

    szBig[16] = 0;
    if (str16Test != szBig)
    {
        strmOut << _CurLn_
                << L"String constructor did not clip initial value" << NewLn;
    }

    szBig[8] = 0;
    if (str8Test != szBig)
    {
        strmOut << _CurLn_
                << L"String constructor did not clip initial value" << NewLn;
    }

    //
    // Reset them all again and do the same thing via an assignment
    //
    szBig[0] = 0;
    for (c4Ind = 0; c4Ind < 30; c4Ind++)
        TRawStr::CatStr(szBig, L"0123456789", c4MaxBufChars(szBig));
    szBig[300] = 0;

    str8Test.Clear();
    str16Test.Clear();
    str32Test.Clear();
    str64Test.Clear();
    str128Test.Clear();
    str256Test.Clear();

    //
    //  After each assignment, we chop down szBig to make it be the 
    //  length that the assigned to string will be after clipping.
    //
    str256Test = szBig;
    szBig[256] = 0;
    if (str256Test.c4Length() != 256)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;
    if (str256Test != szBig)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;

    str128Test = szBig;
    szBig[128] = 0;
    if (str128Test.c4Length() != 128)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;
    if (str128Test != szBig)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;

    str64Test = szBig;
    szBig[64] = 0;
    if (str64Test.c4Length() != 64)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;
    if (str64Test != szBig)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;

    str32Test = szBig;
    szBig[32] = 0;
    if (str32Test.c4Length() != 32)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;
    if (str32Test != szBig)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;

    str16Test = szBig;
    szBig[16] = 0;
    if (str16Test.c4Length() != 16)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;
    if (str16Test != szBig)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;

    str8Test = szBig;
    szBig[8] = 0;
    if (str8Test.c4Length() != 8)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;
    if (str8Test != szBig)
        strmOut << _CurLn_<< L"Assignment failed" << NewLn;

}


//
// FUNCTION/METHOD NAME: __TestConstructors
//
// DESCRIPTION:
//
//  This function will test some string constructor scenarios.
// ---------------------------------------
//   INPUT: strmOut is the stream to send messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestConstructors(TTextStream& strmOut)
{
    TString     strCopyTest(L"12345678", 8, 8);
    TString     strTest(strCopyTest);

    // Make sure that the buffers are not the same address
    if (strCopyTest.pszData() == strTest.pszData())
        strmOut << _CurLn_
                << L"Copy constructor from TString(8) reused buffer" << NewLn;

    if (strTest != strCopyTest)
        strmOut << _CurLn_<< L"Copy constructor from TString(8) failed" << NewLn;

    if (strTest.c4BufChars() != 8)
        strmOut << _CurLn_<< L"Copy from TString(8) set wrong buf size" << NewLn;

    if (strTest.c4MaxChars() != 8)
        strmOut << _CurLn_<< L"Copy from TString(8) set wrong max size" << NewLn;

    //
    //  Load up a couple of msg file based strings. In this case there are
    //  no replacement parameters, so we just provide the resource id and
    //  take the default values for the other parameters.
    //
    TString  strTest1(kTestMsgs::midTest1, facTestCIDLib);
    if (strTest1 != L"This is test string 1")
        strmOut << _CurLn_<< L"Msg file string 1 was not correct" << NewLn;

    TString  strTest2(kTestMsgs::midTest2, facTestCIDLib);
    if (strTest2 != L"This is test string 2")
        strmOut << _CurLn_<< L"Msg file string 2 was not correct" << NewLn;

    //
    //  Load up a resource based string with replacement parameters and
    //  replace them. We have to tell it how much extra room we need,
    //  so give exactly what we need, as a test. If there's a problem,
    //  the string should clip and fail the comparision.
    //
    TString  strTest3
    (
        kTestMsgs::midTest3
        , facTestCIDLib
        , 4
        , TString(L"test")
        , TCardinal(3)
    );
    if (strTest3 != L"This is test string 3")
        strmOut << _CurLn_<< L"Msg file string 3 was not correct" << NewLn;

    //
    //  Test the constructor that supports the concatenation operator. It
    //  takes two raw strings and creates a string that holds the two
    //  strings concatenated together.
    //
    TString strConcat(L"This is ", L"a test string");
    if (strConcat != L"This is a test string")
    {
        strmOut << _CurLn_ << L"Concatenation constructor gave incorrect value"
                << NewLn;
    }

    // Test the constructor from a formattable object
    TString strFormatted(TArea(10,10,100,100));
    if (strFormatted != L"10,10-100,100")
    {
        strmOut << _CurLn_ << L"Construct from formattable object failed"
                << NewLn;
    }
}


//
// FUNCTION/METHOD NAME: __TestCutNPaste
//
// DESCRIPTION:
//
//  This function will test some cut/paste scenarios. It inserts stuff into
//  and cuts stuff out of strings and compares them to the expected result.
// ---------------------------------------
//   INPUT: strmOut is the stream to send message to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestCutNPaste(TTextStream& strmOut)
{
    TString   strCutPaste(L"0123456789ABCDEF", 16, 16);

    strCutPaste.Cut(0, 1);
    strCutPaste.Cut(14, 1);
    if (strCutPaste != L"123456789ABCDE")
        strmOut << _CurLn_<< L"Cut on end failed" << NewLn;

    strCutPaste.Cut(4, 4);
    if (strCutPaste != L"12349ABCDE")
        strmOut << _CurLn_<< L"Cut in middle failed" << NewLn;

    strCutPaste.Insert(L"5678", 4);
    if (strCutPaste != L"123456789ABCDE")
        strmOut << _CurLn_<< L"Insert into middle failed" << NewLn;

    strCutPaste.Prepend(L"0");
    if (strCutPaste != L"0123456789ABCDE")
        strmOut << _CurLn_<< L"Prepend failed" << NewLn;

    strCutPaste.Insert(L"F-", 15);
    if (strCutPaste != L"0123456789ABCDEF")
        strmOut << _CurLn_<< L"Insert at end failed" << NewLn;
}


//
// FUNCTION/METHOD NAME: __TestFormatting
//
// DESCRIPTION:
//
//  This function will test some string formatting functionality. Some of
//  it is done via the string directly, and some by creating a string stream.
//  That part is really testing the formatting of the stream, but it still
//  lets us test for the string stream misusing the string, or a hole in the
//  string's API letting the string stream do something it should not.
// ---------------------------------------
//   INPUT: strmOut is the stream to format our messages to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestFormatting(TTextStream& strmOut)
{
    TString str1;
    TString str2;

    // Format some fundamental data types into strings
    str1 = L"This is a test ";
    str2 = L"of the emergency";
    str1.Append(str2);
    str1.Append(L" broadcasting system");
    if (str1 != L"This is a test of the emergency broadcasting system")
        strmOut << _CurLn_<< L"String into string insert failed" << NewLn;

    str1.Clear();
    str2 = L"234567";
    str1.FormatToFld(str2, 8, tCIDLib::EHJustify_Center);
    if (str1 != L" 234567 ")
        strmOut << _CurLn_<< L"Format to field failed" << NewLn;

    //
    //  Create a string stream to do the formatting part of the test. We
    //  don't let it create the string. We create it ourself and give it
    //  to him and tell him not to adopt it (actually that's the default,
    //  so we don't have to say anything.)
    //
    TString strToStream(kCIDLib::pszEmptyZStr, 64, 128);
    TStringStream   strmTest(&strToStream);

    strmTest << 123456UL;
    if (strToStream != L"123456")
        strmOut << _CurLn_<< L"Raw 32 bit value format failed" << NewLn;

    strmTest.Reset();
    strmTest << TCardinal(0xFACE, tCIDLib::ERadix_Hex);
    if (strToStream != L"FACE")
        strmOut << _CurLn_<< L"Hex TCardinal object format failed" << NewLn;

    strmTest.Reset();
    strmTest << TInteger(-1024, tCIDLib::ERadix_Dec);
    if (strToStream != L"-1024")
        strmOut << _CurLn_<< L"Negative integer format failed" << NewLn;

    strmTest.Reset();
    strmTest << TCardinal(0xFF, tCIDLib::ERadix_Bin);
    if (strToStream != L"11111111")
        strmOut << _CurLn_<< L"Binary cardinal format failed" << NewLn;

    strmTest.Reset();
    strmTest << 999.999;
    if (strToStream != L"999.99")
        strmOut << _CurLn_<< L"Raw float format failed" << NewLn;

    strmTest.Reset();
    strmTest << -999.999;
    if (strToStream != L"-999.99")
        strmOut << _CurLn_<< L"Raw negative float format failed" << NewLn;

    //
    //  This one will just be '0' since the default trailing format for
    //  raw floats is to ignore trailing zeros.
    //
    strmTest.Reset();
    strmTest << 0.000001;
    if (strToStream != L"0")
        strmOut << _CurLn_<< L"Raw very small float format failed" << NewLn;

    strmTest.Reset();
    strmTest << TFloat(11111.11111, 4);
    if (strToStream != L"11111.1111")
        strmOut << _CurLn_<< L"Float object format failed" << NewLn;

    strmTest.Reset();
    strmTest << TFloat(10.0, 0);
    if (strToStream != L"10")
        strmOut << _CurLn_<< L"Float object format failed" << NewLn;

    strmTest.Reset();
    strmTest << TFloat(10.0);
    if (strToStream != L"10.00")
        strmOut << _CurLn_<< L"Float object format failed" << NewLn;

    strmTest.Reset();
    strmTest << TFloat(0.0);
    if (strToStream != L"0.00")
        strmOut << _CurLn_<< L"Float object format failed" << NewLn;

    strmTest.Reset();
    strmTest << TFloat(0.0, 6);
    if (strToStream != L"0.000000")
        strmOut << _CurLn_<< L"Float object format failed" << NewLn;

    strmTest.Reset();
    strmTest << TFloat(12345.01234567, 7);
    if (strToStream != L"12345.0123456")
        strmOut << _CurLn_<< L"Float object format failed" << NewLn;

    // Test the lower/upper casing
    strmTest.Reset();
    strmTest << L"this is a lower case string";
    strToStream.ToUpper();
    if (TRawStr::eCompareStr
    (
        strToStream.pszData()
        , L"THIS IS A LOWER CASE STRING"))
    {
        strmOut << _CurLn_<< L"ToUpper() failed" << NewLn;
    }

    strmTest.Reset();
    strmTest << L"NOW DO THE OPPOSITE OPERATION AND LOWER CASE IT";
    strToStream.ToLower();
    if (TRawStr::eCompareStr
    (
        strToStream.pszData()
        , L"now do the opposite operation and lower case it"))
    {
        strmOut << _CurLn_<< L"ToLower() failed" << NewLn;
    }

    //
    //  For variety we want to create a string that is full and do some
    //  stuff on it, to check for going over the end.
    //
    TString strCase(L"abcdefghijklmnop", 16, 16);
    strCase.ToUpper(2,1);
    if (TRawStr::eCompareStr(strCase.pszData(), L"abCdefghijklmnop"))
        strmOut << _CurLn_<< L"ToUpper() failed on char 2 of string" << NewLn;

    strCase.ToUpper(15,1);
    if (TRawStr::eCompareStr(strCase.pszData(), L"abCdefghijklmnoP"))
        strmOut << _CurLn_<< L"ToUpper() failed on last char" << NewLn;

    strCase.ToLower();
    if (TRawStr::eCompareStr(strCase.pszData(), L"abcdefghijklmnop"))
        strmOut << _CurLn_<< L"ToLower() failed on full string" << NewLn;
}


//
// FUNCTION/METHOD NAME: __TestOperators
//
// DESCRIPTION:
//
//  This function will test the various string operators
// ---------------------------------------
//   INPUT: strmOut is the stream to output messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestOperators(TTextStream& strmOut)
{
    //
    //  Create one with an initial size of 0, which will force it to
    //  use the intial value string to create the initial size.
    //
    TString strTest(L"1234567890", 0, 92);

    //
    //  The buffer must be at least 10 chars since that was our initial
    //  value we passed in.
    //
    if (strTest.c4BufChars() < 10)
    {
        strmOut << _CurLn_
                << L"The init string was 10 chars long but the buffer "
                   L"size is only " << strTest.c4BufChars() << NewLn;
        return;
    }

    if (strTest.c4MaxChars() != 92)
        strmOut << _CurLn_<< L"The initial string size is not 92" << NewLn;

    //
    //  Test the indexing operator and make sure it catches errors in
    //  indexing.
    //
    if (strTest[0] != L'1')
    {
        strmOut << _CurLn_<< L"The character at position 0 was "
                << strTest[0] << L" not 1" << NewLn;
    }

    if (strTest[5] != L'6')
    {
        strmOut << _CurLn_<< L"The character at position 5 was "
                << strTest[5] << L" not 6" << NewLn;
    }

    if (strTest != L"1234567890")
    {
        strmOut << _CurLn_
                << L"The == operator for string to raw string failed" << NewLn;
    }

    //
    //  Test the concatenation operator out. It creates two strings from a
    //  single string.
    //
    TString strConcat(TString(L"This is a") + TString(L" test string value"));
    if (strConcat != L"This is a test string value")
    {
        strmOut << _CurLn_
                << L"Concatenation operator failed to create correct result"
                << NewLn;
    }
}


//
// FUNCTION/METHOD NAME: __TestPatternSearch
//
// DESCRIPTION:
//
//  This function will test the various character and substring search
//  functions.
// ---------------------------------------
//   INPUT: strmOut is the stream to output messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestPatternSearch(TTextStream& strmOut)
{
    //
    //  Create a string in which we can search for various things
    //
    //                        01234567890123456789012345
    TString         strTest(L"0123456789 ABC DEF ABC 123");
    tCIDLib::TCard4 c4Index;

    //
    //  Do some character seaching first. Lets find all of the '1' chars
    //  in the string.
    //
    if (!strTest.bFirstOccurrence(kCIDLib::chDigit1, c4Index))
    {
        strmOut << _CurLn_ << L"Did not find first occurange of '1'" << NewLn;
        return;
    }

    // It should have been at index 1
    if (c4Index != 1)
    {
        strmOut << _CurLn_ << L"First occurrence of '1' was not at 1st index"
                << NewLn;
        return;
    }

    // Find the next occurrence
    if (!strTest.bNextOccurrence(kCIDLib::chDigit1, ++c4Index))
    {
        strmOut << _CurLn_ << L"Did not find next occurange of '1'" << NewLn;
        return;
    }

    // Should have been at index 23
    if (c4Index != 23)
    {
        strmOut << _CurLn_ << L"Next occurrence of '1' was not at 23rd index"
                << NewLn;
        return;
    }

    // Now find the last occurence of 2
    if (!strTest.bLastOccurrence(kCIDLib::chDigit2, c4Index))
    {
        strmOut << _CurLn_ << L"Did not find first occurrence of '2'" << NewLn;
        return;
    }

    // Should have been at index 24
    if (c4Index != 24)
    {
        strmOut << _CurLn_ << L"Last occurrence of '2' was not at 24rd index"
                << NewLn;
        return;
    }

    // And find the previous instance
    if (!strTest.bPrevOccurrence(kCIDLib::chDigit2, --c4Index))
    {
        strmOut << _CurLn_ << L"Did not find previous occurrence of '2'"
                << NewLn;
        return;
    }

    // Should have been at index 2
    if (c4Index != 2)
    {
        strmOut << _CurLn_ << L"Previous occurrence of '2' was not at 2nd index"
                << NewLn;
        return;
    }


    //
    //  Now do some substring searchs like the character searchs above.
    //  Start by finding the first occurrence of "ABC".
    //
    if (!strTest.bFirstOccurrence(L"ABC", c4Index))
    {
        strmOut << _CurLn_ << L"Did not find first occurrence of 'ABC'"
                << NewLn;
        return;
    }

    // It should have been at 11
    if (c4Index != 11)
    {
        strmOut << _CurLn_ << L"First occurrence of 'ABC' was not at 11th index"
                << NewLn;
        return;
    }

    if (!strTest.bNextOccurrence(L"ABC", ++c4Index))
    {
        strmOut << _CurLn_ << L"Did not find next occurrence of 'ABC'" << NewLn;
        return;
    }

    // It should have been at 19
    if (c4Index != 19)
    {
        strmOut << _CurLn_ << L"Next occurrence of 'ABC' was not at 29th index"
                << NewLn;
        return;
    }

    if (!strTest.bLastOccurrence(L"123", c4Index))
    {
        strmOut << _CurLn_ << L"Did not find last occurrence of '123'" << NewLn;
        return;
    }

    // It should have been at 23
    if (c4Index != 23)
    {
        strmOut << _CurLn_ << L"Last occurrence of '123' was not at 23rd index"
                << NewLn;
        return;
    }

    if (!strTest.bPrevOccurrence(L"123", --c4Index))
    {
        strmOut << _CurLn_ << L"Did not find previous occurrence of '123'"
                << NewLn;
        return;
    }

    // It should have been at 1
    if (c4Index != 1)
    {
        strmOut << _CurLn_ << L"Previous occurrence of '123' was not at 1st index"
                << NewLn;
        return;
    }


    //
    //  Now do some tests to insure that we catch invalid index attempts.
    //
    tCIDLib::TBoolean   bCaughtIt;

    try
    {
        bCaughtIt = kCIDLib::False;
        c4Index = 50;
        strTest.bNextOccurrence(kCIDLib::chDigit4, c4Index);
    }

    catch(const TError&)
    {
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
    {
        strmOut << _CurLn_ << L"Invalid next occurrence index not caught"
                << NewLn;
        return;
    }

    try
    {
        bCaughtIt = kCIDLib::False;
        c4Index = 50;
        strTest.bPrevOccurrence(kCIDLib::chDigit4, c4Index);
    }

    catch(const TError&)
    {
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
    {
        strmOut << _CurLn_ << L"Invalid previous occurrence index not caught"
                << NewLn;
        return;
    }

}


//
// FUNCTION/METHOD NAME: __TestStripping
//
// DESCRIPTION:
//
//  This function will test the stripping capabilities of strings. We strip
//  certain preformatted strings and then compare them against the respected
//  result.
// ---------------------------------------
//   INPUT: strmOut is the stream to output messages to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestStripping(TTextStream& strmOut)
{
    const tCIDLib::Tch* pszWhtSpace  = L" 1  2  3  \t4  5  6  ";

    TString str1(pszWhtSpace);

    // First do the white space strip
    str1.StripWhitespace();

    if (str1 != L"1  2  3  \t4  5  6")
        strmOut << _CurLn_<< L"Leading/Trailing whitespace strip failed" << NewLn;

    //
    //  Strip out all middle white space, starting with the previous
    //  string which has no leading/trailing white space (to test the
    //  boundary conditions.)
    //
    str1.StripWhitespace
    (
        tCIDLib::EStripModes
        (
            tCIDLib::EStripMode_Total
            | tCIDLib::EStripMode_Leading
            | tCIDLib::EStripMode_Trailing
            | tCIDLib::EStripMode_Middle
        )
    );

    if (str1 != L"123456")
        strmOut << _CurLn_<< L"Total whitespace strip failed" << NewLn;


    // Do it again, starting with the original string
    str1 = pszWhtSpace;
    str1.StripWhitespace
    (
        tCIDLib::EStripModes
        (
            tCIDLib::EStripMode_Total
            | tCIDLib::EStripMode_Leading
            | tCIDLib::EStripMode_Trailing
            | tCIDLib::EStripMode_Middle
        )
    );
    if (str1 != L"123456")
        strmOut << _CurLn_<< L"Total whitespace strip failed" << NewLn;

    // Test small strings, which are handled specially inside the stripper
    str1 = L" ";
    str1.StripWhitespace(tCIDLib::EStripMode_Leading);
    if (str1 != kCIDLib::pszEmptyZStr)
        strmOut << _CurLn_<< L"Strip of single space failed" << NewLn;

    str1 = L"  \n";
    str1.StripWhitespace(tCIDLib::EStripMode_Trailing);
    if (str1 != kCIDLib::pszEmptyZStr)
        strmOut << _CurLn_<< L"Strip of single space failed" << NewLn;

    str1 = L"1 ";
    str1.StripWhitespace(tCIDLib::EStripMode_Leading);
    if (str1 != L"1 ")
        strmOut << _CurLn_<< L"Strip of single space failed" << NewLn;

    //
    //  Set up a string, strip certain characters out of it, and test the result.
    //
    str1 = L"aabbaaccaaddaaeeaaffaagg";
    str1.Strip
    (
        L"a"
        , tCIDLib::EStripModes
          (
            tCIDLib::EStripMode_Total
            | tCIDLib::EStripMode_Leading
            | tCIDLib::EStripMode_Trailing
            | tCIDLib::EStripMode_Middle
          )
    );
    if (str1 != L"bbccddeeffgg")
        strmOut << _CurLn_<< L"Total strip failed" << NewLn;
}


//
// FUNCTION/METHOD NAME: __TestTokens()
//
// DESCRIPTION:
//
//  This function will test some token replacement scenarios. It will set up
//  some canned strings, replace the tokens in them, and compare the
//  resulting strings against the expected results.
// ---------------------------------------
//   INPUT: strmOut is the stream to output messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestTokens(TTextStream& strmOut)
{
    TCardinal   cTmp1(0UL), cTmp2(0UL);
    TInteger    iTmp1(0L), iTmp2(0L);
    TString     str1(kCIDLib::pszEmptyZStr, 64, 64);
    TString     str2(kCIDLib::pszEmptyZStr, 128, 128);

    // Check a replacement of a token with an asciiz string
    str1 = L"This is a test of the %(a) system";
    str1.ReplaceToken(L"token replacement", L'a');
    if (str1 != L"This is a test of the token replacement system")
        strmOut << _CurLn_<< L"Asciiz token replacement failed" << NewLn;

    // Check a replacement of a token with some string objects
    str1 = L"This is a %(1) test of the %(2) system";
    str2 = L"darned";
    str1.ReplaceToken(str2, L'1');
    str2 = L"emergency broadcasting";
    str1.ReplaceToken(str2, L'2');
    if (str1 != L"This is a darned test of the emergency broadcasting system")
        strmOut << _CurLn_<< L"String object token replacement failed" << NewLn;

    // Check a replacement of tokens with some numeric objects
    cTmp1 = 128;
    iTmp1 = -1;
    str1 = L"The value %(A) is less than the value %(B)";
    str1.ReplaceToken(iTmp1, L'A');
    str1.ReplaceToken(cTmp1, L'B');
    if (str1 != L"The value -1 is less than the value 128")
        strmOut << _CurLn_<< L"Numeric object token replacement failed" << NewLn;

    str1 = L"The value %(A,-4) is less than the value 0x%(B,-4)";
    str1.ReplaceToken(-1L, L'A');
    str1.ReplaceToken(128UL, L'B', tCIDLib::ERadix_Hex);
    if (str1 != L"The value -1   is less than the value 0x80  ")
        strmOut << _CurLn_<< L"Numeric object token replacement failed" << NewLn;

    str1 = L"The value %(A,4) is less than the value %(B,4)";
    str1.ReplaceToken(-1L, L'A');
    str1.ReplaceToken(128UL, L'B');
    if (str1 != L"The value   -1 is less than the value  128")
        strmOut << _CurLn_<< L"Numeric object token replacement failed" << NewLn;
}



// ----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestStrings
//
// DESCRIPTION:
//
//  This method is one of the testing methods of the facility class. It tests
//  out all of the various capabilities of the TString class. This class is
//  very fundamental so it needs to be tested well.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestStrings()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"constructor";
        __TestConstructors(strmOut());

        pszCurTest = L"buffer growth";
        __TestBufferGrowth(strmOut());

        pszCurTest = L"pattern search";
        __TestPatternSearch(strmOut());

         pszCurTest = L"test opeators";
        __TestOperators(strmOut());

        pszCurTest = L"clipping";
        __TestClipping(strmOut());

        pszCurTest = L"cut/paste";
        __TestCutNPaste(strmOut());

        pszCurTest = L"formatting";
        __TestFormatting(strmOut());

        pszCurTest = L"token";
        __TestTokens(strmOut());

        pszCurTest = L"strip";
        __TestStripping(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"An exception occured during the "
                    << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
