//
// NAME: TestCIDLib_Stacks.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  Stack collection classes.
//
//  This module brings in a common set of collection tests that are implemented
//  as templates themselves so that they can be invoked for all types of
//  collections without redundancy. But some of the testing is always specific
//  to the particular collection type.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_CommonCollect.Hpp"


// -----------------------------------------------------------------------------
//  Typedef some instantiations we want to test
// -----------------------------------------------------------------------------
typedef TStack<TInteger>    TStackOfTInteger;
typedef TStack<TPoint>      TStackOfTPoint;


// -----------------------------------------------------------------------------
//  Force full instantiations of at least one specialization of each class so
//  that we can catch any errors
// -----------------------------------------------------------------------------
template class TStack<TInteger>;


// -----------------------------------------------------------------------------
//  In order to avoid warnings about unimplemented methods, provide dummy
//  implementations here.
// -----------------------------------------------------------------------------
DummyTmplImpls(TStack<TInteger>)



// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------
static tCIDLib::TVoid __TestStackBasics(TTextStream& strmOut)
{
    // Create a stack of points and push on some values
    TStackOfTPoint  colTest(4);

    // Push on some values that are easy to check the order of
    colTest.Push(TPoint(0,0));
    colTest.Push(TPoint(1,1));
    colTest.Push(TPoint(2,2));
    colTest.Push(TPoint(3,3));

    //
    //  Now lets test the overflow of the collection by pushing on one too
    //  many values.
    //
    tCIDLib::TBoolean bGotIt = kCIDLib::False;
    try
    {
        colTest.Push(TPoint(4,4));
    }

    catch(...)
    {
        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Did not catch collection full exception" << NewLn;

    //
    //  Pop the values off and make sure that they are in the correct reverse
    //  order of pushing.
    //
    for (tCIDLib::TInt4 i4Index = 3; i4Index >= 0; i4Index--)
    {
        TPoint pntCur = colTest.objPop();

        if (pntCur != TPoint(i4Index, i4Index))
        {
            strmOut << _CurLn_
                    << L"Stack pops were not reverse of pushes. Expected: "
                    << TPoint(i4Index, i4Index)
                    << L" but got:" << pntCur
                    << NewLn;
            break;
        }
    }
}

static tCIDLib::TVoid __TestStackCommon(TTextStream& strmOut)
{
    // Create a stack of TInteger objects. It can hold up to 16 objects
    TStackOfTInteger colTest(16);

    // Check the automatic class name generation
    if (!colTest.bIsA(TClass(L"TStackOfTInteger")))
    {
        strmOut << _CurLn_ << L"Stack class name was bad. It was: "
                << colTest.clsIsA() << NewLn;
    }

    // Throw in some integer objects
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.Push(TInteger(c4Ind+1));

    //
    //  Do the copy and duplication tests. We provide a standard object
    //  equality object for the element comparison.
    //
    TestColCopy(strmOut, colTest, TStdObjEq<TInteger>());

    //
    //  Invoke the basic collection testing template function. It will do
    //  some generic things that all collections should do the same.
    //
    TestColBasics(strmOut, colTest, 16);

    //
    //  Do it again but this time with only 1 element, to check for any
    //  boundary conditions.
    //
    colTest.Push(TInteger(1));
    TestColBasics(strmOut, colTest, 1);
}



// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestStacks
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various aspects of
//  the Stack collections.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestStacks()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        // Get the current object count
        c4CurCount = facCIDLib.c4ObjectCount();
        
        // Invoke the stack tests
        pszCurTest = L"common";
        __TestStackCommon(strmOut());

        pszCurTest = L"basic";
        __TestStackBasics(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Stack tests leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" stack test" << NewLn;
        throw;
    }
}
