//
// NAME: TestCIDLib_RTTI.Cpp
//
// DESCRIPTION:
//
//  This module implements one of the test methods of the facility class.
//  This particular method excercises the RTTI and dynamic typing systems.
//  These are fundamental to so much stuff that they need to be very well
//  tested automatically.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/04/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"



// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __TestDynamicTyping
//
// DESCRIPTION:
//
//  This method will test out the dynamic typing system, by trying to create
//  some objects of various type by name or via their class objects.
// ---------------------------------------
//   INPUT: strmOut is the stream to output messages to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestDynamicTyping(TTextStream& strmOut)
{
    TPathStr*   ppathTest;
    TRGBClr*    prgbTest;
    TTime*      ptmTest;

    // Create a color object using the low level TClass method
    prgbTest = (TRGBClr*)TClass::pobjMakeNewOfType(L"TRGBClr");
    if (!prgbTest->bIsA(TRGBClr::clsThis))
    {
        strmOut << _CurLn_
                << L"Dynamically created TRGBClr object is not of type"
                   L" TRGBClr (via low level method)" << NewLn;
    }
    delete prgbTest;
    prgbTest = 0;

    //
    //  Now create one via the factory template method, which is more
    //  typesafe and checks for validity.
    //
    ::MakeNewOfType(prgbTest, TString(L"TRGBClr"));
    if (!prgbTest->bIsA(TRGBClr::clsThis))
    {
        strmOut << _CurLn_
                << L"Dynamically created TRGBClr object is not of type"
                   L" TRGBClr (via template factory method)" << NewLn;
    }
    delete prgbTest;
    prgbTest = 0;

    //
    //  Create a TTime object via its class object, which is yet another
    //  way to do it.
    //
    ptmTest = (TTime*)TTime::clsThis.pobjMakeNew();
    if (!ptmTest->bIsA(TTime::clsThis))
    {
        strmOut << _CurLn_
                << L"Dynamically created TTime object is not of type"
                   L" TTime (via class object.)" << NewLn;
    }
    delete ptmTest;

    //
    //  And create another type of object via the factory template
    //  method.
    //
    ::MakeNewOfType(ppathTest, TString(L"TPathStr"));
    if (!ppathTest->bIsA(TPathStr::clsThis))
    {
        strmOut << _CurLn_
                << L"Dynamically created TPathStr object is not of type"
                   L" TPathStr (via template factory method)" << NewLn;
    }
    delete ppathTest;
    ppathTest = 0;


    //
    //  Now purposefully try to dynamically create a color object into the
    //  pointer to the path string. This should cause an exception, so make
    //  sure it happens.
    //
    tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
    try
    {
        ::MakeNewOfType(ppathTest, TString(L"TRGBClr"));
    }

    catch(...)
    {
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
    {
        strmOut <<  _CurLn_
                <<  L"Did not catch a dynamic object creation into wrong"
                    L" type of pointer" << NewLn;
    }
}


//
// FUNCTION/METHOD NAME: __TestRTTI()
//
// DESCRIPTION:
//
//  This function will test the runtime type identification system out to
//  make sure that it is basically working ok.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestRTTI(TTextStream& strmOut)
{
    //
    //  Make sure two class objects constructed with the same class name
    //  compare ok.
    //
    {
        TClass clsTest("TestClassName");
        TClass clsTest2("TestClassName");
        if (clsTest != clsTest2)
        {
            strmOut << _CurLn_
                    << L"Class objects of same class did not compare" << NewLn;
        }
    }

    //
    //  Make sure two class objects constructed with different class names
    //  don't compare ok.
    //
    {
        TClass   clsTest("TestClassName");
        TClass   clsTest2("AnotherClassName");
        if (clsTest == clsTest2)
        {
            strmOut << _CurLn_
                    << L"Class objects of different classes were equal"
                    << NewLn;
        }
    }

    //
    //  Test out some known relationships in the class hierarchy.
    //
    TPathStr    pathTest("A test");
    if (!pathTest.bIsDescendantOf(TString::clsThis))
    {
        strmOut << _CurLn_
                <<  L"TPathStr->TString descendence test failed" << NewLn;
    }
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestRTTI
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various
//  aspects of the RTTI and Dynamic Typing systems.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestRTTI()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"Base RTTI";
        __TestRTTI(strmOut());

        pszCurTest = L"Dynamic Typing";
        __TestDynamicTyping(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
