//
// NAME: TestCIDLib_Queues.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  Queue collection classes.
//
//  This module brings in a common set of collection tests that are implemented
//  as templates themselves so that they can be invoked for all types of
//  collections without redundancy. But some of the testing is always specific
//  to the particular collection type.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_CommonCollect.Hpp"


// -----------------------------------------------------------------------------
//  Typedef some instantiations
// -----------------------------------------------------------------------------
typedef TQueue<TSize>       TQueueOfTSize;
typedef TQueue<TFloat>      TQueueOfTFloat;


// -----------------------------------------------------------------------------
//  Force full instantiations of at least one specialization of each class so
//  that we can catch any errors
// -----------------------------------------------------------------------------
template class TQueue<TSize>;


// -----------------------------------------------------------------------------
//  In order to avoid warnings about unimplemented methods, provide dummy
//  implementations here.
// -----------------------------------------------------------------------------
DummyTmplImpls(TQueue<TSize>)



// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------
static tCIDLib::TVoid __TestQueueCommon(TTextStream& strmOut)
{
    // Create a bag of size objects. It can hold up to 16 objects
    TQueueOfTSize colTest(16);

    // Check the automatic class name generation
    if (!colTest.bIsA(TClass(L"TQueueOfTSize")))
    {
        strmOut << _CurLn_ << L"Queue class name was bad. It was: "
                << colTest.clsIsA() << NewLn;
    }

    // Throw in some size objects
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.Put(TSize(c4Ind+1, c4Ind+2));

    //
    //  Do the copy and duplication tests. We provide a standard object
    //  equality object for the element comparison.
    //
    TestColCopy(strmOut, colTest, TStdObjEq<TSize>());

    //
    //  Invoke the basic collection testing template function. It will do
    //  some generic things that all collections should do the same.
    //
    TestColBasics(strmOut, colTest, 16);

    //
    //  Do it again but this time with only 1 element, to check for any
    //  boundary conditions.
    //
    colTest.Put(TSize(1, 2));
    TestColBasics(strmOut, colTest, 1);

    // Refill the queue and do the second basic test
    for (c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.Put(TSize(c4Ind+1, c4Ind+16));

    TestColBasics2(strmOut, colTest, 16);
}

static tCIDLib::TVoid __TestQueueBasics(TTextStream& strmOut)
{
    // Create a queue of float objects. It can hold up to 8 objects
    TQueueOfTFloat queTest(8);

    // Check the automatic class name generation
    if (!queTest.bIsA(TClass(L"TQueueOfTFloat")))
    {
        strmOut << _CurLn_ << L"Queue class name was bad, it was: "
                << queTest.clsIsA() << NewLn;
    }

    // Put two objects on it
    queTest.Put(1.0);
    queTest.Put(8.0);

    // There should be 2 elements in it
    if (queTest.c4ElemCount() != 2)
        strmOut << _CurLn_
                << L"Queue count was not 2 after putting two objects" << NewLn;

    // Now get them off and check their values
    if (queTest.objGetNext() != 1.0)
        strmOut << _CurLn_ << L"Get next from queue was incorrect" << NewLn;

    if (queTest.objGetNext() != 8.0)
        strmOut << _CurLn_ << L"Get next from queue was incorrect" << NewLn;

    // There should be no elements
    if (queTest.c4ElemCount())
    {
        strmOut << _CurLn_
                << L"Queue was not empty after getting objects out" << NewLn;
    }

    // Test the other way
    if (!queTest.bIsEmpty())
    {
        strmOut << _CurLn_
                << L"Queue's bIsEmpty is eTrue but elem count was 0" << NewLn;
    }
}



// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestQueues
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various aspects of
//  the Queue collections.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestQueues()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        // Get the current object count
        c4CurCount = facCIDLib.c4ObjectCount();
        
        // Invoke the bag tests
        pszCurTest = L"queue";
        __TestQueueCommon(strmOut());

        pszCurTest = L"basic";
        __TestQueueBasics(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Queue tests leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" queue test" << NewLn;
        throw;
    }
}
