//
// NAME: TestCIDLib_Process.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the process
//  control classes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/12/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"



// -----------------------------------------------------------------------------
//  Local data types
//
//  TThreadData
//      This is the data buffer that is passed to each instance of the TThread
//      class. It is used to test control between threads.
//
//  TThreadList
//      This is a structure that we use to create an array of threads.
// -----------------------------------------------------------------------------
struct  TThreadData
{
    tCIDLib::EExitCodes eRetVal;
    tCIDLib::TCard4     c4ThreadNum;
    tCIDLib::TCard4     c4Counter;
};

struct  TThreadList
{
    TThread*            pThread;
    TThreadData*        pBuf;
};



// -----------------------------------------------------------------------------
//  Local function prototypes
// -----------------------------------------------------------------------------
tCIDLib::TUInt __uThreadFunc(TThread&, tCIDLib::TVoid*);

// -----------------------------------------------------------------------------
//  Typedef a bag of process info objects
// -----------------------------------------------------------------------------
typedef TBag<TRegProcessInfo>   TBagOfTRegProcessInfo;



// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eThreadFunc
//
// DESCRIPTION:
//
//  This is the thread function for the test threads. These threads just wake
//  up every 1/2 second and bump up their counters by 1.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread.
//          pData is a pointer to the instance data buffer.
//
//  OUTPUT: None
//
//  RETURN: We return what we were told to by our caller
//
tCIDLib::EExitCodes __eThreadFunc(TThread& thrThis, tCIDLib::TVoid* pData)
{
    // Look at the passed data as a thread data
    TThreadData* const pBuf = (TThreadData*)(pData);

    while (1)
    {
        // Check for a sync request
        thrThis.Sync();

        // If we get an exit signal, then exit
        if (thrThis.bCheckShutdownRequest())
            break;

        // Sleep a little
        thrThis.Sleep(100);

        // Bump up the counter
        pBuf->c4Counter++;
    }

    // Return the return value that we were told to
    return pBuf->eRetVal;
}


//
// FUNCTION/METHOD NAME: __TestBasicThreading
//
// DESCRIPTION:
//
//  This process does a basic threading test in which one thread creates
//  and controls a set of worker threads.
// ---------------------------------------
//   INPUT: strmOut is the stream to send messagee to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestBasicThreading(TTextStream& strmOut)
{
    const tCIDLib::TCard4   c4ThreadCount = 1;
    TThreadList             arThreads[c4ThreadCount];
    TBagOfTRegProcessInfo   bagProcInfo;
    tCIDLib::TCard4         c4Ind;

    //
    //  Check that the process registry worked by displaying all of the
    //  processes in the registry.
    //
    TProcessRegistry    ProcReg;

    // Set the process info line and card
    ProcReg.strInfoLine(L"This is a test of the info line");
    ProcReg.c4InfoCard(123456);

    if (!ProcReg.c4GetRegisteredProcs(bagProcInfo))
    {
        strmOut << _CurLn_ << L"No processes registered" << NewLn;
    }
     else
    {
        //
        //  Iterate the process info bag and output the information
        //  for each process.
        //
        TBagOfTRegProcessInfo::TCursor cursProcs(&bagProcInfo);
        strmOut << L"Processes registered:" << NewLn;
        do
        {
            strmOut << L"  " << cursProcs.objCur().strName() << NewLn;
        }   while (cursProcs.bNext());
    }
    strmOut << L"\n";

    // Create 3 copies of the local test thread
    TStringStream   strmTmp(64);
    for (c4Ind = 0; c4Ind < c4ThreadCount; c4Ind++)
    {
        arThreads[c4Ind].pBuf               = new TThreadData;
        arThreads[c4Ind].pBuf->eRetVal      = tCIDLib::EExitCodes(c4Ind);
        arThreads[c4Ind].pBuf->c4ThreadNum  = c4Ind+1;
        arThreads[c4Ind].pBuf->c4Counter    = 1;

        strmTmp.Reset();
        strmTmp << L"Thread" << c4Ind+1;
        arThreads[c4Ind].pThread = new TThread(strmTmp.strData(), __eThreadFunc);

        if (arThreads[c4Ind].pThread->bIsRunning())
        {
            strmOut << _CurLn_
                    << L"Thread " << c4Ind+1
                    << L" should not be running yet" << NewLn;
        }

        // Start the thread now and pass it its data
        arThreads[c4Ind].pThread->Start(arThreads[c4Ind].pBuf);

        if (!arThreads[c4Ind].pThread->bIsRunning())
        {
            strmOut << _CurLn_
                    << L"Thread " << c4Ind+1 << L" should be running << NewLn";
        }
    }

    // Sleep for a little to let the threads count up some
    TThread::pthrCaller()->Sleep(3 Seconds);

    // Sync up with each thread and check its counter
    for (c4Ind = 0; c4Ind < c4ThreadCount; c4Ind++)
    {
        arThreads[c4Ind].pThread->WaitSync();

        if (arThreads[c4Ind].pBuf->c4Counter == 1)
        {
            strmOut << _CurLn_
                    << L"Thread " << c4Ind+1 << L" is not counting up"
                    << NewLn;
        }

        // Let the thread go again
        arThreads[c4Ind].pThread->Release();
    }

    // Sync up with each thread and tell it to exit
    for (c4Ind = 0; c4Ind < c4ThreadCount; c4Ind++)
        arThreads[c4Ind].pThread->RequestShutdown(5 Seconds);

    // Wait for them to die
    for (c4Ind = 0; c4Ind < c4ThreadCount; c4Ind++)
        arThreads[c4Ind].pThread->eWaitForDeath(5 Seconds);

    //
    //  Test the exit codes. Each thread will set its exit code to the
    //  value we gave it for that purpose.
    //
    // Sync up with each thread and check its counter
    //
    for (c4Ind = 0; c4Ind < c4ThreadCount; c4Ind++)
    {
        if (arThreads[c4Ind].pThread->eTermCode()
                                        != arThreads[c4Ind].pBuf->eRetVal)
        {
            strmOut << L"Thread " << c4Ind+1 << L" had a bad exit code. "
                    << L"Expected:" << arThreads[c4Ind].pBuf->eRetVal
                    << L" but got: " << arThreads[c4Ind].pThread->eTermCode()
                    << NewLn;
        }
    }

    // Destroy the threads and thread data objects
    for (c4Ind = 0; c4Ind < c4ThreadCount; c4Ind++)
    {
        delete arThreads[c4Ind].pThread;
        delete arThreads[c4Ind].pBuf;
    }
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestProcess
//
// DESCRIPTION:
//
//  This method tests out the process related classes.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestProcess()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"basic threading";
        __TestBasicThreading(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
