//
// NAME: TestCIDLib_Pointers.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  pointer wrapper classes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"



// -----------------------------------------------------------------------------
//  CLASS: TTester
//
//  This class is used to test the counted pointer template. It lets us know
//  for sure that it got deleted properly by setting a flag.
// -----------------------------------------------------------------------------
class TTester : public TObject
{
    public  :
        TTester(tCIDLib::TBoolean& bGotDeleted) :

            __bGotDeleted(bGotDeleted)
        {
        }

        ~TTester()
        {
            __bGotDeleted = kCIDLib::True;
        }

        tCIDLib::TBoolean&  __bGotDeleted;
        tCIDLib::TInt4      __i4Field1;
        tCIDLib::TInt4      __i4Field2;

        RTTIMacros(TTester,TObject)
};
RTTIData(TTester,TObject)


// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __TestXXXX
//
// DESCRIPTION:
//
//  These methods all run various pointer wrapper tests.
// ---------------------------------------
//   INPUT: strmOut is the text stream to send messages to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestBasicCountedPointer(TTextStream& strmOut)
{
    // A boolean for the tester to set
    tCIDLib::TBoolean   bGotDeleted = kCIDLib::False;

    // Create a counted pointer for a TTester object
    {
        TCntPtr<TTester> cptrTest(new TTester(bGotDeleted));

        // Check that the class name was set up correctly
        if (TCntPtr<TTester>::clsThis != TClass(L"TCntPtrOfTTester"))
            strmOut << _CurLn_ << L"Class name was not correct" << NewLn;

        // Make sure that we can access the point members via the operators
        cptrTest->__i4Field1 = 1;
        cptrTest->__i4Field2 = 2;
    }

    // Make sure it got deleted
    if (!bGotDeleted)
        strmOut << _CurLn_ << L"Test pointer did not get deleted" << NewLn;

    //
    //  Now do a similar test, but allocate and delete a lot of copies of
    //  the counted pointer object, to make sure that the counting is
    //  working correctly.
    //
    {
        const tCIDLib::TCard4 c4MaxCopies = 63;

        bGotDeleted = kCIDLib::False;
        TCntPtr<TTester> cptrTest(new TTester(bGotDeleted));
        TCntPtr<TTester>* acptrCopies[c4MaxCopies];

        for (tCIDLib::TCard4 c4Index = 0; c4Index < c4MaxCopies; c4Index++)
            acptrCopies[c4Index] = new TCntPtr<TTester>(cptrTest);

        // Should not have been deleted yet
        if (bGotDeleted)
            strmOut << _CurLn_ << L"Test pointer was already deleted" << NewLn;

        // The count should be 64 now
        if (cptrTest.c4RefCount() != 64)
        {
            strmOut << _CurLn_ << L"Ref count was " << cptrTest.c4RefCount()
                    << L" instead of 64" << NewLn;
        }

        // Now delete the copies
        for (c4Index = 0; c4Index < c4MaxCopies; c4Index++)
            delete acptrCopies[c4Index];

        // Still should not be deleted
        if (bGotDeleted)
            strmOut << _CurLn_ << L"Test pointer was already deleted" << NewLn;

        // Count should now be 1 again
        if (cptrTest.c4RefCount() != 1)
        {
            strmOut << _CurLn_ << L"Ref count was " << cptrTest.c4RefCount()
                    << L" instead of 1" << NewLn;
        }
    }

    // Make sure it got deleted
    if (!bGotDeleted)
        strmOut << _CurLn_ << L"Test pointer did not get deleted" << NewLn;
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestPointerClasses
//
// DESCRIPTION:
//
//  This method tests out the pointer wrapper classes.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestPointerClasses()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"basic counted pointer tests";
        __TestBasicCountedPointer(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
