//
// NAME: TestCIDLib_ObjArrays.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  object array collection class.
//
//  This module brings in a common set of collection tests that are implemented
//  as templates themselves so that they can be invoked for all types of
//  collections without redundancy. But some of the testing is always specific
//  to the particular collection type.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_CommonCollect.Hpp"


// -----------------------------------------------------------------------------
//  Force some instantiations
// -----------------------------------------------------------------------------
typedef TObjArray<TRGBClr>      TObjArrayOfTRGBClr;
typedef TObjArray<TCardinal>    TObjArrayOfTCardinal;


// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------

static tCIDLib::TVoid __TestObjArrayBasics(TTextStream& strmOut)
{
    TObjArrayOfTRGBClr  objaTest(16);

    for (tCIDLib::TCard1 c1Ind = 0; c1Ind < 16; c1Ind++)
        objaTest[c1Ind] = TRGBClr(c1Ind, c1Ind, c1Ind);

    for (c1Ind = 0; c1Ind < 16; c1Ind++)
    {
        if (objaTest[c1Ind] != TRGBClr(c1Ind, c1Ind, c1Ind))
        {
            strmOut << _CurLn_
                    << L"Color in object array was not equal set value" << NewLn;
            break;
        }
    }

    // Exchange the first and last element
    objaTest.ExchangeElems(0, 15);

    if (objaTest[0] != TRGBClr(15, 15, 15))
        strmOut << _CurLn_
                << L"Exchanged element of object array was incorrect" << NewLn;

    if (objaTest[15] != TRGBClr(0, 0, 0))
        strmOut << _CurLn_
                << L"Exchanged element of object array was incorrect" << NewLn;

    // Make sure it catches an index error
    tCIDLib::TBoolean  bCaught(kCIDLib::False);
    try
    {
        if (objaTest[16] != TRGBClr(0,0,0))
            objaTest[16] = TRGBClr(0,0,0);
    }

    catch(...)
    {
        bCaught = kCIDLib::True;
    }

    if (!bCaught)
        strmOut << _CurLn_ << L"Index error in object array was not caught" << NewLn;

    // Overwrite element 4
    objaTest[4] = TRGBClr(255,255,255);

    if (objaTest[4] != TRGBClr(255,255,255))
        strmOut << _CurLn_
                << L"Element put into object array did overwrite element" << NewLn;

    // Over another one via the index operator
    objaTest[5] = TRGBClr(128,128,128);

    if (objaTest[5] != TRGBClr(128,128,128))
        strmOut << _CurLn_
                << L"Element put into object array did overwrite element" << NewLn;
}

static tCIDLib::TVoid __TestObjArrayCopyDup(TTextStream& strmOut)
{
    static const tCIDLib::TCard4 c4Count = 64;

    // Create and fill a test array
    TObjArrayOfTCardinal    objaTest(c4Count);
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4Count; c4Ind++)
        objaTest[c4Ind] = c4Ind;

    // Create another array by copy constructor
    TObjArrayOfTCardinal objaCopy(objaTest);

    // Compare the elements
    for (c4Ind = 0; c4Ind < c4Count; c4Ind++)
    {
        if (objaTest[c4Ind] != objaCopy[c4Ind])
        {
            strmOut << _CurLn_ << L"Copy of array elements failed" << NewLn;
            break;
        }
    }

    // Do a comparison of the arrays
    if (objaTest != objaCopy)
    {
        strmOut << _CurLn_ << L"Standard array comparision function failed"
                << NewLn;
    }

    //
    //  Dup the array into another dynamically allocated array and do the
    //  same test.
    //
    TObjArrayOfTCardinal* pobjaNew;
    pobjaNew = ::pDupObject<TObjArrayOfTCardinal>(objaTest);
    TJanitor<TObjArrayOfTCardinal> janNewArray(pobjaNew);

    for (c4Ind = 0; c4Ind < c4Count; c4Ind++)
    {
        if (objaTest[c4Ind] != (*pobjaNew)[c4Ind])
        {
            strmOut << _CurLn_ << L"Dup of array failed" << NewLn;
            break;
        }
    }
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestObjArrays
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various aspects of
//  the object array collections.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestObjArrays()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        // Get the current object count
        c4CurCount = facCIDLib.c4ObjectCount();
        
        pszCurTest = L"basic";
        __TestObjArrayBasics(strmOut());

        pszCurTest = L"copy/dup";
        __TestObjArrayCopyDup(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Object array tests leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" object array test" << NewLn;
        throw;
    }
}
