//
// NAME: TestCIDLib_Numeric.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program. This module is called
//  from the program's main function. The functions in this module test the
//  numeric classes (TCardinal, TFloat, and TInteger.)
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/14/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"


// -----------------------------------------------------------------------------
//  Some templatized testing code that can be applied to all of the numeric
//  classes without change.
// -----------------------------------------------------------------------------
template <class T, class FT> tCIDLib::TVoid __CommonTests(TTextStream& strmOut)
{
    // Default construct an object
    T   objTest1;

    // Make sure it has a value of 0
    if (objTest1 != FT(0))
        strmOut << _CurLn_ << L"Default construction is != 0" << NewLn;

    // Create another one with a value of 1
    T   objTest2(FT(1));

    // Make sure that they have the correct relationship
    if (objTest1 > objTest2)
        strmOut << _CurLn_ << L"Operator> failed" << NewLn;

    if (objTest1 >= objTest2)
        strmOut << _CurLn_ << L"Operator>= failed" << NewLn;

    if (objTest2 < objTest1)
        strmOut << _CurLn_ << L"Operator< failed" << NewLn;

    if (objTest2 <= objTest1)
        strmOut << _CurLn_ << L"Operator<= failed" << NewLn;

    // Copy construct from the second object and compare
    T   objTest3(objTest2);

    if (objTest3 != objTest2)
        strmOut << _CurLn_ << L"Copy constructed object != to original" << NewLn;

    if (!(objTest3 == objTest3))
        strmOut << _CurLn_ << L"Copy constructed object != to original" << NewLn;

    // Assign its related fundamental type
    objTest1 = FT(16);
    if (objTest1 != FT(16))
        strmOut << _CurLn_ << L"Assignment of fundamental failed" << NewLn;

    if (!(objTest1 == FT(16)))
        strmOut << _CurLn_ << L"Assignment of fundamental failed" << NewLn;
}


// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------

static tCIDLib::TVoid __BasicNumericTest(TTextStream& strmOut)
{
    // Do common tests on each basic type
    __CommonTests<TCardinal, tCIDLib::TCard4>(strmOut);
    __CommonTests<TFloat, tCIDLib::TFloat8>(strmOut);
    __CommonTests<TInteger, tCIDLib::TInt4>(strmOut);
    __CommonTests<TMoney, tCIDLib::TFloat8>(strmOut);
}

static tCIDLib::TVoid __ConstructorTests(TTextStream& strmOut)
{
    //
    //  Construct each type via each constructor and insure that the values
    //  are set up correctly. We don't test copy construction because that's
    //  done in the basic tests above.
    //
    TCardinal   cDefault;
    if (cDefault.c4Val() != 0)
        strmOut << _CurLn_ << L"Default cardinal ctor != 0" << NewLn;

    if (cDefault.eRadix() != tCIDLib::ERadix_Dec)
        strmOut << _CurLn_ << L"Default cardinal ctor has wrong radix" << NewLn;

    TCardinal   cExplicit(91, tCIDLib::ERadix_Bin);
    if (cExplicit.c4Val() != 91)
        strmOut << _CurLn_ << L"Explicit cardinal ctor failed" << NewLn;

    if (cExplicit.eRadix() != tCIDLib::ERadix_Bin)
        strmOut << _CurLn_ << L"Explicit cardinal ctor failed" << NewLn;


    TInteger    iDefault;
    if (iDefault.i4Val() != 0)
        strmOut << _CurLn_ << L"Default integer ctor != 0" << NewLn;

    if (iDefault.eRadix() != tCIDLib::ERadix_Dec)
        strmOut << _CurLn_ << L"Default integer ctor has wrong radix" << NewLn;

    TInteger   iExplicit(-91, tCIDLib::ERadix_Bin);
    if (iExplicit.i4Val() != -91)
        strmOut << _CurLn_ << L"Explicit integer ctor failed" << NewLn;

    if (iExplicit.eRadix() != tCIDLib::ERadix_Bin)
        strmOut << _CurLn_ << L"Explicit ingeger ctor failed" << NewLn;


    TFloat  fDefault;
    if (fDefault.f8Val() != 0.0)
        strmOut << _CurLn_ << L"Default float ctor != 0.0" << NewLn;

    if (fDefault.c1Precision() != 2)
        strmOut << _CurLn_ << L"Default float ctor has bad precision" << NewLn;

    if (fDefault.eTrailingStyle() != tCIDLib::ETrail_Zeroes)
        strmOut << _CurLn_ << L"Default float ctor has bad trail style" << NewLn;


}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestNumericClasses
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various aspects of
//  the numeric classes.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestNumericClasses()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        // Get the current object count
        c4CurCount = facCIDLib.c4ObjectCount();

        //
        //  Invoke the tests that we want to do memory checking on. These
        //  only do stuff that doesn't cause spurious leak reports.
        //
        {
            TKrnlMemCheck kmchkTest;

            pszCurTest = L"constructor";
            __ConstructorTests(strmOut());

            pszCurTest = L"basic";
            __BasicNumericTest(strmOut());
        }

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Environment tests leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" numeric test" << NewLn;
        throw;
    }
}
