//
// NAME: TestCIDLib_Memory.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program. This module is called
//  from the program's main function. The functions in this module test the
//  memory classes to make sure that they are functioning correctly.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/05/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_ResourceIds.H"


// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
//  This method handles tests that should cause the same results for any
//  memory buffer derivative, so its called for different types of buffers
//
template <class T> static
tCIDLib::TVoid __CommonTests(           TTextStream&        strmOut
                                ,       T&                  mbufToTest
                                , const tCIDLib::TCard4     c4InitSize
                                , const tCIDLib::TCard4     c4MaxSize
                                , const tCIDLib::TCard4     c4ExpandBy
                                , const tCIDLib::TBoolean   bShared)
{
    if (mbufToTest.c4Size() != c4InitSize)
    {
        strmOut << _CurLn_
                << L"Buffer size (" << mbufToTest.c4Size()
                << L") was not != to initial size (" << c4InitSize
                << L")" << NewLn;
        return;
    }

    T mbufCopy(mbufToTest);
    if (mbufCopy != mbufToTest)
        strmOut << _CurLn_ << L"Copied buffer != original" << NewLn;

    T mbufAssigned = mbufToTest;

    if (mbufAssigned != mbufToTest)
        strmOut << _CurLn_ << L"Assigned buffer != original" << NewLn;

    // And test through the one that just checks the buffer
    if (!mbufAssigned.bBuffersEqual(mbufToTest))
    {
        strmOut << _CurLn_ << L"Assigned buffer != original" << NewLn;
    }

    //
    //  This will fail for a shared buffer because they will refer to the
    //  same actual buffer. So test it differently.
    //
    mbufToTest[0] = 16;
    if (bShared)
    {
        if (mbufToTest != mbufCopy)
        {
            strmOut << _CurLn_ << L"Modified shared buffer was != to original"
            << NewLn;
        }
    }
     else
    {
        if (mbufToTest == mbufCopy)
            strmOut << _CurLn_ << L"Modified buffer was == to original" << NewLn;

        mbufCopy = mbufToTest;
        if (mbufCopy != mbufToTest)
            strmOut << _CurLn_ << L"Assigned buffer was != to original" << NewLn;
    }

    // Probe the boundaries and check for bad range checks
    mbufToTest[0]             = 1;
    mbufToTest[c4InitSize-1]  = 255;

    // See if the elements got set
    if (mbufToTest[0] != 1)
        strmOut << _CurLn_ << L"Elem set via operator[] not set" << NewLn;

    if (mbufToTest[c4InitSize-1] != 255)
        strmOut << _CurLn_ << L"Elem set via operator[] not set" << NewLn;

    //
    //  Make sure that it catches an invalid index. It will get logged
    //  but we will trap it to prevent it showing up as an error in this
    //  test.
    //
    tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
    try
    {
        mbufToTest[c4MaxSize] = 0;
    }

    catch(...)
    {
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
        strmOut << _CurLn_ << L"Failed to catch bad index" << NewLn;

    //
    //  Test the automatic reallocation of the buffer. We try setting the
    //  next byte after the initial size. This should expand it by the
    //  expansion increment. Only do it if the max and init are not the
    //  same.
    //
    if (c4InitSize != c4MaxSize)
    {
        mbufToTest[c4InitSize] = 0xAC;

        if (mbufToTest.c4Size() == c4InitSize)
        {
            strmOut << _CurLn_ << L"Buffer failed to expand" << NewLn;
            return;
        }

        if (mbufToTest.c4Size() != c4InitSize + c4ExpandBy)
        {
            strmOut << _CurLn_
                    << L"Buffer failed to expand by expansion increment" << NewLn;
            return;
        }

        // Now try to set the last byte
        mbufToTest[c4MaxSize-1] = 0xCA;

        if (mbufToTest.c4Size() != c4MaxSize)
        {
            strmOut << _CurLn_ << L"Buffer failed to fully expand" << NewLn;
            return;
        }
    }

    //
    //  Now that the buffer is fully allocated, lets just do some basic
    //  access operations.
    //
    mbufToTest[2] = 0xFF;
    if (mbufToTest.c1At(2) != 0xFF)
    {
        strmOut << _CurLn_ << L"c1At returned wrong results" << NewLn;
        return;
    }

    if (mbufToTest.i1At(2) != -1)
    {
        strmOut << _CurLn_ << L"i1At returned wrong results" << NewLn;
        return;
    }

    mbufToTest.PutCard2(0xCAFE, 2);
    if (mbufToTest.c2At(2) != 0xCAFE)
    {
        strmOut << _CurLn_ << L"c2At returned wrong results" << NewLn;
        return;
    }

    mbufToTest.PutInt2(-5122, 20);
    if (mbufToTest.i2At(20) != -5122)
    {
        strmOut << _CurLn_ << L"i2At returned wrong results" << NewLn;
        return;
    }

    mbufToTest.PutFloat4(4.1F, 16);
    if (mbufToTest.f4At(16) != 4.1F)
    {
        strmOut << _CurLn_ << L"f4At returned wrong results" << NewLn;
        return;
    }

    mbufToTest.PutFloat8(8.1, 16);
    if (mbufToTest.f8At(16) != 8.1)
    {
        strmOut << _CurLn_ << L"f8At returned wrong results" << NewLn;
        return;
    }

    mbufToTest.PutInt4(2554, 18);
    if (mbufToTest.i4At(18) != 2554)
    {
        strmOut << _CurLn_ << L"i4At returned wrong results" << NewLn;
        return;
    }

    //
    //  Now do a fill of the buffer with a particular fill byte and
    //  check that it happened.
    //
    mbufToTest.Set(0xFF);

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4InitSize; c4Ind++)
    {
        if (mbufToTest[c4Ind] != 0xFF)
        {
            strmOut << _CurLn_
                    << L"Element at index " << c4Ind << L" was "
                    << mbufToTest[c4Ind] << NewLn;
        }
    }

    //
    //  And do some other fills that probe the end to see if any bogus
    //  exceptions are triggered. And do one that should and make sure that
    //  it does.
    //
    mbufToTest.Set(0xAF, c4InitSize-10);
    mbufToTest.Set(0xAD, c4MaxSize-15, 15);

    tCIDLib::TBoolean bGotIt = kCIDLib::False;
    try
    {
        mbufToTest.Set(0xAD, c4MaxSize-15, 16);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcMBuf_BufOverflow))
        {
            strmOut << _CurLn_ << L"Incorrect exception caught" << NewLn;
            return;
        }

        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Failed to catch overflow buffer" << NewLn;

    //
    //  Do the same for the checksum mechanism. Make sure it does and does
    //  not cause exceptions as appropriate.
    //
    mbufToTest.c4CheckSum(c4MaxSize - 20);
    mbufToTest.c4CheckSum(c4MaxSize - 20, 20);

    bGotIt = kCIDLib::False;
    try
    {
        mbufToTest.c4CheckSum(c4MaxSize - 20, 21);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcMBuf_BufOverflow))
        {
            strmOut << _CurLn_ << L"Incorrect exception caught" << NewLn;
            return;
        }

        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Failed to catch buffer overflow" << NewLn;


    //
    //  And do the same thing for the hash calculator method.
    //
    mbufToTest.hshCalcHash(17);
    mbufToTest.hshCalcHash(17, c4MaxSize - 20);

    bGotIt = kCIDLib::False;
    try
    {
        mbufToTest.hshCalcHash(17, c4MaxSize);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcMBuf_Index))
        {
            strmOut << _CurLn_ << L"Incorrect exception caught" << NewLn;
            return;
        }

        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Failed to catch index error" << NewLn;

    //
    //  Do some copy in and out operations to probe for errors in the
    //  range and index checking.
    //
    tCIDLib::TCard1 ac1TestBuf[16];

    mbufToTest.CopyOut(ac1TestBuf, 16);
    mbufToTest.CopyOut(ac1TestBuf, 16, c4MaxSize-16);

    bGotIt = kCIDLib::False;
    try
    {
        mbufToTest.CopyOut(ac1TestBuf, 16, c4MaxSize-15);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcMBuf_BufOverflow))
        {
            strmOut << _CurLn_ << L"Incorrect exception caught" << NewLn;
            return;
        }

        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Failed to catch buffer overflow" << NewLn;


    bGotIt = kCIDLib::False;
    try
    {
        mbufToTest.CopyOut(ac1TestBuf, 16, c4MaxSize);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcMBuf_Index))
        {
            strmOut << _CurLn_ << L"Incorrect exception caught" << NewLn;
            return;
        }

        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Failed to catch index error" << NewLn;


    mbufToTest.CopyIn(ac1TestBuf, 16);
    mbufToTest.CopyIn(ac1TestBuf, 16, c4MaxSize-16);

    bGotIt = kCIDLib::False;
    try
    {
        mbufToTest.CopyIn(ac1TestBuf, 16, c4MaxSize-15);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcMBuf_BufOverflow))
        {
            strmOut << _CurLn_ << L"Incorrect exception caught" << NewLn;
            return;
        }

        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Failed to catch buffer overflow" << NewLn;


    bGotIt = kCIDLib::False;
    try
    {
        mbufToTest.CopyIn(ac1TestBuf, 16, c4MaxSize);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcMBuf_Index))
        {
            strmOut << _CurLn_ << L"Incorrect exception caught" << NewLn;
            return;
        }

        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
        strmOut << _CurLn_ << L"Failed to catch index error" << NewLn;
}


static tCIDLib::TVoid __TestCommon(TTextStream& strmOut)
{
    // Create a heap buffer and do the common tests on it
    {
        THeapBuf mbufHeap(1024, 2048, 16);

        strmOut << L"Common tests of heap buffers" << NewLn;
        __CommonTests(strmOut, mbufHeap, 1024, 2048, 16, kCIDLib::False);
    }

    // And a system buffer
    {
        TSysBuf mbufSys(1024, 8190, tCIDLib::EMemAcc_ReadWrite);

        strmOut << L"Common tests of sys buffers" << NewLn;
        __CommonTests
        (
            strmOut
            , mbufSys
            , kCIDLib::c4MemPageSize
            , 8192
            , kCIDLib::c4MemPageSize
            , kCIDLib::False
        );
    }

    // And a shared memory buffer
    {
        TResourceName   rsnTest(L"CIDCorp", L"TestCIDLib", L"TestBuffer");
        TSharedMemBuf   mbufShare
        (
            8190
            , rsnTest
            , tCIDLib::EMemAcc_ReadWrite
        );

        strmOut << L"Common tests of share buffers" << NewLn;
        __CommonTests
        (
            strmOut
            , mbufShare
            , 8192
            , 8192
            , kCIDLib::c4MemPageSize
            , kCIDLib::True
        );
    }
}


//
//  This one tests some stuff that is specific to particular types of
//  buffers, so it cannot be tested in the common tests above.
//
static tCIDLib::TVoid __TestSpecialCases(TTextStream& strmOut)
{
    tCIDLib::TBoolean   bGotIt;

    //
    //  Test the sizes that got set. Note that these guys should always
    //  go in system page size increments, so we check for that though we
    //  pass in 2048.
    //
    TSysBuf mbufSysTest(2048);

    // The max and current size should have been set one page size
    if (mbufSysTest.c4Size() != kCIDLib::c4MemPageSize)
    {
        strmOut << _CurLn_
                << L"Buffer size (" << mbufSysTest.c4Size()
                << L") was not != to page size" << NewLn;
    }

    if (mbufSysTest.c4MaxSize() != kCIDLib::c4MemPageSize)
    {
        strmOut << _CurLn_
                << L"Buffer max size (" << mbufSysTest.c4MaxSize()
                << L") did not default to 1 page" << NewLn;
    }

    // The access flags should have defaults to readwrite
    if (mbufSysTest.eMemAccFlags() != tCIDLib::EMemAcc_ReadWrite)
    {
        strmOut << _CurLn_
                << L"Buffer did not default to read/write access" << NewLn;
    }

    //
    //  Now do a shared memory buffer. The default is to create or open
    //  it, so this should work despite it not already existing.
    //
    TResourceName   rsnTest(L"CIDCorp", L"TestCIDLib", L"TestBuffer");
    TSharedMemBuf   mbufShareTest
    (
        kCIDLib::c4MemPageSize
        , rsnTest
        , tCIDLib::EMemAcc_ReadWrite
    );

    if (mbufShareTest.c4Size() != kCIDLib::c4MemPageSize)
    {
        strmOut << _CurLn_
                << L"Buffer size (" << mbufShareTest.c4Size()
                << L") was not != to constructor size" << NewLn;
    }

    if (mbufShareTest.rsnName() != rsnTest)
    {
        strmOut << _CurLn_ << L"Buffer name was: "
                << mbufShareTest.rsnName() << NewLn;
    }

    if (mbufShareTest.eAccess() != tCIDLib::EMemAcc_ReadWrite)
        strmOut << _CurLn_ << L"Access flag was not 'read/write'" << NewLn;

    //
    //  Test that the create actions work correctly for a shared memory
    //  object. Do one that we know does not exist, and do an 'open if
    //  exists' which should fail.
    //
    bGotIt = kCIDLib::False;
    try
    {
        TResourceName   rsnFail(L"CIDCorp", L"TestCIDLib", L"DontExist");
        TSharedMemBuf   mbufFailTest
        (
            kCIDLib::c4MemPageSize
            , rsnFail
            , tCIDLib::EMemAcc_ReadWrite
            , tCIDLib::ECreateAct_OpenIfExists
        );
    }

    catch(...)
    {
        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
    {
        strmOut << _CurLn_ << L"Named buffer should not exist" << NewLn;
    }

    //
    //  For the shared buffer, we need to test a copy of alternatives.
    //  First we create a totally different one with the same name and
    //  access. It should be equal because of the way that shared buffers
    //  work.
    //
    {
        TSharedMemBuf   mbufShareCopy
        (
            kCIDLib::c4MemPageSize
            , rsnTest
            , tCIDLib::EMemAcc_ReadWrite
            , tCIDLib::ECreateAct_OpenIfExists
        );

        if (mbufShareCopy != mbufShareTest)
            strmOut << _CurLn_ << L"Shared buffer != original" << NewLn;
    }
}


static tCIDLib::TVoid __TestBufferExpansion(TTextStream& strmOut)
{
    tCIDLib::TBoolean   bGotIt;
    TSysBuf             mbufSysTest(0xFF, 0x10000);

    // The initial size should have rounded up to a page
    if (mbufSysTest.c4Size() != kCIDLib::c4MemPageSize)
    {
        strmOut << _CurLn_
                << L"Buffer size (" << mbufSysTest.c4Size()
                << L") was not != to page size" << NewLn;
    }

    // The max size should be 64K, which is a multiple of the page size
    if (mbufSysTest.c4MaxSize() != 0x10000)
    {
        strmOut << _CurLn_
                << L"Buffer max size (" << mbufSysTest.c4MaxSize()
                << L") did not round up to 64K" << NewLn;
    }

    //
    //  Commit another page of memory and check the size to make sure that
    //  its now two pages.
    //
    mbufSysTest.ReallocateTo(8192);

    if (mbufSysTest.c4Size() != kCIDLib::c4MemPageSize * 2)
    {
        strmOut << _CurLn_
                << L"Buffer size (" << mbufSysTest.c4Size()
                << L") was not reallocated to 2 pages" << NewLn;
    }


    //
    //  Now make sure it auto-expands via an index access within the
    //  max size but over the current committed size.
    //
    bGotIt = kCIDLib::False;
    try
    {
        mbufSysTest[8195] = 1;
    }

    catch(...)
    {
        bGotIt = kCIDLib::True;
    }

    if (bGotIt)
    {
        strmOut << _CurLn_
                << L"Index operator access failed to reallocate" << NewLn;
    }

    // The size should now be 3 pages
    if (mbufSysTest.c4Size() != kCIDLib::c4MemPageSize * 3)
    {
        strmOut << _CurLn_
                << L"Buffer size (" << mbufSysTest.c4Size()
                << L") was not reallocated to 3 pages" << NewLn;
    }
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFacTestCIDLib::TestMemoryClasses()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"special cases";
        __TestSpecialCases(strmOut());

        pszCurTest = L"common";
        __TestCommon(strmOut());

        pszCurTest = L"buffer expansion";
        __TestBufferExpansion(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"An exception occured during the "
                    << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
