//
// NAME: TestCIDLib_HashSet.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  hash set collection class.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/01/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_CommonCollect.Hpp"


// -----------------------------------------------------------------------------
//  Typedef our test maps
// -----------------------------------------------------------------------------
typedef THashSet<TString> THashSetOfTString;


// -----------------------------------------------------------------------------
//  Force full instantiations of at least one specialization of each class so
//  that we can catch any errors
// -----------------------------------------------------------------------------
template class THashSet<TString>;


// -----------------------------------------------------------------------------
//  In order to avoid warnings about unimplemented methods, provide dummy
//  implementations here.
// -----------------------------------------------------------------------------
DummyTmplImpls2(THashSet<TString>)



// -----------------------------------------------------------------------------
//  Local constants
// -----------------------------------------------------------------------------
static const tCIDLib::TCard4 __c4HashModulus = 17;
static const tCIDLib::TCard4 __c4MaxElems = 6;


// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

static tCIDLib::TVoid __TestSetBasics(TTextStream& strmOut)
{
    tCIDLib::TBoolean bCaughtIt;

    // Create a hash ste of TString objects
    THashSetOfTString colTest
    (
        __c4HashModulus
        , new TStringKeyOps<TString>
        , __c4MaxElems
    );

    // Add in one element, then try to add it again. It should be rejected
    colTest.Add(TString(L"Test String Value"));

    bCaughtIt = kCIDLib::False;
    try
    {
        colTest.Add(TString(L"Test String Value"));
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcCol_DuplicateElem))
        {
            strmOut <<_CurLn_ << L"Wrong exception thrown for duplicate element"
                    << NewLn;
            return;
        }
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
    {
        strmOut << _CurLn_ << L"Failed to catch duplicate element addition"
                << NewLn;
    }
}

static tCIDLib::TVoid __TestSetCommon(TTextStream& strmOut)
{
    // Create a hash ste of TString objects
    THashSetOfTString colTest
    (
        __c4HashModulus
        , new TStringKeyOps<TString>
        , __c4MaxElems
    );

    // Check the automatic class name generation
    if (!colTest.bIsA(TClass(L"THashSetOfTString")))
    {
        strmOut << _CurLn_ << L"Hash set class name was bad. It was: "
                << colTest.clsIsA() << NewLn;
    }

    //
    //  Add in the elements, creating a unique value for each one, which
    //  is required for a set.
    //
    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4MaxElems; c4Index++)
        colTest.Add(TString(L"Testing") + TString(TCardinal(c4Index)));

    //
    //  Do the copy and duplication tests. We provide a standard object
    //  equality object for the element comparison.
    //
    TestColCopy(strmOut, colTest, TStdObjEq<TString>());

    //
    //  Invoke the basic collection testing template function. It will do
    //  some generic things that all collections should do the same.
    //
    TestColBasics(strmOut, colTest, __c4MaxElems);

    // Refill the collection and do the second round of basic tests
    for (c4Index = 0; c4Index < __c4MaxElems; c4Index++)
        colTest.Add(TString(L"Testing") + TString(TCardinal(c4Index)));

    TestColBasics2(strmOut, colTest, __c4MaxElems);
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestHashSet
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various
//  aspects of the hash set class.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestHashSet()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        c4CurCount = facCIDLib.c4ObjectCount();

        pszCurTest = L"common";
        __TestSetCommon(strmOut());

        pszCurTest = L"basic";
        __TestSetBasics(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Hash set test leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" hash set test" << NewLn;
        throw;
    }
}
