//
// NAME: TestCIDLib_HashMap.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  hash map collection class.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/22/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_CommonCollect.Hpp"



// -----------------------------------------------------------------------------
//  Typedef our test map. We use a TFindBuf object since its a very obvious
//  type of object for this kind of collection. The file name will be the
//  key field.
// -----------------------------------------------------------------------------
typedef THashMap<TFindBuf,TPathStr>         THashMapOfTFindBuf;


// -----------------------------------------------------------------------------
//  Force full instantiations of at least one specialization of each class so
//  that we can catch any errors
// -----------------------------------------------------------------------------
template class THashMap<TFindBuf, TPathStr>;


// -----------------------------------------------------------------------------
//  In order to avoid warnings about unimplemented methods, provide dummy
//  implementations here.
// -----------------------------------------------------------------------------
THashMap<TFindBuf, TPathStr>::THashMap() {}

tCIDLib::TVoid THashMap<TFindBuf, TPathStr>::operator=(const THashMap<TFindBuf, TPathStr>&) {}



// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

static const TPathStr& __pathGetKey(const TFindBuf& fndbData)
{
    return fndbData.pathFileName();
}

static tCIDLib::TVoid __TestMapCommon(TTextStream& strmOut)
{
    static const tCIDLib::TCard4 c4MaxElems = 64;

    // Create a hash of THashData objects
    THashMapOfTFindBuf colTest
    (
        17
        , new TStringKeyOps<TPathStr>
        , __pathGetKey
        , c4MaxElems
    );

    // Check the automatic class name generation
    if (!colTest.bIsA(TClass(L"THashMapOfTFindBuf")))
    {
        strmOut << _CurLn_ << L"Hashmap class name was bad. It was: "
                << colTest.clsIsA() << NewLn;
    }

    // Check any params that common testing does not do
    if (colTest.c4HashModulus() != 17)
        strmOut << _CurLn_ << L"Modulus set in constructor not found" << NewLn;

    if (colTest.c4MaxElemCount() != c4MaxElems)
        strmOut << _CurLn_ << L"Max elems set in constructor not found" << NewLn;

    //
    //  Load up a listing of the current directory. We can do this with just
    //  a call to the file system class.
    //
    tCIDLib::TCard4 c4Elems = TFileSys::c4SearchDir
    (
        kCIDLib::pszAllFilesSpec
        , colTest
    );

    if (!c4Elems)
    {
        strmOut << L"Did not find any files for test" << NewLn;
        return;
    }

    //
    //  Do the copy and duplication tests. We provide a standard object
    //  equality object for the element comparison.
    //
    TestColCopy(strmOut, colTest, TStdObjEq<TFindBuf>());

    //
    //  Invoke the basic collection testing template function. It will do
    //  some generic things that all collections should do the same.
    //
    TestColBasics(strmOut, colTest, c4Elems);

    // Load the collection back up
    c4Elems = TFileSys::c4SearchDir(kCIDLib::pszAllFilesSpec, colTest);

    if (!c4Elems)
    {
        strmOut << L"Did not find any files for test" << NewLn;
        return;
    }

    // And do the second round of basic testing
    TestColBasics(strmOut, colTest, c4Elems);
}


static tCIDLib::TVoid __TestMapBasics(TTextStream& strmOut)
{
    static const tCIDLib::TCard4 c4MaxElems = 64;

    // Create a map of our test type.
    THashMapOfTFindBuf colTest
    (
        17
        , new TStringKeyOps<TPathStr>
        , __pathGetKey
        , c4MaxElems
    );

    // Check any params that common testing does not do
    if (colTest.c4HashModulus() != 17)
        strmOut << L"Modulus set in constructor not found" << NewLn;

    if (colTest.c4MaxElemCount() != c4MaxElems)
        strmOut << _CurLn_ << L"Max elems set in constructor not found" << NewLn;

    //
    //  Load up a listing of the current directory. We can do this with just
    //  a call to the file system class.
    //
    tCIDLib::TCard4 c4Elems = TFileSys::c4SearchDir
    (
        kCIDLib::pszAllFilesSpec
        , colTest
    );

    if (!c4Elems)
    {
        strmOut << _CurLn_ << L"Did not find any files for test" << NewLn;
        return;
    }

    //
    //  Check the value of each element by key lookup. We do a separate
    //  iteration of the directory via a directory iterator to get the
    //  elements.
    //
    TDirIter diterTest;
    TFindBuf fndbCur;
    if (!diterTest.bFindFirst(kCIDLib::pszAllFilesSpec, fndbCur))
    {
        strmOut << _CurLn_ << L"Did not find any files for test" << NewLn;
        return;
    }

    // Save this first one for later
    TFindBuf fndbFirst(fndbCur);

    do
    {
        if (colTest.objFindByKey(fndbCur.pathFileName()) != fndbCur)
            strmOut << _CurLn_ << L"The value looked up by key was wrong" << NewLn;
    }   while (diterTest.bFindNext(fndbCur));

    // Remove one of the elements that we know was found
    colTest.RemoveKey(fndbFirst.pathFileName());

    // Make sure the element count was adjusted correctly
    if (colTest.c4ElemCount() != c4Elems-1)
        strmOut << _CurLn_ << L"Removing key did not adjust element count" << NewLn;

    // Make sure that we cannot add a key that is already in the map.
    tCIDLib::TBoolean bGotIt = kCIDLib::False;
    try
    {
        colTest.Add(fndbCur);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcCol_DuplicateKey))
            strmOut << _CurLn_ << L"Expected duplicate key error" << NewLn;
        bGotIt = kCIDLib::True;
    }

    if (!bGotIt)
    {
        strmOut << _CurLn_ << L"Failed to catch duplicate key insert" << NewLn;
    }

    // But we should be able to add back in the one we removed
    colTest.Add(fndbFirst);

    // Flush the map and check the element count
    colTest.Flush();
    if (colTest.c4ElemCount() != 0)
        strmOut << _CurLn_ << L"Flush did not set element count to 0" << NewLn;
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestHashMap
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various
//  aspects of the hash map class.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestHashMap()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        c4CurCount = facCIDLib.c4ObjectCount();

        pszCurTest = L"common";
        __TestMapCommon(strmOut());

        pszCurTest = L"basic";
        __TestMapBasics(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Basic hash map test leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" hash map test" << NewLn;
        throw;
    }
}
